/*
  lxinetd.c -- INETD server for the HP200LX palmtop.

  Copyright (C) 1999  Rod Whitby

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version
  2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be 
  useful, but WITHOUT ANY WARRANTY; without even the implied
  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the GNU General Public License for more
  details.

  You should have received a copy of the GNU General Public
  License along with this program; if not, write to the Free
  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
  USA.

  $Source: A:/SRC/TCP/INETD/RCS/LXINETD.C $
  $Id: LXINETD.C 1.15 2000/04/09 07:15:52 rwhitby Exp $
*/

#include <conio.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "lxinetd.h"

byte buffer[2048];

int default_conn(tcp_Socket *socket, void **datap) {
  (void)socket; (void)datap;
  return 1;
}

int default_tick(tcp_Socket *socket, void **datap) {
  (void)socket; (void)datap;
  return 1;
}

#include "simple.h"
#include "http.h"

static inetd_daemon daemons[] = {
  { "default", 0, NULL, default_conn, default_tick, NULL,     1, 5 },
  { "echo",    7, NULL, NULL, echo_tick,    NULL,             1, 5 },
  { "discard", 9, NULL, NULL, discard_tick, NULL,             1, 5 },
  { "daytime",13, NULL, daytime_conn, daytime_tick, NULL,     1, 5 },
  { "chargen",19, NULL, chargen_conn, chargen_tick, NULL,     1, 5 },
  { "time",   37, NULL, time_conn, time_tick, NULL,           1, 5 },
  { "http",   80, http_boot, http_conn, http_tick, http_quit, 5, 50 },
  { 0 }
};

void usage(char *name)
{
  fprintf(stderr,
          "Usage: %s [<directive>=<value>] ...\n", name);
  exit(1);
}

void (*old_init)(char *directive, char *value);
void new_init   (char *directive, char *value)
{
  if (!strnicmp(directive, "lxinetd.", 8)) {
    strcpy(directive, directive+8);
  }
  /*fprintf(stderr, "Setting %s to %s.\n", directive, value);*/
  if (!stricmp(directive, "debug")) {
    if (atoi(value)) {
      dbug_init();
    }
    tcp_set_debug_state(atoi(value));
  }
  else {
    if (old_init) (*old_init)(directive, value);
  }
}

int inetd_listener(inetd_daemon *daemon, int slot) {
  inetd_slot **slots;
  inetd_slot *s;
  tcp_Socket *t;
  int i;

  slots = daemon->slots;

  /* If we weren't passed a slot, then we have to find an idle slot. */
  if (slot < 0) {

    /* First, look for an idle slot. */
    for (i = 0; i < daemon->init_slots; i++) {
      if (slots[i]->status == S_INIT) {
        /* Record the slot that is going to be the new listener. */
        slot = i;
        break;
      }
    }

    /* If that doesn't succeed, then try to create a new slot. */
    if (slot < 0) {

      /* Check to see if we can create a new slot. */
      if (daemon->init_slots < daemon->max_slots) {

        /* Attempt to allocate room for the new slot. */
        s = (inetd_slot *)malloc(sizeof(inetd_slot));

        if (s == NULL) {
          fprintf(stderr, "%s: no listening on port %d (out of memory)\n",
                  daemon->name, daemon->port);
          return 0;
        }

        /* Record the slot that is going to be the new listener. */
        slot = daemon->init_slots;

        /* Store the new slot at the end of the slots array. */
        slots[daemon->init_slots++] = s;
      }

      /* No more slots allowed - notify the user. */
      else {
        fprintf(stderr, "%s: no listening on port %d (connection limit)\n",
                daemon->name, daemon->port);
        return 0;
      }
    }
  }
        
  /* Grab the socket associated with the idle slot. */
  s = slots[slot]; t = &(s->sockdata);

  /* Not necessary, but doesn't hurt. */
  sock_abort(t);

  /* Start listening on the socket on behalf of this daemon. */
  tcp_listen(t, daemon->port, 0L, 0, NULL, 0);

  /* The daemon is now ready for incoming connections. */
  fprintf(stderr, "%s(%d): listening on port %d\n",
          daemon->name, slot, daemon->port);

  daemon->wait_slots++;
  s->status = S_WAIT;

  return 1;
}

void main(int argc, char *argv[])
{
  int i, j, max_daemons;
  char *directive, *value;
  inetd_daemon *daemon;
  inetd_slot **slots;
  inetd_slot *s;
  tcp_Socket *t;
  sockaddr remote;
  char remote_ip[17];

  fprintf(stderr, "LXINETD %d.%d", VERSION/10, VERSION%10);

  if (RELEASE > 0) {
    fprintf(stderr, "b%d", RELEASE);
  }

  fprintf(stderr, " - %s <%s>\n",
          "HP200LX TCP/IP Suite",
          "http://lxtcp.hplx.net/");

  old_init = usr_init;
  usr_init = new_init;

  if (getenv("LXTCP.CFG")) {
    tcp_config_file(getenv("LXTCP.CFG"));
  }

  sock_init();

  for (i = 1; i < argc; i++) {
    directive = argv[i];
    if ((value = strchr(directive, '=')) != 0) {
      *value = 0; value++; strupr(directive);
      usr_init(directive, value);
    }
    else {
      usage(argv[0]);
    }
  }

  /* Determine how many daemons are installed, and initialise them. */
  for (i = 0; ; i++) {

    /* Work on one daemon entry at a time. */
    daemon = &daemons[i];

    /* NULL indicates the end of the installed daemons. */
    if (daemon->name) {

      /* Initialise the daemon if required. */
      if ((daemon->boot != NULL) && !(*daemon->boot)()) {
        fprintf(stderr, "%s: no listening on port %d (boot failed)\n",
                daemon->name, daemon->port);
        continue;
      }

      /* Allocate room for an array of slots. */
      slots = (inetd_slot **)calloc(sizeof(inetd_slot *), daemon->max_slots);

      /* A memory error at this stage is fatal for this daemon. */
      if (slots == NULL) {
        fprintf(stderr, "%s: no listening on port %d (out of memory)\n",
                daemon->name, daemon->port);
        continue;
      }

      /* The daemon now has a slot array ready for initialisation. */
      daemon->slots = slots;

      /* Initialise the slot array. */
      for (j = 0; j < daemon->max_slots; j++) {
        slots[j] = NULL;
      }

      daemon->init_slots = 0;
      daemon->wait_slots = 0;

      for (j = 0; j < daemon->min_slots; j++) {

        /* Allocate room for each initial slot. */
        s = (inetd_slot *)malloc(sizeof(inetd_slot));

        if (s == NULL) {
          fprintf(stderr, "%s: no listening on port %d (out of memory)\n",
                  daemon->name, daemon->port);
          break;
        }

        slots[j] = s;
        daemon->init_slots++;
        s->status = S_INIT;
      }
    }
    else {

      /* We've run out of daemons, so store the number found. */
      max_daemons = i++;

      break;
    }
  }

  /* Now that we have initialised all the daemons, we enter an endless
     loop of servicing requests on the ports associated with the daemons. */

  while (!(kbhit() && (getch() == 27))) {

    /* Keep the network ticking over. */
    tcp_tick(NULL);

    /* Allow each daemon to operate. */
    for (i = 0; i < max_daemons; i++) {

      /* Work on one daemon entry at a time. */
      daemon = &daemons[i];
      slots = daemon->slots;

      /* Process each slot of active daemons. */
      for (j = 0; j < daemon->init_slots; j++) {

        s = slots[j]; t = &(s->sockdata);

        switch (s->status) {

          case S_INIT :

            /* Listen on this slot. */
            inetd_listener(daemon, j);

            break;

          case S_WAIT :

            if (sock_established(t)) {

              /* Get the remote host's details. */
              getpeername(t, &remote, NULL);
              inet_ntoa(remote_ip, remote.s_ip);

              fprintf(stderr, "%s(%d): connection requested from %s\n",
                      daemon->name, j, remote_ip);

              /* Change the state of this slot to connected. */
              daemon->wait_slots--;
              s->status = S_CONN;

              /* We've lost a listener, so replace it with a new one. */
              if (daemon->wait_slots < daemon->min_slots) {
                inetd_listener(daemon, -1);
              }

              /* Call the daemon's conn routine. */
              if ((daemon->conn != NULL) &&
                  !(*daemon->conn)(t, &(s->userdata))) {

                /* Connection refused by the daemon. */
                fprintf(stderr, "%s(%d): refused connection (conn failed)\n",
                        daemon->name, j);

                /* Close the socket gracefully. */
                sock_close(t);
                s->status = S_CLOS;

                /* Call the daemon's quit routine. */
                if ((daemon->quit != NULL) &&
                    !(*daemon->quit)(t, &(s->userdata))) {
                  fprintf(stderr, "%s(%d): error during close (quit failed)\n",
                          daemon->name, j);
                }

                continue;
              }

              fprintf(stderr, "%s(%d): accepted connection\n",
                      daemon->name, j);
            }

            /* Handle the case where a connection is never made. */
            if (!tcp_tick(t)) {

              fprintf(stderr, "%s(%d): connection closed\n",
                      daemon->name, j);

              /* The socket has been closed by the remote host. */
              daemon->wait_slots--;
              s->status = S_CLOS;

              /* We've lost a listener, so replace it with a new one. */
              if (daemon->wait_slots < daemon->min_slots) {
                inetd_listener(daemon, -1);
              }

              /* Call the daemon's quit routine. */
              if ((daemon->quit != NULL) &&
                  !(*daemon->quit)(t, &(s->userdata))) {
                fprintf(stderr, "%s(%d): error during close (quit failed)\n",
                        daemon->name, j);
              }
            }

            break;

          case S_CONN :

            /* Check to see if the connection is still alive. */
            if (!tcp_tick(t)) {

              fprintf(stderr, "%s(%d): connection closed\n",
                      daemon->name, j);

              /* Remote host closed the connection. */
              s->status = S_CLOS;

              /* Call the daemon's quit routine. */
              if ((daemon->quit != NULL) &&
                  !(*daemon->quit)(t, &(s->userdata))) {
                fprintf(stderr, "%s(%d): error during close (quit failed)\n",
                        daemon->name, j);
              }

              continue;
            }

            /* Call the daemon's tick routine. */
            if ((daemon->tick != NULL) &&
                !(*daemon->tick)(t, &(s->userdata))) {

              fprintf(stderr, "%s(%d): closed connection\n",
                      daemon->name, j);

              /* Close the socket gracefully. */
              sock_close(t);
              s->status = S_CLOS;

              /* Call the daemon's quit routine. */
              if ((daemon->quit != NULL) &&
                  !(*daemon->quit)(t, &(s->userdata))) {
                fprintf(stderr, "%s(%d): error during close (quit failed)\n",
                        daemon->name, j);
              }
            }
            break;

          case S_CLOS :

            /* Wait until the remote host acknowledges the close. */
            if (!tcp_tick(t)) {
              s->status = S_INIT;
            }

            break;
        }
      }
    }
  }

  for (i = 0; i < max_daemons; i++) {  

    /* Work on one daemon entry at a time. */
    daemon = &daemons[i];
    slots = daemon->slots;

    for (j = 0; j < daemon->init_slots; j++) {
      s = slots[j]; t = &(s->sockdata);

      switch (s->status) {

        case S_WAIT :
        case S_CONN :

          fprintf(stderr, "%s(%d): closed connection\n",
                  daemon->name, j);

          /* Forcibly close the socket. */
          sock_abort(t);
          s->status = S_CLOS;

          /* Call the daemon's quit routine. */
          if ((daemon->quit != NULL) &&
              !(*daemon->quit)(t, &(s->userdata))) {
            fprintf(stderr, "%s(%d): error during close (quit failed)\n",
                    daemon->name, j);
          }
          break;

        case S_CLOS :

          /* Forcibly close the socket. */
          sock_abort(t);
          break;
      }

      free(s); slots[j] = NULL;
    }

    free(slots); slots = NULL;

    daemon->init_slots = 0;
    daemon->wait_slots = 0;
  }
}

/* End of lxinetd.c */
