/*
  socklib.c -- Waterloo TCP socket libraries.

  Copyright (C) 1997-1999  Rod Whitby
  Copyright (C) 1992       Walter Andrew Nolan

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version
  2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be 
  useful, but WITHOUT ANY WARRANTY; without even the implied
  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the GNU General Public License for more
  details.

  You should have received a copy of the GNU General Public
  License along with this program; if not, write to the Free
  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
  USA.

  $Source: A:/SRC/TCP/MTA/RCS/SOCKLIB.C $
  $Id: SOCKLIB.C 1.13 2000/04/09 07:02:50 rwhitby Exp $
*/

#include <conio.h>
#include <ctype.h>
#include <mem.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "socklib.h"

/* Global Variables */

int WatTCP_initialized = 0;

char Sock_buffer[SOCK_BUFFER_LEN];
char Sock_last[SOCK_BUFFER_LEN];

int Sock_echo = 0;
int Sock_debug = 0;

/* General Stuff */

char *trim(char *str)
{
  int i;

  if (str == NULL) return(str);

  for (i = strlen(str) - 1; (i >= 0) && isspace(str[i]);
       str[i--] = '\0');

  while (isspace(str[0])) strcpy(str, str + 1);

  return str;
}

char *getaddress(char *str)
{
  char *p, *sp;

  /* This does *very* rudimentary parsing. */

  if ((p = strchr(str, '<')) != NULL) {
    /* Rod Whitby <rwhitby@hplx.net> */
    sp = ++p;
    if (((p = strpbrk(sp, " \"()<>")) != NULL) &&
        (*p == '>')) {
      *p = 0;
      return sp;
    } else { return str; }
  }
  else if ((p = strpbrk(str, " \t\"(")) != NULL) {
    /* rwhitby@hplx.net (Rod Whitby) */
    *p = 0;
    return str;
  }
  else {
    /* rwhitby@hplx.net */
    return str;
  }
}

char *strrep(char *str, char old, char New)
{
  int i;
  for (i = 0; str[i]; i++) {
    if (str[i] == old) {
      str[i] = New;
    }
  }
  return str;
}

int getnumbers(char *ascii, ...)
{
  char *p;
  int count;
  long *dp;
  va_list ap;

  va_start(ap, ascii);
  count = 0;

  /* it must return a number after the white space */
  p = ascii;

  while ((dp = va_arg(ap, long *)) != 0) {
    if ((p = strchr(p, ' ')) == NULL) break;

    /* skip space */
    while (*p == ' ') p++;

    *dp = atol(p);
    count++;
  }

  va_end(ap);

  return(count);
}

char * getnpass (char * prompt, char * buffer, int length)
{
  char inchar;
  int i = 0;

  while (*prompt) {
    putchar(*prompt++);
  }

  while ((i < length-1) && ((inchar = getch()) != '\r')) {
    *(buffer+i) = inchar;
    i++;
    putchar('*');
  }

  *(buffer+i) = 0;
  putchar('\n');

  return buffer;
}

void socklib_init(void)
{
  if (!WatTCP_initialized) {
    sock_init();
    WatTCP_initialized=1;
  }
}

void socklib_debug(int debug)
{
  Sock_debug = debug;
  if (debug) dbug_init();
  tcp_set_debug_state(debug);
}

int socklib_url(char *url, char **protocol,
                char **host, int *port,
                char **user, char **pass,
                char **path)
{
  char *p;

  /* First, determine which protocol should be used. */

  if ((p = strstr(url, "://")) != NULL) {
    *protocol = url;
    *p = 0;
    if (!strcmp(url, "ftp")) {
      *port = FTP_PORT;
      url = p + 3;
    }
    else if (!strcmp(url, "telnet")) {
      *port = TELNET_PORT;
      url = p + 3;
    }
    else if (!strcmp(url, "http")) {
      *port = HTTP_PORT;
      url = p + 3;
    }
    else if (!strcmp(url, "pop3")) {
      *port = POP_PORT;
      url = p + 3;
    }
    else if (!strcmp(url, "nntp")) {
      *port = NNTP_PORT;
      url = p + 3;
    }
    else if (!strcmp(url, "tngate")) {
      char *q;
      if ((q = strchr(p + 3, '/')) != NULL) {
        *p = ':';
        *q = 0;
        url = q + 1;
      }
      else {
        url = p + 3;
      }
      *port = TELNET_PORT;
    }
    else {
      return 0;
    }
  }
  else {
    return 0;
  }
  
  /* Separate the user information and host information. */
  if ((p = strchr(url, '@')) == NULL) {
    *host = url;
  }
  else {
    *p = 0; *host = ++p;
    /* Split the user information into username and password. */
    if ((p = strchr(url, ':')) != NULL) {
      *p = 0; *pass = ++p;
    }
    *user = url;
  }
  
  /* Grab any path off the end of the URL */
  if ((p = strchr(*host, '/')) != NULL) {
    *p = 0; *path = ++p;
  }

  /* Split the host information into hostname and port. */
  if ((p = strchr(*host, ':')) != NULL) {
    *p = 0; *port = atoi(++p);
  }
  
  return 1;
}

int socklib_wait_for(char *service, tcp_Socket *socket, int *status, char *match)
{
  byte *b = (byte *)Sock_buffer;
  byte *p, *s;
  char *q; int i;

  sock_mode( socket, TCP_MODE_BINARY );
  do {
    sock_wait_input(socket, sock_delay, NULL, status);
    i = sock_fastread(socket, b, sizeof(Sock_buffer) -
                      (size_t)(b - (byte *)Sock_buffer) - 1);
    *(b + i) = 0;
    while (((s = (byte *)memchr(b, 0x00, i)) != NULL) &&
           (s > b) && (*(s-1) == 0x0d)) {
      *s = 0x0a;
    }
    while ((s = (byte *)strchr(Sock_buffer, 0xff)) != NULL) {
      p = s; s++;
      if (*s == 0xfd) {
        *s = 0xfc; sock_fastwrite(socket, p, 3);
        s += 2;
      }
      else if (*s == 0xfb) {
        *s = 0xfe; sock_fastwrite(socket, p, 3);
        s += 2;
      }
      else if (*s == 0xfc) {
        s += 2;
      }
      else if (*s == 0xfe) {
        s += 2;
      }
      else {
        s++;
      }
      strcpy((char *)p, (char *)s);
    }
    if ((q = strstr(Sock_buffer, match)) != NULL) {
      strcpy(Sock_buffer, q);
    }
    else if ((q = strrchr(Sock_buffer, '\n')) != NULL) {
      strcpy(Sock_buffer, ++q);
    }
    db_write(service);
    db_write(">>> ");
    db_write((char *)Sock_buffer);
    db_write("\n\n");
    b = (byte *)Sock_buffer + strlen(Sock_buffer);
  } while (!strstr(Sock_buffer, match));
  sock_mode( socket, TCP_MODE_ASCII );
  return(1);
sock_err:
  return(0);
}

int socklib_gets(char *service, tcp_Socket *socket, int *status)
{
  sock_wait_input(socket, sock_delay, NULL, status);
  sock_gets(socket, (byte *)Sock_buffer, sizeof(Sock_buffer));
  db_write(service);
  db_write(">>> ");
  db_write(Sock_buffer);
  db_write("\n\n");
  return(1);
sock_err:
  return(0);
}

void socklib_puts(char *service, tcp_Socket *socket)
{
  db_write(service);
  db_write("<<< ");
  db_write(Sock_buffer);
  db_write("\n\n");
  sock_puts(socket, (byte *)Sock_buffer);
  return;
}

int socklib_connect(char *service, tcp_Socket *socket, int *status, char *proxy, char *host, int port)
{
  longword h;
  char *p;

  char *proxy_proto = NULL;
  char *proxy_host = NULL;
  int   proxy_port = 0;
  char *proxy_user = NULL;
  char *proxy_pass = NULL;
  char *proxy_path = NULL;

  char *tngate_proto = NULL;
  char *tngate_host = NULL;
  int   tngate_port = 0;
  char *tngate_user = NULL;
  char *tngate_pass = NULL;
  char *tngate_path = NULL;

  char *remote_host = host;
  int   remote_port = port;

  char pass[MAX_PASSWORD];

  if (proxy) {
    if (!socklib_url(proxy, &proxy_proto, &proxy_host, &proxy_port,
                     &proxy_user, &proxy_pass, &proxy_path)) {
      fprintf(stderr, "%s> Cannot parse proxy url: %s\n", service, proxy);
      proxy_host = NULL;
      return 0;
    }

    if (proxy_port != TELNET_PORT) {
      fprintf(stderr, "%s> Only telnet proxies are supported\n", service);
      proxy_host = NULL;
      return 0;
    }

    if (!proxy_pass) {
      sprintf(Sock_buffer, "Password for %s@%s:", proxy_user, proxy_host);
      (void)getnpass(Sock_buffer, pass, MAX_PASSWORD);
      proxy_pass = strdup(pass);
    }
  }

  if (proxy_host) {
    host = proxy_host;
    port = proxy_port;
    if (!strncmp(proxy_proto, "tngate", 6)) {
      if (!socklib_url(proxy_proto, &tngate_proto, &tngate_host, &tngate_port,
                       &tngate_user, &tngate_pass, &tngate_path)) {
        fprintf(stderr, "%s> Cannot parse proxy url: %s\n", service, proxy_proto);
      }
      host = tngate_host;
      port = tngate_port;
    }
    fprintf(stderr, "Connecting to proxy (port %d at %s)\n", port, host);
  }

  if (!(h = resolve(host))) {
    fprintf(stderr, "%s> Cannot resolve host %s\n", service, host);
    return(0);
  }

  if (!tcp_open( socket, 0, h, port, NULL )) {
    fprintf(stderr, "%s> Unable to connect to %s\n", service, host);
    return(0);
  }

  sock_wait_established( socket, sock_delay, NULL, status);

  sock_mode( socket, TCP_MODE_ASCII );

  if (proxy) {

    if (!strcmp(proxy_proto, "tngate")) {

      if (!socklib_wait_for(service, socket, status, "NETID Username: ")) goto sock_err;
      fprintf(stderr, "%s%s\n", Sock_buffer, tngate_user);
      sprintf(Sock_buffer, "%s", tngate_user);
      socklib_puts(service, socket);
      if (!socklib_gets(service, socket, status)) goto sock_err;

      if (!socklib_wait_for(service, socket, status, "NETID Password: ")) goto sock_err;
      fprintf(stderr, "%sTurtle Power!\n", Sock_buffer);
      sprintf(Sock_buffer, "%s", tngate_pass);
      socklib_puts(service, socket);
      if (!socklib_gets(service, socket, status)) goto sock_err;

      if ((p = strchr(tngate_host, '.')) != NULL) *p = 0;
      if (!socklib_wait_for(service, socket, status, tngate_host)) goto sock_err;
      fprintf(stderr, "%s", Sock_buffer);

      sprintf(Sock_buffer, "telnet %s %d\n", proxy_host, proxy_port);
      fprintf(stderr, "%s", Sock_buffer);
      socklib_puts(service, socket);
    }

    if (!socklib_wait_for(service, socket, status, "login:")) goto sock_err;
    fprintf(stderr, "%s", Sock_buffer);

    sprintf(Sock_buffer, "%s", proxy_user);
    fprintf(stderr, "%s\n", Sock_buffer);
    socklib_puts(service, socket);

    if (!socklib_wait_for(service, socket, status, "Password:")) goto sock_err;
    fprintf(stderr, "%sTurtle Power!\n", Sock_buffer);
    sprintf(Sock_buffer, "%s", proxy_pass);
    socklib_puts(service, socket);

    sleep(1);

    sprintf(Sock_buffer, "telnet %s %d", remote_host, remote_port);
    fprintf(stderr, "%s@%s>%s\n", proxy_user, proxy_host, Sock_buffer);
    socklib_puts(service, socket);

    if (!socklib_wait_for(service, socket, status, "Escape character is ")) goto sock_err;
  }

  return(1);

sock_err:
  return(0);
}

/* End of socklib.c */
