/*************************************************************************
** interpcom-1.1 (command interpreter - tutorial)                        **
** fourier.c : Fourier transforms  	                                 **
**                                                                       **
** Copyright (C) 1998  Jean-Marc Drezet                                  **
**                                                                       **
**  This library is free software; you can redistribute it and/or        **
**  modify it under the terms of the GNU Library General Public          **
**  License as published by the Free Software Foundation; either         **
**  version 2 of the License, or (at your option) any later version.     **
**									 **
**  This library is distributed in the hope that it will be useful,      **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of       **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    **
**  Library General Public License for more details. 			 **
**									 **
**  You should have received a copy of the GNU Library General Public    **
**  License along with this library; if not, write to the Free		 **
**  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.   **
**                                                                       **
** Please mail any bug reports/fixes/enhancements to me at:              **
**      drezet@math.jussieu.fr                                           **
** or                                                                    **
**      Jean-Marc Drezet                                                 **
**      Institut de Mathematiques                                        **
**      Aile 45-55                                                       **
**      2, place Jussieu                                                 **
**      75251 Paris Cedex 05                                             **
**      France								 **
**                                                                       **
 *************************************************************************/


#include "interp.h"
#include "funct.h"



/*---------------------------------------------------------------------------
    Computation of the Tfourier structure associated to a real function in
    double precision.
    The function is defined by the arrays x_r[] (the x) and F[]
    (the f(x)). 
    NB[0] is the number of points of this function (the indices run from 1
    to NB[0]).
    ifw is the number of points for the FFTs that will be used. It must be
    a power of 2.
    iprec determines the precision of the computations (from 0 to 4). 3
    is a good value.
---------------------------------------------------------------------------*/
Tfourier       *
Tf(int *NB, double *x_r, double *F, int ifw, int iprec)
{
    double          pi,
                    pid,
                    xs,
                    xs2,
                    xs3,
                    x;
    double          bxcof[20];
    int             iser,
                    j,
                    ij;
    int             ik,
                    i,
                    _expos;
    double         *onx,
                   *on,
                   *om,
		   *pz;
    Tfourier       *Fo;

    Fo = (Tfourier *) malloc(sizeof(Tfourier));
    if (iprec < 0)
	iprec = 0;
    if (iprec > 4)
	iprec = 4;
    Fo->isermax = 2 * iprec + 7;
    Fo->ifft = ifw;
    Fo->npoints_2 = ifw / 2;
    ik = 3 + Fo->ifft / 2;
    j = Fo->isermax + 1;

    Fo->a = double_alloc2(ik, j);
    Fo->b = double_alloc2(ik, j);

    pi = 3.141592653589793;
    Fo->bmin = x_r[1];


/*
 * Approximation de cos(PI*x), 0<=x<1 
 */

    switch (iprec) {
    case 0:
	bxcof[0] = 0.9985257976844458L;
	bxcof[2] = 4.8874541171280601L;
	bxcof[4] = 3.8172707288719550L;
	bxcof[6] = 0.9296558086560129L;
	break;
       /*
	* 0.0014742023155540  
	*/

    case 1:
	bxcof[0] = 0.9999582316201327L;
	bxcof[2] = 4.9327104910735358L;
	bxcof[4] = 4.0418400661565741L;
	bxcof[6] = 1.2871567633369711L;
	bxcof[8] = 0.1781076114527940L;
	break;
       /*
	* 0.0000417683798670  
	*/

    case 2:
	bxcof[0] = 0.9999991998413435L;
	bxcof[2] = 4.9347445439653335L;
	bxcof[4] = 4.0580367399959316L;
	bxcof[6] = 1.3323569316706978L;
	bxcof[8] = 0.2296364873558310L;
	bxcof[10] = 0.0205717084058903L;
	break;
       /*
	* 0.0000008001586550  
	*/

    case 3:
	bxcof[0] = 0.9999999889445757L;
	bxcof[2] = 4.9348011166440138L;
	bxcof[4] = 4.0586947455214943L;
	bxcof[6] = 1.3351580223042279L;
	bxcof[8] = 0.2350290226239222L;
	bxcof[10] = 0.0253582857538241L;
	bxcof[12] = 0.0015936782134291L;
	break;
       /*
	* 0.0000000110554235  
	*/

    case 4:
	bxcof[0] = 0.9999999999999938L;
	bxcof[2] = 4.9348022005434435L;
	bxcof[4] = 4.0587121263759709L;
	bxcof[6] = 1.3352627683318374L;
	bxcof[8] = 0.2353306269553867L;
	bxcof[10] = 0.0258068786519750L;
	bxcof[12] = 0.0019295452505552L;
        bxcof[14] = 0.0001045969891168L;
        bxcof[16] = 0.0000042677655230L;
        bxcof[18] = 0.0000001218310627L;
	break;
       /*
	* 0.0000000000000062
	*/
    }

/*
 * Approximation de sin(PI*x), 0<=x<=1  
 */

    switch (iprec) {
    case 0:
	bxcof[1] = pi * 0.9992497106511437L;
	bxcof[3] = pi * 1.6348586794245885L;
	bxcof[5] = pi * 0.7747701153220409L;
	bxcof[7] = pi * 0.1392370618055386L;
	break;
       /*
 	* 0.0002611506680863    
 	*/

    case 1:
	bxcof[1] = pi * 0.9999788400893682L;
	bxcof[3] = pi * 1.6445094676981539L;
	bxcof[5] = pi * 0.8093444287199553L;
	bxcof[7] = pi * 0.1851766230170661L;
	bxcof[9] = pi * 0.0203646241201746L;
	break;
       /*
 	* 0.0000060351688488  
 	*/

    case 2:
	bxcof[1] = pi * 0.9999995958788578L;
	bxcof[3] = pi * 1.6449227317817565L;
	bxcof[5] = pi * 0.8116513505303372L;
	bxcof[7] = pi * 0.1904370314970014L;
	bxcof[9] = pi * 0.0256152983002450L;
	bxcof[11] = pi * 0.0019065110581533L;
	break;
       /*
 	* 0.0000000975912972  
 	*/

    case 3:
	bxcof[1] = pi * 0.9999999944282595L;
	bxcof[3] = pi * 1.6449338586286357L;
	bxcof[5] = pi * 0.8117401709882369L;
	bxcof[7] = pi * 0.1907410423089234L;
	bxcof[9] = pi * 0.0261212960574211L;
	bxcof[11] = pi * 0.0023108605007544L;
	bxcof[13] = pi * 0.0001243003223813L;
	break;
       /*
 	* 0.0000000011664671  
 	*/

    case 4:
	bxcof[1] = pi * 0.9999999999999968L;
	bxcof[3] = pi * 1.6449340668479986L;
	bxcof[5] = pi * 0.8117424252784339L;
	bxcof[7] = pi * 0.1907518240733058L;
	bxcof[9] = pi * 0.0261478475516592L;
	bxcof[11] = pi * 0.0023460801628395L;
	bxcof[13] = pi * 0.0001484269966023L;
        bxcof[15] = pi * 0.0000069735337047L;
        bxcof[17] = pi * 0.0000002512455112L;
        bxcof[19] = pi * 0.0000000064543548L;
	break;	
       /*
	* 0.0000000000000005
	*/
    }

    ifw = Fo->ifft;
    ik = Fo->ifft + 3;
    _expos = (int) floor(log((double) Fo->npoints_2) / log(2.) + .001);

    onx = double_alloc1(ik);
    on = double_alloc1(ik);
    pz = double_alloc1(ik);
    om = double_alloc1(ik);

    Fo->d = (x_r[NB[0]] - x_r[1]) / Fo->ifft;
    on[1] = F[1];
    on[Fo->ifft + 1] = F[NB[0]];
    Fo->ondeb = on[1];
    Fo->onfin = on[Fo->ifft + 1];
    x = x_r[1];
    ij = 1;

    for (i = 2; i <= Fo->ifft; i++) {
	x += Fo->d;

	for (j = ij; j <= NB[0]; j++) {
	    if (x_r[j] > x)
		break;
	}

	ij = j;
	j = ij - 1;
	on[i] = ((x_r[ij] - x) * F[j] + (x - x_r[j]) * F[ij]) /
 	    (x_r[ij] - x_r[j]);
    }

    pid = 2. / (double) Fo->ifft;
    pz[1] = 0.0;
    for (i = 2; i <= Fo->ifft; i++)
	pz[i] = pid + pz[i - 1];

    for (iser = 0; iser <= Fo->isermax; iser++) {
	xs = 0.;
        j = 0;
        
        for (i = 0; i < Fo->ifft; i++) {
	    onx[i] = on[i + 1];
	    xs += onx[i];
	}

	xs2 = 0.;
	xs3 = 0.;

	for (i = 1; i < Fo->ifft; i += 2) {
	    xs3 = on[i] - on[i + 1];
	    xs2 += xs3;
	}

        realfft(Fo->ifft, onx);

        for (i = 1; i <= Fo->npoints_2; i++) {
            Fo->a[i][iser] = onx[i - 1] * bxcof[iser];
	    Fo->b[i][iser] = onx[Fo->ifft - i + 1] * bxcof[iser];
        }

	Fo->a[1][iser] = xs * bxcof[iser];
	Fo->b[1][iser] = 0.;
	Fo->a[Fo->npoints_2 + 1][iser] = xs2 * bxcof[iser];
	Fo->b[Fo->npoints_2 + 1][iser] = 0.;

	if (iser < Fo->isermax) 
	    for (i = 1; i <= Fo->ifft; i++)
		on[i] *= pz[i];
    }

    for (iser = 2; iser <= Fo->isermax; iser = iser + 4) {
	for (i = 1; i <= Fo->npoints_2 + 1; i++) {
	    Fo->a[i][iser] = -Fo->a[i][iser];
	    if (iser < Fo->isermax)
		Fo->a[i][iser + 1] = -Fo->a[i][iser + 1];
	    Fo->b[i][iser] = -Fo->b[i][iser];
	    Fo->b[i][iser - 1] = -Fo->b[i][iser - 1];
	}
    }

    Fo->ttm = pi / Fo->d;
    Fo->dt = Fo->ttm / Fo->npoints_2;
    Fo->ttmax = 2. * Fo->ttm;
    Fo->nom_func = NULL;

    XFREE(on);
    XFREE(om);
    XFREE(pz);
    XFREE(onx);

    return Fo;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Computes the Fourier transform of a real function at the point z
    from the associated Tfourier structure (obtained with the functions
    Tf or fTf .
---------------------------------------------------------------------------*/
dcomplex 
Trans(Tfourier * Fo, double z)
{
    dcomplex        xx,
                    zz;
    double          cx1,
                    sx1,
                    xr1,
                    xr2,
                    xi1,
                    xi2,
                    u2,
                    u,
                    e00,
                    xr,
                    xi,
                    xr0,
                    xi0,
                    xr00,
                    xi00,
                    t,
                    ta,
                    x,
                    uz;
    int             i,
                    j,
                    ig,
                    ilp;

    t = z - floor(z / Fo->ttmax) * Fo->ttmax;
    ilp = 1;
    ta = t;
    if (t > Fo->ttm) {
	t = Fo->ttmax - t;
	ilp = -1;
    }
    e00 = .000001;

    ig = (int) floor(t / Fo->dt);
    x = (t - Fo->dt * ig) / Fo->dt;
    if (x > .5) {
	ig++;
	x -= 1.;
    }
    ig++;
    xr = Fo->b[ig][Fo->isermax];
    xi = Fo->a[ig][Fo->isermax];

    for (i = Fo->isermax - 1; i > 1; i -= 2) {
	j = i - 1;
	xr = xr * x + Fo->a[ig][i];
	xr = xr * x + Fo->b[ig][j];
	xi = xi * x + Fo->b[ig][i];
	xi = xi * x + Fo->a[ig][j];
    }

    xr = xr * x + Fo->a[ig][0];
    xi = xi * x + Fo->b[ig][0];
    xi *= ilp;
    xr0 = xr * Fo->d;
    xi0 = xi * Fo->d;
    u = ta * Fo->d;
    uz = z * Fo->d;
    u2 = uz * z;
    if (u2 > e00) {
	sx1 = sin(u);
	cx1 = cos(u);
	xr1 = (1. - cx1) / u2;
	x = (xr1 + xr1) / Fo->d;
	xi1 = (sx1 - uz) / u2;
	xr2 = Fo->onfin * cos(Fo->ifft * u) - Fo->ondeb;
	xi2 = Fo->onfin * sin(Fo->ifft * u);
	xr00 = xr0 * x + xr1 * xr2 - xi1 * xi2;
	xi00 = xi0 * x + xr1 * xi2 + xr2 * xi1;
    } else {
	xr00 = xr0;
	xi00 = xi0;
    }
    xx.r = xr00;
    xx.i = xi00;

    zz.r = cos(Fo->bmin * z);
    zz.i = sin(Fo->bmin * z);
    xx = dCmul(xx, zz);
    return xx;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Computes the real part of the Fourier transform of a real function at 
    the point z (i.e. the 'cosine' transform) from the associated Tfourier
    structure (obtained with the functions Tf or fTf .
---------------------------------------------------------------------------*/
double 
TransR(Tfourier * Fo, double z)
{
    double          cx1,
                    sx1,
                    xr1,
                    xr2,
                    xi1,
                    xi2,
                    u2,
                    u,
                    e00,
                    xr,
                    xr0,
                    xr00,
                    t,
                    ta,
                    x,
                    uz,
                   *a,
		   *b;
    int             i,
                    ig,
                    ilp;

    t = z - floor(z / Fo->ttmax) * Fo->ttmax;
    ilp = 1;
    ta = t;
    if (t > Fo->ttm) {
	t = Fo->ttmax - t;
	ilp = -1;
    }
    e00 = .000001;

    ig = (int) floor(t / Fo->dt);
    x = (t - Fo->dt * ig) / Fo->dt;
    if (x > .5) {
	ig++;
	x -= 1.;
    }
    ig++;
    xr = Fo->b[ig][Fo->isermax];

    a = Fo->a[ig];
    b = Fo->b[ig];
    i = Fo->isermax - 1;

    while (i != 0) {
        xr *= x;
        xr += a[i--];
	xr *= x;
        xr += b[i--];
    }

    xr = xr * x + a[0];
    xr0 = xr * Fo->d;
    u = ta * Fo->d;
    uz = z * Fo->d;
    u2 = uz * z;
    if (u2 > e00) {
	sx1 = sin(u);
	cx1 = cos(u);
	xr1 = (1. - cx1) / u2;
	x = (xr1 + xr1) / Fo->d;
	xi1 = (sx1 - uz) / u2;
	xr2 = Fo->onfin * cos(Fo->ifft * u) - Fo->ondeb;
	xi2 = Fo->onfin * sin(Fo->ifft * u);
	xr00 = xr0 * x + xr1 * xr2 - xi1 * xi2;
    } else {
	xr00 = xr0;
    }

    return xr00;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Computes the imaginary part of the Fourier transform of a real function at 
    the point z (i.e. the 'sine' transform) from the associated Tfourier
    structure (obtained with the functions Tf or fTf .
---------------------------------------------------------------------------*/
double 
TransI(Tfourier * Fo, double z)
{
    double          cx1,
                    sx1,
                    xr1,
                    xr2,
                    xi1,
                    xi2,
                    u2,
                    u,
                    e00,
                    xi,
                    xi0,
                    xi00,
                    t,
                    ta,
                    x,
                    uz;
    int             i,
                    j,
                    ig,
                    ilp;

    t = z - floor(z / Fo->ttmax) * Fo->ttmax;
    ilp = 1;
    ta = t;
    if (t > Fo->ttm) {
	t = Fo->ttmax - t;
	ilp = -1;
    }
    e00 = .000001;

    ig = (int) floor(t / Fo->dt);
    x = (t - Fo->dt * ig) / Fo->dt;
    if (x > .5) {
	ig++;
	x -= 1.;
    }
    ig++;
    xi = Fo->a[ig][Fo->isermax];

    for (i = Fo->isermax - 1; i > 1; i -= 2) {
	j = i - 1;
	xi = xi * x + Fo->b[ig][i];
	xi = xi * x + Fo->a[ig][j];
    }

    xi = xi * x + Fo->b[ig][0];
    xi *= ilp;
    xi0 = xi * Fo->d;
    u = ta * Fo->d;
    uz = z * Fo->d;
    u2 = uz * z;
    if (u2 > e00) {
	sx1 = sin(u);
	cx1 = cos(u);
	xr1 = (1. - cx1) / u2;
	x = (xr1 + xr1) / Fo->d;
	xi1 = (sx1 - uz) / u2;
	xr2 = Fo->onfin * cos(Fo->ifft * u) - Fo->ondeb;
	xi2 = Fo->onfin * sin(Fo->ifft * u);
	xi00 = xi0 * x + xr1 * xi2 + xr2 * xi1;
    } else {
	xi00 = xi0;
    }

    return xi00;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    This frees the memory occupied by a Tfourier structure.
---------------------------------------------------------------------------*/
void 
Detruit(Tfourier * Fo)
{
    XFREE(Fo->a);
    XFREE(Fo->b);
    if (Fo->nom_func != NULL)
	free(Fo->nom_func);
    free(Fo);
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Computation of the Tfourier structure associated to a real function in
    simple precision.
    The function is defined by the arrays x_r[] (the x) and F[]
    (the f(x)). 
    NB[0] is the number of points of this function (the indices run from 1
    to NB[0]).
    ifw is the number of points for the FFTs that will be used. It must be
    a power of 2.
    iprec determines the precision of the computations (from 0 to 4). 3
    is a good value.
---------------------------------------------------------------------------*/
Tfourier       *
fTf(int *NB, float *x_r, float *F, int ifw, int iprec)
{
    int             i;
    double         *b,
                   *on;
    Tfourier       *Fo;

    b = double_alloc1(*NB + 1); 
    on = double_alloc1(*NB + 1);

    for (i = 1; i <= *NB; i++) {
	b[i] = (double) x_r[i];
	on[i] = (double) F[i];
    }

    Fo = Tf(NB, b, on, ifw, iprec);
    XFREE(b);
    XFREE(on);
    return Fo;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Computation of the CTfourier structure associated to a complex function 
    in double precision.
    The function is defined by the arrays x_r[] (the x) and F[]
    (the f(x)). 
    NB[0] is the number of points of this function (the indices run from 1
    to NB[0]).
    ifw is the number of points for the FFTs that will be used. It must be
    a power of 2.
    iprec determines the precision of the computations (from 0 to 4). 3
    is a good value.
---------------------------------------------------------------------------*/
CTfourier      *
dCTf(int *NB, double *x_r, dcomplex * F, int ifw, int iprec)
{
    CTfourier      *Fo;
    double         *onre,
                   *onim;
    int             i;

    Fo = (CTfourier *) malloc(sizeof(Tfourier));
    onre = double_alloc1(*NB + 1);
    onim = double_alloc1(*NB + 1);

    for (i = 1; i <= *NB; i++) {
	onre[i] = F[i].r;
	onim[i] = F[i].i;
    }

    Fo->Reel = Tf(NB, x_r, onre, ifw, iprec);
    Fo->Imag = Tf(NB, x_r, onim, ifw, iprec);
    XFREE(onre);
    XFREE(onim);
    return Fo;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Computation of the CTfourier structure associated to a complex function 
    in simple precision.
    The function is defined by the arrays x_r[] (the x) and F[]
    (the f(x)). 
    NB[0] is the number of points of this function (the indices run from 1
    to NB[0]).
    ifw is the number of points for the FFTs that will be used. It must be
    a power of 2.
    iprec determines the precision of the computations (from 0 to 4). 3
    is a good value.
---------------------------------------------------------------------------*/
CTfourier      *
CTf(int *NB, float *x_r, fcomplex * F, int ifw, int iprec)
{
    int             i;
    double         *b;
    dcomplex       *on;
    CTfourier      *Fo;

    b = double_alloc1(*NB + 1);
    on = dcomplex_alloc1(*NB + 1);

    for (i = 1; i <= *NB; i++) {
	b[i] = (double) x_r[i];
	on[i].r = (double) F[i].r;
	on[i].i = (double) F[i].i;
    }

    Fo = dCTf(NB, b, on, ifw, iprec);
    XFREE(b);
    XFREE(on);
    return Fo;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Computes the Fourier transform of a real function at the point z
    from the associated Tfourier structure (obtained with the functions
    Tf or fTf . This is the simple precision version of Trans.
---------------------------------------------------------------------------*/
fcomplex 
fTrans(Tfourier * Fo, float x)
{
    fcomplex        z;
    dcomplex        y;

    y = Trans(Fo, (double) x);
    z.r = (float) y.r;
    z.i = (float) y.i;
    return z;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Computes the real part of the Fourier transform of a real function at 
    the point z (i.e. the 'cosine' transform) from the associated Tfourier
    structure (obtained with the functions Tf or fTf . This is the simple
    precision version of TransR.
---------------------------------------------------------------------------*/
float 
fTransR(Tfourier * Fo, float x)
{
    float  	    z;
    double          y;

    y = TransR(Fo, (double) x);
    z = (float) y;
    return z;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Computes the imaginary part of the Fourier transform of a real function at 
    the point z (i.e. the 'sine' transform) from the associated Tfourier
    structure (obtained with the functions Tf or fTf . This is the simple
    precision version of TransI.
---------------------------------------------------------------------------*/
float 
fTransI(Tfourier * Fo, float x)
{
    float  	    z;
    double          y;

    y = TransI(Fo, (double) x);
    z = (float) y;
    return z;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Computes the Fourier transform of a complex function at the point z
    from the associated CTfourier structure (obtained with the functions
    dCTf or CTf . 
---------------------------------------------------------------------------*/
dcomplex 
dCTrans(CTfourier * Fo, double x)
{
    dcomplex        zr,
                    zi,
                    z;

    zr = Trans(Fo->Reel, x);
    zi = Trans(Fo->Imag, x);
    z.r = zr.r - zi.i;
    z.i = zr.i + zi.r;
    return z;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    Computes the Fourier transform of a complex function at the point z
    from the associated CTfourier structure (obtained with the functions
    dCTf or CTf . This is the simple precision version of dCTrans.
---------------------------------------------------------------------------*/
fcomplex 
CTrans(CTfourier * Fo, float x)
{
    dcomplex        y;
    fcomplex        z;

    y = dCTrans(Fo, (double) x);
    z.r = (float) y.r;
    z.i = (float) y.i;
    return z;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
    This frees the memory occupied by a CTfourier structure.
---------------------------------------------------------------------------*/
void 
CDetruit(CTfourier * Fo)
{
    Detruit(Fo->Reel);
    Detruit(Fo->Imag);
    free(Fo);
}
/*-------------------------------------------------------------------------*/
