#pragma linesize(132)	/* listing linewidth = 132 */
/* hpgl2.c */

/* Copyright 1991-1992, Robert C. Becker */

/* rev. C 8/18/92

	Added code for TD (transparent data) to hpgl2.c */

/* rev. B 8/13/92

	Relocated some code from hpgl1.c to hpgl2.c to make file sizes more
equal and reduce compile time of hpgl1.c . */

/* rev. A  8/3/92

	Found a bug in labeling routine.  If no label terminator char
was found, the routine would not stop reading at the EOF.  Cause was traced
to comparing an unsigned char with (char) EOF: changed to (unsigned char) EOF.
*/

/* this file contains the following functions:

	begin_plot ()
	calc_csize ()
	char_plot ()
	define_path ()
	draw_arc ()
	draw_wedge ()
	extra_space ()
	get_xy ()
	get_val ()
	label_dir ()
	label_graph ()
	label_origin ()
	label_term ()
	line_pattern ()
	page_adv ()
	recalc_csize ()
	set_csize ()
	set_slant ()
	symbol_mark ()
	xparent_data ()
*/

#include <stdio.h>
#include <math.h>
#define NO_VIDEO		/* define no video for hpgl.h */
#include "hpgl.h"
#include "graph.h"

extern struct csizes cs;
extern struct cplot cp;
extern int chord_type, p_status, symbol_mode;
extern char symbol;

extern double p1x, p2x, p1y, p2y;
extern unsigned endlabel;

static char copyright[] = "Copyright 1991-1992, Robert C. Becker, Lantern Systems";
static unsigned char label_bfr[LABEL_BFR];
static double es_x, es_y;
static char bfr[256];			/* scratch char buffer for get_val */

/*--------------------------------------*/

void label_origin (FILE * infile)
	{
	double x;
	int l;

	if (!get_val (infile, &x) )
		{
		cp.lorg = LORG_DEFAULT;
		return;
		}
	l = (int) x;
	if (l > 19 || l < 1 || l == 10)	/* no lorg (10) */
		{
		print_string ("LO: value out of range\n");
		return;		/* no changes */
		}
	cp.lorg = l;	/* new label origin */
	return;
	}

/*--------------------------------------*/

void line_pattern (FILE * infile)
	{
	double x, y;
	int l;

	DB (printf ("line_pattern\n");)
	if ( !get_val (infile, &x) )
		{	/* default line type */
		DB (printf ("line_pattern: default pattern\n");)
		line_type (1, 0);
		return;
		}
	else
		{
		DB (printf ("line_pattern: line type = %lf\n", x);)
		if (x < 0)		/* line type w/pattern # */
			{
			print_noinstr ("Adaptive line type");
			return;
			}
		l = (int) (MAX (0.0, MIN (6.0, x)) ) + 2;
			/* translate HPGL patterns to library patterns */
		if (!get_val (infile, &y))
			{
			DB (printf ("line_pattern: line pattern = %d\n", l);)
			line_type (l, 0);
			return;
			}
		else	/* line type w/pattern # and repeat length */
			{
			DB (printf ("line_pattern: line pattern = %d, length = %lf\n", l, MIN (y, 127.9999));)
			line_type (l, MAX (0.0, MIN ( y, 127.9999)));
			return;
			}
		}
	return;
	}

/*--------------------------------------*/

void set_csize (FILE * infile, int type)
	{	/* type: 0 -> absolute, 1 -> relative */
		/* set height (y) and width (x) of char (in cm) */
	double x, y;

	if (type == RELATIVE)
		{		/* relative size */
		cs.rcsize_x = x = 0.75;	/* width: 0.75% (p2x - p1x) */
		cs.rcsize_y = y = 1.50;	/* height: 1.5% (p2y - p1y) */
		}
	else
		{		/* absolute size */
		cs.acsize_x = x = 0.19;	/* default width (cm) */
		cs.acsize_y = y = 0.27;
		}
	if (get_val (infile, &x) )	/* got width value: need height */
		{
		if ( !get_val (infile, &y))	/* missing height */
			{
			if (type == RELATIVE)
				print_string ("SR: missing height parameter\n");
			else
				print_string ("SI: missing height parameter\n");
			return;
			}
		}

	x = MAX (-128.00, (MIN (127.9999, x)));
	y = MAX (-128.00, (MIN (127.9999, y)));

	calc_csize (x, y, type);	/* set char size */

	return;
	}

/*--------------------------------------*/

void recalc_csize (void)
	{

	DB (printf ("recalc_csize: cs.cscale_mode = %d\n", cs.cscale_mode);)
	if (cs.cscale_mode) 
		{
		DB (printf ("recalc_csize: cs.rcsize_x = %lf, cs.rcsize_y = %lf\n", cs.rcsize_x, cs.rcsize_y);)
		calc_csize (cs.rcsize_x, cs.rcsize_y, cs.cscale_mode);
		}	/* rescale char size for relative scaling */
	else
		{
		DB (printf ("recalc_csize: cs.acsize_x = %lf, cs.acsize_y = %lf\n", cs.acsize_x, cs.acsize_y);)
		calc_csize (cs.acsize_x, cs.acsize_y, cs.cscale_mode);
		}	/* rescale char size for absolute scaling */

	return;
	}

/*--------------------------------------*/

void calc_csize (double x, double y, int type)
	{
	double ar;

	cs.cscale_mode = 0;	/* default to abs. char size scaling */
	if (type == RELATIVE)	/* relative char size: convert to cm */
		{
		cs.cscale_mode = 1;	/* turn on relative char size scaling */
		cs.rcsize_x = x;
		cs.rcsize_y = y;	/* save relative csizes for changes in IP */
		x = x * fabs (p2x - p1x) * 0.00254 / 100.0;
		y = y * fabs (p2y - p1y) * 0.00254 / 100.0;
		}	/* P1, P2 are in mils.  0.00254 cm/mil */

	cs.acsize_x = x;
	cs.acsize_y = y;	/* save absolute char sizes */

	x = cs.csize_x = cs.csize_basis * (10.0) * ((p1x > p2x) ? -x : x);	/* convert from cm to mm's for csize () */
	y = cs.csize_y = cs.csize_basis * (10.0) * ((p1y > p2y) ? -y : y);	/* correct for reflection/inversion of P2 - P1 */
		/* cs.csize_x, cs.csize_y save char size for SL function */
		/* cs.csize_basis changes char size to maintain proportion to paper size */

	ar = x / fabs (y);			/* aspect ratio */
	/* need fabs() to isolate AR from possibly negative y-size, but retain sign of x */

	if (type == ABSOLUTE) cs.csize_y = (y *= 0.725);
		/* factor of 0.725 corrects default graphix lib font size to 
		HPGL plotter font size for absolute font sizes (this is a patch) */

	csize (y, ar, cs.slant);	/* height (in cm), aspect ratio, slant (deg)  */
	return;
	}

/*--------------------------------------*/

void set_slant ( FILE *infile )
	{
	double ar, x;

	DB (printf ("SL\n");)
	x = 0.0;
	get_val (infile, &x);	/* if no value present, x is unchanged */
	x = MAX (-128.00, (MIN (127.9999, x)));
	cs.slant = RAD_2_DEG * atan (x);	/* sl is in degrees */
	DB (printf ("SL: angle = %lf\n", cs.slant);)

	ar = cs.csize_x / fabs (cs.csize_y);	/* aspect ratio */
	/* need fabs() to isolate AR from possibly negative y-size, but retain sign of x */

	csize (cs.csize_y, ar, cs.slant);	/* height (in cm), aspect ratio, slant (deg)  */
	return;
	}

/*--------------------------------------*/

void extra_space ( double x, double y)
	{

	es_x = x;	/* save extra space for char_plot */
	es_y = y;
	char_xspace (100.0 * x, 100.0 * y);	/* char_xspace takes % values */
	return;
	}

/*--------------------------------------*/

void char_plot (FILE *infile)
	{
	double x, y, x_1, y_1;
	double lsin, lcos, xdir, ydir;

	if (get_val (infile, &x))	/* get # spaces */
		{	/* got x-value: try for y-value */
		if (!get_val (infile, &y)) y = 0.0;	/* get # linefeeds (if any) */
		}
	else	/* got no x-value: default to 1 linefeed */
		{
		move (cp.x, cp.y);
		labelf ("\n");
		where (&cp.x, &cp.y);
		return;
		}

	/* move x spaces "horizontally" and y lines "vertically" */
	/* directions depend on current label direction & mirroring */

	DB (printf ("CP: x = %lf, y = %lf, cp.dv = %d\n", x, y, cp.dv);)
	DB (printf ("CP1: cp.x = %lf, cp.y = %lf\n", cp.x, cp.y);)

	move (cp.x, cp.y);
	setgu ();		/* this looks like a kludge, but it gets */
	where (&x_1, &y_1);	/* around problems of not knowing character */
				/* sizes in both plotter and user units */

	lcos = cos ( (cp.ldir + cp.dvdir) / RAD_2_DEG );
	lsin = sin ( (cp.ldir + cp.dvdir) / RAD_2_DEG );	/* label directions */

	xdir = (p1x > p2x) ? -1.0 : 1.0;
	ydir = (p1y > p2y) ? -1.0 : 1.0;	/* correct for P1/P2 mirroring */

	switch (cp.dv)	/* calculate x, y move from # x, y spaces */
		{
		case 0:	/* horizontal */
			x = x * cs.csize_x * CP_XSCALE * xdir * (1.0 + es_x);
			y = y * cs.csize_y * CP_YSCALE * ydir * (1.0 + es_y);
			break;
		case 1:	/* -90 degrees: 'x' chars of y-size, 'y' chars of x-size */
			x = x * cs.csize_y * CP_YSCALE * ydir * (1.0 + es_y);
			y = y * cs.csize_x * CP_XSCALE * xdir * (1.0 + es_x);
			break;
		case 2:	/* 180 degrees */
			x = x * cs.csize_x * CP_XSCALE * xdir * (1.0 + es_x);
			y = y * cs.csize_y * CP_YSCALE * ydir * (1.0 + es_y);
			break;
		case 3:	/* 90 degrees: 'x' chars of y-size, 'y' chars of x-size */
			x = x * cs.csize_y * CP_YSCALE * ydir * (1.0 + es_y);
			y = y * cs.csize_x * CP_XSCALE * xdir * (1.0 + es_x);
			break;
		default: break;
		}

	/* now rotate char offset to label direction */
	x_1 += x * lcos - y * lsin;
	y_1 += y * lcos + x * lsin;

	move (x_1, y_1);	/* new position */
	setuu ();
	where (&cp.x, &cp.y);

	DB (printf ("CP2: cp.x = %lf, cp.y = %lf\n", cp.x, cp.y);)

	return;
	}

/*--------------------------------------*/

void label_graph (FILE * infile)
	{
	double xpos, ypos, x_offset, y_offset;
	unsigned char c2;
	int i;

	DB (printf ("LB\n");)
	setgu ();			/* switch to GDU's */
	where ( &xpos, &ypos );		/* get current position in GDU's */

	x_offset = y_offset = 0.0;
	if (cp.lorg < 10)	/* anchor label to edges of label */
		{
		switch (cp.lorg)	/* these offsets derived by experimentation */
			{
			case 1:
				y_offset = -0.386 * cs.csize_y;
				x_offset = -0.10 * cs.csize_x;
				break;
			case 2:
				x_offset = -0.10 * cs.csize_x;
				break;
			case 3:
				y_offset = 0.303 * cs.csize_y;
				x_offset = -0.10 * cs.csize_x;
				break;
			case 4:
				y_offset = -0.386 * cs.csize_y;
				break;
			case 5: break;	/* centered in (x,y) */
			case 6:
				y_offset = 0.303 * cs.csize_y;
				break;
			case 7:
				x_offset = 0.09 * cs.csize_x;
				y_offset = -0.386 * cs.csize_y;
				break;
			case 8:
				x_offset = 0.09 * cs.csize_x;
				break;
			case 9:
				y_offset = 0.303 * cs.csize_y;
				x_offset = 0.09 * cs.csize_x;
				break;
			default: break;
			}
		move (xpos + x_offset, ypos + y_offset);	/* correct for normally offset label origins */
		lorg (cp.lorg);
		}
	else	/* normal label origins:  offset from edges of labels */
		{
		lorg (cp.lorg - 10);
		}
	setuu ();	/* back to user units */
	i = 0;
	while ( 1 )
		{
		label_bfr[i] = c2 = getc (infile);
		DB (printf ("LB: c2 = 0x%02X\n", c2);)
		if (i >= LABEL_BFR - 1 || c2 == '\n')
		 	{
			label_bfr[++i] = '\0';		/* terminate label string */
			DB (printf ("LB: i = %d, labeling plot\n", i);)
			labelf ("%s", label_bfr);	/* label the plot */
			if (c2 == '\n')
				{
				setgu ();
				imove (-x_offset, -y_offset);	/* remove for label origin offset */
				setuu ();
				where (&cp.x, &cp.y);		/* get label position: new <cr> position for CP fn. */
				setgu ();
				imove (x_offset, y_offset);	/* restore for label origin offset */
				setuu ();
				}
			i = 0;
			continue;
			}
		if (c2 == endlabel || c2 == (unsigned char) EOF)
		 	{
			DB (printf ("LB:endlabel: c2 = 0x%02X\n", c2);)
			label_bfr[i] = '\0';		/* terminate label string */
			labelf ("%s", label_bfr);	/* label the plot */
			setgu ();
			imove (-x_offset, -y_offset);	/* remove for label origin offset */
			setuu ();
			break;
			}
		label_bfr[i++] = c2;
		 }
	DB (printf ("end: LB\n");)
	plotted_on (1);		/* we drew something on this plot */
	return;
	}

/*--------------------------------------*/

void define_path (FILE * infile)
	{
	double path, line;
	int l, p;

	if (!get_val (infile, &path))	/* no parameters-> defaults */
		{
		cp.dv = 0;	/* store label direction for CP fn. */
		cp.dvdir = 0.0;	/* clear stacking direction */
		cdir (0);	/* char direction along label direction */
		ldir (cp.ldir);	/* horizontal label directions */
		return;
		}
	if (path < 0.0 || path > 3.0)
		return;		/* illegal values: do nothing */
	p = (int) path;
	if ( get_val (infile, &line))
		{	/* have a linefeed type param */
		print_string ("DV: linefeed parameter not implimented\n");
		if (line != 0.0 || line != 1.0)
			l = 0;	/* force default for illegal value */
		else
			l = (int) line;
		}

	switch (p)
		{
		case 0:			/* normal directions */
			cdir (0.0);
			cp.dvdir = 0.0;	/* normal stacking direction */
			cp.dv = 0;	/* store direction code for CP fn. */
			break;
		case 1:			/* down with vertical stacking */
			cdir (90.0);
			cp.dvdir = -90.0;	/* downward label stacking */
			cp.dv = 1;	/* store label direction for CP fn. */
			break;
		case 2:			/* left, with reversed lettering */
			cdir (180.0);
			cp.dvdir = 180.0;	/* left label stacking */
			cp.dv = 2;	/* store label direction for CP fn. */
			break;
		case 3:			/* up with vertical stacking, reversed */
			cdir (-90.0);
			cp.dvdir = 90.0;	/* upward label stacking */
			cp.dv = 3;	/* store label direction for CP fn. */
			break;
		default: break;
		}
	ldir (cp.ldir + cp.dvdir);	/* rotate DV angle by label direction */
	return;
	}

/*--------------------------------------*/

void symbol_mark (unsigned char symbol)
	{

	lorg (5);
	labelf ("%c", symbol);
	if (cp.lorg < 10)
		lorg (cp.lorg);
	else
		lorg (cp.lorg - 10);
	return;
	}

/*--------------------------------------*/

void label_term (FILE * infile)
	{
	unsigned l;
	double x;

	l = (unsigned) getc (infile);
	/* test for illegal values */
	if ( l < 1 || l > 127 || l == 5 || l == 27) return;
	endlabel = l;
	return;
	}

/*--------------------------------------*/

void label_dir (FILE * infile, int type)
	{	/* set relative character direction */
		/* type 0 -> abs, type 1 -> relative */
	double rise, run;

	if ( !get_val (infile, &run))	/* get run value */
	 	{	/* No value: ldir defaults to 0.0 */
	 	cp.ldir = 0.0;	/* save for CP fn. */
		ldir (cp.dvdir);	/* label in direction of text stacking */
		return;
		}
	if ( !get_val (infile, &rise))	/* get rise value */
		{
		return;	/* no rise coordinate: error: do nothing */
		}
	if (type == ABSOLUTE)	/* absolute label direction */
		{		/* set absolute angle, & save for CP fn. */
		cp.ldir = RAD_2_DEG * atan2 (rise, run);
		ldir ( cp.ldir + cp.dvdir );
		return;
		}

	if (rise < -128.0 || rise > 127.9999 || run < -128.0 || run > 127.9999) return;	/* fault */

	/* relative angle determined by delta P1:P2 plot scaling takes care of 
	case where P1 > P2.  Must use fabs () to eliminate 2nd compensation 
	for P1 > P2 in for relative directions */

	rise = rise * fabs (p2y - p1y) / 100.0;
	run = run * fabs (p2x - p1x) / 100.0;

	/* set relative angle & save for CP fn. */

	cp.ldir = RAD_2_DEG * atan2 (rise, run);
	ldir ( cp.ldir + cp.dvdir );

	return;
	}

/*--------------------------------------*/

void begin_plot (FILE * infile)
	{
	double x;
	unsigned l;
	unsigned char c;


	if ( plotted_on (0) )
		{
		twait (infile, PG_DELAY);	/* if we have a plot already up, delay screen clear */
		gclear ();
		}
	initialize (infile);		/* initialize plotter */

	if (!get_val (infile, &x)) return;	/* no arguments */

	do		/* process arguments */
		{
		l = (int) x;
		switch (l)
			{
			case 1:	/* newstring (in quotes) */
				c = getc (infile);	/* find 1st '"' char */
				while (c  && c != (unsigned char) EOF && c != '"') c = getc (infile);
				c = getc (infile);	/* dump everything through next '"' char */
				while (c  && c != (unsigned char) EOF && c != '"') c = getc (infile);
				break;
			case 2:	/* number of copies: NOP */
				if (!get_val (infile, &x))	/* mus supply a value, anyway */
					{
					print_string ("BP: missing argument to number of copies\n");
					return;
					}
				break;
			case 3:	/* file disposition code: NOP */
				if (!get_val (infile, &x))	/* must supply a disposition code, anyway */
					{
					print_string ("BP: missing file disposition code\n");
					return;
					}
				break;
			case 4:	/* render last plot if unfinished */
				if (!get_val (infile, &x))
					{
					print_string ("BP: incomplete 'unfinished plot' rendering code\n");
					break;
					}
			default:
				print_string ("BP: invalid kind parameter\n");
				return;	/* stop processing BP at invalid kind */
				break;
			}
		}
	while (get_val (infile, &x));

	return;
	}

/*--------------------------------------*/

void xparent_data (FILE * infile)
	{
	double x;

	x = 0.0;		/* value remains unchanged if nothing to read */
	get_val (infile, &x);	/* read one possible value */
	if (x != 1.0 && x != 0.0)
		{
		print_string ("TD: value out of range\n");
		return;
		}
	if (x == 0.0)
		disp_fns (DFN_OFF);	/* function code display off */
	else
		disp_fns (DFN_ON);	/*function code display on */

	return;
	}

/*--------------------------------------*/

void page_adv (FILE * infile)
	{
	double x;

	get_val (infile, &x);	/* get possibly one value */
	setgu ();
	move (0.0, 0.0);	/* move to lower left corner of hard clip */
	setuu ();
	twait (infile, PG_DELAY);	/* delay PG_DELAY seconds */
	gclear ();
	plotted_on (0);		/* new page */
	return;
	}

/*--------------------------------------*/

void draw_arc (FILE * infile, int type )
	{	/* type: 0 -> absolute, 1 -> relative */
	double radius, x, y, x_1, y_1, x_2, y_2, val4, ang1, ang2, ang3;
	unsigned char c2;
	int segs;

	DB (if (type == RELATIVE) printf ("AR\n"); else printf ("AA\n");)
	if (!get_xy (infile, &x, &y))
		{
		if (type == RELATIVE)
			print_string ("AR: missing x- or y-value\n");
		else
			print_string ("AA: missing x- or y-value\n");
		return;	/* illegal instr */
		}
	if (!get_val (infile, &ang1))	/* arc angle */
		{
		if (type == RELATIVE)
			print_string ("AR: missing arc angle\n");
		else
			print_string ("AA: missing arc angle\n");
		return;	/* illegal instr */
		}
	DB (printf ("input value: x = %lf, y = %lf, ang1 = %lf\n", x, y, ang1);)
	where (&x_1, &y_1);	/* get starting point */
	if (type == RELATIVE)
		{		/* relative */
		DB (printf ("current location: (x,y) = (%lf, %lf)\n", x_1, y_1);)
		x_2 = x_1 + x;
		y_2 = y_1 + y;
		x_1 = -x;
		y_1 = -y;
		}
	else
		{		/* absolute */
		x_2 = x;
		y_2 = y;	/* save center position */
		x_1 -= x;
		y_1 -= y;	/* delta-x, -y */
		}
	DB (printf ("arc center: (x,y) = (%lf, %lf), angle = %lf\n", x_2, y_2, ang1);)
	move (x_2, y_2);	/* move to center of arc */
	radius = sqrt (x_1*x_1 + y_1*y_1);

	ang2 = val4 = DEF_CHORD;
	if (get_val (infile, &val4))
		{
		val4 = fabs (val4);	/* only positive values allowed */
		if (chord_type == ANG || val4 == 0.0)		/* using angles, not deviations */
			{
			if (val4 == 0.0) val4 = DEF_CHORD;	/* set default chord angle or deviation */
			ang2 = MIN (MAX_CHORD, (MAX (val4, MIN_CHORD)));
			}
		else		/* chord angle determined by deviation of segments from circle radius */
			{	/* at this point, val4 is length in current units, not angle in degrees */
			if (val4 > 2.0 * radius) val4 = 2.0 * radius;	/* limit deviation: resulting chord angle < 180 deg. */
			ang2 = acos ( (radius - val4) / radius) * RAD_2_DEG;
			}
		}

	segs = (int) (fabs (ang1) / ang2 + 0.5);	/* # segments in "ang1" deg. */
		/* sign of 'segs' changes direction of arc () */
	DB (printf ("chord = %lf, # segments = %d\n", ang2, segs);)
	ang3 = RAD_2_DEG * atan2 (y_1, x_1);
	if (p_status == PENDOWN)	/* draw the arc only if pen is down */
		{
		DB (printf ("radius = %lf, start angle = %lf, stop angle = %lf\n", radius, ang3, ang1 + ang3);)
		arc ( radius, segs, ang3, ang1 + ang3);
		}
	if (symbol_mode)
		{
		move (x_2, y_2);
		symbol_mark (symbol);
		}
	plotted_on (1);		/* we drew something on this plot */
	return;
	}

/*--------------------------------------*/

void draw_wedge (FILE * infile)
	{
	double radius, xc, yc, x_1, y_1, val3, ang1, ang2, ang3;
	int segs;

	DB (printf ("EW\n");)
	if (!get_val (infile, &radius))
		{
		print_string ("EW: missing radius\n");
		return;	/* illegal instr */
		}
	if (!get_val (infile, &ang1))
		{
		print_string ("EW: missing start_ang\n");
		return;	/* illegal instr */
		}
	if (!get_val (infile, &ang2))	/* arc angle */
		{
		print_string ("EW: missing sweep angle\n");
		return;	/* illegal instr */
		}
	where (&xc, &yc);	/* get starting point */
	DB (printf ("center: (x,y) = (%lf, %lf), start angle = %lf\n", xc, yc, ang1);)

	val3 = ang3 = DEF_CHORD;
	if (get_val (infile, &val3))
		{
		val3 = fabs (val3);	/* only positive values allowed */
		if (chord_type == ANG || val3 == 0.0)		/* using angles, not deviations */
			{
			if (val3 == 0.0) val3 = DEF_CHORD;	/* set default chord angle or deviation */
			ang3 = MIN (MAX_CHORD, (MAX (val3, MIN_CHORD)));
			}
		else		/* chord angle determined by deviation of segments from circle radius */
			{	/* at this point, val3 is length in current units, not angle in degrees */
			if (val3 > 2.0 * radius) val3 = 2.0 * radius;	/* limit deviation: resulting chord angle < 180 deg. */
			ang3 = acos ( (radius - val3) / radius) * RAD_2_DEG;
			}
		}
	segs = (int) (fabs (ang2) / ang3 + 0.5);	/* # segments in "ang1" deg. */
		/* sign of 'segs' changes direction of arc () */

	DB (printf ("chord = %lf, # segments = %d\n", ang3, segs);)
	DB (printf ("radius = %lf, start angle = %lf, stop angle = %lf\n", radius, ang1, ang1 + ang2);)
	x_1 = xc + radius * cos (ang1 / RAD_2_DEG);
	y_1 = yc + radius * sin (ang1 / RAD_2_DEG);	/* position of start of arc */

	draw (x_1, y_1);			/* draw line from center to start of arc */
	move (xc, yc);
	arc ( radius, segs, ang1, ang1 + ang2);	/* draw arc: leaves pen at end of arc */
	draw (xc, yc);				/* draw line from end of arc to center */

	if (symbol_mode)
		{
		symbol_mark (symbol);
		}

	plotted_on (1);		/* we drew something on this plot */
	return;
	}

/*--------------------------------------*/

int get_xy (FILE *infile, double *xv, double *yv)
	{

	if (get_val (infile, xv))	/* get x values */
	 	{
		if ( !get_val (infile, yv))	/* get y value */
			{
			print_string ("get_xy():  missing y-value\n");
			return (0);
			}
		return (1);	/* have 2 values */
		}
	return (0);	/* no values */
	}

/*--------------------------------------*/

int get_val (FILE *infile, double *rval)
	{
	int j;
	char c;

	c = getc (infile);
	while ( c  && c != (char) EOF && isspace (c) ) c = getc (infile);
	/* remove leading white-space */

	if ( c == TERMCHAR || c == (char) EOF) return (0);	/* terminated: return # params read */
	if ( ! (isdigit (c) || c == ',' || c == '-' || c == '+' || c == '.'))
		{
		ungetc (c, infile);	/* put back what we grabbed */
		return (0);		/* next char not a number */
		}
	if (c == ',') c = getc (infile);

	j = 0;
	while ( isdigit ( c ) || c == '-' || c == '+' || c == '.' )
		{
		bfr[j] = c;
		c = getc (infile);
		++j;
		}
	bfr[j] = '\0';	/* terminate string */
	if (c != ',' && !isspace (c) ) ungetc (c, infile);	/* DO NOT INCLUDE termchar IN THIS TEST!! */

	/* including termchar in the test to unget a char will cause a failure if termchar was read 
	by this routine and it is called again immediately thereafter (will return 1 argument of zero value [error]) */

	if (j == 0)	/* trap zero length arguments */
		{
		print_string ("? Null argument in get_val ()\n");
		return (0);
		}

	DB (printf ("get_val: instring = %s\n", bfr);)
	*rval = atof (bfr);
	return (1);
	}

/*--------------------------------------*/
