/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       FUTIL.C
**     SYSTEM   NAME:       
**     ORIGINAL AUTHOR(S):  Wim van Campen
**     VERSION  NUMBER:     1.0
**     CREATION DATE:       1990/9/30
**
** DESCRIPTION: Contains some functions to read strings from
**              command files.
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.1  $
** WORKFILE:    $Workfile:   FUTIL.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/FUTIL/VCS/FUTIL.C_V  $
**              
**                 Rev 1.1   01 Feb 1991 10:11:30   etstjan
**              No explicit note
**              
**                 Rev 1.0   20 Nov 1990 15:12:52   etstjan
**              Initial revision.
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/CPROG/BEHOLDER/FUTIL/VCS/FUTIL.C_V   1.1   01 Feb 1991 10:11:30   etstjan  $";
#endif

#include  <stdio.h>
#include  <stdlib.h>
#include  <string.h>
#include  <ctype.h>

#include  "futil.h"

/* global variables */
char  ConfigFile[255] = "BEHOLDER.INI";   /* configuration file           */
int   LineNumber;

/* prototypes */
static int FGetCooked(FILE *ComFile, char *Buffer, int BufLen);

/**************************************************************
** NAME:        FGetCooked                                
** SYNOPSIS:    static int FGetCooked(FILE *ComFile,
**                                    char *Buffer,
**                                    int BufLen)
**
** DESCRIPTION: Reads the next line from file 'ComFile'.
**              Comments are stripped from the line
**              and leading whitespaces are removed.
**              All non printable characters are replaced
**              with spaces.
**              The trailing '\n' from the fgets() library
**              call is removed.
** RETURNS:     FNOREAD   -->   nothing left to read
**              else      -->   number of characters
**************************************************************/
static int FGetCooked(FILE *ComFile, char *Buffer, int BufLen)
{
  char  FileBuf[255];
  char  *FirstToken;

  Buffer[0] = '\0';
  if (fgets(FileBuf, 255, ComFile) != NULL) {
    LineNumber++;
    /* first, remove comment */
    FileBuf[strcspn(FileBuf, "#;\n")] = '\0';
    for (FirstToken = FileBuf; *FirstToken != '\0'; FirstToken++) 
      if (!isprint(*FirstToken)) 
        *FirstToken = ' ';
    for (FirstToken = FileBuf; *FirstToken == ' '; FirstToken++) 
      ;
    strncat(Buffer, FirstToken, BufLen - 1);
    return strlen(Buffer);
    }
  return FNOREAD;
}

/**************************************************************
** NAME:        FFindSection                              [API]
** SYNOPSIS:    int FFindSection(FILE *ComFile,
**                               char *SectionStr);
**
** DESCRIPTION: Searches in file 'ComFile' for '[<SectionStr>]'.
**              The first occurence of section '[<SectionStr>]'
**              will be chosen. 
**              ComFile should be opened before calling
**              this function.
**              String compares are case insensitive.
**              The new position in the file is the line
**              immediate after the section indicator.
** RETURNS:     FNOPEN   -->   file not open
**              SNFOUND  -->   section not found
**              else     -->   line number of section
**************************************************************/
int FFindSection(FILE *ComFile, char *SectionStr)
{
  char  SearchStr[255] = "[";
  char  FileStr[255];
  int   SearchLen = strlen(SectionStr) + 2;

  if (fseek(ComFile, 0, SEEK_SET) != 0)           /* set to start of file */
    return FNOPEN;
  LineNumber = 0;

  strncat(SearchStr, SectionStr, 252);              /* make search string */
  strcat(SearchStr, "]");
  StrUpper(SearchStr, SearchStr);

  while (FGetCooked(ComFile, FileStr, 255) >= 0) {
    if (strncmp(StrUpper(FileStr, FileStr), SearchStr, SearchLen) == 0) {
      return LineNumber;
      }
    }
  return SNFOUND;
}


/**************************************************************
** NAME:        FGetNext                                  [API]
** SYNOPSIS:    int  FGetNext(FILE *ComFile, char *AnswerStr,
**                            int AnswLen, int Case);
**
** DESCRIPTION: Reads the next line of command file 'ComFile'
**              into AnswerStr. At most 'AnswLen' characters
**              are returned in 'AnswerStr'. When a line is
**              empty, or completely commented, a new line
**              is read.
**              A line's leading spaces are removed.
**              The trailing '\n' from the fgets() library
**              call is removed and the string is converted
**              to upper case.
**              If a line contains a comment sign (# or ;),
**              the rest of the line is skipped.
**              FGetNext returns an error code when all of
**              the section has been read. ComFile must be
**              opened before calling this function.
**              Case gives you the opportunity to select
**              the case of the returned string. Case = UPC
**              returns upper case only, Case = MIX returns
**              mixed case.
**
** RETURNS:     FNOREAD    -->  nothing left to read
**              FENDSEC    -->  passed end of section
**              else       -->  current line number in file
**************************************************************/
int  FGetNext(FILE *ComFile, char *AnswerStr, int AnswLen, int Case)
{
  int r;

  while ((r=FGetCooked(ComFile, AnswerStr, AnswLen)) == 0);
  if (r>0)
  {
    if (AnswerStr[0] == '[')                     /* start of next section */
        return FENDSEC;
    if (!Case)
        StrUpper(AnswerStr, AnswerStr);
    return LineNumber;
  }
  return FNOREAD;
}

/**************************************************************
** NAME:        FGetVar                                   [API]
** SYNOPSIS:    int FGetVar(FILE *ComFile, char *VarString,
**                          char *Result, int ResultLen,
**                          int Case);
**            
** DESCRIPTION: Searches in file 'ComFile' for a variable.
**              A search will be done from the current
**              position to the end of the section. After
**              reading the variable, the initial file
**              position is reset.
**              All string compares are case insensitive.
**              A variable is a name followed by ' ' or '='.
**              Spaces before or after '=' are allowed.
**              The string returned consists of the value
**              of the variable, that is, with the variable
**              name, '=' and spaces removed.
**              Case gives you the opportunity to select
**              the case of the returned string. Case = UPC
**              returns upper case only, Case = MIX returns
**              mixed case.
**
** EXAMPLE:     Size = 3400    ;Set size to 3400 bytes
**              When looking for the variable 'Size',
**              the string '3400' is returned.
**
** RETURNS:     FILERR   -->   general file error
**              VNFOUND  -->   variable not found
**              else     -->   current line number in file
**************************************************************/
int FGetVar(FILE *ComFile, char *VarString, char *Result,
            int ResultLen, int Case)
{
  long    CurrentPos;
  char    FileStr[255];
  char    SearchStr[255]; 
  int     LineNr, SaveLineNr;
  int     RetCode=VNFOUND;
  int     EndVar;
  int     VarLen = strlen(VarString);

  if ((CurrentPos = ftell(ComFile)) == -1)
    return FILERR;
  SaveLineNr = LineNumber;
  StrUpper(SearchStr, VarString);
  while ((LineNr = FGetNext(ComFile, FileStr, 255, 1)) > 0) {
    if ((EndVar = strcspn(FileStr, " =")) == VarLen) {
      if (Case) FileStr[EndVar++] = '\0';
      StrUpper(FileStr, FileStr);
      if (strncmp(FileStr, SearchStr, VarLen) == 0) {
        while((isspace(FileStr[EndVar])) || (FileStr[EndVar] == '='))
           EndVar++;
        Result[0] = '\0';
        strncat(Result, FileStr + EndVar, ResultLen - 1);
        RetCode = LineNr;
        break;
        }
      }
    }
  if (fseek(ComFile, CurrentPos, SEEK_SET) != 0)
    return FILERR;
  LineNumber = SaveLineNr;
  return RetCode;
}

/**************************************************************
** NAME:        StrUpper                                  [API]
** SYNOPSIS:    char *StrUpper(char *Destin, char *Source);
**    
** DESCRIPTION: Converts Source to upper case in Destin. 
** RETURNS:     Destin
**************************************************************/
char *StrUpper(char *Destin, char *Source)
{
  int i;
  for (i = 0; (Destin[i] = (char)toupper(Source[i])) != '\0'; i++)
    ;
  return Destin;
}

/**************************************************************
** NAME:        StrLower                                  [API]
** SYNOPSIS:    char *StrLower(char *Destin, char *Source);
**    
** DESCRIPTION: Converts Source to lower case in Destin.
** RETURNS:     Destin
**************************************************************/
char *StrLower(char *Destin, char *Source)
{
  int i;
  for (i = 0; (Destin[i] = (char)tolower(Source[i])) != '\0'; i++)
    ;
  return Destin;
}
