/*
    Input.c

    These are some subrotines to obtain user input.
*/

#include <stdio.h>      /* standard I/O functions header file */
#include <string.h>     /* standard string functions header file */
#include <stdlib.h>     /* standard library functions header file */
#include <ctype.h>      /* standard character functions header file */
#include <dos.h>        /* standard DOS functions header file */
#include <bios.h>       /* standard BIOS functions header file */
#include <direct.h>     /* standard directory functions header file */
#include "video.h"      /* video display routines */
#include "memmgr.h"     /* memory management routines */
#include "modstuf.h"    /* music module loading/playing routines */
#include "input.h"      /* user input routine */


/*************************************************************************/
/*                                                                       */
/* Macros.                                                               */
/*                                                                       */
/*************************************************************************/

    /* Maximum number of drives, directories, and files to display. */
#define MAXDIRENTRIES 256


/*************************************************************************/
/*                                                                       */
/* Types.                                                                */
/*                                                                       */
/*************************************************************************/

    /* Type of directory entry for selector. */
struct screenentry {
    char name[13];      /* name of drive, directory, or file */
    char type;          /* drive, directory or file code */
};


/*************************************************************************/
/*                                                                       */
/* Global variables.                                                     */
/*                                                                       */
/*************************************************************************/

static char instring[_MAX_PATH];       /* input string returned */

    /* List of entries in the current directory. */
static struct screenentry dirlist[MAXDIRENTRIES];

static int ndrives = 0;     /* number of drives in the directory list */
static int ndirs = 0;       /* number of subdirectories in directory list */
static int nfiles = 0;      /* number of files in directory list */
static int totalentries = 0; /* total number of entries in directory list */
static int lastcol;         /* last column in the directory list */
static int inlastcol;       /* number of entries in last column */
static int leftcolumn;      /* leftmost displayed column */
static int highlightrow;    /* row of highlighted entry */
static int highlightcol;    /* column of highlighted entry */

    /* List of valid keys in selector screen, to display to the user. */
static char keyprompt1[] =
"Keys:  <enter>, <esc>, <up arrow>, \
<down arrow>, <left arrow>, <right arrow>,";
static char keyprompt2[] =
"<home>, <end>, <page up>, <page down>, <ctrl>-<home>, <ctrl>-<end>, r(eread)";


/*************************************************************************/
/*                                                                       */
/* Function prototypes.                                                  */
/*                                                                       */
/*************************************************************************/

static void filldrives( void );
static int cmpscrentries( struct screenentry *first,
    struct screenentry *second );
static void filldirs( void );
static void fillfiles( void );
static void showdircol( int column, int scrcolumn );
static void showdirlist( void );
static void compute_cols( void );
static void scroll_to_column( int newcol );
static void move_highlight( int newrow, int newcol );


/*************************************************************************/
/*                                                                       */
/* getinput() routine.  Obtains a string from the user with echo and     */
/* editing.  Returns a pointer to the null-terminated string in inbuf.   */
/* Returns 0 if input has been entered, -1 if the user cancelled out of  */
/* the input by pressing <esc>.                                          */
/*                                                                       */
/*************************************************************************/

int getinput( char **inbuf )
{
    int stringlen;              /* length of input string */
    int cursorpos;              /* cursor position in string */
    int insertmode;             /* 1 if in insert mode */
    unsigned inchar;            /* user's keystroke */
    int i;                      /* for looping over characters */

    /* Set inbuf to our string buffer. */
    *inbuf = instring;

    /* Make the string empty. */
    instring[0] = '\0';
    stringlen = 0;
    cursorpos = 0;

    /* Clear the input line onscreen and set its color to indicate input is 
       pending.  Place the screen cursor at the start of the line. */
    vidstartinput();

    /* Initially in overtype mode - use underline cursor. */
    insertmode = 0;
    vidlinecurs();

    /* Get keystrokes from the user and process them. */
    do
    {
        inchar = vidgetchar();
        switch ( inchar )
        {
            case ESC:
                break;

            case BACKSPACE:
                if ( cursorpos )
                {
                    vidmovecurs( --cursorpos );
                    for ( i = cursorpos; i < stringlen; i++ )
                        instring[i] = instring[i+1];
                    stringlen--;
                    vidputinput( instring );
                }
                break;

            case INSERT:
                if ( insertmode ^= 1 )
                    vidblockcurs();
                else
                    vidlinecurs();
                break;

            case HOME:
                vidmovecurs( cursorpos = 0 );
                break;

            case DELETE:
                if ( cursorpos < stringlen )
                {
                    for ( i = cursorpos; i < stringlen; i++ )
                        instring[i] = instring[i+1];
                    stringlen--;
                    vidputinput( instring );
                }
                break;

            case END:
                vidmovecurs( cursorpos = stringlen );
                break;

            case ENTER:
                break;

            case LEFT:
                if ( cursorpos )
                    vidmovecurs( --cursorpos );
                break;

            case RIGHT:
                if ( cursorpos < stringlen )
                    vidmovecurs( ++cursorpos );
                break;

            default:
                inchar &= 0xff;
                if ( inchar >= 0x20 && inchar <= 0x7e )
                {
                    if ( insertmode )
                    {
                        if ( stringlen < 79 )
                        {
                            for ( i = ++stringlen; i > cursorpos; i-- )
                                instring[i] = instring[i-1];
                            instring[cursorpos] = inchar;
                            vidmovecurs( ++cursorpos );
                            vidputinput( instring );
                        }
                    }
                    else
                    {
                        if ( cursorpos < stringlen )
                        {
                            instring[cursorpos] = inchar;
                            vidmovecurs( ++cursorpos );
                            vidputinput( instring );
                        }
                        else if ( stringlen < 79 )
                        {
                            instring[cursorpos] = inchar;
                            vidmovecurs( ++cursorpos );
                            instring[++stringlen] = '\0';
                            vidputinput( instring );
                        }
                    }
                }
                break;
        }
    } while ( inchar != ESC && inchar != ENTER );

    /* Reset the color of the input line to indicate that input is no 
       longer pending; the cursor will become invisible. */
    videndinput();

    /* Set the return code according to whether <enter> or <esc> was 
       pressed. */
    if ( inchar == ESC )
        return( -1 );
    else /* inchar == ENTER */
        return( 0 );
}


/*************************************************************************/
/*                                                                       */
/* chkdrive() function.  This function verifies that the drive specified */
/* in the path parameter is current.  On a single-floppy system, it will */
/* prompt the user to insert the diskette if needed to access drive A:   */
/* or B:, for example.  Returns 0 if successful, -1 if the drive is      */
/* invalid and no attempt should be made to access the path.  Note:  the */
/* drive might still really be invalid if this function returns 0, but   */
/* at least DOS won't try to prompt for the drive B: diskette.           */
/*                                                                       */
/*************************************************************************/

int chkdrive( char *path )
{
    char *colonplace;           /* position of colon in path */
    char drivechar;             /* drive character */
    unsigned drivenum;          /* number of drive in path */
    union REGS inregs, outregs; /* register structs for DOS calls */
    unsigned dosver;            /* DOS version number */
    int nfloppies;              /* number of floppy drives in the system */
    char tempstr[80];           /* message string */

    /* Check for a colon.  If none, or if it isn't the second character, 
       either the path is invalid or it refers to the current drive, so 
       we're OK. */
    if ( (colonplace = strchr( path, ':' )) == NULL )
        return( 0 );
    else if ( colonplace != path+1 )
        return( 0 );

    /* Get the drive number.  If invalid, just return. */
    drivechar = toupper( path[0] );
    if ( !isupper( drivechar ) )
        return( 0 );
    drivenum = (unsigned) drivechar - 'A' + 1;

    /* Get the DOS version. */
    inregs.x.ax = 0x3000;
    intdos( &inregs, &outregs );
    dosver = ((unsigned) outregs.h.al << 8) + outregs.h.ah;

    /* If DOS earlier than 3.2, drive B: is invalid in a single-floppy 
       system. */
    if ( dosver < 0x0314 )
    {
        /* OK if not B: in this case. */
        if ( drivenum != 2 )
            return( 0 );

        /* It's drive B: - see how many floppies there are.  If only one, 
           it's invalid. */
        nfloppies = ((_bios_equiplist() >> 6) & 3) + 1;
        if ( nfloppies == 1 )
            return( -1 );

        /* More than one floppy.  It's OK. */
        return( 0 );
    }

    /* DOS 3.2 or later.  Use IOCTL Get Logical Drive Map to determine if 
       the drive being requested is current. */
    inregs.x.ax = 0x440e;
    inregs.h.bl = drivenum;
    intdos( &inregs, &outregs );

    /* If the call returned with carry set, either the driver doesn't 
       support the call, or the drive is invalid. */
    if ( outregs.x.cflag )
    {
        /* If the drive is valid, but the driver doesn't support the call, 
           it's OK; device drivers written for DOS prior to DOS 3.2 do not 
           support multiple logical drives. */
        if ( outregs.x.ax == 1 )
            return( 0 );

        /* Otherwise, the drive is invalid. */
        return( -1 );
    }

    /* If there is only one logical drive letter associated with the 
       requested drive, it's valid. */
    if ( !outregs.h.al )
        return( 0 );

    /* If there is more than one logical drive letter associated with the 
       requested drive, but the requested drive is current, it's valid. */
    if ( outregs.h.al == drivenum )
        return( 0 );

    /* Drive is not current.  Get the user to insert a new disk. */
    sprintf( tempstr, "Insert diskette for drive %c: and press any key.",
        drivechar );
    vidputmsg( tempstr );
    vidgetchar();

    /* The user pressed a key, so we assume he put the diskette in.  Now 
       make the requested drive current. */
    inregs.x.ax = 0x440f;
    inregs.h.bl = drivenum;
    intdos( &inregs, &outregs );

    /* Return "valid." */
    return( 0 );
}


/*************************************************************************/
/*                                                                       */
/* filldrives() routine.  Fills in the directory list with whatever      */
/* drives are in the system.  Returns nothing.                           */
/*                                                                       */
/*************************************************************************/

static void filldrives( void )
{
    union REGS inregs, outregs; /* register structs for DOS calls */
    unsigned dosver;            /* DOS version number */
    unsigned currentdrive;      /* current drive */
    unsigned drivenum;          /* drive to test */
    unsigned actual;            /* actual drive after drive change */
    unsigned drives;            /* number of drives returned (ignored) */
    char valid[27];             /* valid flags for the drives (0 unused) */
    int nfloppies;              /* number of floppy drives in the system */

    /* Get the DOS version. */
    inregs.x.ax = 0x3000;
    intdos( &inregs, &outregs );
    dosver = ((unsigned) outregs.h.al << 8) + outregs.h.ah;

    /* If DOS 3.2 or later, use IOCTL Get Logical Drive Map to determine 
       which drives exist. */
    if ( dosver >= 0x0314 )
        for ( drivenum = 1; drivenum <= 26; drivenum++ )
        {
            /* Call DOS. */
            inregs.x.ax = 0x440e;
            inregs.h.bl = drivenum;
            intdos( &inregs, &outregs );

            /* If the call returns with carry clear, the drive is valid.  
               Otherwise, if the call returns with carry set, and the 
               driver does not support the call, the drive is valid still. 
               */
            valid[drivenum] = (!outregs.x.cflag || outregs.x.ax == 1);
        }

    /* If DOS 3.1 or earlier, use Select Disk to find out which drives we 
       have. */
    else
    {
        /* Get and save the current drive. */
        _dos_getdrive( &currentdrive );

        /* Loop over the possible drives, attempting to change to each one. 
           */
        for ( drivenum = 1; drivenum <= 26; drivenum++ )
        {
            /* Special case:  We don't want to try to change to drive B: in 
               a single-floppy system under DOS earlier than 3.2, since 
               there is no way to keep DOS from displaying the "Insert disk 
               for drive B:" message (which would mess up our screen).  So 
               - drive B: is invalid if there is only one floppy. */
            if ( drivenum == 2 )
            {
                nfloppies = ((_bios_equiplist() >> 6) & 3) + 1;
                if ( nfloppies == 1 )
                {
                    valid[drivenum] = 0;
                    continue;
                }
            }

            /* If able to change to the drive, it's valid; otherwise not. 
               */
            _dos_setdrive( drivenum, &drives );
            _dos_getdrive( &actual );
            valid[drivenum] = (actual == drivenum);
        }

        /* Change back to the (original) current drive. */
        _dos_setdrive( currentdrive, &drives );
    }

    /* We have a list of valid drives.  Put them in the directory list. */
    ndrives = 0;
    for ( drivenum = 1; drivenum <= 26; drivenum++ )
        if ( valid[drivenum] )
        {
            sprintf( dirlist[ndrives].name, "%c:", drivenum + 'A' - 1 );
            dirlist[ndrives++].type = DRIVETYPE;
        }
}


/*************************************************************************/
/*                                                                       */
/* cmpscrentries() routine.  Compares the names of two entries in the    */
/* directory list and returns -1 if the first comes before the second    */
/* alphabetically, 0 if they are the same (should never happen), and 1   */
/* if the first comes after the second.  For use with qsort().           */
/*                                                                       */
/*************************************************************************/

static int cmpscrentries( struct screenentry *first,
    struct screenentry *second )
{
    return( strcmp( first->name, second->name ) );
}


/*************************************************************************/
/*                                                                       */
/* filldirs() routine.  Adds to the directory list all subdirectories of */
/* the current directory, including the parent directory if the current  */
/* directory is not the root directory.  filldrives() must be called     */
/* before this routine.  The directories are sorted alphabetically.      */
/* Returns nothing.                                                      */
/*                                                                       */
/*************************************************************************/

static void filldirs( void )
{
    char currentdir[_MAX_DIR];  /* full path of current directory */
    int isroot;                 /* 1 if current directory is root */
    int currententry;           /* current position in directory list */
    struct find_t findbuf;      /* buffer for findfirst, findnext */

    /* Get the full path of the current working directory and use it to 
       determine if we're in the root directory. */
    getcwd( currentdir, _MAX_DIR );
    isroot = (strlen( currentdir ) == 3);

    /* Start where the drives end. */
    currententry = ndrives;
    ndirs = 0;

    /* If the current directory is not the root directory, the first 
       subdirectory is ".", which we omit, and the second is "..", which we 
       include. */
    if ( !isroot )
    {
        _dos_findfirst( "*.*", _A_SUBDIR, &findbuf );
        _dos_findnext( &findbuf );
        strcpy( dirlist[currententry].name, ".." );
        dirlist[currententry++].type = DIRTYPE;
        ndirs++;
    }

    /* Otherwise, if the current directory is root, the first subdirectory, 
       if any, is a real subdirectory. */
    else
    {
        /* If no subdirectories, we're done. */
        if ( _dos_findfirst( "*.*", _A_SUBDIR, &findbuf ) )
            return;

        /* If it's a subdirectory, put it in. */
        if ( findbuf.attrib & _A_SUBDIR )
        {
            strcpy( dirlist[currententry].name, findbuf.name );
            dirlist[currententry++].type = DIRTYPE;
            ndirs++;
        }
    }

    /* Find more subdirectories and put them in the list too.  Make sure 
       you don't go past the end of the list, though. */
    while( currententry < MAXDIRENTRIES && !_dos_findnext( &findbuf ) )
        if ( findbuf.attrib & _A_SUBDIR )
        {
            strcpy( dirlist[currententry].name, findbuf.name );
            dirlist[currententry++].type = DIRTYPE;
            ndirs++;
        }

    /* Put the subdirectories in alphabetical order. */
    if ( isroot )
    {
        /* Root directory - need at least 2 to sort :). */
        if ( ndirs < 2 )
            return;
        qsort( &dirlist[ndrives], ndirs, sizeof( struct screenentry ),
            cmpscrentries );
    }
    else
    {
        /* Not root directory - need 2 in addition to "..". */
        if ( ndirs < 3 )
            return;

        /* Sort directories other than "..". */
        qsort( &dirlist[ndrives+1], ndirs-1, sizeof( struct screenentry ),
            cmpscrentries );
    }
}


/*************************************************************************/
/*                                                                       */
/* fillfiles() routine.  Adds to the directory list all files in the     */
/* current directory.  filldrives() and filldirs() must be called before */
/* this routine.  The files are sorted alphabetically.  Returns nothing. */
/*                                                                       */
/*************************************************************************/

static void fillfiles( void )
{
    int currententry;           /* current position in directory list */
    struct find_t findbuf;      /* buffer for findfirst, findnext */

    /* Start where the directories end. */
    currententry = ndrives + ndirs;
    nfiles = 0;

    /* Get the first filename.  If no files, or if the directory list is 
       already full, we're done. */
    if ( currententry >= MAXDIRENTRIES ||
            _dos_findfirst( "*.*", _A_NORMAL, &findbuf ) )
        return;

    /* Put the first filename in. */
    strcpy( dirlist[currententry].name, findbuf.name );
    dirlist[currententry++].type = FILETYPE;
    nfiles++;

    /* Get the rest of the filenames.  Make sure you don't go past the end 
       of the directory list. */
    while( currententry < MAXDIRENTRIES && !_dos_findnext( &findbuf ) )
    {
        strcpy( dirlist[currententry].name, findbuf.name );
        dirlist[currententry++].type = FILETYPE;
        nfiles++;
    }

    /* Sort the files, if we got at least 2. */
    if ( nfiles < 2 )
        return;
    qsort( &dirlist[ndrives+ndirs], nfiles, sizeof( struct screenentry ),
        cmpscrentries );
}


/*************************************************************************/
/*                                                                       */
/* showdircol() routine.  Displays one column from the current directory */
/* list, using the requested list column and screen column.  Returns     */
/* nothing.                                                              */
/*                                                                       */
/*************************************************************************/

static void showdircol( int column, int scrcolumn )
{
    int row;                /* current row in the directory list */
    int currententry;       /* current entry in the list */

    /* Figure out where to start. */
    currententry = column * NDIRROWS;

    /* Display the files in the column. */
    for ( row=0; row < NDIRROWS && currententry < totalentries;
            row++, currententry++ )
        vidputdirent( dirlist[currententry].name,
            dirlist[currententry].type, row, scrcolumn );

    /* Clear out any remaining entries in the column. */
    for ( ; row < NDIRROWS; row++ )
        vidputdirent( " ", FILETYPE, row, scrcolumn );
}


/*************************************************************************/
/*                                                                       */
/* showdirlist() routine.  Displays the current directory list starting  */
/* from the current leftmost column and highlights the currently         */
/* selected entry.  Returns nothing.                                     */
/*                                                                       */
/*************************************************************************/

static void showdirlist( void )
{
    int scrcolumn;          /* current screen column */
    int column;             /* current column in directory list */

    /* Clear out the current command area. */
    vidstartsel();

    /* Loop over the screen columns, displaying them. */
    for ( scrcolumn = 0, column = leftcolumn;
            scrcolumn < NDIRCOLS; scrcolumn++, column++ )
        showdircol( column, scrcolumn );

    /* Highlight the selected entry. */
    vidhilight( highlightrow, highlightcol - leftcolumn );
}


/*************************************************************************/
/*                                                                       */
/* compute_cols() routine.  This routine is called when the directory    */
/* list has just been (re)filled.  It determines the total number of     */
/* entries in the directory list, the number of the last column          */
/* (including nondisplayed columns), and the number of entries in the    */
/* last column.  It also sets the leftmost displayed column, the         */
/* highlighted column, and the highlighted row to 0.  Returns nothing.   */
/*                                                                       */
/*************************************************************************/

static void compute_cols( void )
{
    /* Get the total number of entries in the directory list. */
    totalentries = ndrives + ndirs + nfiles;

    /* Determine the number of columns and number in the last one. */
    if ( inlastcol = totalentries % NDIRROWS )
        lastcol = totalentries / NDIRROWS;
    else
    {
        inlastcol = NDIRROWS;
        lastcol = totalentries/NDIRROWS - 1;
    }

    /* Set the leftmost displayed column, the highlighted column, and the 
       highlighted row. */
    leftcolumn = highlightrow = highlightcol = 0;
}


/*************************************************************************/
/*                                                                       */
/* scroll_to_column() routine.  Scrolls the directory list display left  */
/* or right as needed to make the new highlighted column visible.        */
/* Returns nothing.                                                      */
/*                                                                       */
/*************************************************************************/

static void scroll_to_column( int newcol )
{
    int rightcolumn;        /* rightmost displayed column */
    int scrcolumn;          /* current screen column */
    int column;             /* current column in directory list */

    /* Determine the rightmost displayed column. */
    rightcolumn = leftcolumn + NDIRCOLS - 1;

    /* Case 1:  The desired column is to the left of the leftmost displayed 
       column. */
    if ( newcol < leftcolumn )
    {
        leftcolumn = newcol;
        for ( scrcolumn = 0, column = leftcolumn;
                scrcolumn < NDIRCOLS; scrcolumn++, column++ )
            showdircol( column, scrcolumn );
    }

    /* Case 2:  The desired column is to the right of the rightmost 
       displayed column. */
    else if ( newcol > rightcolumn )
    {
        leftcolumn = newcol - NDIRCOLS + 1;
        for ( scrcolumn = 0, column = leftcolumn;
                scrcolumn < NDIRCOLS; scrcolumn++, column++ )
            showdircol( column, scrcolumn );
    }

    /* Case 3:  The desired column is already displayed - do nothing. */
}


/*************************************************************************/
/*                                                                       */
/* move_highlight() routine.  Moves the highlight, scrolling the         */
/* directory list as necessary.  (newrow, newcol) is where the highlight */
/* should be placed.                                                     */
/*                                                                       */
/*************************************************************************/

static void move_highlight( int newrow, int newcol )
{
    /* Remove the existing highlight. */
    vidunhilight( highlightrow, highlightcol - leftcolumn );

    /* Scroll the display if necessary to make the new highlighted column 
       visible. */
    scroll_to_column( newcol );

    /* Set the new highlight location. */
    highlightrow = newrow;
    highlightcol = newcol;

    /* Place the new highlight. */
    vidhilight( highlightrow, highlightcol - leftcolumn );
}


/*************************************************************************/
/*                                                                       */
/* fileselect() routine.  Displays a list of filenames, drives, and      */
/* directories for the user to select from.  If the user selects a file, */
/* its pathname is returned in path, and fileselect() returns 0.  The    */
/* function returns 1 if the user cancels, -1 if there was an error of   */
/* some sort.  In case of errors, errmsg is set to the error message     */
/* string.                                                               */
/*                                                                       */
/*************************************************************************/

int fileselect( char **path, char **errmsg )
{
    int farright;           /* rightmost entry on line in directory list */
    int rightcolumn;        /* rightmost displayed entry on line */
    int newcol;             /* column to move highlight to */
    int currententry;       /* currently-highlighted entry */
    unsigned drivenum;      /* drive to change to */
    unsigned actual;        /* actual drive after drive change */
    unsigned drives;        /* number of drives returned (ignored) */
        /* Error messages. */
    static char nolistmsg[] = "Error constructing directory list.";
    static char dirchgmsg[] = "Error during directory change.";
    static char drivechgmsg[] = "Error during drive change.";

    /* Display the keys to the user. */
    vidstartsel();
    vidputmsg( keyprompt1 );
    vidputmsg2( keyprompt2 );

    /* Fill the directory list. */
    filldrives();
    filldirs();
    fillfiles();
    compute_cols();
    if ( !totalentries )
    {
        *errmsg = nolistmsg;
        videndinput();
        vidputcmds();
        return( -1 );
    }

    /* Display the directory list. */
    showdirlist();

    /* Loop and process keystrokes. */
    while ( 1 )
        switch ( vidgetchar() )
        {
            /* Enter key. */
            case ENTER:
                /* Get the currently-highlighted entry. */
                currententry = highlightcol*NDIRROWS + highlightrow;

                /* Process according to what it is. */
                switch ( dirlist[currententry].type )
                {
                    /* Drive - change to it. */
                    case DRIVETYPE:
                        /* Verify the drive. */
                        if ( chkdrive( dirlist[currententry].name ) )
                        {
                            *errmsg = drivechgmsg;
                            videndinput();
                            vidputcmds();
                            return( -1 );
                        }
                        vidputmsg( keyprompt1 );

                        /* Set the drive. */
                        drivenum = (unsigned) dirlist[currententry].name[0]
                            - 'A' + 1;
                        _dos_setdrive( drivenum, &drives );
                        _dos_getdrive( &actual );
                        if ( actual != drivenum )
                        {
                            *errmsg = drivechgmsg;
                            videndinput();
                            vidputcmds();
                            return( -1 );
                        }

                        /* Fill the directory list (except for the drives, 
                           which don't change). */
                        filldirs();
                        fillfiles();
                        compute_cols();

                        /* Display the directory list. */
                        showdirlist();
                        break;

                    /* Directory - change to it. */
                    case DIRTYPE:
                        if ( chdir( dirlist[currententry].name ) )
                        {
                            *errmsg = dirchgmsg;
                            videndinput();
                            vidputcmds();
                            return( -1 );
                        }

                        /* Fill the directory list (except for the drives, 
                           which don't change). */
                        filldirs();
                        fillfiles();
                        compute_cols();

                        /* Display the directory list. */
                        showdirlist();
                        break;

                    /* Regular file.  I'm going to make a full path out of 
                       this, though I really don't have to, so that the 
                       current directory will be displayed on the screen as 
                       part of the filename. */
                    case FILETYPE:
                        getcwd( instring, _MAX_PATH );
                        if ( strlen( instring ) > 3 )
                            strcat( instring, "\\" );
                        strcat( instring, dirlist[currententry].name );
                        *path = instring;

                        /* Clear the input area and redisplay the command 
                           screen. */
                        videndinput();
                        vidputcmds();
                        return( 0 );
                }
                break;

            /* Escape key - user cancels. */
            case ESC:
                /* Clear the input area and redisplay the command screen. */
                videndinput();
                vidputcmds();
                return( 1 );

            /* <up arrow> key. */
            case UP:
                /* If we are on the top row ... */
                if ( highlightrow == 0 )
                {
                    /* If we're in the first column, go to the last file in 
                       the list. */
                    if ( highlightcol == 0 )
                        move_highlight( inlastcol-1, lastcol );

                    /* Otherwise, move to the last row in the previous 
                       column. */
                    else
                        move_highlight( NDIRROWS-1, highlightcol-1 );
                }

                /* We're not on the top row.  Move up one row. */
                else
                    move_highlight( highlightrow-1, highlightcol );
                break;

            /* <down arrow> key. */
            case DOWN:
                /* If we are on the last row in the last column, go to the 
                   first row in the first column. */
                if ( highlightcol == lastcol && highlightrow == inlastcol-1 )
                    move_highlight( 0, 0 );

                /* If we are on the last row in another column, go to the 
                   first row in the next column. */
                else if ( highlightrow == NDIRROWS-1 )
                    move_highlight( 0, highlightcol+1 );

                /* Otherwise, just move down one row. */
                else
                    move_highlight( highlightrow+1, highlightcol );
                break;

            /* <left arrow> key. */
            case LEFT:
                /* If we are already on the leftmost entry on the line, 
                   skip it. */
                if ( highlightcol == 0 )
                    break;

                /* Move the highlight. */
                move_highlight( highlightrow, highlightcol-1 );
                break;

            /* <right arrow> key. */
            case RIGHT:
                /* If we are already on the rightmost entry on the line, 
                   skip it. */
                if ( highlightcol == lastcol )
                    break;
                if ( highlightcol == lastcol-1 && highlightrow >= inlastcol )
                    break;

                /* Move the highlight. */
                move_highlight( highlightrow, highlightcol+1 );
                break;

            /* <home> key. */
            case HOME:
                /* If we're already in the leftmost column in the list, 
                   skip it. */
                if ( highlightcol == 0 )
                    break;

                /* If we're already in the leftmost displayed column on the 
                   screen ... */
                if ( highlightcol == leftcolumn )
                {
                    /* Determine the column where the highlight will be 
                       placed. */
                    newcol = leftcolumn - NDIRCOLS + 1;
                    if ( newcol < 0 )
                        newcol = 0;

                    /* Move the highlight. */
                    move_highlight( highlightrow, newcol );
                }

                /* Otherwise, move the highlight to the leftmost displayed 
                   column. */
                else
                    move_highlight( highlightrow, leftcolumn );
                break;

            /* <end> key. */
            case END:
                /* Determine the rightmost displayed entry on the line and 
                   the rightmost entry on the line in the directory list. */
                if ( highlightrow < inlastcol )
                    farright = lastcol;
                else
                    farright = lastcol - 1;
                rightcolumn = leftcolumn + NDIRCOLS - 1;
                if ( rightcolumn > farright )
                    rightcolumn = farright;

                /* If we're already in the rightmost column in the list, 
                   skip it. */
                if ( highlightcol == farright )
                    break;

                /* If we're already in the rightmost displayed column on 
                   the screen ... */
                if ( highlightcol == rightcolumn )
                {
                    /* Determine the column where the highlight will be 
                       placed. */
                    newcol = rightcolumn + NDIRCOLS - 1;
                    if ( newcol > farright )
                        newcol = farright;

                    /* Move the highlight. */
                    move_highlight( highlightrow, newcol );
                }

                /* Otherwise, move the highlight to the rightmost displayed 
                   column. */
                else
                    move_highlight( highlightrow, rightcolumn );
                break;

            /* <page up> key. */
            case PGUP:
                /* Move the highlight to the top row in the column. */
                move_highlight( 0, highlightcol );
                break;

            /* <page down> key. */
            case PGDOWN:
                /* If we are in the last column, move to the last entry in 
                   the last column. */
                if ( highlightcol == lastcol )
                    move_highlight( inlastcol-1, highlightcol );

                /* Otherwise, move to the last entry in the column. */
                else
                    move_highlight( NDIRROWS-1, highlightcol );
                break;

            /* <control>-<home> key. */
            case CTRLHOME:
                /* Move the highlight to the first entry in the list. */
                move_highlight( 0, 0 );
                break;

            /* <control>-<end> key. */
            case CTRLEND:
                /* Move the highlight to the last entry in the list. */
                move_highlight( inlastcol-1, lastcol );
                break;

            /* Lower-case "r" (reread disk, for when the user changes 
               floppies). */
            case LOWERR:
                /* Fill the directory list (except for the drives, which 
                   don't change). */
                filldirs();
                fillfiles();
                compute_cols();

                /* Display the directory list. */
                showdirlist();
                break;

            /* Toggle EMS enable. */
            case ALTX:
                emsenable ^= 1;
                vidputemsen();
                break;

            /* Toggle save memory. */
            case ALTS:
                savemem ^= 1;
                vidputsavemem();
                break;

            /* Toggle loading optimization. */
            case ALTO:
                optimize ^= 1;
                vidputloadopt();
                break;

            default:
                break;
        }
}
