/* --------------------------------- radar.c -------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* paint the radar info onto the Head Up Display.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include "fly.h"
#include "plane.h"
#include "hud.h"

extern void	FAR SetKillCorrection (OBJECT *p, OBJECT *target, VECT R,
	VECT A, int *tti);

static void FAR	show_all_intel (VIEW *view, OBJECT *pov, OBJECT *obj, int sx,
	int sy, int bspeed, int orgx, int orgy, int maxx, int maxy, int ss,
	int mode, int shifty);
static void FAR	show_intel (VIEW *view, OBJECT *pov, OBJECT *obj,
	OBJECT *target, int sx, int sy, int orgx, int orgy, int maxx, int maxy,
	int ss, int mode, int shifty);
static void FAR	show_box (OBJECT *obj, OBJECT *target, int x, int y, int dx,
	int dy, int mode, int off_screen, int dist, int tti,
	int closure, int bottom, int right, int hbottom, int hleft, int ss,
	int shifty);
static void FAR	show_range (OBJECT *obj, int dist, int closure, int orgx,
	int orgy, int sx, int sy, int tx, int ss);
static void FAR check_shoot (OBJECT *p, int orgx, int orgy, int sx, int sy,
	int dx, int dy, int ty, int ss);
#if 0
static void FAR	show_reticule (OBJECT *p, int x, int y, int tx, int ty,
	int orgx, int orgy, int sx, int sy, int off_screen, int c, int ss);
#endif
static void FAR	show_ross (OBJECT *p, int x, int y, int tx, int ty, int orgx,
	int orgy, int sx, int sy, int off_screen, int c, int ss);
static int FAR	show_select (VIEW *view, OBJECT *p, int radar, int orgx,
	int orgy, int sx, int sy, int maxx, int shifty);
static void FAR show_bombs (VIEW *view, OBJECT *p, OBJECT *pov, int orgx, int orgy,
	int sx, int sy, int maxx, int maxy, int clipx, int clipy, int shifty);
static void FAR	show_cross (int x, int y, int dx, int dy);
static void FAR show_target (OBJECT *p, OBJECT *target, VECT RR, int dist,
	int closure, int tti, int orgx, int orgy, int maxx, int maxy, int ss);

#undef	CP
#undef	CW
#undef	CS

#define CP	view->viewport
#define CW	view->window
#define CS	view->screen

extern void FAR
show_radar (VIEW *view, OBJECT *p, OBJECT *pov, int orgx, int orgy,
	int maxx, int maxy, int tx, int ty, int ss, int clipx, int clipy,
	int sx, int sy, int shifty, int VVD[2], int mode, int hon)
{
	int	dx, dy, ds, x, y, closure, have_reticule, f156cl, fa18;
	int	i, dist, off_screen, toff, bspeed, rv, dt, dti, tti, tbox;
	int	bx, by, radar, selecting, savehud, dmode;
	OBJECT	*target;
	VECT	RRT, RR, V, TA;
	LVECT	LL, IP;
	int	D[2], DD[2];
	long	ldist;

/* bomb impact points
*/
	if (WE_MK82 == EE(p)->weapon)
		show_bombs (view, p, pov, orgx, orgy, sx, sy, maxx, maxy, clipx,
			clipy, shifty);

	radar = EE(p)->radar;
	if (!(radar & R_ON))
		return;

	i = EE(p)->hud1 & HUD_TYPES;
	f156cl = i == HUD_F16 || i == HUD_F15 || i == HUD_CLASSIC;
	fa18 = i == HUD_FA18;
	have_reticule = 0;
	dx = ds = toff = 0;	/* avoid compiler warning */

	if (WE_MK82 == EE(p)->weapon)
		bspeed = BombSpeed (p, V)/VONE;
	else
		bspeed = BulletSpeed (p, V)/VONE;

	if ((radar & R_INTEL) ||
	    (mode == HDD_MAP || mode == HDD_RADAR)) {
		show_all_intel (view, pov, p, sx, sy, bspeed, orgx, orgy,
			maxx, maxy, ss, mode, (mode == HDD_HUD) ? shifty: 0);
		return;
	}
/* Check if we have a target at all.
*/
	if (!((target = EE(p)->target)) || target->id != EE(p)->tid) {
		target = 0;
		EE(p)->target = 0;
	}
/* if selecting, show selection circle and best target (if we have one).
*/
	selecting = (radar & (R_SELECT3|R_SELECT20|R_SELECT5))
			&& !(target && (radar & R_LOCK));

	if (selecting && hon)
		dy = show_select (view, p, radar, orgx, orgy, sx, sy, maxx,
				shifty);
	else
		dy = 0;	/* avoid compiler warning */

	if (!target)
		goto ret;

/* get target position, distance, closure and time to intercept.
*/
	objects_show (view, pov, 0, target->R, RRT);
	Vsub (LL, p->R, target->R);
	ldist = lhypot3d (LL);
	dti = dist = (int)(ldist/VONE);
	if (WE_MK82 == EE(p)->weapon) {
		closure = (int)((p->R[Z]-target->R[Z])/VONE);
		dt = BombIP (p->R, V, target->R[Z], IP);
		tti = (dt < 6000) ? dt/10 : 600;
	} else {
		if (dist)
			closure = (int)(-(LL[X]*(p->V[X] - EE(p)->tvnew[X]) +
					  LL[Y]*(p->V[Y] - EE(p)->tvnew[Y]) +
					  LL[Z]*(p->V[Z] - EE(p)->tvnew[Z])
					 ) / (ldist*VONE));
		else
			closure = 0;
		tti = (dist/60 < closure) ? muldiv (dist, 10, closure) : 600;
	}

	tbox = fa18 ? F18TBOX : F15TBOX;
	bx = fmul (sx, tbox);
	by = fmul (sy, tbox);
	if (selecting) {
		if (hon)
			show_data (p, orgx-3*ss, orgy+dy+ss, 1,
				EE(p)->hud1 & HUD_KNOTS, dist, closure,
				get_name (p, target, 0), tti, 0, ss);
		off_screen = clip_to_screen (D, RRT, maxx, maxy, bx, by, clipx,
				clipy, shifty);
		show_ross (0, D[X], D[Y], bx, by, orgx, orgy, 0, 0,
				off_screen, st.hfg, 0);
		goto ret;
	}
/* target highlighting box
*/
	if (mode == HDD_TARGET || mode == HDD_PAN) {
		show_target (p, target, RRT, dist, closure, tti, orgx, orgy,
				maxx, maxy, ss);
		goto ret;
	}

	off_screen = clip_to_screen (D, RRT, maxx, maxy, bx, by, clipx, clipy,
			shifty);

	if (HDD_HUD != mode && !(HDD_FRONT == mode && EE(p)->hud & HUD_RETICLE))
		goto no_reticule;

	rv = fa18 ? F18RAIM : F15RAIM;
	dx = fmul (sx, rv);
	dy = fmul (sy, rv);
	ds = 3;

/* bomb aiming reticule
*/
	if (WE_MK82 == EE(p)->weapon) {
		objects_show (view, pov, 0, IP, RR);

		toff = clip_to_screen (DD, RR, maxx, maxy, dx, dy,
			clipx, clipy, shifty);
		add_line (orgx+VVD[X], orgy-VVD[Y], T_MOVE);
		add_line (orgx+DD[X],  orgy-DD[Y],  st.hfg);

		TA[X] = TA[Y] = TA[Z] = 0;
		have_reticule = 1;
		goto no_reticule;
	}
/* target-speed corrected gun aiming reticule
*/
	dt = rv = 0;

	if (dist/4 > bspeed)
		goto no_reticule;
	have_reticule = 1;
	SetKillCorrection (p, target, RR, TA, &dt);

	if (EE(p)->hud & HUD_ROSS) {

/* Ross style aiming reticule
*/
		LVcopy (LL, target->R);
		Vinc (LL, RR);
		objects_show (view, p, 0, LL, RR);

		toff = clip_to_screen (DD, RR, maxx, maxy, 2*tx, 2*ty,
			clipx, clipy, shifty);
		show_ross (0, DD[X], DD[Y], tx*2, ty*2, orgx, orgy,
			sx, sy, toff, st.hfgi, ss);
		DD[X] = DD[Y] = 0;	/* reticule at center */
		toff = 0;

	} else {

/* Eyal style aiming reticule
*/
		Vsub (LL, p->R, RR);
		LL[X] += muldiv (V[X], dt, VONE*1000) * (long)VONE;
		LL[Y] += muldiv (V[Y], dt, VONE*1000) * (long)VONE;
		LL[Z] += muldiv (V[Z], dt, VONE*1000) * (long)VONE;

		objects_show (view, p, 0, LL, RR);
		toff = clip_to_screen (DD, RR, maxx, maxy, dx, dy,
					clipx, clipy, shifty);
	}
	tti = dt/100;
no_reticule:

/* Highlight the target with a box and maybe data too.
*/
	if (EE(p)->hud1 & HUD_CORNER) {
		dmode = 2;
		x = orgx-sx+2;
		y = orgy+sy-2;
		if (EE(p)->hud & HUD_BIG) {
			x += 6*tx;
			if (!(EE(p)->hud1 & HUD_TOP))
				y -= 6*ty + ss;
		} else {
			if (!(EE(p)->hud1 & HUD_TOP))
				y -= ss;
		}
	} else {
		dmode = 0;
		x = y = 0;
	}

	savehud = EE(p)->hud;
	if (have_reticule) {
		show_piper (p, target, orgx+DD[X], orgy-DD[Y],
			dx, dy, ds, dmode, toff, dti, tti, closure,
			orgx, orgy, clipx, clipy, y, x, TA, ss, shifty);
		check_shoot (p, orgx, orgy, sx, sy, DD[X], DD[Y], dy,
			ss);
		EE(p)->hud &= ~HUD_DATA;	/* don't show data again */
	}
	if (EE(p)->hud & HUD_TARGET) {
		if ((mode == HDD_FRONT || mode == HDD_HUD) &&
		    !((EE(p)->hud2 & HUD_HIDETGT) && have_reticule &&
		      iabs (D[X]-DD[X]) < (Uint)2*bx &&
		      iabs (D[Y]-DD[Y]) < (Uint)2*by))
			show_box (p, target, orgx+D[X], orgy-D[Y],
				bx, by, dmode, off_screen, dti, tti,
				closure, orgy+clipy, orgx+clipx, y, x,
				ss, shifty);
	}
	if (f156cl && hon &&
	    (have_reticule || (target && (EE(p)->hud & HUD_TARGET))))
		show_range (p, dti, closure, orgx, orgy, sx, sy, tx, ss);
	EE(p)->hud = savehud;
ret:	;
	return;
}

static void FAR
show_all_intel (VIEW *view, OBJECT *pov, OBJECT *obj, int sx, int sy,
	int bspeed, int orgx, int orgy, int maxx, int maxy, int ss, int mode,
	int shifty)
{
	OBJECT	*target;

	for (target = CO; target; target = target->next) {
		if (!(target->shflags & SH_BEHIT))
			continue;
		if ((EE(obj)->radar&R_INTELCC) && !(target->flags&F_CC))
			continue;
		if (!(st.flags1&SF_EXTVIEW) && target == obj)
			continue;
		show_intel (view, pov, obj, target, sx, sy, orgx, orgy,
			maxx, maxy, ss, mode, shifty);
	}
}

static void FAR
show_intel (VIEW *view, OBJECT *pov, OBJECT *obj, OBJECT *target, int sx,
	int sy, int orgx, int orgy, int maxx, int maxy, int ss, int map,
	int shifty)
{
	VECT	RR;
	int	D[2], off_screen, size, dx, dy, tti, speed, c, s, t;
	long	dist;

	if (map == HDD_MAP)
		map = 1;
	else if (map == HDD_RADAR)
		map = 2;
	else
		map = 0;

	if (map) {
		if (CC == target)
			dist = CC->R[Z]/VONE;
		else
			dist = (target->R[Z] - CC->R[Z])/VONE;
		tti = 0;
	} else {
		dist = ldist3d (obj->R, target->R)/VONE;
#if 0
		Vcopy (V, obj->V);
		Vdec (V, EE(obj)->tvnew);
		speed = est_hyp (V[X], V[Y], V[Z])/VONE; /* closure speed */
#else
		speed = iabs(obj->speed/VONE);		/* linear speed */
#endif
		if (dist/100 >= speed)
			tti = 999;
		else if (dist > VONE)
			tti = (int)(dist*10/speed);
		else
			tti = muldiv ((int)dist, 10, speed);
		if (dist > 31999)
			dist = 31999;
	}

	objects_show (view, pov, 0, target->R, RR);

	if (map || tti == 999) {
		dx = fmul (sx, MAPBOX/4);
		dy = fmul (sy, MAPBOX/4);
	} else {
		size = muldiv (MAPBOX, 6*600-4*tti, 6*600);
		dx = fmul (sx, size);
		dy = fmul (sy, size);
	}
	
	off_screen = clip_to_screen (D, RR, maxx, maxy, dx, dy, maxx, maxy,
					shifty);
#if 1
	if (off_screen)
		return;
#endif
	D[X] = orgx + D[X];
	D[Y] = orgy - D[Y];
	show_box (obj, target, D[X], D[Y], dx, dy, 1, off_screen,
		(int)dist, tti, 0, orgy+maxy+shifty, orgx+maxx, 0, 0, ss, 0);
	if (map) {
		t = 25*VONE;		/* one boxfull for 100 knots */
	    	dx = muldiv (target->V[X], dx, t);
	    	dy = muldiv (target->V[Y], dy, t);
	    	if (dx || dy) {
	    		if (2 == map) {
	    			s = SIN (-pov->a[Z]);
	    			c = COS (-pov->a[Z]);
	    			t  = fmul (dx, c) - fmul (dy, s);
	    			dy = fmul (dx, s) + fmul (dy, c);
	    			dx = t;
	    		}
			add_segment (D[X], D[Y], D[X]+dx, D[Y]-dy,
				st.hfgi, orgx, orgy, maxx, maxy, 0);
		}
	}
}

extern void FAR
show_data (OBJECT *obj, int datax, int datay, int detail, int knots, int dist,
	int closure, char *name, int tti, int mode, int ss)
{
	long	tt;

	if (detail) {
		if (knots)
			tt = 3L*dist + fmul (dist, 4601); /* feet */
		else
			tt = dist;
		if (tt > 10000)
			stroke_frac (datax, datay, tt/100, 0, 1, ss, st.hfg);
		else
			stroke_frac (datax, datay, tt/10, 3, 2, ss, st.hfg);
		datay += ss;

		if (!(1&mode)) {
			tt = closure;
			if (knots) {
				if (WE_M61 == EE(obj)->weapon)	/* knots */
					tt += fmul (closure, 15465);
				else				/* feet */
					tt = 3*tt + fmul (closure, 4601);
			}
			stroke_num (datax, datay, tt, ss, st.hfg);
			datay += ss;
			stroke_frac (datax, datay, (long)tti, 0, 1, ss, st.hfg);
			datay += ss;
		}

	}
	if (name)
		stroke_str (datax, datay, name, ss, st.hfg);
}

extern char * FAR
get_name (OBJECT *obj, OBJECT *target, int mode)
{
	char	*name;
	static char	title[40];

	if (target->flags & F_CC) {
		if (target->flags & F_IMPORTED)
			name = target->rplayer->name;
		else
			name = st.nikname;
	} else if (mode != 1 || (EE(obj)->hud1 & HUD_INAME)) {
		title[0] = '*';
		if (ET_PLANE == target->e_type)
			strcpy (title+1, EEP(target)->name);
		else
			strcpy (title+1, TITLE(target));
		name = title;
	} else
		name = 0;
	return (name);
}

static void FAR
show_box (OBJECT *obj, OBJECT *target, int x, int y, int dx, int dy, int mode,
	int off_screen, int dist, int tti, int closure, int bottom, int right,
	int hbottom, int hleft, int ss, int shifty)
{
	int	t, datax, datay, nlines, knots;
	char	*name;

	knots = EE(obj)->hud1 & HUD_KNOTS;

	if (NEWTGT(obj))
		t = NEWTGT(obj)--&1;
	else
		t = 1;

	if (t) {
		if (EE(obj)->target == target && (EE(obj)->radar&R_LOCK)) {
			add_line (x-dx, y+dy, T_MOVE);
			add_line (x-dx, y-dy, st.hfg);
			add_line (x+dx, y-dy, st.hfg);
			add_line (x+dx, y+dy, st.hfg);
			add_line (x-dx, y+dy, st.hfg);
		} else {
			show_ross (0, x, -y, dx, dy, 0, 0, 0, 0,
				0, st.hfg, 0);
		}

		if (off_screen == 1) {			/* ahead */
			add_line (x,    y+dy, T_MOVE);
			add_line (x,    y-dy, st.hfg);
			add_line (x+dx, y,    T_MOVE);
			add_line (x-dx, y,    st.hfg);
		} else if (off_screen == 2){		/* behind */
			add_line (x-dx, y+dy, T_MOVE);
			add_line (x+dx, y-dy, st.hfg);
			add_line (x-dx, y-dy, T_MOVE);
			add_line (x+dx, y+dy, st.hfg);
		}
	}

	if (EE(obj)->hud & HUD_DATA) {
		name = get_name (obj, target, mode);
		nlines = ((1&mode) ? 1 : 3) + !!name;

		if (2&mode) {
			datax = hleft;
			datay = hbottom+shifty-(nlines-1)*ss;
		} else {
			if (y+dy+nlines*ss >= bottom+shifty)
				datay = y-dy-2-(nlines-1)*ss;
			else
				datay = y+dy+ss;
			datax = num_size (9L, ss) * (dist<0 ? 5 : 4);
			if (name) {
				t = stroke_size (name, ss);
				if (t < datax)
					t = datax;
			} else
				t = datax;

			t = right-t;
			datax = x-dx;
			if (datax > t)
				datax = t;
		}

		show_data (obj, datax, datay, 1, knots, dist, closure, name,
				tti, mode, ss);
	}
}

/*
 * F15 radar range scale
*/

static void FAR
show_range (OBJECT *obj, int dist, int closure, int orgx, int orgy, int sx,
	int sy, int tx, int ss)
{
	int	i, x, y, y0, t, dm, range, ss2, hud1, hudtype;

	hud1 = EE(obj)->hud1;
	hudtype = hud1 & HUD_TYPES;

	ss2 = ss/2;
	if (hud1 & HUD_KNOTS) {
		dist = fmul (dist, FCON(0.54));		/* nm/1000 */
		if (WE_M61 == EE(obj)->weapon)		/* knots */
			closure += fmul (closure, FCON(0.943));
		else					/* feet/sec */
			closure = 3*closure + fmul (closure, FCON(0.28));
	}
	if (hudtype == HUD_F15 || hudtype == HUD_CLASSIC) {
		if (dist > 10000)			/* full scale range */
			range = 20000;
		else
			range = 10000;

		x = orgx + (hudtype == HUD_F15
				? fmul (sx, F15RDR) : (sx-2*ss-2*tx));
		y0 = orgy + muldiv ( sy, 8, 15);
		y = muldiv (sy, 10, 15);
		add_line (x, y0,   T_MOVE);
		add_line (x, y0-y, st.hfg);
	    	t = range/1000;				/* radar range */
		stroke_num (x+tx+2, y0-y/2+ss2, (long)t/2, ss, st.hfg);
		stroke_num (x-ss2,  y0-y-1,     (long)t,   ss, st.hfg);
		for (i = 0; i < 5; ++i) {
			t = y0 - muldiv (i, y, 5-1);
			add_line (x,    t, T_MOVE);
			add_line (x+tx, t, st.hfg);
		}

		y = y0 - muldiv (y, dist, range);
	} else if (hudtype == HUD_F16) {
		x = orgx + fmul (sx, F16ALT);
		y = orgy + muldiv (sy, 8, 16);
	} else {
		x = orgx + fmul (sx, F15ALT);
		y = orgy + muldiv (sy, 8, 15);
	}
	add_line (x-ss2, y-ss2, T_MOVE);
	add_line (x,     y,     st.hfg);
	add_line (x-ss2, y+ss2, st.hfg);

	dm = num_size ((long)closure, ss);
	stroke_num (x-dm-ss2, y+ss2, (long)closure, ss, st.hfg);
}

static void FAR
check_shoot (OBJECT *p, int orgx, int orgy, int sx, int sy, int dx, int dy,
	int ty, int ss)
{
	int	t, x, y;

	if ((EE(p)->radar&R_SHOOT)) {
		t = stroke_size ("SHOOT", ss)/2;
		x = orgx+dx;
		if (x-t < orgx-sx)
			x = orgx-sx;
		else if (x+t > orgx+sx)
			x = orgx+sx-t;
		else
			x -= t;
		t = ss;
		y = orgy-dy-ty;
		if (y-t < orgy-sy)
			y = orgy-sy+t;
		stroke_str (x, y, "SHOOT", ss, st.hfgi);
	}
}

#if 0		/* keep this one for old times sake... */
static void FAR
show_reticule (OBJECT *p, int x, int y, int tx, int ty, int orgx, int orgy,
	int sx, int sy, int off_screen, int c, int ss)
{
	int	dx, dy;

	check_shoot (p, orgx, orgy, sx, sy, x, y, ty, ss);

	x = orgx + x;
	y = orgy - y;

	dx = tx/2;
	dy = ty/2;

	add_line (x+tx,   y+dy-1, T_MOVE);
	add_line (x+tx,   y+dy,   c);
	add_line (x+dx-1, y+ty,   T_MOVE);
	add_line (x+dx,   y+ty,   c);
	add_line (x-dx+1, y+ty,   T_MOVE);
	add_line (x-dx,   y+ty,   c);
	add_line (x-tx,   y+dy-1, T_MOVE);
	add_line (x-tx,   y+dy,   c);
	add_line (x-tx,   y-dy+1, T_MOVE);
	add_line (x-tx,   y-dy,   c);
	add_line (x-dx+1, y-ty,   T_MOVE);
	add_line (x-dx,   y-ty,   c);
	add_line (x+dx-1, y-ty,   T_MOVE);
	add_line (x+dx,   y-ty,   c);
	add_line (x+tx,   y-dy+1, T_MOVE);
	add_line (x+tx,   y-dy,   c);
	if (off_screen) {
		add_line (x+tx, y,    T_MOVE);
		add_line (x-tx, y,    c);
		add_line (x,    y+ty, T_MOVE);
		add_line (x,    y-ty, c);
	}
}
#endif

static void FAR
show_ross (OBJECT *p, int x, int y, int tx, int ty, int orgx, int orgy,
	int sx, int sy, int off_screen, int c, int ss)
{
	int	tx2, ty2;

	if (p)
		check_shoot (p, orgx, orgy, sx, sy, x, y, ty*2, ss);

	x = orgx + x;
	y = orgy - y;
	tx2 = tx/2;
	ty2 = ty/2;
	add_line (x+tx2, y+ty, T_MOVE);
	add_line (x+tx,  y+ty, c);
	add_line (x+tx,  y+ty2,   c);
	add_line (x+tx,  y-ty2,   T_MOVE);
	add_line (x+tx,  y-ty, c);
	add_line (x+tx2, y-ty, c);
	add_line (x-tx2, y-ty, T_MOVE);
	add_line (x-tx,  y-ty, c);
	add_line (x-tx,  y-ty2,   c);
	add_line (x-tx,  y+ty2,   T_MOVE);
	add_line (x-tx,  y+ty, c);
	add_line (x-tx2, y+ty, c);
	if (off_screen) {
		add_line (x+tx, y,      T_MOVE);
		add_line (x-tx, y,      c);
		add_line (x,    y+ty, T_MOVE);
		add_line (x,    y-ty, c);
	}
}

#define SEL3D	(D90/54)		/* 3.3 degrees wide circle */

static int FAR
show_select (VIEW *view, OBJECT *p, int radar, int orgx, int orgy, int sx,
	int sy, int maxx, int shifty)
{
	int	i, rx, ry, dx, dy, ox, oy;
	ANGLE	a;

	if (radar & R_SELECT3) {
		i = muldiv (CP->maxx, sx, maxx);
		a = ATAN (i, CP->z);
		if (a < SEL3D)
			i = FONE;
		else
			i = muldiv (FONE, SEL3D, a);

		dx = fmul (sx, i);
		dy = fmul (sy, i);

		add_5op (T_ELLIPSE, orgx, orgy, dx, dy, st.hfg);
	} else if (radar & R_SELECT20) {
		orgy += shifty;
		rx = fmul (sx, RSELECT20);
		ry = fmul (sy, RSELECT20);

		ox = rx;
		oy = 0;
		for (a = 0, i = 0; i < 12*2; ++i) {
			a += D90/12;
			dx = fmul (COS (a), rx);
			dy = fmul (SIN (a), ry);
			add_line (orgx+ox,        orgy-oy,        T_MOVE);
			add_line (orgx+(dx+ox)/2, orgy-(dy+oy)/2, st.hfg);
			add_line (orgx-ox,        orgy+oy,        T_MOVE);
			add_line (orgx-(dx+ox)/2, orgy+(dy+oy)/2, st.hfg);
			ox = dx;
			oy = dy;
		}
		dy = sy/2;
	} else {
		orgy += shifty;
		dx = fmul (sx, RSELECT5);
		add_dash (orgx-dx, orgy,    orgx-dx, orgy-sy, 16, FONE/2, st.hfg);
		add_dash (orgx-dx, orgy,    orgx,    orgy,     4, FONE/2, st.hfg);
		add_dash (orgx+dx, orgy,    orgx+dx, orgy-sy, 16, FONE/2, st.hfg);
		add_dash (orgx+dx, orgy,    orgx,    orgy,     4, FONE/2, st.hfg);
		dy = 0;
	}
	return (dy);
}

static void FAR
show_diamond (VIEW *view, LVECT IP, OBJECT *pov, int orgx, int orgy, int dx,
	int dy, int maxx, int maxy, int clipx, int clipy, int shifty)
{
	int	x, y, toff;
	VECT	RR;
	int	DD[2];

	objects_show (view, pov, 0, IP, RR);
	toff = clip_to_screen (DD, RR, maxx, maxy, dx, dy, clipx, clipy,
		shifty);
	if (!toff) {
		x = orgx+DD[X];
		y = orgy-DD[Y];
		add_line (x+dx, y,    T_MOVE);
		add_line (x,    y-dy, st.hfg);
		add_line (x-dx, y,    st.hfg);
		add_line (x,    y+dy, st.hfg);
		add_line (x+dx, y,    st.hfg);
	}
}

static void FAR
show_bombs (VIEW *view, OBJECT *p, OBJECT *pov, int orgx, int orgy, int sx,
	int sy, int maxx, int maxy, int clipx, int clipy, int shifty)
{
	int	dx, dy;
	OBJECT	*b;
	LVECT	IP;

	dx = fmul (sx, IPDIAMOND);
	dy = fmul (sy, IPDIAMOND);

	for (b = CO; b; b = b->next) {
		if (b->owner != CC || ET_BOMB != b->e_type)
			continue;
		show_diamond (view, EBM(b)->IP, pov, orgx, orgy, dx, dy, maxx,
			maxy, clipx, clipy, shifty);
	}

	if (!(EE(p)->radar & R_ON)) {
		BombIP (p->R, p->V, 0L, IP);
		show_diamond (view, IP, pov, orgx, orgy, dx, dy, maxx, maxy,
			clipx, clipy, shifty);
	}
}

static void FAR
show_cross (int x, int y, int dx, int dy)
{
	int	g;

	g = dx/8;
	add_line (x-dx, y,    T_MOVE);
	add_line (x-g,  y,    st.hfg);

	add_line (x+dx, y,    T_MOVE);
	add_line (x+g,  y,    st.hfg);

	g = dy/8;
	add_line (x,    y-dy, T_MOVE);
	add_line (x,    y-g,  st.hfg);

	add_line (x,    y+dy, T_MOVE);
	add_line (x,    y+g,  st.hfg);
}

static void FAR
show_target (OBJECT *p, OBJECT *target, VECT RR, int dist, int closure,
	int tti, int orgx, int orgy, int maxx, int maxy, int ss)
{
	int	dx, dy, x, i;
	long	tx, ty, tz;
	int	D[2];

	dx = fmul (maxx, FCON(0.25));
	dy = fmul (maxy, FCON(0.25));
	if (!clip_to_screen (D, RR, maxx, maxy, dx, dy, maxx, maxy, 0))
		show_cross (orgx+D[X], orgy-D[Y], dx, dy);

	x = orgx - maxx + ss;
	show_data (p, x, orgy+maxy-5*ss, 1, EE(p)->hud1 & HUD_KNOTS,
		dist, closure, get_name (p, target, 0), tti, 0, ss);


	tx = (target->R[X] - p->R[X])/VONE;
	ty = (target->R[Y] - p->R[Y])/VONE;
	tz = (target->R[Z] - p->R[Z])/VONE;
	if (labs(tx) > 25000L || labs(ty) > 25000L || labs(tz) > 25000L)
		return;				/* too far out */

	dx = stroke_size ("P", ss);
	dy = orgy - maxy + 2*ss;
	stroke_char (x, dy, 'P', ss, st.hfg);
	i = ihypot2d ((int)tx, (int)ty);
	i = ANG2DEG(p->a[X] - ATAN ((int)tz, i));
	stroke_frac (x+dx*3/2, dy, (long)iabs(i), 3, 0, ss, st.hfg);
	stroke_char (x+dx*9/2, dy, (i<0) ? 'U' : 'D', ss, st.hfg);

	dy += ss;
	stroke_char (x, dy, 'H', ss, st.hfg);
	i = ANG2DEG(p->a[Z] + ATAN ((int)tx, (int)ty));
	stroke_frac (x+dx*3/2, dy, (long)iabs(i), 3, 0, ss, st.hfg);
	stroke_char (x+dx*9/2, dy, (i<0) ? 'L' : 'R', ss, st.hfg);
}
