/*
================================================================================
LapCalc - Lap Time Calculator for Microprose Formula One Grand Prix (F1GP).
(version 0.1.1)

LapCalc is a command-line program that calculates the lap times of the
computer-controlled cars, for a given set of performance parameters
(AI Grip, Car BHP, Driver Qualifying Skill, Track), using a database file.

Author: Hrvoje Stimac

Date modified: 2019-02-12
================================================================================
*/


/*
--------------------------------------------------------------------------------
DEVELOPER INFORMATION:

The database file used to calculate the lap times is based on the data obtained through a series of simulations. The simulations are performed by running a F1GP qualifying session with a duration of 120 minutes. Each simulation is performed for a grid consisting of 26 drivers. Each driver uses a unique combination of the performance parameters. The qualifying results are used to extract the lap time for each combination of the performance parameters. A large series of simulations is performed to obtain the data for the specified range of the performance parameters.


Each simulation is performed for a grid consisting of 26 DRIVERS, divided into 13 teams.

Each team uses a unique BHP value for their cars. The first team uses the value stored in BHPMAXARR, the second team uses a value smaller by BHPSTEP, the third team uses a value smaller by another BHPSTEP, etc. 13 different BHP values are used in one simulation.

The first driver in each team uses a SKILL value stored in SKILLMINARR, the second driver in the team uses a value larger by SKILLSTEP. The same is true for all teams. 2 different SKILL values are used in one simulation.

The GRIP level is set to a constant value for all 26 cars.
The simulation is performed for a certain TRACK.
The simulation iteration has a certain ITER value.
The simulation has a certain CODE value. Usually all simulations performed in one large series have the same CODE value.

The simulation is repeated for LGRIP number of GRIP values, from GRIPMIN to GRIPMAX.
The simulation is repeated for LBHPMAXARR sets of BHP data, one for each element in BHPMAXARR.
The simulation is repeated for LSKILLMINARR sets of SKILL data, one for each element in SKILLMINARR.
The simulation is repeated for LTRACK number of TRACKs, from TRACKMIN to TRACKMAX.
The simulation is repeated for LITER number of ITERations, from ITERMIN to ITERMAX.
The simulation is repeated for LCODE number of CODEs, from CODEMIN to CODEMAX.


Example of the parameters used for one simulation:

Driver  Team  Grip  BHP   Skill  Track  Code  Iter
01      01    001   0720  000    01     001   001
02      01    001   0720  015    01     001   001
03      02    001   0710  000    01     001   001
04      02    001   0710  015    01     001   001
05      03    001   0700  000    01     001   001
06      03    001   0700  015    01     001   001
07      04    001   0690  000    01     001   001
08      04    001   0690  015    01     001   001
09      05    001   0680  000    01     001   001
10      05    001   0680  015    01     001   001
11      06    001   0670  000    01     001   001
12      06    001   0670  015    01     001   001
13      07    001   0660  000    01     001   001
14      07    001   0660  015    01     001   001
15      08    001   0650  000    01     001   001
16      08    001   0650  015    01     001   001
17      09    001   0640  000    01     001   001
18      09    001   0640  015    01     001   001
19      10    001   0630  000    01     001   001
20      10    001   0630  015    01     001   001
21      11    001   0620  000    01     001   001
22      11    001   0620  015    01     001   001
23      12    001   0610  000    01     001   001
24      12    001   0610  015    01     001   001
25      13    001   0600  000    01     001   001
26      13    001   0600  015    01     001   001


In this version of the program, the performance parameters are limited to the following range of values:
Track: 1-16 = 16 values
Grip: 1-22 = 22 values
BHP: 470-1240 (multiples of 10) = 78 values
Skill: 0-59 = 60 values
Iterations: 10
Code: 1

Total number of lap time entries: 16*22*78*60*10*1 = 16 473 600
Total number of unique performance parameter combinations for each track: 22*78*60 = 102 960


The numerical constants and the indexing formula are organized in a generalized way, to simplify the process of upgrading the program code as the lap time database evolves.


Known issues:
The lap times are stored in the lap time database file in the float format. This makes the lap time data susceptible to floating point errors. It is recommended that the lap times are corrected and reduced to the nominal range (sss.ttt) before using them. In this version of the program, the lap times are converted to integers in the tttttt format.
--------------------------------------------------------------------------------
*/


#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <math.h>

#define LFILE 256 // number of characters in a file name

#define TRACKMIN 1 // track minimum value [must be 1 for getindex() to work]
#define TRACKMAX 16 // track maximum value
#define LTRACK (TRACKMAX-TRACKMIN+1) //number of tracks

#define GRIPMIN 1 // grip minimum value [must be 1 for getindex() to work]
#define GRIPMAX 22 // grip maximum value
#define LGRIP (GRIPMAX-GRIPMIN+1) // number of grip values

#define BHPMAX 1240 // bhp maximum value
#define BHPMIN 470 // bhp minimum value
#define BHPSTEP 10 // difference between two bhp values in a simulation
const int BHPMAXARR[] = {1240, 1110, 980, 850, 720, 590}; // array of maximum bhp values for each simulation set
const int LBHPMAXARR = sizeof (BHPMAXARR) / sizeof (BHPMAXARR[0]); // number of elements in the BHPMAXARR array

#define SKILLMIN 0 // skill minimum value
#define SKILLMAX 59 // skill maximum value
#define SKILLSTEP 15 // difference between two skill values in a simulation
const int SKILLMINARR[] = {0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44}; // array of minimum skill values for each simulation set
const int LSKILLMINARR = sizeof (SKILLMINARR) / sizeof (SKILLMINARR[0]); // number of elements in the SKILLMINARR array

#define ITERMIN 1 // simulation iteration minimum value [must be 1 for getindex() to work]
#define ITERMAX 10 // simulation iteration maximum value
#define LITER (ITERMAX-ITERMIN+1) // number of iteration values

#define CODEMIN 1 // simulation code minimum value [must be 1 for getindex() to work]
#define CODEMAX 1 // simulation code maximum value
#define LCODE (CODEMAX-CODEMIN+1) // number of code values

#define DRIVERMIN 1 // driver minimum value [must be 1 for getindex() to work]
#define DRIVERMAX 26 // driver maximum value
#define LDRIVER (DRIVERMAX-DRIVERMIN+1) // number of drivers


// Calculate index of lap times in the performance database file for a set of performance parameters
// indexing order: grip (ascending), bhp (descending), skill (ascending), track (ascending)
// index = 1 to LGRIP*LBHP*LSKILL*LTRACK if valid
// index = -1 if invalid
// GRIPMIN, TRACKMIN, ITERMIN, CODEMIN, DRIVERMIN must be set to 1 !!!
long long getindex (int grip, int bhp, int skill, int track)
{
    long long index;
    int bhp_index, skill_index;
    int i, j, i_bhp, i_skill, driver, temp;
    int LBHP = LBHPMAXARR * (LDRIVER / 2);
    int bhparr[LBHP]; // array of all bhp values
    int LSKILL = LSKILLMINARR * 2;
    int skillarr[LSKILL]; // array of all skill values

    // check performance parameters
    if ( (grip > GRIPMAX || grip < GRIPMIN) || (bhp > BHPMAX || bhp < BHPMIN || bhp % BHPSTEP != 0) || (skill > SKILLMAX || skill < SKILLMIN) || (track > TRACKMAX || track < TRACKMIN) )
    {
        return (long long) - 1;
    }

    // Calculate bhp index (bhp_index = 1 to LBHP if valid, bhp_index = -1 if invalid)
    // generate bhp array
    i = 0;
    for (i_bhp = 0; i_bhp < LBHPMAXARR; i_bhp++)
    {
        for (driver = DRIVERMIN; driver <= LDRIVER; driver += 2)
        {
            bhparr[i] = BHPMAXARR[i_bhp] - BHPSTEP * (int) ( (driver - 1) / 2);
            i++;
        }
    }
    // sort bhp array in descending order
    for (i = 0; i < LBHP - 1; i++)
    {
        for (j = 0; j < LBHP - 1 - i; j++)
        {
            if (bhparr[j + 1] > bhparr[j])
            {
                temp = bhparr[j];
                bhparr[j] = bhparr[j + 1];
                bhparr[j + 1] = temp;
            }
        }
    }
    // calculate bhp index
    bhp_index = -1;
    for (i_bhp = 0; i_bhp < LBHP; i_bhp++)
    {
        if (bhparr[i_bhp] == bhp)
        {
            bhp_index = i_bhp + 1;
            break;
        }
    }
    // check bhp index
    if (bhp_index == -1)
    {
        return (long long) - 1;
    }

    // Calculate skill index (skill_index = 1 to LSKILL if valid, skill_index = -1 if invalid)
    // generate skill array
    i = 0;
    for (i_skill = 0; i_skill < LSKILLMINARR; i_skill++)
    {
        for (driver = DRIVERMIN; driver <= DRIVERMIN + 1; driver++)
        {
            skillarr[i] = SKILLMINARR[i_skill] + SKILLSTEP * ( (driver - 1) % 2);
            i++;
        }
    }
    // sort skill array in ascending order
    for (i = 0; i < LSKILL - 1; i++)
    {
        for (j = 0; j < LSKILL - 1 - i; j++)
        {
            if (skillarr[j + 1] < skillarr[j])
            {
                temp = skillarr[j];
                skillarr[j] = skillarr[j + 1];
                skillarr[j + 1] = temp;
            }
        }
    }
    // calculate skill index
    skill_index = -1;
    for (i_skill = 0; i_skill < LBHP; i_skill++)
    {
        if (skillarr[i_skill] == skill)
        {
            skill_index = i_skill + 1;
            break;
        }
    }
    // check skill index
    if (skill_index == -1)
    {
        return (long long) - 1;
    }

    // calculate index
    index = (long long) track +
            (LTRACK * (skill_index - 1) ) +
            (LTRACK * LSKILL * (bhp_index - 1) ) +
            (LTRACK * LSKILL * LBHP * (grip - 1) );

    return index;
}


// Format time output [m:ss.ttt or s.ttt (if < 60 s)]
char *timeformat (int time, char *timestr)
{
    int min;
    int sec;
    int tho;

    // extract minutes, seconds and thousandths
    tho = time % 1000;
    time = (int) time / 1000;
    sec = time % 60;
    min = (int) time / 60;

    // determine output format
    if (min > 0)
    {
        sprintf (timestr, "%d:%02d.%03d", min, sec, tho);
    }
    else
    {
        sprintf (timestr, "%d.%03d", sec, tho);
    }

    // return formatted time string
    return timestr;
}


// Check if command-line argument is -h (help)
int arghelp (const char *arg)
{
    if ( (arg[0] == '-' || arg[0] == '/') && (arg[1] == '?' || arg[1] == 'h' || arg[1] == 'H') && arg[2] == '\0')
    {
        return 1;
    }
    else
    {
        return 0;
    }
}

// Check for unknown command-line arguments (starting with - or /)
int argunknown (const char *arg)
{
    if (arg[0] == '-' || arg[0] == '/')
    {
        return 1;
    }
    else
    {
        return 0;
    }
}

// Program usage instructions
void usage()
{
    printf ("\nLAPCALC v0.1.1 - F1GP Lap Time Calculator\n");
    printf ("Programmed by Hrvoje Stimac.\n\n");
    printf ("Usage: LAPCALC <grip>,<bhp>,<skill>,<track>\n");
    printf ("-?/-h                         Show this message.\n");
    printf ("<grip>,<bhp>,<skill>,<track>  Combination of 4 performance parameters.\n\n");
    printf ("<grip> - AI Grip value (1 to 22)\n");
    printf ("<bhp> - Car BHP value (multiple of 10, from 470 to 1240)\n");
    printf ("<skill> - Driver Qualifying Skill value (0 to 59)\n");
    printf ("<track> - Track value (1 to 16)\n\n");
    printf ("Example: LAPCALC 1,470,0,1\n");
    printf ("Example: LAPCALC 22,1240,59,16\n");
    return;
}



// MAIN PROGRAM
int main (int argc, char *argv[])
{
    FILE *fin;
    char inname[] = "times.bin"; //lap time database filename

    int grip, bhp, skill, track;
    int i, j;
    int begin, end, length, value;
    char valuestr[LFILE] = "";

    long long index;
    long long offset;

    int LTIME = LITER * LCODE; // number of lap time entries for each combination of the 4 performance parameters
    float times_read[LTIME]; // lap times read from database in sss.ttt format (stored as float)
    int times[LTIME], times_sorted[LTIME]; // lap times for processing in tttttt format (stored as integer, to avoid floating point errors)
    int i_median;
    int time_avg, time_dmax, time_stdev, time_median, time_max, time_min, time_temp;
    char *trackname[] = {"United States", "Brazil", "San Marino", "Monaco", "Canada", "Mexico", "France", "Great Britain", "Germany", "Hungary", "Belgium", "Italy", "Portugal", "Spain", "Japan", "Australia"};
    char timestr[10] = "";

    //==================================================================

    // Process command line arguments

    switch (argc)
    {
    // 1 argument
    case 2:
        // help argument
        if (arghelp (argv[1]) )
        {
            usage();
            return EXIT_SUCCESS;
        }
        // unknown argument
        if (argunknown (argv[1]) )
        {
            printf ("\nERROR! Unknown argument passed: %s\n", argv[1]);
            usage();
            return EXIT_FAILURE;
        }
        // performance parameters argument
        // continue to main program
        break;

    // no or too many arguments
    default:
        // too many arguments
        if (argc != 1)
        {
            printf ("\nERROR! Too many arguments passed: %d\n", argc - 1);
        }
        // no arguments
        usage();
        return EXIT_FAILURE;
    }

    //==================================================================

    // Extract performance parameters from command-line argument

    begin = 0; // performance parameter begin index
    end = 0; // performance parameter end index
    length = 0; // performance parameter length
    value = 0; // number of performance parameters found
    // go through the whole command-line argument string
    for (i = 0; i < strlen (argv[1]); i++)
    {
        // current character in string is invalid (not digit or comma)
        if (!isdigit (argv[1][i]) && argv[1][i] != ',')
        {
            printf ("\nERROR! Invalid format of performance parameters: %s\n", argv[1]);
            return EXIT_FAILURE;
        }
        // next character in string is comma or end of string
        if (argv[1][i + 1] == ',' || i + 1 == strlen (argv[1]) )
        {
            end = i;
            length = end - begin + 1;
            // performance parameter length is zero or parameter is too long to store
            if (length == 0 || length + 1 >= LFILE)
            {
                printf ("\nERROR! Invalid format of performance parameters: %s\n", argv[1]);
                return EXIT_FAILURE;
            }
            // copy performance parameter to temporary string
            strncpy (valuestr, &argv[1][begin], length);
            valuestr[length] = '\0';
            value++;

            // convert performance parameter into number
            switch (value)
            {
            // grip
            case 1:
                grip = atoi (valuestr);
                break;

            // bhp
            case 2:
                bhp = atoi (valuestr);
                break;

            // skill
            case 3:
                skill = atoi (valuestr);
                break;

            // track
            case 4:
                track = atoi (valuestr);
                break;

            // too many performance parameters
            default:
                printf ("\nERROR! Too many performance parameters passed: %s\n", argv[1]);
                return EXIT_FAILURE;
            }

            begin = i + 2; // skip to after comma
        }
    }

    // too few performance parameters
    if (value < 4)
    {
        printf ("\nERROR! Too few performance parameters passed: %s\n", argv[1]);
        return EXIT_FAILURE;
    }

    //==================================================================

    // Check performance parameters

    // check grip value
    if (grip > GRIPMAX || grip < GRIPMIN)
    {
        printf ("\nERROR! Invalid <grip> value: %d\n", grip);
        return EXIT_FAILURE;
    }

    // check bhp value
    if (bhp > BHPMAX || bhp < BHPMIN || bhp % BHPSTEP)
    {
        printf ("\nERROR! Invalid <bhp> value: %d\n", bhp);
        return EXIT_FAILURE;
    }

    // check skill value
    if (skill > SKILLMAX || skill < SKILLMIN)
    {
        printf ("\nERROR! Invalid <skill> value: %d\n", skill);
        return EXIT_FAILURE;
    }

    // check track value
    if (track > TRACKMAX || track < TRACKMIN)
    {
        printf ("\nERROR! Invalid <track> value: %d\n", track);
        return EXIT_FAILURE;
    }

    //==================================================================

    // Read lap times from database file

    // open input database file for reading
    fin = fopen (inname, "rb");
    if (fin == NULL)
    {
        printf ("\nERROR! Cannot open database file: %s\n", inname);
        return EXIT_FAILURE;
    }

    // calculate database file index
    index = getindex (grip, bhp, skill, track);
    if (index == -1)
    {
        printf ("\nERROR! Invalid index value.\n");
        return EXIT_FAILURE;
    }

    // reset lap time values
    for (i = 0; i < LTIME; i++)
    {
        times_read[i] = (float) - 1;
    }

    // read lap times from database file
    offset = (long long) sizeof (times_read) * (index - 1);
    fseek (fin, offset, SEEK_SET);
    if (fread (times_read, sizeof (times_read), 1, fin) != 1)
    {
        printf ("\nERROR! Binary read from database failed: %s\n", inname);
        printf ("offset: %lld, size: %d\n", (long long) offset, sizeof (times_read) );
        return EXIT_FAILURE;
    }

    //close input file
    fclose (fin);

    //==================================================================

    // Correct and check lap time values

    // convert lap times from float to integer to correct floating point errors
    for (i = 0; i < LTIME; i++)
    {
        times[i] = (int) round (times_read[i] * 1000);
    }

    // check for invalid lap time values
    for (i = 0; i < LTIME; i++)
    {
        // invalid lap time value found
        if (times[i] < 0)
        {
            break;
        }
    }

    // invalid lap times exist
    if (i < LTIME)
    {
        printf ("\nERROR! Invalid lap times read from database: %s\n", inname);
        // output all lap times
        for (i = 0; i < LTIME; i++)
        {
            // invalid lap time value
            if (times[i] < 0)
            {
                printf ("(Invalid!)\n");
            }
            // correct lap time value
            else
            {
                printf ("%s (OK)\n", timeformat (times[i], timestr) );
            }
        }
        return EXIT_FAILURE;
    }

    //==================================================================

    // Process lap time data

    // create a sorted times array
    for (i = 0; i < LTIME; i++)
    {
        times_sorted[i] = times[i];
    }
    // sort array in ascending order
    for (i = 0; i < LTIME; i++)
    {
        for (j = 0; j < LTIME - 1 - i; j++)
        {
            if (times_sorted[j] > times_sorted[j + 1])
            {
                time_temp = times_sorted[j];
                times_sorted[j] = times_sorted[j + 1];
                times_sorted[j + 1] = time_temp;
            }
        }
    }

    // calculate lap time maximum and minimum
    time_max = times_sorted[LTIME - 1];
    time_min = times_sorted[0];

    // calculate lap time average
    time_avg = 0;
    for (i = 0; i < LTIME; i++)
    {
        time_avg += times[i];
    }
    time_avg = (int) round ( (double) time_avg / LTIME);

    // calculate lap time maximum deviation
    time_dmax = time_max - time_avg;
    time_temp = time_avg - time_min;
    if (time_temp > time_dmax)
    {
        time_dmax = time_temp;
    }

    // calculate median
    i_median = (int) (LTIME / 2);
    if (LTIME % 2)
    {
        time_median = times_sorted[i_median];
    }
    else
    {
        time_median = (int) round ( (double) (times_sorted[i_median - 1] + times_sorted[i_median]) / 2);
    }

    // calculate lap time standard deviation
    time_stdev = 0;
    for (i = 0; i < LTIME; i++)
    {
        time_stdev += (int) round ( (double) pow ( (double) (times[i] - time_avg), (double) 2) );
    }
    time_stdev = (int) round ( (double) sqrt ( (double) time_stdev / LTIME) );

    //==================================================================

    // Output data

    // output performance parameters
    printf ("\nGrip: %d, ", grip);
    printf ("BHP: %d, ", bhp);
    printf ("Skill: %d, ", skill);
    printf ("Track: %d (%s)\n\n", track, trackname[track - 1]);

    // output processed lap time data
    printf ("Average time: %s\n", timeformat (time_avg, timestr) );
    printf ("Maximum absolute deviation: %s\n", timeformat (time_dmax, timestr) );
    printf ("Standard deviation: %s\n", timeformat (time_stdev, timestr) );
    printf ("Median time: %s\n", timeformat (time_median, timestr) );
    printf ("Maximum time: %s\n", timeformat (time_max, timestr) );
    printf ("Minimum time: %s\n\n", timeformat (time_min, timestr) );

    // output individual lap times
    printf ("Lap times:\n");
    for (i = 0; i < LTIME; i++)
    {
        printf ("%s\n", timeformat (times[i], timestr) );
    }

    //==================================================================

    return EXIT_SUCCESS;
}
