// TrackPropertiesView.cpp : implementation file
//

#include "stdafx.h"
#include "TrackEd.h"
#include "TrackSeg.h"
#include "Track.h"
#include "TrackPropertiesView.h"
#include "ChangePropertyDialog.h"
#include "EditCommandDialog.h"
#include "EditTrackSegmentDialog.h"
#include "EditBestLineSegmentDialog.h"
#include "TrackHeader.h"
#include "BestLineSeg.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CTrackPropertiesView

IMPLEMENT_DYNCREATE(CTrackPropertiesView, CFormView)

CTrackPropertiesView::CTrackPropertiesView()
	: CFormView(CTrackPropertiesView::IDD)
{
	//{{AFX_DATA_INIT(CTrackPropertiesView)
	//}}AFX_DATA_INIT
	m_hTrackHeader = 0;
	m_hTrackSegments = 0;
	m_hPitlaneSegments = 0;
}

CTrackPropertiesView::~CTrackPropertiesView()
{
}

void CTrackPropertiesView::DoDataExchange(CDataExchange* pDX)
{
	CFormView::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CTrackPropertiesView)
	DDX_Control(pDX, IDC_DATA_TREE, m_treeData);
	DDX_Control(pDX, IDC_TRACK_HEADER_PROPS, m_listTrackHeaderProps);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CTrackPropertiesView, CFormView)
	//{{AFX_MSG_MAP(CTrackPropertiesView)
	ON_NOTIFY(NM_DBLCLK, IDC_TRACK_HEADER_PROPS, OnDblclkTrackHeaderProps)
	ON_WM_CONTEXTMENU()
	ON_COMMAND(ID_INSERT_COMMAND, OnInsertCommand)
	ON_COMMAND(ID_EDIT_COMMAND, OnEditCommand)
	ON_COMMAND(ID_EDIT_PIT_SEGMENT, OnEditPitSegment)
	ON_COMMAND(ID_EDIT_TRACK_SEGMENT, OnEditTrackSegment)
	ON_COMMAND(ID_DELETE_COMMAND, OnDeleteCommand)
	ON_NOTIFY(TVN_SELCHANGED, IDC_DATA_TREE, OnSelchangedDataTree)
	ON_COMMAND(ID_EDIT_BEST_LINE_SEGMENT, OnEditBestLineSegment)
	ON_COMMAND(ID_INSERT_BEST_LINE_SEGMENT, OnInsertBestLineSegment)
	ON_COMMAND(ID_DELETE_BEST_LINE_SEGMENT, OnDeleteBestLineSegment)
	ON_COMMAND(ID_DELETE_PIT_SEGMENT, OnDeletePitSegment)
	ON_COMMAND(ID_DELETE_TRACK_SEGMENT, OnDeleteTrackSegment)
	ON_COMMAND(ID_INSERT_PIT_SEGMENT, OnInsertPitSegment)
	ON_COMMAND(ID_INSERT_TRACK_SEGMENT, OnInsertTrackSegment)
	ON_COMMAND(ID_PREFERENCES_NEW_CCLINE, OnPreferencesNewCcline)
	ON_UPDATE_COMMAND_UI(ID_PREFERENCES_NEW_CCLINE, OnUpdatePreferencesNewCcline)
	ON_COMMAND(ID_PREFERENCES_OLD_CCLINE, OnPreferencesOldCcline)
	ON_UPDATE_COMMAND_UI(ID_PREFERENCES_OLD_CCLINE, OnUpdatePreferencesOldCcline)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CTrackPropertiesView diagnostics

#ifdef _DEBUG
void CTrackPropertiesView::AssertValid() const
{
	CFormView::AssertValid();
}

void CTrackPropertiesView::Dump(CDumpContext& dc) const
{
	CFormView::Dump(dc);
}

CTrack* CTrackPropertiesView::GetDocument() // non-debug version is inline
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CTrack)));
	return (CTrack*)m_pDocument;
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CTrackPropertiesView message handlers

void CTrackPropertiesView::OnUpdate(CView* pSender, LPARAM lHint, CObject* pHint) 
{
	CTrackSeg * pTs = NULL;
	HTREEITEM hItem = NULL;
	CString strAux;
	// ----------------------------------------------------------------------
	switch ( lHint )
	{
	case UPD_HINT_TRACK_SEG_INSERTED:
		pTs = reinterpret_cast<CTrackSeg *>( pHint );
		// New segment was inserted after pTs.
		if ( pTs->m_pNext != NULL )
		{
			// Find tree item for segment.
			hItem = GetTreeItem( m_hTrackSegments, pTs );
			if ( hItem != NULL )
			{
				hItem = m_treeData.InsertItem( "Dummy", m_hTrackSegments, hItem );
				m_treeData.SetItemData( hItem, (DWORD) pTs->m_pNext );
				// add commands
				UpdateTreeCommands( hItem, pTs->m_pNext->cmd );
			}
			// Relabel the subtree
			RelabelTrackSegments(m_hTrackSegments);
		}
		break;
	case UPD_HINT_PIT_SEG_INSERTED:
		pTs = reinterpret_cast<CTrackSeg *>( pHint );
		// New segment was inserted after pTs.
		if ( pTs->m_pNext != NULL )
		{
			// Find tree item for segment.
			hItem = GetTreeItem( m_hPitlaneSegments, pTs );
			if ( hItem != NULL )
			{
				hItem = m_treeData.InsertItem( "Dummy", m_hPitlaneSegments, hItem );
				m_treeData.SetItemData( hItem, (DWORD) pTs->m_pNext );
				// add commands
				UpdateTreeCommands( hItem, pTs->m_pNext->cmd );
			}
			// Relabel the subtree
			RelabelTrackSegments(m_hPitlaneSegments);
		}
		break;

	case UPD_HINT_TRACK_SEG_DELETED:
		pTs = reinterpret_cast<CTrackSeg *>( pHint );
		// Find tree item for segment.
		hItem = GetTreeItem( m_hTrackSegments, pTs );
		if ( hItem != NULL )
		{
			m_treeData.DeleteItem( hItem );
			// Relabel the subtree
			RelabelTrackSegments(m_hTrackSegments);
		}
		break;
	case UPD_HINT_PIT_SEG_DELETED:
		pTs = reinterpret_cast<CTrackSeg *>( pHint );
		// Find tree item for segment.
		hItem = GetTreeItem( m_hPitlaneSegments, pTs );
		if ( hItem != NULL )
		{
			m_treeData.DeleteItem( hItem );
			// Relabel the subtree
			RelabelTrackSegments(m_hPitlaneSegments);
		}
		break;

	default:
		HTREEITEM hItem = m_treeData.GetSelectedItem();
		// Only the selected item could be altered. Check type and update.
		HTREEITEM hItemParent = m_treeData.GetParentItem( hItem );
		if ( hItemParent == NULL )
		{
			// root item
		}
		else if ( hItemParent == m_hTrackSegments )
		{
			// it is a track item
			UpdateTrackProps( reinterpret_cast<CTrackSeg*>(m_treeData.GetItemData( hItem )) );
		}
		else if ( hItemParent == m_hPitlaneSegments )
		{
			// it is a pitlane item
			UpdatePitlaneProps( reinterpret_cast<CTrackSeg*>(m_treeData.GetItemData( hItem )) );
		}
		else if ( hItemParent == m_hBestLineSegments )
		{
			// it is a best line item
			// nothing to do right now.... @@@
		}
		else
		{
			// check if parent item is some kind of track segment
			CObject * pObj = reinterpret_cast<CObject *>( m_treeData.GetItemData( hItemParent ) );
			if ( pObj != NULL )
				if ( pObj->IsKindOf( RUNTIME_CLASS( CTrackSeg ) ) )
				{
					// Selected item then should be a command.
					// Update the item text to reflect possible type changes
					UpdateCommandProps( hItem );
				}
		}
		
		m_treeData.EnsureVisible( m_treeData.GetSelectedItem() );

	} // switch( lHint )

	UpdateData( FALSE );
}

void CTrackPropertiesView::OnInitialUpdate() 
{
	CFormView::OnInitialUpdate();

	// Construct Track Properties
	ConstructTrackProps();
	UpdateTrackProps(GetDocument()->GetTrackSegs());

	// Construct Pitlane Properties
	ConstructPitlaneProps();
	UpdatePitlaneProps(GetDocument()->GetPitSegs());

	// Construct Best Line Properties
	ConstructBestLineProps();

	// Construct Track Header Properties
	ConstructTrackDataProps();
	UpdateTrackDataProps(GetDocument()->m_TrackHeader);
}

void CTrackPropertiesView::ConstructPitlaneProps()
{
	// insert pitlane root in tree
	m_hPitlaneSegments = m_treeData.InsertItem( "Pitlane segments" );
	// insert pitlane segments under root
	CTrackSeg * pTs = GetDocument()->GetPitSegs();
	HTREEITEM hItem;
	int i = 1;
	CString strAux;
	while ( pTs != NULL )
	{
		strAux.Format( "%d", i++ );
		if ( pTs->curvature == 0 )
			strAux += " Straight";
		else if ( pTs->curvature > 0 )
			strAux += " Turn right";
		else
			strAux += " Turn left";
		hItem = m_treeData.InsertItem( strAux, m_hPitlaneSegments );
		m_treeData.SetItemData( hItem, (DWORD) pTs );
		// add commands
		UpdateTreeCommands( hItem, pTs->cmd );
		pTs = pTs->m_pNext; // next segment
	}
}

/// Make sure this pitlane segment is visible and expanded

void CTrackPropertiesView::UpdatePitlaneProps(CTrackSeg * pTs)
{
	if ( GetDocument()->GetPitSegs() == NULL )
		// no pitlane defined
		return;

	if ( m_hPitlaneSegments == 0 )
		return;

	HTREEITEM hItem = GetTreeItem( m_hPitlaneSegments, pTs );
	if ( hItem != NULL )
	{
		CString strAux;
		// Get current text
		strAux = m_treeData.GetItemText( hItem );
		// First part is number of segment: leave as is
		strAux = strAux.Left( strAux.Find( ' ' ) );
		// Add segment description
		if ( pTs->curvature == 0 )
			strAux += " Straight";
		else if ( pTs->curvature > 0 )
			strAux += " Turn right";
		else
			strAux += " Turn left";
		m_treeData.SetItemText( hItem, strAux );
/*
		// remove and add commands
		HTREEITEM hCmdItem = m_treeData.GetChildItem( hItem );
		HTREEITEM hCmdItemNext;
		while ( hCmdItem != NULL )
		{
			hCmdItemNext = m_treeData.GetNextSiblingItem( hCmdItem );
			m_treeData.DeleteItem( hCmdItem );
			hCmdItem = hCmdItemNext;
		}
		UpdateTreeCommands( hItem, pTs->cmd );
*/
	}
}

void CTrackPropertiesView::ConstructTrackProps()
{
	// insert track root in tree
	m_hTrackSegments = m_treeData.InsertItem( "Track segments" );
	// insert track segments under root
	CTrackSeg * pTs = GetDocument()->GetTrackSegs();
	HTREEITEM hItem;
	int i = 1;
	CString strAux;
	while ( pTs != NULL )
	{
		strAux.Format( "%d", i++ );
		if ( pTs->curvature == 0 )
			strAux += " Straight";
		else if ( pTs->curvature > 0 )
			strAux += " Turn right";
		else
			strAux += " Turn left";
		hItem = m_treeData.InsertItem( strAux, m_hTrackSegments );
		m_treeData.SetItemData( hItem, (DWORD) pTs );
		// add commands
		UpdateTreeCommands( hItem, pTs->cmd );
		pTs = pTs->m_pNext; // next segment
	}
}

void CTrackPropertiesView::UpdateTrackProps(CTrackSeg * pTs)
{
	if ( GetDocument()->GetTrackSegs() == NULL )
		// no track defined
		return;

	if ( m_hTrackSegments == 0 )
		return;

	HTREEITEM hItem = GetTreeItem( m_hTrackSegments, pTs );
	if ( hItem != NULL )
	{
		CString strAux;
		// Get current text
		strAux = m_treeData.GetItemText( hItem );
		// First part is number of segment: leave as is
		strAux = strAux.Left( strAux.Find( ' ' ) );
		// Add segment description
		if ( pTs->curvature == 0 )
			strAux += " Straight";
		else if ( pTs->curvature > 0 )
			strAux += " Turn right";
		else
			strAux += " Turn left";
		m_treeData.SetItemText( hItem, strAux );

		// Update the command list
		UpdateTreeCommands( hItem, pTs->cmd );
	}
}

void CTrackPropertiesView::ConstructTrackDataProps()
{
	m_listTrackHeaderProps.InsertColumn(0, "Label", LVCFMT_LEFT, 60);
	m_listTrackHeaderProps.InsertColumn(1, "Value", LVCFMT_LEFT, 50, 1);

	m_listTrackHeaderProps.InsertItem(  0, "sAng:" );
	m_listTrackHeaderProps.InsertItem(  1, "sHei:" );
	m_listTrackHeaderProps.InsertItem(  2, "stY:" );
	m_listTrackHeaderProps.InsertItem(  3, "stX:" );
	m_listTrackHeaderProps.InsertItem(  4, "stZ:" );
	m_listTrackHeaderProps.InsertItem(  5, "sWidth:" );
	m_listTrackHeaderProps.InsertItem(  6, "PoleWi:" );
	m_listTrackHeaderProps.InsertItem(  7, "Pitside:" );
	m_listTrackHeaderProps.InsertItem(  8, "Surround:" );
	m_listTrackHeaderProps.InsertItem(  9, "r. Bank:" );
	m_listTrackHeaderProps.InsertItem( 10, "l. Bank:" );
	m_listTrackHeaderProps.InsertItem( 11, "KerbCNum:" );
	m_listTrackHeaderProps.InsertItem( 12, "Unk1:" );
	m_listTrackHeaderProps.InsertItem( 13, "KerbTC1:" );
	m_listTrackHeaderProps.InsertItem( 14, "KerbBC1:" );
	m_listTrackHeaderProps.InsertItem( 15, "Unk2:" );
	m_listTrackHeaderProps.InsertItem( 16, "LapNum:" );
	m_listTrackHeaderProps.InsertItem( 17, "KerbTC2:" );
	m_listTrackHeaderProps.InsertItem( 18, "KerbBC2:" );
}

void CTrackPropertiesView::UpdateTrackDataProps(const CTrackHeader &th)
{
	CString strAux;
	strAux.Format( "%d", th.startangle );
	m_listTrackHeaderProps.SetItemText(  0, 1, strAux );
	strAux.Format( "%d", th.startheight );
	m_listTrackHeaderProps.SetItemText(  1, 1, strAux );
	strAux.Format( "%d", th.startY );
	m_listTrackHeaderProps.SetItemText(  2, 1, strAux );
	strAux.Format( "%d", th.startZ );
	m_listTrackHeaderProps.SetItemText(  3, 1, strAux );
	strAux.Format( "%d", th.startX );
	m_listTrackHeaderProps.SetItemText(  4, 1, strAux );
	strAux.Format( "%d", th.startWidth );
	m_listTrackHeaderProps.SetItemText(  5, 1, strAux );
	strAux.Format( "%d", th.poleWidth );
	m_listTrackHeaderProps.SetItemText(  6, 1, strAux );
	strAux.Format( "%d", th.pitside );
	m_listTrackHeaderProps.SetItemText(  7, 1, strAux );
	strAux.Format( "%d", th.trsurround );
	m_listTrackHeaderProps.SetItemText(  8, 1, strAux );
	strAux.Format( "%d", th.rbank );
	m_listTrackHeaderProps.SetItemText(  9, 1, strAux );
	strAux.Format( "%d", th.lbank );
	m_listTrackHeaderProps.SetItemText( 10, 1, strAux );
	strAux.Format( "%d", th.kerbcnum );
	m_listTrackHeaderProps.SetItemText( 11, 1, strAux );
	strAux.Format( "%d", th.unk1 );
	m_listTrackHeaderProps.SetItemText( 12, 1, strAux );
	strAux.Format( "%d", th.kerbtopcolor );
	m_listTrackHeaderProps.SetItemText( 13, 1, strAux );
	strAux.Format( "%d", th.kerbbottomcolor );
	m_listTrackHeaderProps.SetItemText( 14, 1, strAux );
	strAux.Format( "%d", th.unk2 );
	m_listTrackHeaderProps.SetItemText( 15, 1, strAux );
	strAux.Format( "%d", -1 ); // Lap number @@@
	m_listTrackHeaderProps.SetItemText( 16, 1, strAux );
	strAux.Format( "%d", th.kerbtopcolor2 );
	m_listTrackHeaderProps.SetItemText( 17, 1, strAux );
	strAux.Format( "%d", th.kerbbottomcolor2 );
	m_listTrackHeaderProps.SetItemText( 18, 1, strAux );
}

void CTrackPropertiesView::OnDblclkTrackHeaderProps(NMHDR* pNMHDR, LRESULT* pResult) 
{
	NMITEMACTIVATE * pNMITEMACTIVATE;
	pNMITEMACTIVATE = reinterpret_cast<NMITEMACTIVATE *>( pNMHDR );

	// allow change of current value
	int nItem = pNMITEMACTIVATE->iItem;
	CString strName = m_listTrackHeaderProps.GetItemText( nItem, 0 );
	CString strValue = m_listTrackHeaderProps.GetItemText( nItem, 1 );
	int nValue = atoi( strValue );

	CChangePropertyDialog * pDlg = new CChangePropertyDialog( this );
	pDlg->SetName( strName );
	pDlg->SetValue( strValue );
	if ( pDlg->DoModal() == IDOK )
	{
		strValue = pDlg->GetValue();
		// set changed value into list ctrl
		m_listTrackHeaderProps.SetItemText( nItem, 1, strValue );
		nValue = atoi( strValue );
		switch( nItem )
		{
		case 0: // Start Angle
			GetDocument()->m_TrackHeader.startangle = nValue;
			break;
		case 1: // Start Height
			GetDocument()->m_TrackHeader.startheight = nValue;
			break;
		case 2: // 
			GetDocument()->m_TrackHeader.startY = nValue;
			break;
		case 3: // 
			GetDocument()->m_TrackHeader.startZ = nValue;
			break;
		case 4: // 
			GetDocument()->m_TrackHeader.startX = nValue;
			break;
		case 5: // 
			GetDocument()->m_TrackHeader.startWidth = nValue;
			break;
		case 6: // 
			GetDocument()->m_TrackHeader.poleWidth = nValue;
			break;
		case 7: // 
			GetDocument()->m_TrackHeader.pitside = nValue;
			break;
		case 8: // 
			GetDocument()->m_TrackHeader.trsurround = nValue;
			break;
		case 9: // 
			GetDocument()->m_TrackHeader.rbank = nValue;
			break;
		case 10: // 
			GetDocument()->m_TrackHeader.lbank = nValue;
			break;
		case 11: // 
			GetDocument()->m_TrackHeader.kerbcnum = nValue;
			break;
		case 12: // 
			GetDocument()->m_TrackHeader.unk1 = nValue;
			break;
		case 13: // 
			GetDocument()->m_TrackHeader.kerbtopcolor = nValue;
			break;
		case 14: // 
			GetDocument()->m_TrackHeader.kerbbottomcolor = nValue;
			break;
		case 15: // 
			GetDocument()->m_TrackHeader.unk2 = nValue;
			break;
		case 16: // LapNumber - where is it stored?
			// @@@  = nValue;
			break;
		case 17: // 
			GetDocument()->m_TrackHeader.kerbtopcolor2 = nValue;
			break;
		case 18: // 
			GetDocument()->m_TrackHeader.kerbbottomcolor2 = nValue;
			break;
		}
		// recalculate all track data
		GetDocument()->RefreshTrack();
		GetDocument()->RefreshCCLine();
		GetDocument()->SetModifiedFlag();
		GetDocument()->UpdateAllViews( this );
	}
	delete pDlg;
	pDlg = NULL;

	*pResult = NULL;
}

/**
  Update a list of commands.
  Missing commands are inserted, commands that are no longer in the list are
  deleted, and all text labels are updated.
*/

void CTrackPropertiesView::UpdateTreeCommands(HTREEITEM hItem, CCommand *pCmdList)
{
	CString strAux;
	HTREEITEM hCmd;
	HTREEITEM hRemove;
	hCmd = m_treeData.GetChildItem( hItem );
	while ( pCmdList != NULL )
	{
		// Generate text label for next command
		strAux.Format( "0x%02X: %s", pCmdList->cmd, pCmdList->GetName() );
		// Check if current tree item corresponds with command
		if ( hCmd != NULL )
		{
			// There are still tree items left
			if ( m_treeData.GetItemData( hCmd ) == (DWORD) pCmdList )
			{
				// Only update the label
				m_treeData.SetItemText( hCmd, strAux );
			}
			else
			{
				// look through the rest of the list to see if element exists
				HTREEITEM hItemAux = hCmd;
				do
				{
					hItemAux = m_treeData.GetNextSiblingItem( hItemAux );
					if ( hItemAux != NULL )
					{
						if ( m_treeData.GetItemData( hItemAux ) == (DWORD) pCmdList )
						{
							// Found the corresponding item. Everything from
							// hCmd to hItemAux has to be removed from the list
							while ( hCmd != hItemAux )
							{
								hRemove = hCmd;
								hCmd = m_treeData.GetNextSiblingItem( hCmd );
								m_treeData.DeleteItem( hRemove );
							}
							// Update the text label
							m_treeData.SetItemText( hCmd, strAux );
							hItemAux = NULL; // stop loop
						}
					}
				} while ( hItemAux != NULL );
				// Check once again, if hCmd is the right item.
				if ( m_treeData.GetItemData( hCmd ) == (DWORD) pCmdList )
				{
					// Yes: some tree items were removed. We are ready.
				}
				else
				{
					// Did not find a corresponding tree item.
					// Create new one and insert into tree.
					hItemAux = m_treeData.GetPrevSiblingItem( hCmd );
					if ( hItemAux != NULL )
						// Insert after hItemAux
						hCmd = m_treeData.InsertItem(TVIF_TEXT, strAux, 
									0, 0, 0, 0, 0,
									hItem, hItemAux );
					else
						// Insert as first item beneath "hItem"
						hCmd = m_treeData.InsertItem( strAux, hItem, TVI_FIRST );
					// store pointer to command object
					m_treeData.SetItemData( hCmd, reinterpret_cast<DWORD>( pCmdList ) );
				}
			}
		}
		else
		{
			// No more tree items left: add command as last item in list
			hCmd = m_treeData.InsertItem( strAux, hItem );
			// store pointer to command object
			m_treeData.SetItemData( hCmd, reinterpret_cast<DWORD>( pCmdList ) );
		}

		// Go to next item in tree and list
		hCmd = m_treeData.GetNextSiblingItem( hCmd );
		pCmdList = pCmdList->m_pNext;
	} // while commands left

	// Reached end of command list. When we have tree items left, they are removed
	// from the list
	while ( hCmd != NULL )
	{
		hRemove = hCmd;
		hCmd = m_treeData.GetNextSiblingItem( hCmd );
		m_treeData.DeleteItem( hRemove );
	}
}

/// Create context menu
void CTrackPropertiesView::OnContextMenu(CWnd* pWnd, CPoint point) 
{
	// Check if mouse button was pushed over data tree control
	if ( pWnd->m_hWnd == m_treeData.m_hWnd )
	{
		// context menu for the tree control
		// convert point to client coordinates
		CPoint ptClient = point;
		m_treeData.ScreenToClient( &ptClient );
		// get item over which the button was clicked
		HTREEITEM hItem = m_treeData.HitTest( ptClient );
		if ( hItem != NULL )
		{
			// construct context menu
			CMenu menu;
			menu.CreatePopupMenu();
			// find out which kind of item it is by looking for parent
			HTREEITEM hItemParent;
			hItemParent = m_treeData.GetParentItem( hItem );
			if ( hItemParent == NULL )
			{
				// root item
			}
			else if ( hItemParent == m_hTrackSegments )
			{
				// it is a track item
				menu.InsertMenu( -1, MF_BYPOSITION, ID_INSERT_COMMAND, "Insert Command" );
				menu.InsertMenu( -1, MF_BYPOSITION, ID_EDIT_TRACK_SEGMENT, "Edit Track Segment" );
			}
			else if ( hItemParent == m_hPitlaneSegments )
			{
				// it is a pitlane item
				menu.InsertMenu( -1, MF_BYPOSITION, ID_INSERT_COMMAND, "Insert Command" );
				menu.InsertMenu( -1, MF_BYPOSITION, ID_EDIT_PIT_SEGMENT, "Edit Pit Segment" );
			}
			else if ( hItemParent == m_hBestLineSegments )
			{
				// it is a best line item
				menu.InsertMenu( -1, MF_BYPOSITION, ID_EDIT_BEST_LINE_SEGMENT, "Edit Best Line Segment" );
				menu.InsertMenu( -1, MF_BYPOSITION, ID_INSERT_BEST_LINE_SEGMENT, "Insert Best Line Segment" );
				menu.InsertMenu( -1, MF_BYPOSITION, ID_DELETE_BEST_LINE_SEGMENT, "Delete Best Line Segment" );
			}
			else
			{
				// check if parent item is some kind of track segment
				CObject * pObj = reinterpret_cast<CObject *>( m_treeData.GetItemData( hItemParent ) );
				if ( pObj != NULL )
					if ( pObj->IsKindOf( RUNTIME_CLASS( CTrackSeg ) ) )
					{
						// selected item then should be a command
						menu.InsertMenu( -1, MF_BYPOSITION, ID_EDIT_COMMAND, "Edit Command" );
						menu.InsertMenu( -1, MF_BYPOSITION, ID_DELETE_COMMAND, "Delete Command" );
					}
			}

			// When menu has at least one item, open it.
			if ( menu.GetMenuItemCount() > 0 )
				menu.TrackPopupMenu( TPM_LEFTALIGN | TPM_RIGHTBUTTON,
									 point.x, point.y, this ); // use screen coordinates again
		}

	}
}

/// a new command is to be inserted into the current track or pit lane segment

void CTrackPropertiesView::OnInsertCommand() 
{
	GetDocument()->InsertCommand();
}

/// command is going to be edited.

void CTrackPropertiesView::OnEditCommand() 
{
	// first, get the currently selected command from the tree control
	HTREEITEM hItem;
	hItem = m_treeData.GetSelectedItem();
	CObject * pObj;
	pObj = reinterpret_cast<CObject *>( m_treeData.GetItemData( hItem ) );
	if ( !pObj->IsKindOf( RUNTIME_CLASS( CCommand ) ) )
		// currently selected item is not a command
		return;
	CCommand * pCmd = reinterpret_cast<CCommand*>(pObj);
	EditCommand( pCmd );
}

void CTrackPropertiesView::EditCommand(CCommand *pCmd)
{
	CEditCommandDialog * pDlg = new CEditCommandDialog( this );
	pDlg->SetCommand( pCmd );
	if ( pDlg->DoModal() == IDOK )
	{
		GetDocument()->UpdateAllViews( NULL );
	}
	delete pDlg;
}

// change properties of a pit lane segment

void CTrackPropertiesView::OnEditPitSegment() 
{
	// first, get the currently selected pit lane segment from the tree control
	HTREEITEM hItem;
	hItem = m_treeData.GetSelectedItem();
	if ( hItem == NULL )
		return;
	// Check if it is a pit lane segment
	CObject * pObj = reinterpret_cast<CObject *>(m_treeData.GetItemData( hItem ));
	// Object should be of type CTrackSeg
	if ( !pObj->IsKindOf( RUNTIME_CLASS( CTrackSeg ) ) )
		// not a track segment
		return;
	// Check if track segment.belongs to pit lane
	if ( m_treeData.GetParentItem( hItem ) != m_hPitlaneSegments )
		return;
	// Selected object should now be the current track segment
	ASSERT( GetDocument()->GetCurrentPitSeg() == reinterpret_cast<CTrackSeg *>(pObj) );
	// now edit pit lane segment
	EditTrackSegment( reinterpret_cast<CTrackSeg*>(pObj) );
}


void CTrackPropertiesView::EditTrackSegment(CTrackSeg *pTs)
{
	CEditTrackSegmentDialog * pDlg = new CEditTrackSegmentDialog( this );
	pDlg->SetTrackSeg( pTs );
	if ( pDlg->DoModal() == IDOK )
	{
		GetDocument()->SetModifiedFlag();
		GetDocument()->RefreshTrack();
		GetDocument()->UpdateAllViews( NULL );
	}
	delete pDlg;
}

void CTrackPropertiesView::OnEditTrackSegment() 
{
	// first, get the currently selected track segment from the tree control
	HTREEITEM hItem;
	hItem = m_treeData.GetSelectedItem();
	if ( hItem == NULL )
		return;
	CObject * pObj = reinterpret_cast<CObject *>(m_treeData.GetItemData( hItem ));
	if ( !pObj->IsKindOf( RUNTIME_CLASS( CTrackSeg ) ) )
		// not a track segment
		return;
	// Check if it is a track segment.
	if ( m_treeData.GetParentItem( hItem ) != m_hTrackSegments )
		return;
	// selected object should now be the current track segment
	ASSERT( GetDocument()->GetCurrentTrackSeg() == reinterpret_cast<CTrackSeg *>(pObj) );
	// now edit track segment
	EditTrackSegment( reinterpret_cast<CTrackSeg*>(pObj) );
}

/// identify currently selected command and remove it from list

void CTrackPropertiesView::OnDeleteCommand() 
{
	// first, get the currently selected command from the tree control
	HTREEITEM hItem;
	hItem = m_treeData.GetSelectedItem();
	if ( hItem == NULL )
		return;
	// is this a command item?
	CObject * pObj = reinterpret_cast<CObject *>(m_treeData.GetItemData( hItem ));
	if ( !pObj->IsKindOf( RUNTIME_CLASS( CCommand ) ) )
		// it is no command
		return;
	
	// now determine which command instance has to be deleted.
	// Get track/pit segment that contains the command from parent item
	HTREEITEM hItemParent;
	hItemParent = m_treeData.GetParentItem( hItem );
	ASSERT( hItemParent != NULL ); // should always exist....
	CCommand * pCmd = reinterpret_cast<CCommand *>( pObj );
	pObj = reinterpret_cast<CObject *>(m_treeData.GetItemData( hItemParent ));
	if ( pObj == NULL )
		return;
	if ( !pObj->IsKindOf( RUNTIME_CLASS( CTrackSeg ) ) )
		return;
	CTrackSeg * pTs = reinterpret_cast<CTrackSeg *>( pObj );
	
	// remove command from list by changing the pointers of previous and next command
	if ( pCmd->m_pPrev != NULL )
		pCmd->m_pPrev->m_pNext = pCmd->m_pNext;
	if ( pCmd->m_pNext != NULL )
		pCmd->m_pNext->m_pPrev = pCmd->m_pPrev;
	// Maybe it was the first command in the list
	if ( pTs->cmd == pCmd )
		pTs->cmd = pCmd->m_pNext;
	// now delete the command
	delete pCmd;
	// Notify document of change
	GetDocument()->SetModifiedFlag();
	// Update UI
	m_treeData.DeleteItem( hItem );
}

// looks below hItemRoot for a tree item belonging to the track segment

HTREEITEM CTrackPropertiesView::GetTreeItem(HTREEITEM hItemRoot, CTrackSeg *pTs)
{
	HTREEITEM hItem;
	hItem = m_treeData.GetChildItem( hItemRoot );
	while ( hItem != 0 )
	{
		if ( m_treeData.GetItemData( hItem ) == (DWORD) pTs )
			return hItem;
		hItem = m_treeData.GetNextItem( hItem, TVGN_NEXT );
	}
	// did not find appropriate item
	return 0;
}

void CTrackPropertiesView::OnSelchangedDataTree(NMHDR* pNMHDR, LRESULT* pResult) 
{
	NM_TREEVIEW* pNMTreeView = (NM_TREEVIEW*)pNMHDR;
	// which kind of data was selected?
	HTREEITEM hItem = pNMTreeView->itemNew.hItem;
	HTREEITEM hItemParent = m_treeData.GetParentItem( hItem );
	if ( hItemParent == NULL )
	{
		// Item has no parent: root item selected.
		// no further action.
	}
	else
	{
		// Find out which kind of parent
		if ( hItemParent == m_hTrackHeader )
		{
		}
		else if ( hItemParent == m_hTrackSegments )
		{
			// Track segment was selected
			CTrackSeg * pCurrentSeg = reinterpret_cast<CTrackSeg *>(m_treeData.GetItemData( hItem ));
			// Make sure we are in track editing mode
			GetDocument()->SetEditMode( CTrack::emTrack );
			// Notify document of new current track segment
			GetDocument()->SetCurrentTrackSeg( pCurrentSeg );
		}
		else if ( hItemParent == m_hPitlaneSegments )
		{
			// Pitlane segment was selected
			CTrackSeg * pCurrentSeg = reinterpret_cast<CTrackSeg *>(m_treeData.GetItemData( hItem ));
			// Make sure we are in pit editing mode
			GetDocument()->SetEditMode( CTrack::emPit );
			// Notify document of new current pit segment
			GetDocument()->SetCurrentPitSeg( pCurrentSeg );
		}
		else if ( hItemParent == m_hBestLineSegments )
		{
			CBestLineSeg * pBestLineSeg = reinterpret_cast<CBestLineSeg *>(m_treeData.GetItemData( hItem ));
			// Make sure we are in CCLine editing mode
			GetDocument()->SetEditMode( CTrack::emBestLine );
			// Notify documnet of new current best line segment
			GetDocument()->SetCurrentBestLineSeg( pBestLineSeg );
		}
	}
	*pResult = 0;
}

void CTrackPropertiesView::ConstructBestLineProps()
{
	// insert best line root in tree
	m_hBestLineSegments = m_treeData.InsertItem( "Best line" );
	// insert best segments under root
	CBestLineSeg * pBl = GetDocument()->GetBestLineSegs();
	HTREEITEM hItem;
	int i = 1;
	CString strAux;
	while ( pBl != NULL )
	{
		strAux.Format( "%d", i++ );
		int iRadius;
		if ( pBl->type == 0 )
			iRadius = pBl->a2;
		else
			iRadius = pBl->a3;
		if ( iRadius == 0 )
			strAux += " Straight";
		else if ( iRadius > 0 )
			strAux += " Turn right";
		else
			strAux += " Turn left";
		hItem = m_treeData.InsertItem( strAux, m_hBestLineSegments );
		m_treeData.SetItemData( hItem, (DWORD) pBl );
		pBl = pBl->m_pNext; // next segment
	}
}

void CTrackPropertiesView::OnEditBestLineSegment() 
{
	// first, get the currently selected command from the tree control
	HTREEITEM hItem;
	hItem = m_treeData.GetSelectedItem();
	if ( hItem == NULL )
		return;
	CObject * pObj = reinterpret_cast<CObject*>(m_treeData.GetItemData( hItem ));
	if ( pObj == NULL )
		return;
	if ( !pObj->IsKindOf( RUNTIME_CLASS( CBestLineSeg ) ) )
		return;
	// now edit current CCLine segment
	EditBestLineSegment( reinterpret_cast<CBestLineSeg*>(pObj) );
}

void CTrackPropertiesView::EditBestLineSegment( CBestLineSeg * pBl )
{
	if ( pBl == NULL )
		return;

	CEditBestLineSegmentDialog * pDlg = new CEditBestLineSegmentDialog( this );
	pDlg->SetBestLineSeg( pBl );
	if ( pDlg->DoModal() == IDOK )
	{
		GetDocument()->SetModifiedFlag();
		GetDocument()->RefreshCCLine();
		UpdateCCLine();
		GetDocument()->UpdateAllViews( NULL );
	}
	delete pDlg;
}

void CTrackPropertiesView::OnInsertBestLineSegment() 
{
	// get the currently selected best line segment
	// first, get the currently selected item from the tree control
	HTREEITEM hItem;
	hItem = m_treeData.GetSelectedItem();
	if ( hItem == NULL )
		return;
	if ( hItem == m_hBestLineSegments )
	{
		// Insert as first CCline segment
		// create new segment
		CBestLineSeg * pBlNew = new CBestLineSeg;
		pBlNew->tlu = 1;
		// insert item for new segment into tree view
		HTREEITEM hItemNew = m_treeData.InsertItem( "New", m_hBestLineSegments, TVI_FIRST );
		m_treeData.SetItemData( hItemNew, reinterpret_cast<DWORD>(pBlNew) );

		// Put segment into list
		GetDocument()->InsertCCLineSegment( pBlNew, NULL );
	}
	else
	{
		// Insert after other CCline segment
		CObject * pObj = reinterpret_cast<CObject*>(m_treeData.GetItemData( hItem ));
		if ( pObj == NULL )
			return;
		if ( !pObj->IsKindOf( RUNTIME_CLASS( CBestLineSeg ) ) )
			return;

		// create new segment
		CBestLineSeg * pBlNew = new CBestLineSeg;
		pBlNew->tlu = 1;
		// insert item for new segment into tree view
		HTREEITEM hItemNew = m_treeData.InsertItem( "New", m_treeData.GetParentItem( hItem ), hItem );
		m_treeData.SetItemData( hItemNew, reinterpret_cast<DWORD>(pBlNew) );

		// put new segment into list
		CBestLineSeg * pBlCurrent = reinterpret_cast<CBestLineSeg *>(pObj);
		pBlNew->m_pPrev = pBlCurrent;
		pBlNew->m_pNext = pBlCurrent->m_pNext;
		if ( pBlNew->m_pNext != NULL )
			pBlNew->m_pNext->m_pPrev = pBlNew;
		pBlCurrent->m_pNext = pBlNew;
	}

	// tell track about the change, recalculate CCLine and update views.
	CTrack * pTrack = GetDocument();
	pTrack->SetModifiedFlag();
	pTrack->RefreshCCLine();
	UpdateCCLine();
	pTrack->UpdateAllViews( NULL );
}

// Update text for all CCLine segments in the tree view (e.g. after changes in CCLine)

void CTrackPropertiesView::UpdateCCLine()
{
	HTREEITEM hItem;
	int i = 1;
	CString strName;
	CBestLineSeg * pBl;

	hItem = m_treeData.GetChildItem( m_hBestLineSegments );
	while ( hItem != NULL )
	{
		strName.Format( "%d", i++ );
		int iRadius;
		pBl = reinterpret_cast<CBestLineSeg *>(m_treeData.GetItemData( hItem ));
		if ( pBl->type == 0 )
			iRadius = pBl->a2;
		else
			iRadius = pBl->a3;
		if ( iRadius == 0 )
			strName += " Straight";
		else if ( iRadius > 0 )
			strName += " Turn right";
		else
			strName += " Turn left";
		m_treeData.SetItemText( hItem, strName );

		// next best line item
		hItem = m_treeData.GetNextSiblingItem( hItem );
	}
}

void CTrackPropertiesView::OnDeleteBestLineSegment() 
{
	// get the currently selected best line segment
	HTREEITEM hItem;
	hItem = m_treeData.GetSelectedItem();
	if ( hItem == NULL )
		return;
	CObject * pObj = reinterpret_cast<CObject*>(m_treeData.GetItemData( hItem ));
	if ( pObj == NULL )
		return;
	if ( !pObj->IsKindOf( RUNTIME_CLASS( CBestLineSeg ) ) )
		return;

	// cast to correct pointer type
	CBestLineSeg * pBl = reinterpret_cast<CBestLineSeg *>(pObj);
	// remove from list
	GetDocument()->SetEditMode( CTrack::emBestLine );
	GetDocument()->SetCurrentBestLineSeg( pBl );
	GetDocument()->DeleteSegment();
	m_treeData.DeleteItem( hItem );
	GetDocument()->SetModifiedFlag();
	GetDocument()->RefreshCCLine();
	UpdateCCLine();
	GetDocument()->UpdateAllViews( NULL );
}

void CTrackPropertiesView::UpdateCommandProps( HTREEITEM hItem )
{
	CString strAux;
	CCommand * pCmd = reinterpret_cast<CCommand *>( m_treeData.GetItemData( hItem ) );
	strAux.Format( "0x%02X: %s", pCmd->cmd, pCmd->GetName() );
	m_treeData.SetItemText( hItem, strAux );

}

void CTrackPropertiesView::OnDeletePitSegment() 
{
	CTrackSeg * pTs = GetSelectedTrackSegment( m_hPitlaneSegments );
	if ( pTs == NULL )
		// Unable to find a selected pitlane segment
		return;

	// Remove the segment from the list of pitlane segments.
	// Will also call GUI update methods.
	GetDocument()->DeletePitSegment( pTs );
}

void CTrackPropertiesView::OnDeleteTrackSegment() 
{
	CTrackSeg * pTs = GetSelectedTrackSegment( m_hTrackSegments );
	if ( pTs == NULL )
		// Unable to find a selected pitlane segment
		return;

	// Remove the segment from the list of pitlane segments.
	// Will also call GUI update methods.
	GetDocument()->DeleteTrackSegment( pTs );
}

void CTrackPropertiesView::OnInsertPitSegment() 
{
	CTrackSeg * pTs = GetSelectedTrackSegment( m_hPitlaneSegments );
	if ( pTs == NULL )
		// Unable to find a selected pitlane segment
		return;

	// Add a new pitlane segment after the selected one.
	// Will also call GUI update methods.
	GetDocument()->InsertPitSegment( pTs );
}

void CTrackPropertiesView::OnInsertTrackSegment() 
{
	CTrackSeg * pTs = GetSelectedTrackSegment( m_hTrackSegments );
	if ( pTs == NULL )
		// Unable to find a selected pitlane segment
		return;

	// Add a new pitlane segment after the selected one.
	// Will also call GUI update methods.
	GetDocument()->InsertTrackSegment( pTs );
}

/**
  Get the currently selected track segment that belongs to the
  given parent, if any.
*/

CTrackSeg * CTrackPropertiesView::GetSelectedTrackSegment(HTREEITEM hParent)
{
	// Find currently selected item
	HTREEITEM hItem = m_treeData.GetSelectedItem();
	if ( hItem == NULL )
		// Nothing selected
		return NULL;

	if ( m_treeData.GetParentItem( hItem ) != hParent )
		// Not a child of given parent
		return NULL;

	// Check if connected object is a track segment
	CObject * pObj = reinterpret_cast<CObject *>( m_treeData.GetItemData( hItem ) );
	if ( !pObj->IsKindOf( RUNTIME_CLASS( CTrackSeg ) ) )
		// Wrong type
		return NULL;

	// Now can safely be casted to CTrackSeg class
	CTrackSeg * pTs = reinterpret_cast<CTrackSeg *>( pObj );
	return pTs;
}

void CTrackPropertiesView::RelabelTrackSegments(HTREEITEM hParent)
{
	HTREEITEM hItem;
	CTrackSeg * pTs;
	int i = 1;
	CString strAux;
	// -------------------------------------------------------------------
	if ( hParent == NULL )
		return;

	hItem = m_treeData.GetChildItem( hParent );
	while ( hItem != NULL )
	{
		pTs = reinterpret_cast<CTrackSeg *>( m_treeData.GetItemData( hItem ) );
		strAux.Format( "%d", i++ );
		if ( pTs->curvature == 0 )
			strAux += " Straight";
		else if ( pTs->curvature > 0 )
			strAux += " Turn right";
		else
			strAux += " Turn left";
		// Change the label
		m_treeData.SetItemText( hItem, strAux );
		// Proceed to next segment
		hItem = m_treeData.GetNextSiblingItem( hItem );
	}
}

void CTrackPropertiesView::OnPreferencesNewCcline() 
{
	GetDocument()->SetOldCCLineStyle( false );
	GetDocument()->RefreshCCLine();
	GetDocument()->UpdateAllViews(NULL);
}

void CTrackPropertiesView::OnUpdatePreferencesNewCcline(CCmdUI* pCmdUI) 
{
	if ( GetDocument()->GetOldCCLineStyle() )
		// old style: remove check
		pCmdUI->SetCheck( 0 );
	else
		// new style: set check
		pCmdUI->SetCheck( 1 );
}

void CTrackPropertiesView::OnPreferencesOldCcline() 
{
	GetDocument()->SetOldCCLineStyle( true );
	GetDocument()->RefreshCCLine();
	GetDocument()->UpdateAllViews(NULL);
}

void CTrackPropertiesView::OnUpdatePreferencesOldCcline(CCmdUI* pCmdUI) 
{
	if ( GetDocument()->GetOldCCLineStyle() )
		// old style: set check
		pCmdUI->SetCheck( 1 );
	else
		// new style: remove check
		pCmdUI->SetCheck( 0 );
}

