/* f1gpfile.c - extract data from MicroProse F1GP save files

   Copyright (C) 1994 David P Gymer

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation,
   675 Mass Ave, Cambridge, MA 02139, USA.  */

/* At present we only scan the track records and setups files.  It would
   be nice if we could alternatively pull the driver and team names out of
   both names, single games, and championship files.  */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "f1gpfile.h"

/* Default directories (in prefix form).  */
static const char *default_dirs[] = {
  "c:/f1gp/",
  0
};
/* Default file names.  */
static const char *records_file = "records";
static const char *setups_file = "setups ";

/* Helper function to look for and open a file.  */
static FILE *
find_and_open (const char *name, const char *mode)
{
  FILE *fp;
#ifndef FILENAME_MAX		/* Stupid bloody C library.  */
#define FILENAME_MAX 256
#endif
  char file[FILENAME_MAX];
  const char **dir;

  dir = default_dirs;
  fp = fopen (name, mode);
  while (!fp && *dir)
    {
      strcpy (file, *dir++);
      strcat (file, name);
      fp = fopen (file, mode);
    }
  return fp;
}

/* It'd be nice to be able to check that there really are 16 of them, but
   C doesn't make it easy for us; the best way is to wrap them up into a
   struct.  Perhaps I should've used C++?  Nah... */
TRACK *
f1gp_new_tracks (TRACK *tracks)
{
  static const TRACK basic_tracks[NUM_TRACKS] = {
    { { "Phoenix", "USA" }, 2280, 81 },
    { { "Interlagos", "Brazil" }, 2688, 71 },
    { { "Imola", "San Marino" }, 3132, 61 },
    { { "Monte Carlo", "Monaco" }, 2068, 78 },
    { { "Montreal", "Canada" }, 2753, 69 },
    { { "Mexico City", "Mexico" }, 2748, 69 },
    { { "Magny Cours", "France" }, 2654, 72 },
    { { "Silverstone", "Great Britain" }, 3203, 60 },
    { { "Hockenheim", "Germany" }, 4228, 45 },
    { { "Hungaroring", "Hungary" }, 2466, 77 },
    { { "Spa", "Belgium" }, 4314, 44 },
    { { "Monza", "Italy" }, 3605, 53 },
    { { "Estoril", "Portugal" }, 2703, 71 },
    { { "Barcelona", "Spain" }, 2950, 65 },
    { { "Suzuka", "Japan" }, 3642, 53 },
    { { "Adelaide", "Australia" }, 2349, 81 }
  };

  if (!tracks)
    tracks = malloc (sizeof (TRACK) * NUM_TRACKS);
  if (!tracks)
    return 0;
  memcpy (tracks, basic_tracks, sizeof (TRACK) * NUM_TRACKS);
  return tracks;
}

static int
read_setup (FILE *fp, SETUP *setup)
{
  fread (setup, sizeof (SETUP), 1, fp);
  return 0;
}

int
f1gp_read_setup (const char *file, TRACK *tracks, int track, int setup)
{
  FILE *fp;
  int rc;

  fp = find_and_open (file, "rb");
  if (!fp)
    {
      perror (file);
      return 1;
    }
  rc = read_setup (fp, tracks[track].setups + setup);
  fclose (fp);
  return rc;
}

int
f1gp_read_setups (const char *file, TRACK *tracks)
{
  FILE *fp;
  int track;

  if (!file)
    file = setups_file;
  fp = find_and_open (file, "rb");
  if (!fp)
    {
      perror (file);
      return 1;
    }

  fseek (fp, 2L, SEEK_CUR);
  track = 0;
  while (track < NUM_TRACKS)
    read_setup (fp, &tracks[track++].setups[0]);
  track = 0;
  while (track < NUM_TRACKS)
    read_setup (fp, &tracks[track++].setups[1]);
  fclose (fp);

  return 0;
}

int
read_record (FILE *fp, RECORD *record)
{
  unsigned short date;

  fread (record->name, sizeof (record->name) - 1, 1, fp);
  record->name[sizeof (record->name) - 1] = '\0';
  fread (record->team, sizeof (record->team) - 1, 1, fp);
  record->team[sizeof (record->team) - 1] = '\0';
  /* FIXME: major portability problems!  */
  fread (&record->lap_time, 4, 1, fp);
  fread (&date, 2, 1, fp);
  record->date = (date + 2922) * 86400;
  return 0;
}

static int
read_and_replace (FILE *fp, RECORD *current)
{
  RECORD record;

  read_record (fp, &record);
  if (current->lap_time == 0 || current->lap_time > record.lap_time)
    memcpy (current, &record, sizeof (RECORD));
  return 0;
}

/* Read the lap records.  This is special in that we don't want to simply
   overwrite any existing ones.  */
int
f1gp_read_records (const char *file, TRACK *tracks)
{
  FILE *fp;
  int track;

  if (!file)
    file = records_file;
  fp = find_and_open (file, "rb");
  if (!fp)
    {
      perror (file);
      return 1;
    }

  fseek (fp, 2L, SEEK_CUR);
  track = 0;
  while (track < NUM_TRACKS)
    {
      read_and_replace (fp, &tracks[track].records[0]);
      read_and_replace (fp, &tracks[track].records[1]);
      ++track;
      fseek (fp, 2L, SEEK_CUR);
    }
  fclose (fp);

  return 0;
}
