/* gplist.c - list data from MicroProse F1GP save files

   Copyright (C) 1994 David P Gymer

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation,
   675 Mass Ave, Cambridge, MA 02139, USA.  */

/* At present we only scan the track records and setups files.  It would
   be nice if we could alternatively pull the driver and team names out of
   both names, single games, and championship files.  */

#include <assert.h>
#include <ctype.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "f1gpfile.h"
#include "f1gpprn.h"

#define GPINI "gplist.ini"

static int print_setups_p = 1;
static int print_records_p = 1;

static void
display_setup (SETUP *setup)
{
  int n;
  int c;

  printf (" %02d,%02d", setup->wings[0], setup->wings[1]);
  n = 0;
  c = ' ';
  while (n < NUM_RATIOS)
  {
    printf ("%c%02d", c, setup->ratios[n++]);
    c = ',';
  }
  printf (" %c %02d%c", 'A' + setup->tyre_compound,
	  abs(setup->brake_balance),
	  (setup->brake_balance < 0 ? 'R' :
	   (setup->brake_balance > 0 ? 'F' : ' ')));
}

static void
display_record (RECORD *record, int lap_len)
{
  struct tm *tm;
  long mph;
  char buf[16];

  if (record->date)
  {
    tm = gmtime (&record->date);
    strftime (buf, sizeof (buf), "%Y/%m/%d", tm);
  }
  else
    strcpy (buf, " <unsaved>");
  /* FIXME: MAGIC NUMBERS!  */
  printf (" %-24.24s %-14.14s", record->name, record->team);
  if (record->lap_time)
  {
    mph = 36000L * lap_len / record->lap_time;
    printf (" %ld:%02ld.%03ld  %03ld.%ld  ", record->lap_time / 60000,
	    (record->lap_time / 1000) % 60, record->lap_time % 1000,
	    mph / 10, mph % 10);
  }
  else
    printf (" -:--.---  ---.-  ");
  printf ("%s", buf);
}

static void
display_track_line (TRACK *track, int which)
{
  if (print_setups_p)
    {
      display_setup (&track->setups[which]);
      putchar (' ');
    }
  if (print_records_p)
    display_record (&track->records[which], track->lap_length);
  putchar ('\n');
}

static void
display_tracks (TRACK *tracks)
{
  int t;

  t = 0;
  while (t < NUM_TRACKS)
    {
      TRACK *track = tracks + t;

      if (t++)
      	prn_code (PRN_HALF, stdout);

      /* FIXME: magic numbers in string lengths.  */
      printf ("%02d %-13s  %02d  Q: ", t, track->names[0], track->lap_count);
      display_track_line (track, 0);
      printf ("%13s  %d.%03d  R: ", track->names[1],
	      track->lap_length / 1000, track->lap_length % 1000);
      display_track_line (track, 1);
    }
}

/* Try in the current directory first, then the binary's directory.  */
static FILE *
open_ini (const char *exe)
{
  FILE *fp;
  char path[PATH_MAX], *s;

  fp = fopen (GPINI, "r");
  if (fp)
    return fp;
  strcpy (path, exe);
  s = strrchr (path, '/');
  assert (s);
  strcpy (s + 1, GPINI);
  fp = fopen (path, "r");
  return fp;
}

static char *
read_ini_line (FILE *fp)
{
  char buf[256];		/* Argh!  Magic number!  */
  char *s;

  if (!fp)
    return 0;
  while ((s = fgets (buf, sizeof (buf), fp)))
    {
      if (*s != '#')
	{
	  s = strchr (s, '\n');
	  if (s)
	    *s = '\0';
	  return *buf ? buf : 0;
	}
    }
  return 0;
}

static int
cheat (char *s, TRACK *tracks)
{
  int t;
  int t_m, t_s, t_f;
  char qr;
  long time;
  RECORD *rec;

  s = strtok (s, ",");
  if (!s)
    return 1;

  sscanf (s, "%d %c %d:%d.%d", &t, &qr, &t_m, &t_s, &t_f);
  if (t < 1 || t > 16 || (qr != 'Q' && qr != 'R'))
    return 1;
  --t;
  qr = qr == 'R';
  rec = tracks[t].records + qr;
  time = t_f + (t_s + t_m * 60L) * 1000;

  if (time < rec->lap_time)
    {
      rec->date = 0;

      s = strtok (0, ",");
      if (!s)
      	return 1;
      strncpy (rec->name, s, sizeof (rec->name));

      s = strtok (0, ",");
      if (!s)
      	return 1;
      strncpy (rec->team, s, sizeof (rec->team));

      rec->lap_time = time;
    }

  return 0;
}

static void
usage (const char *prog)
{
  fprintf (stderr, "Usage: %s [-I] [-R] [-S] [-r FILE] [-s FILE]\n", prog);
  fprintf (stderr, "  Options: \n");
  fprintf (stderr, "    -I        Ignore \"" GPINI "\"\n");
  fprintf (stderr, "    -R        Don't print lap records\n");
  fprintf (stderr, "    -S        Don't print setups\n");
  fprintf (stderr, "    -p NAME   Name a printer for codes; one of:\n");
  prn_list (stderr, "      ");
  fprintf (stderr, "    -r FILE   Additional lap records file\n");
  fprintf (stderr, "    -s FILE   Setups file\n");
  exit (1);
}

int
main (int argc, char **argv)
{
  TRACK tracks[16];
  int c;
  const char *f_setup = 0;
  const char *printer = 0;
  int ignore_ini_p = 0;
  FILE *ini_fp = 0;
  int read_records_p = 0;
  char *s;

  f1gp_new_tracks (tracks);

  while ((c = getopt (argc, argv, "IRSp:r:s:")) != EOF)
    switch (c)
      {
      case 'I':
	ignore_ini_p = 1;
	break;
      case 'R':
	print_records_p = 0;
	break;
      case 'S':
	print_setups_p = 0;
	break;
      case 'p':
	printer = optarg;
	break;
      case 'r':
	if (f1gp_read_records (optarg, tracks))
	  return 1;
	read_records_p = 1;
	break;
      case 's':
	f_setup = optarg;
	break;
      default:
	usage (*argv);
      }

  if (argc != optind)
    usage (*argv);

  if (!(ignore_ini_p || (ini_fp = open_ini (*argv))))
    fprintf (stderr, "%s: warning, can't find \"" GPINI "\"\n", *argv);

  if (prn_select (printer))
    usage (*argv);

  while (!ignore_ini_p && (s = read_ini_line (ini_fp)))
    {
      if (s[0] == 'S' && s[1] == '=')
	{
	  if (print_setups_p)
	    f1gp_read_setups (s + 2, tracks);
	}
      else if (s[0] == 'R' && s[1] == '=')
	{
	  if (print_records_p && !f1gp_read_records (s + 2, tracks))
	    read_records_p = 1;
	}
      else if (s[0] == 'C' && s[1] == '=')
	{
	  if (cheat (s + 2, tracks))
	    fprintf (stderr, "%s: malformed Cheat line\n", *argv);
	}
      else if (isdigit (s[0]) && isdigit (s[1]) && s[2] == '=')
	{
	  int t, q_p = 0, r_p = 0;

	  t = atoi (s) - 1;
	  if (t >= 40)
	  {
	    t -= 40;
	    r_p = 1;
	  }
	  else if (t >= 20)
	  {
	    t -= 20;
	    q_p = 1;
	  }
	  else
	    r_p = q_p = 1;

	  if (t < 0 || t > 15)
	    continue;
	  if (q_p)
	    f1gp_read_setup (s + 3, tracks, t, 0);
	  if (r_p)
	    f1gp_read_setup (s + 3, tracks, t, 1);
	}
      /* ... else we are confused and will ignore the line.  */
    }

  if (print_records_p && !read_records_p && f1gp_read_records (0, tracks))
    return 1;

  if (ini_fp)
    fclose (ini_fp);
  
  prn_code (PRN_OPEN, stdout);
  display_tracks (tracks);
  prn_code (PRN_CLOSE, stdout);

  return 0;
}
