;=======================================
;
; Music Quest Programmer's ToolKit
;
; Clock and Timer Services
;
; Written for use with small, medium, compact, and large memory models
;
; Requires Turbo Assembler or Microsoft Assember 5.0 or later
;
; Supports
;   Turbo Assembler
;   Microsoft Assembler
;   Turbo C
;   Quick C and Microsoft C
;   Turbo Pascal
;   Quick Basic
;
; Copyright 1987, 1990 by Music Quest, Inc.
; All rights reserved
;
;=======================================
;
; Check for valid language OR model specification
;
        DOSSEG
        IFDEF   _SMALL_
        .model  small
        ENDIF
        IFDEF   _MEDIUM_
        .model  medium
        ENDIF
        IFDEF   _COMPACT_
        .model  compact
        ENDIF
        IFDEF   _LARGE_
        .model  large
        ENDIF
        IFDEF   _PASCAL_
        .model  medium
        ENDIF
        IFDEF   _QBASIC_
        .model  medium
        ENDIF

        IFNDEF  @codesize
        %out    ***ERROR***  Valid language or model name not specified
        .ERR
        END
        ENDIF

;=======================================
; External references
;=======================================
        IF      @codesize GE 1
        extrn   mcccommand:far
        ELSE
        extrn   mcccommand:near
        ENDIF

        .data
;=======================================
; Call parameter definitions
;=======================================
        if      @codesize GE 1
p1      equ     6                       ;parameter 1 medium, large, huge
p2      equ     8                       ;parameter 2 medium, large, huge
p3      equ     10                      ;parameter 3 medium, large, huge
        else
p1      equ     4                       ;parameter 1 small model
p2      equ     6                       ;parameter 2 small model
p3      equ     8                       ;parameter 3 small model
        endif
;=======================================
; Timer Request Block
;=======================================
trqblok struc
trquse  db      0                       ;in use flag
trqcnt  dw      0                       ;tic count for event
        IF      @datasize
trqefw  dd      0                       ;ptr to EFW
        ELSE
trqefw  dw      0                       ;ptr to EFW
        ENDIF
trqblok ends
;=======================================
; Table of available TRQ blocks
;=======================================
trqdim  =       16
trq     trqblok trqdim dup(<0,0,0>)     ;array of trq bloks
;=======================================
; Clock data
;=======================================
clk_starts db   0                       ;count of number of starts
clocktics dw    0                       ;count of tics since last time
clockrate dw    24                      ;clock tic decrement value
;=======================================
        .code
;=======================================
; _mclk_init
;
; Initialize clock service
;
; C Model:
;       mclk_init(ticvalue);
;               int ticvalue;
;
; Notes:
;   The tic value is the number of time units to
;   be counted on every timer tic.  It MUST match
;   the current settings of the MCC.  For example.
;   if the MCC clock resolution is 96 PPB and the
;   clock-to-PC rate is set to 48, then every
;   clock-to-PC interrupt counts as 12 clock tics.
;
;   The clock-to-PC interrupt occurs every M tics of
;   the MCC's clock, where M is the clock-to-PC rate
;   divided by 4.  The value of M is the number of
;   tics that the local clock will get for each
;   clock-to-PC interrupt.
;
;   Thus, the granularity of the local clock is
;   totally dependent on the frequency of clock-to-PC
;   interrupts and the number of tics each interrupt
;   represents.
;
;=======================================
        public  _mclk_init
_mclk_init proc
        push    bp
        mov     bp,sp
;
        mov     ax,[bp+p1]              ;new tic rate value
        mov     clockrate,ax
        mov     clocktics,0             ;clear local clock
;
        pop     bp
        ret
_mclk_init endp
;=======================================
; _mclk_start
;
; Start clock service.  When the clock
; starts, clokc-to-PC interrupts are
; enabled and the local clock begins
; to "tic".
;
; Note: This routine uses mcccomand to
; start clock-to-PC interrupts.  The
; current interrupt handler must be
; able to handle any data captured by
; the mcccommand function.  If the interrupt
; handler can not handle all conditions,
; unpredicatble results may occur.
;
; C Model:
;       mclk_start();
;
;=======================================
        public  _mclk_start
_mclk_start proc
        push    bp
        mov     bp,sp
;
        mov     clocktics,0             ;reset local clock
        inc     clk_starts              ;bump count of starts
        mov     ah,95H                  ;enable clock to PC
        call    mcccommand
;=======================================
; Sync local clock to MCC clock
;=======================================
        mov     cx,-1
mcs1:
        cmp     clocktics,0             ;wait for first tic
        jne     mcs2
        loop    mcs1                    ;avoid hang
mcs2:
        cli
        mov     clocktics,0             ;clock now synched
        sti
;=======================================
; Exit with local clock started and synced
;=======================================
        pop     bp
        ret
_mclk_start endp
;=======================================
; _mclk_stop
;
; Stop clock service.  The clock-to-PC is
; disabled, so that the local clock no
; longer "tics".
;
; Note: This routine uses mcccomand to
; start clock-to-PC interrupts.  The
; current interrupt handler must be
; able to handle any data captured by
; the mcccommand function.  If the interrupt
; handler can not handle all conditions,
; unpredicatble results may occur.
;
; C Model:
;       mclk_stop();
;
;=======================================
        public  _mclk_stop
_mclk_stop proc
        mov     ah,94H                  ;disable clock-to-PC interrupts
        call    mcccommand
        dec     clk_starts              ;decrement count of starts
        ret
_mclk_stop endp
;=======================================
; _midi_clock
;
; Read the current value of the local clock.
; After the clock is read, it is reset to 0.
;
; C Model:
;       value=midi_clock();
;               unsigned value;
;
;=======================================
        public  _midi_clock
_midi_clock proc
        cli
        mov     ax,clocktics            ;get current clock value
        mov     clocktics,0             ;reset clock
        sti
        ret
_midi_clock endp
;=======================================
; clock_tic
;
; Clock tic interrupt handler
; Call by MCC co-processor SLIH
;
; Notes:
;   Assumes machine disabled at entry.
;
;=======================================
        public  clock_tic
clock_tic proc
;
        mov     dx,clockrate            ;tics/interrupt value
        add     clocktics,dx            ;update running timer
;=======================================
; Run through TRQ table, decrementing timers
;=======================================
        sub     bx,bx
        mov     cx,trqdim
;
tbtop:
        test    trq.trquse[bx],1
        jz      tbnext
        mov     ax,trq.trqcnt[bx]       ;decrement this timer
        sub     ax,dx
        mov     trq.trqcnt[bx],ax
        jz      tbexpire                ;timer has expired
        jnc     tbnext                  ;still > 0
tbexpire:
        IF      @datasize
        les     si,trq.trqefw[bx]       ;post the event
        mov     word ptr es:[si],0FFh
        ELSE
        mov     si,trq.trqefw[bx]       ;post the event
        mov     word ptr [si],0FFh
        ENDIF
;
tbnext:
        add     bx,SIZE trqblok
        loop    tbtop
;=======================================
; Exit after all timers decremented
;=======================================
        ret
clock_tic endp
;=======================================
; _set_trq
;
; Set up a Timer Request.  A TRQ block is
; allocated and an initial count down time
; is set.
;
; C Model:
;       hx=set_trq(&efw,time);
;               int hx;
;                       TRQ handle or return code
;                       <0 = no TRQ available
;                       >=0 = TRQ set
;               int *efw;
;                       event flag word
;               int time;
;                       time value when the
;                       event occurs.  When this
;                       time counts to 0, the
;                       event is posted.
;
;=======================================
        public  _set_trq
_set_trq proc
        push    bp
        mov     bp,sp
        push    si
;=======================================
; Look for a free TRQ block
;=======================================
        sub     bx,bx
        mov     cx,trqdim
;
trqtop:
        test    trq.trquse[bx],1        ;find unused trq blok
        jnz     trqnext
;=======================================
; Free TRQ block found
;=======================================
        cli
        mov     trq.trquse[bx],1        ;mark it used
        IF      @datasize
        les     si,[bp+p1]
        mov     word ptr trq.trqefw[bx],si ;copy efw ptr
        mov     word ptr trq.trqefw[bx+2],es
        mov     word ptr es:[si],0      ;clear efw
        mov     ax,[bp+p3]
        mov     trq.trqcnt[bx],ax       ;copy starting time
        ELSE
        mov     si,[bp+p1]
        mov     trq.trqefw[bx],si       ;copy efw ptr
        mov     word ptr [si],0         ;clear efw
        mov     ax,[bp+p2]
        mov     trq.trqcnt[bx],ax       ;copy starting time
        ENDIF
        or      ax,ax                   ;if time==0, set efw
        jnz     trqexit
        IF      @datasize
        mov     word ptr es:[si],0FFh
        ELSE
        mov     word ptr [si],0FFh
        ENDIF
        jmp     trqexit
;
trqnext:
        add     bx,SIZE trqblok
        loop    trqtop
;=======================================
; No TRQ blocks free
;=======================================
        mov     bx,-1                   ;all trqs in use
;=======================================
; Exit with return code in AX
;=======================================
trqexit:
        sti
        mov     ax,bx
        pop     si
        pop     bp
        ret
_set_trq endp
;=======================================
; _set_trq_time
;
; Set TRQ timer value for an existing TRQ
; block.
;
; C Model:
;       set_trq_time(hx,time);
;               int hx;
;                       TRQ block handle
;               int time;
;                       New time value
;
;=======================================
        public  _set_trq_time
_set_trq_time proc
        push    bp
        mov     bp,sp
        mov     bx,[bp+p1]              ;the handle points to TRQ
        mov     ax,[bp+p2]              ;new time value
;=======================================
; Update TRQ block
;=======================================
        cli
        mov     trq.trqcnt[bx],ax
        or      ax,ax                   ;time==0?
        jnz     stt1
        IF      @datasize
        les     bx,trq.trqefw[bx]
        mov     word ptr es:[bx],0FFh   ;set efw
        ELSE
        mov     bx,trq.trqefw[bx]
        mov     word ptr [bx],0FFh      ;set efw
        ENDIF
stt1:
        sti
        pop     bp
        ret
_set_trq_time endp
;=======================================
; _add_trq_time
;
; Add to TRQ time value.  This is the preferred
; way of setting a new timer value, as it
; does not loose an tics.  The new value is added
; to the current TRQ timer.
;
; C Model:
;       add_trq_time(hx,time);
;               int hx;
;                       TRQ block handle
;               int time;
;                       Additional time value
;=======================================
        public  _add_trq_time
_add_trq_time proc
        push    bp
        mov     bp,sp
        mov     bx,[bp+p1]              ;the handle points to the TRQ
        mov     ax,[bp+p2]              ;timer value adder
;=======================================
; Update the TRQ block
;=======================================
        cli
        add     trq.trqcnt[bx],ax
        js      short att1              ;already expired if negative
        jnz     short att2              ;not expired
att1:
        IF      @datasize
        les     bx,trq.trqefw[bx]
        mov     word ptr es:[bx],0FFh   ;set efw
        ELSE
        mov     bx,trq.trqefw[bx]
        mov     word ptr [bx],0FFh      ;set efw
        ENDIF
att2:
        sti
        pop     bp
        ret
_add_trq_time endp
;=======================================
; _end_trq
;
; Terminate a TRQ timer.  The designated timer
; is freed.
;
; C Model:
;       end_trq(hx);
;               int hx;
;                       The TRQ handle.

;=======================================
        public  _end_trq
_end_trq proc
        push    bp
        mov     bp,sp
;=======================================
; Free TRQ block for re-use
;=======================================
        cli
        mov     bx,[bp+p1]              ;the handle points to the TRQ
        mov     trq.trquse[bx],0
        sti
        pop     bp
        ret
_end_trq endp
;=======================================
; _clear_efw
;
; Clear an EFW.  This provides a "safe"
; way to reset an event flag word.
;
; C Model:
;       clear_efw(&efw);
;               int *efw;
;                       A pointer to the EFW.
;=======================================
        public  _clear_efw
_clear_efw proc
        push    bp
        mov     bp,sp
;=======================================
; EFWs must be cleared in a disabled state
;=======================================
        pushf
        cli
        IF      @datasize
        les     bx,[bp+p1]              ;get address of EFW to be cleared
        mov     word ptr es:[bx],0
        ELSE
        mov     bx,[bp+p1]              ;get address of EFW to be cleared
        mov     word ptr [bx],0
        ENDIF
        popf
        pop     bp
        ret
_clear_efw endp
;=======================================
        END
