{*===================================*}
{*                                   *}
{* Music Quest Programmer's ToolKit  *}
{*                                   *}
{* Interpreted Trace                 *}
{*                                   *}
{* Copyright (c) 1988                *}
{* By Music Quest, Inc.              *}
{*                                   *}
{*===================================*}

unit inttrace;                          {stand-alone unit}

interface

  uses Crt, mcc;

  function midigetw: integer;
  procedure tk_itrace;

implementation

{$I mccconst.inc}

{*===================================*}
{*                                   *}
{* Convert to hex                    *}
{*                                   *}
{*===================================*}

type
  hex2 = string[2];

function tohex(b: integer): hex2;

const
  hextab:       array [0..15] of char = '0123456789ABCDEF';

begin
  tohex[1]:=hextab[(b and $F0) shr 4];
  tohex[2]:=hextab[b and $F];
  tohex[0]:=chr(2);
end;

{*===================================*}
{*                                   *}
{* Wait for the next MIDI data byte  *}
{*                                   *}
{*===================================*}
function midigetw: integer;

var
  loopcnt:      longint;                {limits loop time}
  mbyte:        integer;

begin
  loopcnt:=500000;                      {about 1-2 sec spin time}
  mbyte:=-1;
  while (loopcnt > 0) and (mbyte < 0) do {* spin until data or time out *}
    begin
      mbyte:=_mcc_get;                  {attempt to get byte}
      loopcnt:=loopcnt-1;               {bump spin counter}
    end;
  midigetw:=mbyte;
end;

{*===================================*}
{*                                   *}
{* Formatted trace                   *}
{*                                   *}
{*===================================*}
procedure tk_itrace;

var
  rc:           integer;
  midi_c:       integer;
  k:            integer;
  byte1, byte2: integer;
  channel:      integer;
  cmdx:         integer;
  sysex_sw:     integer;
  rstatus:      integer;                {* MIDI running status *}
  hexdata:      hex2;                   {used with tohex function}
  kbc:          char;

begin
  writeln('Press any key to end trace.');
  _mcc_set_receiveslih;                 {* set up our slih *}
  rc:=_mcc_command(UART_MODE);          {* put MCC into pass-thru mode *}

  channel:=0;
  midi_c:=0;
  sysex_sw:=0;
  rstatus:=1;                           {* running status = note on *}
  while not KeyPressed do               {* until escape *}
    begin
      if sysex_sw=0 then
        midi_c:=_mcc_get;
      if (sysex_sw > 0) or (midi_c >= 0) then {* got a byte to start a message *}
        begin
          sysex_sw:=0;
          if (midi_c and $80) = $80 then {* new status? *}
            if midi_c < $F0 then        {* non-system message? *}
              begin
                channel:=(midi_c and $0F)+1;
                rstatus:=(midi_c and $70) shr 4;
                cmdx:=rstatus;
                byte1:=midigetw;
              end
            else                        {* system message *}
              cmdx:=7
          else                            {* running status *}
            begin
              byte1:=midi_c;
              cmdx:=rstatus
            end;
          case cmdx of
            0:
              begin
                byte2:=midigetw;
                writeln('Note off            channel=',channel,' key=',byte1,' velocity=',byte2);
              end;
            1:
              begin
                byte2:=midigetw;
                writeln('Note on             channel=',channel,' key=',byte1,' velocity=',byte2);
              end;
            2:
              begin
                byte2:=midigetw;
                writeln('Key aftertouch      channel=',channel,' key=',byte1,' pressure=',byte2);
              end;
            3:
              begin
                byte2:=midigetw;
                writeln('Control change      channel=',channel,' controller=',byte1,' value=',byte2);
              end;
            4:
              begin
                writeln('Program change      channel=',channel,' program=',byte1);
              end;
            5:
              begin
                writeln('Channel aftertouch  channel=',channel,' pressure=',byte1);
              end;
            6:
              begin
                byte2:=midigetw;
                writeln('Pitch bend          channel=',channel,' bend=',(byte2 shl 7)+byte1);
              end;
            7:
              begin
                case midi_c and $0F of
                  0:               {* sys ex *}
                    begin
                      write(Output,'System exclusive    ');
                      k:=16;
                      repeat
                        begin
                          midi_c:=midigetw;
                          hexdata:=tohex(midi_c);
                          write(Output,hexdata,' ');
                          k:=k-1;
                          if k=0 then
                            begin
                              writeln;
                              write(Output,'                    ');
                              k:=16
                            end;
                        end;
                      until (midi_c and $80) = $80;
                      writeln;
                      if midi_c <> $F7 then     {* ended on new command, not eox *}
                        sysex_sw:=1;
                    end;
                  1:               {* midi time code *}
                    begin
                      byte1:=midigetw;
                      writeln('MIDI time code      code=',byte1);
                    end;
                  2:               {* song position *}
                    begin
                      byte1:=midigetw;
                      byte2:=midigetw;
                      writeln('Song position       beats=',(byte2 shl 7)+byte1);
                    end;
                  3:               {* song select *}
                    begin
                      byte1:=midigetw;
                      writeln('Song select         song=',byte1);
                    end;
                  6:               {* tune request *}
                    begin
                      writeln('Tune request');
                    end;
                  8:               {* clock *}
                    begin
                    { Note: MIDI clock happens too frequently to
                      do anything here }
                    end;
                  9:
                    begin
                    end;
                  10:              {* start *}
                    begin
                      writeln('Start');
                    end;
                  11:              {* continue *}
                    begin
                      writeln('Continue');
                    end;
                  12:
                    begin
                      writeln('Stop');
                    end;
                  13:
                    begin
                    end;
                  14:              {* active sensing *}
                    { Note: Active sensing happens too frequently to
                      do anything here }
                    begin
                    end;
                  15:
                    begin
                      writeln('System reset');
                    end;
                  else
                    begin
                      writeln('Undefined           status=',midi_c);
                    end;
                end;
              end;
            else
              begin
              end
          end;
        end;
    end;

  kbc:=ReadKey;                         {* clear exit key *}
  _mcc_set_noslih;                      {* default slih *}
  _mcc_reset;                           {* return to coproc mode *}
end;                                    {* end of procedure *}

end.                                    { end of unit }
