;=======================================
;
; Music Quest Programmer's ToolKit
;
; Co-processor mode second level interrupt handler
; This routine is suitable for use as
; a SLIH for the ToolKit FLIH.
;
; It provides the following functions:
;   - In memmory recording of event data
;   - Event flag words for MCC events
;   - Clock tics for the ToolKit clock services
;
; This module is written specifically for:
;   Microsoft Quick Basic (medium) model.
;
; Copyright (c) 1988
; Music Quest, Inc.
;
;=======================================
DATA    SEGMENT WORD PUBLIC
;=======================================
; Small model values
;=======================================
p1of1           equ     6               ;offset to parameter 1
p1of2           equ     8               ;offset to parameter 1
p2of2           equ     6               ;offset to parameter 2
p1of3           equ     10              ;offset to parameter 1
p2of3           equ     8               ;offset to parameter 2
p3of3           equ     6               ;offset to parameter 3
;=======================================
res_cnt         db      ?               ;residual data count expected
data_in         db      ?               ;midi-in data
;
eventp          dw      ?               ;event assembly area
event_buff      db      16 dup(?)       ;the assembly area
spp_byte1       db      ?               ;first byte of SPP
;
rbuff           dw      ?               ;recording buffer
rbuffseg        dw      ?               ;
rbx             dw      ?               ;recording index
rbsize          dw      ?               ;buffer size (should be 1K multiple)
;=======================================
; Decoded values
;=======================================
last_spp        dw      ?               ;decoded value from last SPP
last_smpte      db      4 dup(?)        ;last SMPTE frame address
;=======================================
DATA    ENDS
CODE    SEGMENT BYTE PUBLIC
;=======================================
; SLIH state data
;=======================================
mqstatew        dw      new_event       ;current, state of recording
rstatec         db      2               ;current running status data count
data_cnt        db      2,2,2,2,1,1,2,0 ;data count for status codes
;
eot_mark        db      0,0FCh          ;MCC end of track marker
;
eventcases      dw      toflow_event    ;jump table for mcc msgs f8-ff
                dw      cdr_event
                dw      $fa_event
                dw      $fb_event
                dw      $fc_event
                dw      $fd_event
                dw      $fe_event
                dw      $ff_event
;=======================================
; Event Flag words
;=======================================
pagerdy         db      0               ;1K page ready flag
track_efw       dw      8 dup(0)        ;EFWs for each track
conductor_efw   dw      0               ;conductor EFW
ackn_efw        dw      0               ;command acknowledge efw
playend_efw     dw      0               ;all play tracks ended
recordend_efw   dw      0               ;recording completed
me_efw          dw      0               ;measure end
clk_efw         dw      0               ;clock-to-PC event
cpt_efw         dw      0               ;cue point event
spp_efw         dw      0               ;SPP event
smpte_efw       dw      0               ;SMPTE frame address received
realtime_efw    dw      0               ;MIDI realtime EFW
strk_end_efw    dw      0               ;SMPTE track end EFW
rbfull_flag     dw      0               ;buffer full flagged
;=======================================
        ASSUME  CS:CODE,DS:DATA
;=======================================
; External references
;=======================================
        extrn   clock_tic:far           ;clock manager (mcctkckb.asm)
;=======================================
; rec_data
;
; Assemble event data into an event
;
; Inputs:
;   al = data to be recorded
; Outputs:
;   None.
;
; Notes:
;   assumes interrupts disabled
;   NOT callable by Pascal
;
;=======================================
rec_data proc   near
        mov     bx,eventp               ;get ptr to assembly area
        mov     [bx],al                 ;assemble byte
        inc     eventp                  ;bump assembly ptr
        ret
rec_data endp
;=======================================
; rec_asmdata
;
; Record assembled event in the recording buffer
;
; Inputs:
;   al = data to be recorded
; Outputs:
;   None.
;
; Notes:
;   assumes interrupts disabled
;   NOT callable by Pascal
;
;=======================================
rec_asmdata proc   near
        push    es
;
        cmp     rbfull_flag,0           ;any room?
        jne     rec_1                   ;no
;
        mov     es,rbuffseg             ;set up ds:si and es:di for move
        mov     di,rbuff
        add     di,rbx
        mov     si,offset event_buff
;
        mov     cx,eventp
        sub     cx,si                   ;length of event
        add     rbx,cx                  ;bump record offset
        sub     rbsize,cx               ;adjust remaining space
        rep     movsb                   ;put event in record buffer
;
        cmp     rbsize,12               ;record buffer filled?
        jnb     rec_1                   ;no
;
        mov     si,offset eot_mark      ;force an end mark on the track
        mov     cx,2
        rep     movsb
        mov     rbfull_flag,1           ;buffer full, stop recording
rec_1:
        pop     es
        ret
rec_asmdata endp
;=======================================
; _rec_overflow
;
; Test for recording buffer overflow
;
; Model:
; function _rec_overflow: integer;
;
; returned value
;   1 = buffer overflow/full
;   0 = buffer not full
;
;=======================================
        public  _rec_overflow
_rec_overflow proc  far
        cli
        mov     ax,rbfull_flag
        mov     rbfull_flag,0
        sti
        ret
_rec_overflow endp
;=======================================
; _track_efw
;
; Track Event Flag Word state
;
; Model:
; function _track_efw: integer;
;
; returned value
;   0 = event not posted
;   1 = event posted
;
;=======================================
        public  _track_efw
_track_efw proc far
        push    bp
        mov     bp,sp
        mov     bx,[bp+p1of1]
        add     bx,bx
        cli
        mov     ax,track_efw[bx]
        mov     track_efw[bx],0
        sti
        pop     bp
        ret     2
_track_efw endp
;=======================================
; _conductor_efw
;
; Conductor EFW status
;
; Model:
; function _conductor_efw: integer;
;
; returned value
;   0 = event not posted
;   1 = event posted
;
;=======================================
        public  _conductor_efw
_conductor_efw proc far
        cli
        mov     ax,conductor_efw
        mov     conductor_efw,0
        sti
        ret
_conductor_efw endp
;=======================================
; _clock_efw
;
; Clock-to-PC EFW status
;
; Model:
; function _clock_efw: integer;
;
; returned value
;   0 = event not posted
;   1 = event posted
;
;=======================================
        public  _clock_efw
_clock_efw proc far
        cli
        mov     ax,clk_efw
        mov     clk_efw,0
        sti
        ret
_clock_efw endp
;=======================================
; _ackn_efw
;
; Command acknowledge EFW status
;
; Model:
; function _ackn_efw: integer;
;
; returned value
;   0 = event not posted
;   1 = event posted
;
;=======================================
        public  _ackn_efw
_ackn_efw proc far
        cli
        mov     ax,ackn_efw
        or      ax,ax
        jz      ackx
        dec     ackn_efw
ackx:
        sti
        ret
_ackn_efw endp
;=======================================
; _playend_efw
;
; All playback tracks ended EFW status
;
; Model:
; function _playend_efw: integer;
;
; returned value
;   0 = event not posted
;   1 = event posted
;
;=======================================
        public  _playend_efw
_playend_efw proc far
        cli
        mov     ax,playend_efw
        mov     playend_efw,0
        sti
        ret
_playend_efw endp
;=======================================
; _recordend_efw
;
; Recording ended EFW status
;
; Model:
; function _recordend_efw: integer;
;
; returned value
;   0 = event not posted
;   1 = event posted
;
;=======================================
        public  _recordend_efw
_recordend_efw proc far
        cli
        mov     ax,recordend_efw
        mov     recordend_efw,0
        sti
        ret
_recordend_efw endp
;=======================================
; _measurend_efw
;
; Measure end EFW status
;
; Model:
; function _measurend_efw: integer;
;
; returned value
;   0 = event not posted
;   1 = event posted
;
;=======================================
        public  _measurend_efw
_measurend_efw proc far
        cli
        mov     ax,me_efw
        mov     me_efw,0
        sti
        ret
_measurend_efw endp
;=======================================
; _spp_efw
;
; SPP EFW status
;
; Model:
; procedure _spp_efw(var efw, spp: integer);
;                      bp+10  bp+6
;               efw
;               0 = event not posted
;               1 = event posted
;               spp = song position pointer if event posted
;
;=======================================
        public  _spp_efw
_spp_efw proc far
        push    bp
        mov     bp,sp
        push    es
        cli
        mov     ax,spp_efw              ;return EFW status
        mov     spp_efw,0
        or      ax,ax                   ;EFW set?
        jz      sppefwx                 ;no
        les     bx,[bp+6]               ;get ptr to result
        mov     cx,last_spp
        mov     es:[bx],cx
sppefwx:
        sti
        les     bx,[bp+10]              ;pass return code
        mov     es:[bx],ax
        pop     es
        pop     bp
        ret     8
_spp_efw endp
;=======================================
; _smpte_efw
;
; SMPTE frame message EFW status
;
; Model:
; procedure smpte_efw(var efw, integer; fid: array of byte);
;                       bp+10  bp+6
;               0 = event not posted
;               1 = event posted
;               fid: array of char
;
;=======================================
        public  _smpte_efw
_smpte_efw proc far
        push    bp
        mov     bp,sp
        push    es
        cli
        mov     ax,smpte_efw            ;return EFW stae
        mov     smpte_efw,0
        or      ax,ax                   ;if EFW set, return address
        jz      smpte_efwx
        les     bx,[bp+6]               ;get ptr to result
        mov     cx,word ptr last_smpte  ;move last frame address to result
        mov     word ptr es:[bx],cx
        mov     cx,word ptr last_smpte+2
        mov     word ptr es:[bx+2],cx
smpte_efwx:
        sti
        les     bx,[bp+10]              ;pass return code
        mov     es:[bx],ax
        pop     es
        pop     bp
        ret     8
_smpte_efw endp
;=======================================
; _cuepoint_efw
;
; Cue Point EFW status
;
; Model:
; function cuepoint_efw: integer;
;               0 = event not posted
;               1 = event posted
;
;=======================================
        public  _cuepoint_efw
_cuepoint_efw proc far
        cli
        mov     ax,cpt_efw              ;return EFW status
        mov     cpt_efw,0
        sti
        ret
_cuepoint_efw endp
;=======================================
; _realtime_efw
;
; MIDI realtime command reception status
;
; Model:
; function realtime_efw: integer;
;               0 = event not posted
;               0xFA = start
;               0xFB = continue
;               0xFC = stop
;
;=======================================
        public  _realtime_efw
_realtime_efw proc far
        cli
        mov     ax,realtime_efw         ;return EFW status
        mov     realtime_efw,0
        sti
        ret
_realtime_efw endp
;=======================================
; _strk_end_efw
;
; SMPTE track end status
;
; Model:
; function strk_end_efw: integer;
;               0 = event not posted
;               1 = event posted
;
;=======================================
        public  _strk_end_efw
_strk_end_efw proc far
        cli
        mov     ax,strk_end_efw         ;return EFW status
        mov     strk_end_efw,0
        sti
        ret
_strk_end_efw endp
;=======================================
; _rec_bytes
;
; Return amount of data recorded in current
; recording buffer.
;
; Model:
; function _rec_bytes: integer;
;
; returned value
;   number of bytes used in record buffer
;
;=======================================
        public  _rec_bytes
_rec_bytes proc  far
        mov     ax,rbx
        ret
_rec_bytes endp
;=======================================
; _rec_init
;
; Intialize MIDI event recording
;
; Note: Buffer size is expected to be
; at least 16 bytes smaller than the record
; buffer, so that recording end can be
; correctly stored in track.
;
; Model:
; procedure _rec_init(buffseg,buffoff,size: integer);
;
; buffseg - buffer segment address
; buffoff - buffer offset address
; size - size of buffer, in bytes
;
;=======================================
        public  _rec_init
_rec_init proc  far
        push    bp
        mov     bp,sp
        cli
        mov     ax,[bp+p1of3]           ;buffer segment address
        mov     rbuffseg,ax
        mov     ax,[bp+p2of3]           ;buffer address/size
        mov     rbuff,ax
        mov     ax,[bp+p3of3]
        mov     rbsize,ax
        sub     ax,ax
        mov     rbx,ax                  ;reset buffer index
        mov     eventp,offset event_buff ;reset event assembler
; clear all efws
        mov     pagerdy,al              ;reset page ready
        mov     rbfull_flag,ax
        mov     ackn_efw,ax
        mov     playend_efw,ax
        mov     recordend_efw,ax
        mov     conductor_efw,ax
        mov     me_efw,ax
;
        mov     cx,8
        sub     bx,bx
ri1:
        mov     track_efw[bx],ax
        add     bx,2
        loop    ri1
;
        sti
        pop     bp
        ret     6
_rec_init endp
;=======================================
; _coproc_slih
;
; Co-processor mode SLIH entry point
;
; Inputs:
;   al = received character
; Outputs:
;   None.
; Notes:
;   Clock tic interrupts are passed to
;       the clock services manager.
;   Event flag words are posted when
;       event messages arrive.
;   NOT callable from Pascal
;=======================================
        public  _coproc_slih
_coproc_slih proc near
        sub     ah,ah
        mov     data_in,al              ;save incoming char
        mov     dx,ax
        jmp     word ptr mqstatew       ;branch directly to current state routine
;=======================================
; Start of a new event
;=======================================
new_event:
        mov     eventp,offset event_buff ;reset assembly ptr
        cmp     al,0F0h                 ;midi data or mark?
        jb      midi_event              ;yes
        cmp     al,0F8h                 ;track data request
        jb      tdr_event               ;yes
;
        sub     ax,00F8h                ;compute index to jump table
        shl     ax,1
        mov     bx,ax
        jmp     word ptr eventcases[bx] ;jump to case handler
;=======================================
; Track data request event
;=======================================
tdr_event:
        and     ax,0007h                ;isolate track number
        shl     ax,1
        mov     bx,ax
        mov     track_efw[bx],1         ;post efw
        mov     mqstatew,offset new_event ;new event is next state
        jmp     rpihxt
;=======================================
; Midi event
;=======================================
midi_event:
        call    rec_data                ;record timing byte
        mov     mqstatew,offset midi1   ;next state
        jmp     rpihxt
;=======================================
; MIDI event or mark
;=======================================
midi1:
        test    al,80h                  ;new running status?
        jz      same_event              ;no, use running status
rih_new:
        push    ax
        call    rec_data                ;record first (status) byte
        pop     bx
;
        cmp     bl,0F0h                 ;normal midi event?
        jl      rih_midi                ;yes
        cmp     bl,0F9h                 ;measure end?
        je      rih_new2                ;yes
        cmp     bl,0FCh                 ;end of recording mark?
        je      rih_new1                ;yes
;
;=======================================
; First byte (possibly new status)
;=======================================
rih_midi:
        and     bx,0070h                ;isolate state
        mov     cl,4
        shr     bx,cl                   ;new running status
        mov     ah,data_cnt[bx]         ;get count for this code
;
        mov     res_cnt,ah              ;save new count
        mov     rstatec,ah
        mov     mqstatew,offset rcvdata ;establish new state to receive data
        jmp     rpihxt
;
rih_new1:
        mov     recordend_efw,1         ;post end efw
        mov     mqstatew,offset new_event ;establish new state for next event
        jmp     rpirec                  ;go record this event
rih_new2:
        mov     me_efw,1                ;post end efw
        mov     mqstatew,offset new_event ;establish new state for next event
        jmp     rpirec                  ;go record this event
;
; receive data after status byte
;
rcvdata:                                ;receive event data 80-e0
        call    rec_data                ;record data byte
        dec     res_cnt                 ;decrement
        jz      rcvd1                   ;all data rec'd
        jmp     rpihxt
rcvd1:
        mov     mqstatew,offset new_event ;new event, with timing byte is next
        jmp     rpirec                  ;go record this event
;
; record data for same event (running status)
;
same_event:
        mov     al,data_in              ;record first byte of running status
        call    rec_data
        mov     ah,rstatec
        dec     ah
        jz      rpi_rs                  ;single byte event
        mov     res_cnt,ah              ;remaining count of data to be rec'd
        mov     mqstatew,offset rcvdata ;next state
        jmp     rpihxt
rpi_rs:
        mov     mqstatew,offset new_event ;new event is next state
        jmp     rpirec                  ;go record this event
;=======================================
; Timer overflow event
;=======================================
toflow_event:
        mov     al,data_in
        call    rec_data                ;record event
        mov     mqstatew,offset new_event ;new event is next state
        jmp     rpirec                  ;go record this event
;=======================================
; Conductor data request event
;=======================================
cdr_event:
        mov     conductor_efw,1         ;post efw
        mov     mqstatew,offset new_event ;new event is next state
        jmp     rpihxt
;=======================================
; Command acknowledge event
;=======================================
$fe_event:
        inc     ackn_efw                ;post efw
        mov     mqstatew,offset new_event ;new event is next state
        jmp     rpihxt
;=======================================
; Host clock event
;=======================================
$fd_event:
        inc     clk_efw                 ;post efw
        call    clock_tic               ;notify clock manager
        mov     mqstatew,offset new_event ;new event is next state
        jmp     rpihxt
;=======================================
; System message event
;=======================================
$ff_event:
        mov     al,data_in
        call    rec_data                ;record sysex data
        mov     mqstatew,offset system_event ;next state
        jmp     rpihxt
;=======================================
; First byte of system event
;
; Note: Beware, system events can be recorded
; but they can not be played back through
; a play track.
;=======================================
system_event:
        call    rec_data                ;record system msg marker
        cmp     al,0FFh                 ;extended system message?
        jne     sys_ev                  ;no
        mov     mqstatew,offset sys_eventx ;next state for extended msg
        jmp     rpihxt
sys_ev:
        cmp     al,0F0h                 ;sysex?
        jne     sys_ev1                 ;no
        mov     mqstatew,offset sysex_event ;next state
        jmp     rpihxt
sys_ev1:
        cmp     al,0F2h                 ;SPP?
        jne     sys_f3                  ;no
        mov     mqstatew,offset rcvspp1 ;set up to receive next SPP byte
        jmp     rpihxt
sys_f3:
        mov     ah,1                    ;try F3=song select
        cmp     al,0F3h
        jne     sys_rt                  ;maybe real time
;
        mov     res_cnt,ah              ;expected data length
        mov     mqstatew,offset rcvdata ;set up to receive the rest of data
        jmp     rpihxt
;
sys_rt:
        cmp     al,0FAh                 ;real time?
        jb      sys_ignore              ;no, ignore
        cmp     al,0FDh
        jnb     sys_ignore              ;nope
        sub     ah,ah                   ;yes, set the EFW
        mov     realtime_efw,ax
sys_ignore:
        mov     mqstatew,offset new_event ;next state
        jmp     rpihrsb
;=======================================
; Continuation of system exclusive
;=======================================
sysex_event:                                    ;sys ex
        call    rec_data                ;record sysex data
        test    data_in,80h             ;EOX or new status?
        jnz     sysex_end               ;yes
        jmp     rpihxt                  ;no, continue this state
sysex_end:
        cmp     byte ptr event_buff+2,7Fh ;possible SMPTE msg?
        jne     sysex_end9              ;no
        mov     ax,word ptr event_buff+4 ;get sub id
        cmp     ax,0101h                ;SMPTE msg?
        jne     sysex_end9              ;no
        mov     ax,word ptr event_buff+6 ;save the frame address data
        mov     word ptr last_smpte,ax
        mov     ax,word ptr event_buff+8
        mov     word ptr last_smpte+2,ax
        mov     smpte_efw,1             ;signal frame received
        mov     mqstatew,offset new_event ;next state
        jmp     short rpihrsb
;
sysex_end9:
        mov     mqstatew,offset new_event ;next state
        jmp     short rpirec
;=======================================
; Continuation of SPP
;=======================================
rcvspp1:
        and     al,7Fh                  ;isolate low order 7 bits
        mov     spp_byte1,al            ;save data
        mov     mqstatew,offset rcvspp2 ;set up to receive next SPP byte
        jmp     short rpihxt
rcvspp2:
        mov     cl,7                    ;left adjust high order bits
        sub     ah,ah
        shl     ax,cl
        or      al,spp_byte1
        mov     last_spp,ax             ;save calculated SPP
        mov     spp_efw,1               ;set EFW to signal SPP received
        mov     mqstatew,offset new_event ;next state
        jmp     short rpihrsb
;=======================================
; Continuation extended system message
;=======================================
sys_eventx:
        or      al,al                   ;cue point hit?
        jnz     sys_xm1                 ;no
        mov     cpt_efw,1               ;set cue point EFW
        jmp     short sys_xmx
sys_xm1:
        cmp     al,1                    ;end of SMPTE track?
        jne     sys_xm2                 ;no
        mov     strk_end_efw,1          ;set SMPTE track end EFW
sys_xm2:
sys_xmx:
        mov     mqstatew,offset new_event ;next state
        jmp     short rpihrsb
;=======================================
; All end event
;=======================================
$fc_event:
        mov     playend_efw,1
        jmp     short rpihxt
;=======================================
; Invalid/unexpected events
;=======================================
$fa_event:
$fb_event:
        jmp     short rpihxt
;=======================================
; Record the assembled event
;=======================================
rpirec:
        call    rec_asmdata
;=======================================
; The grand exit from interrupt handling!
;=======================================
rpihrsb:
rpihxt:
        ret
_coproc_slih endp
;
CODE    ENDS
        END
