{
    ViewEdit1.pas - View editor
    Copyright (C) 1997-1999 Peter Kelly <peter@area51.org.au>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
}

{$H+}   {use long strings - required for description}
unit ViewEdit1;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, defines, ComCtrls, Grids, Calendar, Buttons, Menus,
  Gauges, ToolWin, SRGrad;


type
  TViewToolType = (None,DrawTool,FillTool);
  TViewEditUndoType = (Draw,SizeChange,MoveCel);

  TViewEdit = class(TForm)
    Panel1: TPanel;
    Panel2: TPanel;
    LoopDisplay: TLabel;
    CelDisplay: TLabel;
    Label1: TLabel;
    Label2: TLabel;
    HeightDisplay: TEdit;
    WidthDisplay: TEdit;
    LoopAdjust: TUpDown;
    CelAdjust: TUpDown;
    WidthAdjust: TUpDown;
    HeightAdjust: TUpDown;
    Label3: TLabel;
    Label4: TLabel;
    ZoomInBut: TBitBtn;
    ZoomOutBut: TBitBtn;
    DescriptionEditBut: TButton;
    DescriptionCheckBox: TCheckBox;
    DrawRadio: TRadioButton;
    FillRadio: TRadioButton;
    PaletteImage: TImage;
    MainMenu1: TMainMenu;
    ViewMenu: TMenuItem;
    LoopMenu: TMenuItem;
    CelMenu: TMenuItem;
    LoopNext: TMenuItem;
    LoopPrev: TMenuItem;
    LoopFirst: TMenuItem;
    LoopLast: TMenuItem;
    N1: TMenuItem;
    LoopInsert: TMenuItem;
    LoopDelete: TMenuItem;
    LoopClear: TMenuItem;
    CelNext: TMenuItem;
    CelPrev: TMenuItem;
    CelFirst: TMenuItem;
    CelLast: TMenuItem;
    N2: TMenuItem;
    CelInsert: TMenuItem;
    CelDelete: TMenuItem;
    CelClear: TMenuItem;
    ViewClose: TMenuItem;
    EditMenu: TMenuItem;
    EditUndo: TMenuItem;
    N3: TMenuItem;
    ViewSaveToFile: TMenuItem;
    SaveDialog1: TSaveDialog;
    LoadingGauge: TGauge;
    Label5: TLabel;
    TransSetBut: TButton;
    TransColDisplay: TLabel;
    CelAppend: TMenuItem;
    LoopAppend: TMenuItem;
    N4: TMenuItem;
    CelFlipH: TMenuItem;
    CelFlipV: TMenuItem;
    N5: TMenuItem;
    EditCopyCel: TMenuItem;
    EditPasteCel: TMenuItem;
    MirrorPanel: TPanel;
    ViewNew: TMenuItem;
    ViewLoadFromFile: TMenuItem;
    CelUpBut: TBitBtn;
    CelRightBut: TBitBtn;
    CelDownBut: TBitBtn;
    CelLeftBut: TBitBtn;
    OpenDialog1: TOpenDialog;
    ViewSaveToGame: TMenuItem;
    ViewSaveToGameAs: TMenuItem;
    LoopMirrorSelectCombo: TComboBox;
    Label6: TLabel;
    ViewPanel: TScrollBox;
    ViewImage: TImage;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure LoopAdjustClick(Sender: TObject; Button: TUDBtnType);
    procedure CelAdjustClick(Sender: TObject; Button: TUDBtnType);
    procedure ZoomInButClick(Sender: TObject);
    procedure ZoomOutButClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure DescriptionCheckBoxClick(Sender: TObject);
    procedure DescriptionEditButClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure DrawRadioClick(Sender: TObject);
    procedure FillRadioClick(Sender: TObject);
    procedure ViewImageMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure ViewImageMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure ViewImageMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure PaletteImageMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure ViewCloseClick(Sender: TObject);
    procedure LoopNextClick(Sender: TObject);
    procedure LoopPrevClick(Sender: TObject);
    procedure LoopFirstClick(Sender: TObject);
    procedure LoopLastClick(Sender: TObject);
    procedure CelNextClick(Sender: TObject);
    procedure CelPrevClick(Sender: TObject);
    procedure CelFirstClick(Sender: TObject);
    procedure CelLastClick(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure EditUndoClick(Sender: TObject);
    procedure TransSetButClick(Sender: TObject);
    procedure WidthAdjustClick(Sender: TObject; Button: TUDBtnType);
    procedure HeightAdjustClick(Sender: TObject; Button: TUDBtnType);
    procedure WidthDisplayExit(Sender: TObject);
    procedure WidthDisplayKeyPress(Sender: TObject; var Key: Char);
    procedure WidthDisplayEnter(Sender: TObject);
    procedure WidthDisplayClick(Sender: TObject);
    procedure HeightDisplayExit(Sender: TObject);
    procedure HeightDisplayKeyPress(Sender: TObject; var Key: Char);
    procedure HeightDisplayEnter(Sender: TObject);
    procedure HeightDisplayClick(Sender: TObject);
    procedure CelInsertClick(Sender: TObject);
    procedure CelDeleteClick(Sender: TObject);
    procedure CelAppendClick(Sender: TObject);
    procedure CelClearClick(Sender: TObject);
    procedure LoopInsertClick(Sender: TObject);
    procedure LoopAppendClick(Sender: TObject);
    procedure LoopDeleteClick(Sender: TObject);
    procedure LoopClearClick(Sender: TObject);
    procedure EditCopyCelClick(Sender: TObject);
    procedure EditPasteCelClick(Sender: TObject);
    procedure Loop1MirrorCheck1Click(Sender: TObject);
    procedure ViewNewClick(Sender: TObject);
    procedure CelUpButClick(Sender: TObject);
    procedure CelDownButClick(Sender: TObject);
    procedure CelLeftButClick(Sender: TObject);
    procedure CelRightButClick(Sender: TObject);
    procedure CelFlipHClick(Sender: TObject);
    procedure CelFlipVClick(Sender: TObject);
    procedure ViewLoadFromFileClick(Sender: TObject);
    procedure ViewSaveToFileClick(Sender: TObject);
    procedure ViewSaveToGameAsClick(Sender: TObject);
    procedure SaveAsResource(ResNum:byte);
    procedure ViewSaveToGameClick(Sender: TObject);
    procedure LoopMirrorSelectComboChange(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
  private
    { Private declarations }
  public
    { Public declarations }
    MainView            : TView; {the view we are working on}
                                 {--This variable holds all the information about loops, cels & associated properties}
                                   {and the images themselves!}
    ViewEditorNum       : word;  {number of the view editor windows - used to distinguish them apart}
    ViewNum             : byte;  {number of the view being edited}
    LoadedFromGame      : boolean; {true if loaded from the game, false if new view or loaded from external file}
    ViewLoaded          : boolean; {whether or not the view has loaded yet}
                                   {--added due to problems when setting loopadjust.max and celadjust.max during view loading progress}
                                     {where they wanted to run DisplayView when the value was changed.}
    ViewModified        : boolean;

    SelectedLoop        : byte;    {current loop being displayed}
    SelectedCel         : byte;    {current cel being displayed}
    ScaleFactor         : byte;    {integral multiple of size of original image - default=2}
    NewlyOpenedWindow   : boolean;
    SelectedTool        : TViewToolType;
    MouseIsDown         : boolean; {is true when EITHER mouse button is held down in the drawing area}
    LeftButPressed      : boolean; {if false, right button pressed -> tells us which colour to use}
    OldViewX            : byte; {previous x,y in drawing field}
    OldViewY            : byte; {--these help tell us if the mouse has moved to a different AGI pixel in the drawing area}
    LCol, RCol          : byte; {EGA colours assigned to the left and right mouse buttons}
    TransCol            : byte; {Transparency colour of the current cel}
    LoopMirrorSelectLoopno : array[0..8] of byte; {determines which item in LoopMirrorSelectCombo is associated with what loop no.}
    LastLoopStatusDisplayed : byte; {the last loop the status was displayed for, regardless of the cel no.}
                                    {--used to decide whether to update the LoopMirrorSelectCombo or not}
                                    {--if value is $FF, no loop status has been displayed}

    UndoBitmap          : TBitmap; {bitmap to store the undo image}
    UndoLoopno          : byte;  {loop no. of UndoBitmap}
    UndoCelno           : byte;  {cel no. of UndoBitmap}
    UndoType            : TViewEditUndoType;
    UndoWidth           : byte; {width/height of undo image}
    UndoHeight          : byte; {--come to think of it, shouldn't I be able to use UndoBitmap.Width and UndiBitmap.Height?}

    function AskClose : boolean;
    procedure UpdateStatus(Loopno,Celno:byte);
    procedure DisplayView(Loopno,Celno:byte);
    function LoadView(ResourceData:TResource) : boolean;
    function SaveView : TResource;
    procedure SaveUndo(NewUndoType:TViewEditUndoType);
    procedure SaveExistingCel;
    procedure ReCalculateMaxHW;

    function MirrorCel(Loopno,Celno:byte) : boolean;
    procedure UnMirrorLoop(Loopno:byte);
    procedure NewView;
    function ClearCel(Loopno,Celno:byte) : boolean;
    function AppendCel(Loopno:byte) : boolean;
    function ClearLoop(Loopno:byte) : boolean;
    function AppendLoop : boolean;
    function DeleteLastCel(Loopno:byte) : boolean;
    function InsertCel(Loopno,Celno:byte) : boolean;
    function DeleteCel(Loopno,Celno:byte) : boolean;
    function CopyCel(SourceLoop,SourceCel,TargetLoop,TargetCel:byte) : boolean;
    function CopyLoop(SourceLoop,TargetLoop:byte;MirrorTarget:boolean) : boolean;
    function DeleteLastLoop : boolean;
    function InsertLoop(Loopno:byte) : boolean;
    function DeleteLoop(Loopno:byte) : boolean;
  end;

var  MaxViewEditWins : byte = 6;
     ViewEditWin: array[1..6] of TViewEdit;
     ViewEditWinUsed : array[1..6] of boolean;
     DefaultViewEditWidth : word;
     DefaultViewEditHeight : word;

procedure EditView(ViewNum:byte);

implementation

uses ViewEditDescription1, VolMan2, GetResourceNum1, main1, Clipbrd,
  ResourceWin1;

const  DefaultCelWidth : byte = 16;
       DefaultCelHeight : byte = 32;
       MaxCelWidth = 160;
       MaxCelHeight = 168;

{$R *.DFM}

{*************************************************************}
function TViewEdit.AskClose : boolean;
{*************************************************************}
var SaveMessageResult : integer;
begin
  AskClose := True;
  if ViewModified then
  begin
    if LoadedFromGame then
    begin
      SaveMessageResult := Application.MessageBox('Do you want to save changes before closing? (view will be saved to game)','View Editor',MB_YESNOCANCEL);
      if SaveMessageResult = IDYES then ViewSaveToGameClick(Application)
      else if SaveMessageResult = IDCANCEL then AskClose := False;
      if (SaveMessageResult = IDYES) or (SaveMessageResult = IDNO) then ViewModified := False;
    end
    else
    begin
      SaveMessageResult := Application.MessageBox('Do you want to save changes before closing?  (view will be saved to a file)','View Editor',MB_YESNOCANCEL);
      if SaveMessageResult = IDYES then ViewSaveToFileClick(Application)
      else if SaveMessageResult = IDCANCEL then AskClose := False;
      if (SaveMessageResult = IDYES) or (SaveMessageResult = IDNO) then ViewModified := False;
    end;
  end;
end;

{*************************************************************}
procedure EditView(ViewNum:byte);
{*************************************************************}
var ResourceData : TResource;
    CurWinNum : byte;
    ThisWinNum : byte;
    ResourceLoadSuccess : boolean;
begin
  ThisWinNum := 0;
  for CurWinNum := 1 to MaxViewEditWins do
    if not ViewEditWinUsed[CurWinNum] then ThisWinNum := CurWinNum;
  if ThisWinNum > 0 then
  begin
    ViewEditWinUsed[ThisWinNum] := True;
    ViewEditWin[ThisWinNum] := TViewEdit.Create(ResourceWin);
    ViewEditWin[ThisWinNum].ScaleFactor := 2;
    ViewEditWin[ThisWinNum].NewlyOpenedWindow := true;
    ViewEditWin[ThisWinNum].Caption := 'View Editor - ' + ResourceName(VIEW,ViewNum);
    ViewEditWin[ThisWinNum].ViewNum := ViewNum;
    ViewEditWin[ThisWinNum].LoadedFromGame := True;
    ViewEditWin[ThisWinNum].MainView.NumLoops := 0;
    if ViewEditWin[ThisWinNum].ClientHeight < ViewEditWin[ThisWinNum].PaletteImage.Top + ViewEditWin[ThisWinNum].PaletteImage.Height then
      ViewEditWin[ThisWinNum].ClientHeight := ViewEditWin[ThisWinNum].PaletteImage.Top + ViewEditWin[ThisWinNum].PaletteImage.Height;
    ViewEditWin[ThisWinNum].OnResize(ResourceWin);
    ResourceData := ReadResource(VIEW,ViewNum);
    if ResourceData.Size > 0 then
    begin
      ViewEditWin[ThisWinNum].ViewEditorNum := ThisWinNum;
      if ViewEditWin[ThisWinNum].LoadView(ResourceData) then {loadview frees the memory after it has used it.}
        ViewEditWin[ThisWinNum].Show
      else
      begin
        ShowMessage('Error loading View. The resource may be corrupt.');
        ViewEditWin[ThisWinNum].MainView.NumLoops := 0;
        ViewEditWin[ThisWinNum].Close;
      end;
    end
    else
    begin
      ShowMessage('Error: Could not load resource. The files may be open or the resource may be corrupt.');
      ViewEditWin[ThisWinNum].MainView.NumLoops := 0;
      ViewEditWin[ThisWinNum].Close;
    end;
  end;
end;

{*************************************************************}
function GetEGACol(Col24Bit:integer) : byte;
{*************************************************************}
var CurCol : byte;
begin
  GetEGACol := 0;
  for CurCol := 0 to 15 do
    if EGACOL[CurCol] = Col24Bit then GetEGACOL := CurCol;
end;

{*************************************************************}
procedure FixPalette(Bitmap:TBitmap);
{*************************************************************}
begin
  Bitmap.Assign(BaseBitmap);
  Bitmap.Palette := CreateEGAPalette;
  SelectPalette(Bitmap.Canvas.Handle, EGAPalHandle, True);
end;

{*************************************************************}
procedure MirrorBitmap(var MBitmap:TBitmap);
{*************************************************************}
begin
  MBitmap.Canvas.StretchDraw(Rect(MBitmap.Width-1,0,-1,MBitmap.Height),MBitmap);
end;

{*************************************************************}
procedure TViewEdit.UpdateStatus(Loopno,Celno:byte);
{*************************************************************}
var CurLoop : byte;
begin
{  if (Loopno = 1) and (not MirrorPanel.Visible) then
   MirrorPanel.Visible := True;
  if (Loopno <> 1) and MirrorPanel.Visible then
    MirrorPanel.Visible := False;}
  TransColDisplay.Color := EGACOL[MainView.CelTransCol[Loopno,Celno]];
  LoopAdjust.Max := MainView.NumLoops-1;
  CelAdjust.Max := MainView.NumCels[Loopno]-1;
  LoopAdjust.Position := Loopno;
  CelAdjust.Position := Celno;
  LoopDisplay.Caption := IntToStr(Loopno) + '/' + IntToStr(MainView.NumLoops-1);
  CelDisplay.Caption := IntToStr(Celno) + '/' + IntToStr(MainView.NumCels[Loopno]-1);
  NewlyOpenedWindow := false;
  WidthAdjust.Position := ViewImage.Picture.Bitmap.Width;
  HeightAdjust.Position := ViewImage.Picture.Bitmap.Height;
  WidthDisplay.Text := IntToStr(ViewImage.Picture.Bitmap.Width);
  HeightDisplay.Text := IntToStr(ViewImage.Picture.Bitmap.Height);
  if LastLoopStatusDisplayed <> Loopno then
  begin
    if Loopno >= 8 then
    begin
      Label6.Enabled := False;
      LoopMirrorSelectCombo.Items.Clear;
      LoopMirrorSelectCombo.Enabled := False;
    end
    else
    begin
      Label6.Enabled := True;
      LoopMirrorSelectCombo.Enabled := True;
      LoopMirrorSelectCombo.Items.Clear;
      LoopMirrorSelectCombo.Items.Add('No other loop');
      LoopMirrorSelectLoopno[0] := $FF; {no other loop}
      if not MainView.MirrorLoop[Loopno] then LoopMirrorSelectCombo.ItemIndex := 0;
      for CurLoop := 0 to MainView.NumLoops - 1 do
      begin
        if (CurLoop<8) and (CurLoop<>Loopno) and ((not MainView.MirrorLoop[CurLoop]) or (MainView.MirrorLoop[CurLoop] and (MainView.MirrorLoopNum[CurLoop]=Loopno))) then
        begin
          LoopMirrorSelectCombo.Items.Add('Loop '+IntToStr(CurLoop));
          LoopMirrorSelectLoopno[LoopMirrorSelectCombo.Items.Count-1] := CurLoop;
        end;
        if (MainView.MirrorLoop[CurLoop] and (MainView.MirrorLoopNum[CurLoop]=Loopno)) then
          LoopMirrorSelectCombo.ItemIndex := LoopMirrorSelectCombo.Items.Count-1;
      end;
    end;
  end;
  LastLoopStatusDisplayed := Loopno;
end;

{*************************************************************}
procedure TViewEdit.DisplayView(Loopno,Celno:byte);
{*************************************************************}
begin
  if (Loopno <= MainView.NumLoops - 1) and (Celno <= MainView.NumCels[Loopno]-1) then
  begin
    ViewImage.Picture.Bitmap.Assign(MainView.CelBitmap[Loopno,Celno]);
    ViewImage.Width := MainView.CelBitmap[Loopno,Celno].Width*2*ScaleFactor;
    ViewImage.Height := MainView.CelBitmap[Loopno,Celno].Height*ScaleFactor;
    ViewImage.Top := MainView.MaxHeight[Loopno]*ScaleFactor - ViewImage.Height;
    TransCol := MainView.CelTransCol[Loopno,Celno];
    TransColDisplay.Color := EGACOL[TransCol];
    if MainView.MirrorLoop[Loopno] and (MainView.MirrorLoopNum[Loopno]<Loopno) then
      ViewImage.Left := MainView.MaxWidth[Loopno]*ScaleFactor*2-ViewImage.Width
    else if ViewImage.Left > 0 then ViewImage.Left := 0;
    UpdateStatus(Loopno,Celno);
  end
  else
  begin
    ShowMessage('FATAL ERROR (DisplayView): Could not display cel.');
    Halt;
  end;
end;

{*************************************************************}
function TViewEdit.LoadView(ResourceData:TResource) : boolean;
{*************************************************************}

var ResPos               : word;  {current position in resource buffer}
    LoopLoc              : array[0..MaxLoops-1] of word;  {location of each loop relative to start of resource}
    CelLoc               : array[0..MaxLoops-1,0..MaxCels-1] of word;  {location of each cel relative to start of loop}
    CurLoop, CurCel      : byte;  {current loop and cel being worked with}
    CelX, CelY           : byte;  {x and y location of current pixel when drawing cel}
    MirrorCel            : boolean;  {whether or not the current cel being drawn is mirrored}
    CurByte              : byte;  {current byte just read from resource buffer}
    CelWidth, CelHeight  : byte;  {width and height of current cel}
    TransCol             : byte;  {transparency colour of current cel}
    ChunkCol, ChunkWidth : byte;  {colour and width of current chunk of pixels when drawing cel}
    TempStr              : string;  {temp. string used for printing numerical values as strings}
    ErrorMessage         : string;  {error message - null if no errors}
    TempBitmap           : TBitmap;  {bitmap used to draw cel before transferred to MainView.CelBitmap[loop,cel]}
    TmpLoopno            : byte;  {used for identifying duplicate loops}
    DescLoc              : word;    {location of description in resource}
    Description          : string; {remember this is a null-terminated string, so we can use > 255 bytes}

  function ReadByte : byte;
  begin
    if ResPos < ResourceData.Size then
    begin
      ReadByte := ResourceData.Data^[ResPos];
      ResPos := ResPos + 1;
    end
    else ReadByte := 0;
  end;

  function ReadWord : word;
  var lsbyte,msbyte : byte;
  begin
    lsbyte := ReadByte;
    msbyte := ReadByte;
    ReadWord := msbyte * 256 + lsbyte;
  end;

begin
  ErrorMessage := '';
  ViewLoaded := False;
  if MainView.NumLoops > 0 then
  for CurLoop := 0 to MainView.NumLoops - 1 do
    for CurCel := 0 to MainView.NumCels[CurLoop] - 1 do
      MainView.CelBitmap[CurLoop,CurCel].Free;
  LoadingGauge.MaxValue := 0;
  ResPos := 2;
  MainView.NumLoops := ReadByte;
  DescLoc := ReadWord;
  if MainView.NumLoops > MaxLoops then ErrorMessage := 'Error: Too many loops';
{read loop locations}
  if ErrorMessage = '' then
  begin
    ResPos := 5;
    for CurLoop := 0 to MainView.NumLoops - 1 do
    begin
      LoopLoc[CurLoop] := ReadWord;
      if (LoopLoc[CurLoop] > ResourceData.Size) and (ErrorMessage = '') then
        ErrorMessage := 'Error: Loop start beyond end of resource';
    end;
  end;
{read cel locations}
  if ErrorMessage = '' then
  begin
    LoadingGauge.MaxValue := 0;
    for CurLoop := 0 to MainView.NumLoops - 1 do
    begin
      MainView.MaxWidth[CurLoop] := 0;
      MainView.MaxHeight[CurLoop] := 0;
      ResPos := LoopLoc[CurLoop];
      MainView.NumCels[CurLoop] := ReadByte;
      for CurCel := 0 to MainView.NumCels[CurLoop] - 1 do
      begin
        CelLoc[CurLoop,CurCel] := ReadWord;
        if (CelLoc[CurLoop,CurCel] > ResourceData.Size) and (ErrorMessage = '') then
          ErrorMessage := 'Error: Cel start beyond end of resource';
        LoadingGauge.MaxValue := LoadingGauge.MaxValue + 1;
      end;
    end;
  end;
{work out what loops are mirrors of what}
  if ErrorMessage = '' then
  begin
    for CurLoop := 0 to MainView.NumLoops - 1 do
      MainView.MirrorLoop[CurLoop] := False;
    for CurLoop := 0 to MainView.NumLoops - 1 do
    begin
      if not MainView.MirrorLoop[CurLoop] then
        for TmpLoopNo := MainView.NumLoops - 1 downto CurLoop + 1 do
          if LoopLoc[TmpLoopNo] = LoopLoc[CurLoop] then
          begin
            MainView.MirrorLoop[CurLoop] := True;
            MainView.MirrorLoopNum[CurLoop] := TmpLoopNo;
            MainView.MirrorLoop[TmpLoopNo] := True;
            MainView.MirrorLoopNum[TmpLoopNo] := CurLoop;
          end;
    end;
  end;
{read actual cel data}
  if ErrorMessage = '' then
  begin
    LoadingGauge.Progress := 0;
    LoadingGauge.Visible := true;
    TempBitmap := TBitmap.Create;
    FixPalette(TempBitmap);
    for CurLoop := 0 to MainView.NumLoops - 1 do
    if not (MainView.MirrorLoop[CurLoop] and (MainView.MirrorLoopNum[CurLoop]<CurLoop)) then
    begin
      for CurCel := 0 to MainView.NumCels[CurLoop] - 1 do
      begin
        ResPos := LoopLoc[CurLoop] + CelLoc[CurLoop,CurCel];
        MainView.CelWidth[CurLoop,CurCel] := ReadByte;
        MainView.CelHeight[CurLoop,CurCel] := ReadByte;
        CelWidth := MainView.CelWidth[CurLoop,CurCel];
        CelHeight := MainView.CelHeight[CurLoop,CurCel];
        CurByte := ReadByte;
        MainView.CelTransCol[CurLoop,CurCel] := CurByte mod $10;
        TransCol := MainView.CelTransCol[CurLoop,CurCel];

        if CelWidth > MainView.MaxWidth[CurLoop] then MainView.MaxWidth[CurLoop] := CelWidth;
        if CelHeight > MainView.MaxHeight[CurLoop] then MainView.MaxHeight[CurLoop] := CelHeight;

        TempBitmap.Canvas.Brush.Style := bsSolid;
        TempBitmap.Width := CelWidth;
        TempBitmap.Height := CelHeight;
        TempBitmap.Canvas.Brush.Color := EGACOL[TransCol];
        TempBitmap.Canvas.FillRect(Rect(0,0,CelWidth,CelHeight));
        CelY := 0;
        repeat
        begin
          CelX := 0;
          repeat
          begin
            CurByte := ReadByte;
            if CurByte > 0 then
            begin
              ChunkCol := Curbyte div $10;
              ChunkWidth := Curbyte mod $10;
              for CelX := CelX to CelX + ChunkWidth - 1 do
              begin
                TempBitmap.Canvas.Pixels[CelX,CelY] := EGACOL[ChunkCol];
              end;
            end;
          end until CurByte = 0;
          CelY := CelY + 1;
        end; until CelY >= CelHeight;
        MainView.CelBitmap[CurLoop,CurCel] := TBitmap.Create;
        MainView.CelBitmap[CurLoop,CurCel].Assign(TempBitmap);
        LoadingGauge.Progress := LoadingGauge.Progress + 1;
      end; {for CurCel := 0 to MainView.NumCels[CurLoop] - 1}
    end { if not (MainView.MirrorLoop[CurLoop] and (MainView.MirrorLoopNum[CurLoop]<CurLoop))}
    else
    begin
      MainView.NumCels[CurLoop] := MainView.NumCels[MainView.MirrorLoopNum[CurLoop]];
      MainView.MaxWidth[CurLoop] := MainView.MaxWidth[MainView.MirrorLoopNum[CurLoop]];
      MainView.MaxHeight[CurLoop] := MainView.MaxHeight[MainView.MirrorLoopNum[CurLoop]];
      for CurCel := 0 to MainView.NumCels[CurLoop] - 1 do
      begin
        MainView.CelWidth[CurLoop,CurCel] := MainView.CelWidth[MainView.MirrorLoopNum[CurLoop],CurCel];
        MainView.CelHeight[CurLoop,CurCel] := MainView.CelHeight[MainView.MirrorLoopNum[CurLoop],CurCel];
        MainView.CelTransCol[CurLoop,CurCel] := MainView.CelTransCol[MainView.MirrorLoopNum[CurLoop],CurCel];
        MainView.CelBitmap[CurLoop,CurCel] := TBitmap.Create;
        MainView.CelBitmap[CurLoop,CurCel].Assign(MainView.CelBitmap[MainView.MirrorLoopNum[CurLoop],CurCel]);
        MirrorBitmap(MainView.CelBitmap[CurLoop,CurCel]);
      end;
    end;
    LoopAdjust.Max := MainView.NumLoops-1;
    SelectedLoop := 0;
    SelectedCel := 0;
    LastLoopStatusDisplayed := $FF;
    DisplayView(0,0);
    TempBitmap.Free;
    LoadingGauge.Visible := false;
  end;
{read description}
  if ErrorMessage = '' then
  begin
    if DescLoc = 0 then
    begin
      MainView.Description.Clear;
      DescriptionCheckBox.Checked := false;
      DescriptionEditBut.Enabled := false;
    end
    else if DescLoc < PreviewView.Resource.Size - 1 then
    begin
      Description := '';
      MainView.Description.Clear;
      ResPos := DescLoc;
      repeat
      begin
        CurByte := ReadByte;
        if (CurByte > $0A) then Description := Description + chr(CurByte)
        else if CurByte = $0A then
        begin
          Description := Description + '\n';
{          MainView.Description.Add(Description);
          Description := '';}
        end;
      end until (ResPos > ResourceData.Size - 1) or (CurByte=0){ or (Length(Description)>=255)};
      MainView.Description.Add(Description);
      DescriptionCheckBox.Checked := true;
      DescriptionEditBut.Enabled := true;
    end;
  end;

  if Length(ErrorMessage) > 0 then
  begin
    ShowMessage(ErrorMessage);
    Close;
    LoadView := False;
  end
  else LoadView := True;
  FreeMem(ResourceData.Data,ResourceData.Size);
  ViewLoaded := True;
  ViewModified := False;
end;

{*************************************************************}
function TViewEdit.SaveView : TResource;
{*************************************************************}
type TErrorType     = (NoError,ResTooBigError);

var ViewResource    : TResource;
    OutputResource  : TResource;
    ResPos          : word;
    ResSize         : word;
    ErrorType       : TErrorType;

    CurLoop         : byte;
    CurCel          : byte;
    LoopLoc         : array[0..MaxLoops-1] of word;
    CelLoc          : array[0..MaxLoops-1,0..MaxCels-1] of word;

    CurX, CurY      : byte;
    ChunkLength     : byte;
    ChunkCol        : byte;
    ChunkCol24Bit   : integer;
    ColDiff         : boolean;
    CelSize         : word;   {no. of bytes that the cel takes up}
    CelMirrorSize   : word;   {no. of bytes that the cel's mirror takes up}
                              {--the no. of bytes that the cel takes up in the}
                                {resource is the greater of the above two values.}

    CurDescLine     : byte;
    CurDescChar     : word;
    DescLoc         : word;

    tmp1            : word;

  procedure WriteByte(TheByte : byte);
  begin
    if ResPos >= MaxResourceSize then
      ErrorType := ResTooBigError
    else
    begin
      ViewResource.Data^[ResPos] := TheByte;
      ResPos := ResPos + 1;
      if ResPos > ResSize then ResSize := ResPos;
    end;
  end;

begin
  ErrorType := NoError;
  ResSize := 0;
  ResPos := 0;
  GetMem(ViewResource.Data,MaxResourceSize);

  LoadingGauge.MaxValue := 1;
  for CurLoop := 0 to MainView.NumLoops - 1 do
    LoadingGauge.MaxValue := LoadingGauge.MaxValue + MainView.NumCels[CurLoop];
  LoadingGauge.Progress := 0;
  LoadingGauge.Visible := True;

  WriteByte($01);   {what do these two bytes do?}
  WriteByte($01);
  WriteByte(MainView.NumLoops);
  WriteByte($00);   {write description later, if it exists}
  WriteByte($00);
  LoopLoc[0] := 5 + MainView.NumLoops*2;
  for CurLoop := 0 to MainView.NumLoops - 1 do
  begin
    ResPos := LoopLoc[CurLoop];
    if not ((MainView.MirrorLoop[CurLoop]) and (MainView.MirrorLoopNum[CurLoop]<CurLoop)) then
    begin
      WriteByte(MainView.NumCels[CurLoop]);
      CelLoc[CurLoop,0] := 1 + MainView.NumCels[CurLoop]*2;
      for CurCel := 0 to MainView.NumCels[CurLoop] - 1 do
      begin
        ResPos := LoopLoc[CurLoop] + CelLoc[CurLoop,CurCel];
        WriteByte(MainView.CelWidth[CurLoop,CurCel]);
        WriteByte(MainView.CelHeight[CurLoop,CurCel]);
        if MainView.MirrorLoop[CurLoop] and (CurLoop<8) then
           WriteByte($80+CurLoop*$10+MainView.CelTransCol[CurLoop,CurCel])
        else WriteByte(MainView.CelTransCol[CurLoop,CurCel]);
        CelSize := 3;
        CelMirrorSize := 3;
        for CurY := 0 to MainView.CelHeight[CurLoop,CurCel]-1 do
        begin
          CurX := 0;
          repeat
          begin
            ChunkCol24Bit := MainView.CelBitmap[CurLoop,CurCel].Canvas.Pixels[CurX,CurY];
            ChunkCol := GetEGACol(ChunkCol24Bit);
            ChunkLength := 0;
            repeat
            begin
              ChunkLength := ChunkLength + 1;
              ColDiff := (MainView.CelBitmap[CurLoop,CurCel].Canvas.Pixels[CurX+ChunkLength,CurY] <> ChunkCol24Bit);
            end;
            until ColDiff or (ChunkLength >= 15) or (CurX + ChunkLength = MainView.CelWidth[CurLoop,CurCel]);
            if (CurX>0) or ((CurX=0) and (ChunkCol<>MainView.CelTransCol[CurLoop,CurCel])) then CelMirrorSize := CelMirrorSize + 1;
            CurX := CurX + ChunkLength;
            if (CurX < MainView.CelWidth[CurLoop,CurCel]) or
               ((CurX = MainView.CelWidth[CurLoop,CurCel]) and (ChunkCol <> MainView.CelTransCol[CurLoop,CurCel])) then
            begin
              WriteByte(ChunkCol*$10+ChunkLength);
              CelSize := CelSize + 1;
            end;
          end; until CurX = MainView.CelWidth[CurLoop,CurCel];
          WriteByte($00); {end of line}
        end;
        if MainView.MirrorLoop[CurLoop] then {write extra few bytes so mirrored cel will fit}
        begin
          if CelSize <> CelMirrorSize then
          begin
              for tmp1 := 1 to abs(CelSize - CelMirrorSize) do WriteByte($00)
          end;
        end;
        if CurCel < MainView.NumCels[CurLoop] - 1 then CelLoc[CurLoop,CurCel+1] := ResSize - LoopLoc[CurLoop];
        LoadingGauge.Progress := LoadingGauge.Progress + 1;
      end;
    end {if not ((MainView.MirrorLoop[CurLoop]) and (MainView.MirrorLoopNum[CurLoop]<CurLoop))}
    else LoadingGauge.Progress := LoadingGauge.Progress + MainView.NumCels[CurLoop];
    if CurLoop < MainView.NumLoops - 1 then LoopLoc[CurLoop+1] := ResSize;
  end; {for CurLoop := 0 to MainView.NumLoops - 1}
  for CurLoop := 1 to MainView.NumLoops - 1 do
    if (MainView.MirrorLoop[CurLoop]) and (MainView.MirrorLoopNum[CurLoop]<CurLoop) then
      LoopLoc[CurLoop] := LoopLoc[MainView.MirrorLoopNum[CurLoop]];
  ResPos := ResSize;
  DescLoc := ResPos;
  if MainView.Description.Count > 0 then
  begin
    for CurDescLine := 0 to MainView.Description.Count - 1 do
    begin
      if Length(MainView.Description[CurDescLine]) > 0 then
      begin
        CurDescChar := 0;
        repeat
        begin
          CurDescChar := CurDescChar + 1;
          if copy(MainView.Description[CurDescLine],CurDescChar,2) = '\n' then
          begin
            WriteByte($0A);
            CurDescChar := CurDescChar + 1;
          end
          else WriteByte(ord(MainView.Description[CurDescLine][CurDescChar]));
        end until CurDescChar >= Length(MainView.Description[CurDescLine]);
      end; {if Length(MainView.Description[CurDescLine]) > 0}
    end; {for CurDescLine := 0 to MainView.Description.Count - 1}
    WriteByte($00);
    ResPos := 3;
    WriteByte(DescLoc mod 256);
    WriteByte(DescLoc div 256);
  end; {if MainView.Description.Count > 0}
  LoadingGauge.Visible := False;
  for CurLoop := 0 to MainView.NumLoops - 1 do
  begin
    ResPos := 5 + CurLoop*2;
    WriteByte(LoopLoc[CurLoop] mod 256);
    WriteByte(LoopLoc[CurLoop] div 256);
    if not ((MainView.MirrorLoop[CurLoop]) and (MainView.MirrorLoopNum[CurLoop]<CurLoop)) then      for CurCel := 0 to MainView.NumCels[CurLoop] - 1 do
    begin
      ResPos := LoopLoc[CurLoop] + 1 + CurCel*2;
      WriteByte(CelLoc[CurLoop,CurCel] mod 256);
      WriteByte(CelLoc[CurLoop,CurCel] div 256);
    end;
  end;
  begin
    GetMem(OutputResource.Data,ResSize);
    Move(ViewResource.Data^,OutputResource.Data^,ResSize);
    SaveView.Data := OutputResource.Data;
  end;
  SaveView.Size := ResSize;
  FreeMem(ViewResource.Data,MaxResourceSize);
  if ErrorType = ResTooBigError then
    ShowMessage('Error: Resource too big! (>'+IntToStr(MaxResourceSize)+')');
end;

{*************************************************************}
procedure TViewEdit.SaveUndo(NewUndoType:TViewEditUndoType);
{*************************************************************}
begin
  UndoType := NewUndoType;
  if (UndoType = Draw) or (UndoType = SizeChange) or (UndoType = MoveCel) then
  begin
    UndoBitmap.Assign(ViewImage.Picture.Bitmap);
    UndoLoopno := SelectedLoop;
    UndoCelno := SelectedCel;
    UndoWidth := ViewImage.Picture.Bitmap.Width;
    UndoHeight := ViewImage.Picture.Bitmap.Height;
    EditUndo.Enabled := True;
  end;
  ViewModified := True;
end;

{*************************************************************}
procedure TViewEdit.SaveExistingCel;
{*************************************************************}
begin
  MainView.CelBitmap[SelectedLoop,SelectedCel].Assign(ViewImage.Picture.Bitmap);
  MainView.CelWidth[SelectedLoop,SelectedCel] := ViewImage.Picture.Bitmap.Width;
  MainView.CelHeight[SelectedLoop,SelectedCel] := ViewImage.Picture.Bitmap.Height;
  MainView.CelTransCol[SelectedLoop,SelectedCel] := TransCol;

  if MainView.MirrorLoop[SelectedLoop] then
  begin
    MainView.CelBitmap[MainView.MirrorLoopNum[SelectedLoop],SelectedCel].Assign(MainView.CelBitmap[SelectedLoop,SelectedCel]);
    MirrorBitmap(MainView.CelBitmap[MainView.MirrorLoopNum[SelectedLoop],SelectedCel]);
    MainView.CelWidth[MainView.MirrorLoopNum[SelectedLoop],SelectedCel] := MainView.CelWidth[SelectedLoop,SelectedCel];
    MainView.CelHeight[MainView.MirrorLoopNum[SelectedLoop],SelectedCel] := MainView.CelHeight[SelectedLoop,SelectedCel];
    MainView.CelTransCol[MainView.MirrorLoopNum[SelectedLoop],SelectedCel] := MainView.CelTransCol[SelectedLoop,SelectedCel];
    MainView.MaxWidth[MainView.MirrorLoopNum[SelectedLoop]] := MainView.MaxWidth[SelectedLoop];
    MainView.MaxHeight[MainView.MirrorLoopNum[SelectedLoop]] := MainView.MaxHeight[SelectedLoop];
  end;
end;

{*************************************************************}
procedure TViewEdit.ReCalculateMaxHW;
{*************************************************************}
var CurCel       : byte;
    NewMaxHeight : byte;
    NewMaxWidth  : byte;
begin
  SaveExistingCel;
{Calculate new MaxHeight}
  NewMaxHeight := 0;
  for CurCel := 0 to MainView.NumCels[SelectedLoop] - 1 do
    if MainView.CelHeight[SelectedLoop,CurCel] > NewMaxHeight then
      NewMaxHeight := MainView.CelHeight[SelectedLoop,CurCel];
  MainView.MaxHeight[SelectedLoop] := NewMaxHeight;
{Calculate new MaxWidth}
  NewMaxWidth := 0;
  for CurCel := 0 to MainView.NumCels[SelectedLoop] - 1 do
    if MainView.CelWidth[SelectedLoop,CurCel] > NewMaxWidth then
      NewMaxWidth := MainView.CelWidth[SelectedLoop,CurCel];
  MainView.MaxWidth[SelectedLoop] := NewMaxWidth;
end;

















{*************************************************************}
function TViewEdit.MirrorCel(Loopno,Celno:byte) : boolean;
{*************************************************************}
begin
  MirrorCel := False;
  if (Loopno <= MainView.NumLoops - 1) and (Celno <= MainView.NumCels[Loopno] - 1) then
  begin
    MirrorBitmap(MainView.CelBitmap[Loopno,Celno]);
    MirrorCel := True;
  end;
end;

{*************************************************************}
procedure TViewEdit.UnMirrorLoop(Loopno:byte);
{*************************************************************}
begin
  if MainView.MirrorLoop[Loopno] then MainView.MirrorLoop[MainView.MirrorLoopNum[Loopno]] := False;
  MainView.MirrorLoop[Loopno] := False;
  LastLoopStatusDisplayed := $FF;
end;


{*************************************************************}
procedure TViewEdit.NewView;
{*************************************************************}
var CurLoop : byte;
    CurCel  : byte;
begin
  if MainView.NumLoops > 0 then
  for CurLoop := 0 to MainView.NumLoops - 1 do
    for CurCel := 0 to MainView.NumCels[CurLoop] - 1 do
      MainView.CelBitmap[CurLoop,CurCel].Free;
  MainView.NumLoops := 1;
  MainView.NumCels[0] := 1;
  MainView.CelWidth[0,0] := DefaultCelWidth;
  MainView.CelHeight[0,0] := DefaultCelHeight;
  MainView.CelTransCol[0,0] := 0;
  MainView.CelBitmap[0,0] := TBitmap.Create;
  FixPalette(MainView.CelBitmap[0,0]);
  MainView.CelBitmap[0,0].Width := DefaultCelWidth;
  MainView.CelBitmap[0,0].Height := DefaultCelHeight;
  MainView.CelBitmap[0,0].Canvas.Brush.Color := EGACOL[0];
  MainView.CelBitmap[0,0].Canvas.FillRect(Rect(0,0,DefaultCelWidth,DefaultCelHeight));
  MainView.MaxWidth[0] := DefaultCelWidth;
  MainView.MaxHeight[0] := DefaultCelHeight;
  MainVIew.MirrorLoop[0] := False;
  MainView.Description := TStringList.Create;
  MainView.Description.Clear;
  SelectedLoop := 0;
  SelectedCel := 0;
  LastLoopStatusDisplayed := $FF;

  UpdateStatus(SelectedLoop,SelectedCel);
  EditUndo.Enabled := False;
  ViewNum := 0;
  LoadedFromGame := False;
  ViewLoaded := True;
  ViewModified := False;
  Caption := 'View Editor';
  DisplayView(0,0);
end;

{*************************************************************}
function TViewEdit.ClearCel(Loopno,Celno:byte) : boolean;
{*************************************************************}
begin
  ClearCel := False;
  if (Loopno <= MainView.NumLoops - 1) and (Celno <= MaxCels-1)
     and (Celno <= MainView.NumCels[Loopno]) then
  begin
    if Celno = MainView.NumCels[Loopno] then
    begin
      MainView.CelBitmap[Loopno,Celno] := TBitmap.Create;
      FixPalette(MainView.CelBitmap[Loopno,Celno]);
    end;
    MainView.CelWidth[Loopno,Celno] := MainView.CelWidth[Loopno,0];
    MainView.CelHeight[Loopno,Celno] := MainView.CelHeight[Loopno,0];
    MainView.CelTransCol[Loopno,Celno] := MainView.CelTransCol[Loopno,0];
    MainView.CelBitmap[Loopno,Celno].Width := MainView.CelWidth[Loopno,Celno];
    MainView.CelBitmap[Loopno,Celno].Height := MainView.CelHeight[Loopno,Celno];
    MainView.CelBitmap[Loopno,Celno].Canvas.Brush.Color := EGACOL[MainView.CelTransCol[Loopno,0]];
    MainView.CelBitmap[Loopno,Celno].Canvas.FillRect(Rect(0,0,MainView.CelWidth[Loopno,Celno],MainView.CelHeight[Loopno,Celno]));
    ClearCel := True;
  end;
end;

{*************************************************************}
function TViewEdit.AppendCel(Loopno:byte) : boolean;
{*************************************************************}
begin
  AppendCel := False;
  if (Loopno <= MainView.NumLoops - 1) and (MainView.NumCels[Loopno] < MaxCels) then
  begin
    MainView.NumCels[Loopno] := MainView.NumCels[Loopno] + 1;
    MainView.CelBitmap[Loopno,MainView.NumCels[Loopno]-1] := TBitmap.Create;
    FixPalette(MainView.CelBitmap[Loopno,MainView.NumCels[Loopno]-1]);
    ClearCel(Loopno,MainView.NumCels[Loopno]-1);
    if MainView.MirrorLoop[Loopno] then
    begin
      MainView.NumCels[MainView.MirrorLoopNum[Loopno]] := MainView.NumCels[MainView.MirrorLoopNum[Loopno]] + 1;
      MainView.CelBitmap[MainView.MirrorLoopNum[Loopno],MainView.NumCels[MainView.MirrorLoopNum[Loopno]]-1] := TBitmap.Create;
      FixPalette(MainView.CelBitmap[MainView.MirrorLoopNum[Loopno],MainView.NumCels[MainView.MirrorLoopNum[Loopno]]-1]);
      ClearCel(MainView.MirrorLoopNum[Loopno],MainView.NumCels[MainView.MirrorLoopNum[Loopno]]-1);
    end;
    AppendCel := True;
  end;
end;

{*************************************************************}
function TViewEdit.ClearLoop(Loopno:byte) : boolean;
{*************************************************************}
var CurCel : byte;

begin
  ClearLoop := False;
  if Loopno <= MainView.NumLoops - 1 then
  begin
    for CurCel := 0 to MainView.NumCels[Loopno] - 1 do
      MainView.CelBitmap[Loopno,CurCel].Free;
    MainView.NumCels[Loopno] := 1;
    MainView.CelBitmap[Loopno,0] := TBitmap.Create;
    FixPalette(MainView.CelBitmap[Loopno,0]);
    if not ClearCel(Loopno,0) then
    begin
      ShowMessage('FATAL ERROR: ClearCel failed.');
      Halt;
    end
    else ClearLoop := True;
  end
end;

{*************************************************************}
function TViewEdit.AppendLoop : boolean;
{*************************************************************}
begin
  AppendLoop := False;
  if MainView.NumLoops < MaxLoops then
  begin
    MainView.NumLoops := MainView.NumLoops + 1;
    MainView.NumCels[MainView.NumLoops-1] := 1;

    MainView.CelBitmap[MainView.NumLoops-1,0] := TBitmap.Create;
    FixPalette(MainView.CelBitmap[MainView.NumLoops-1,0]);
    MainView.CelWidth[MainView.NumLoops-1,0] := MainView.CelWidth[0,0];
    MainView.CelHeight[MainView.NumLoops-1,0] := MainView.CelHeight[0,0];
    MainView.CelTransCol[MainView.NumLoops-1,0] := MainView.CelTransCol[0,0];
    MainView.CelBitmap[MainView.NumLoops-1,0].Width := MainView.CelWidth[MainView.NumLoops-1,0];
    MainView.CelBitmap[MainView.NumLoops-1,0].Height := MainView.CelHeight[MainView.NumLoops-1,0];
    MainView.CelBitmap[MainView.NumLoops-1,0].Canvas.Brush.Color := EGACOL[MainView.CelTransCol[MainView.NumLoops-1,0]];
    MainView.CelBitmap[MainView.NumLoops-1,0].Canvas.FillRect(Rect(0,0,MainView.CelWidth[MainView.NumLoops-1,0],MainView.CelHeight[MainView.NumLoops-1,0]));

    MainView.MaxWidth[MainView.NumLoops-1] := MainView.CelWidth[MainView.NumLoops-1,0];
    MainView.MaxHeight[MainView.NumLoops-1] := MainView.CelHeight[MainView.NumLoops-1,0];
    MainView.MirrorLoop[MainView.NumLoops-1] := False;
    AppendLoop := True;
  end;
end;

{*************************************************************}
function TViewEdit.DeleteLastCel(Loopno:byte) : boolean;
{*************************************************************}
begin
  DeleteLastCel := False;
  if (Loopno <= MainView.NumLoops - 1) and (MainView.NumCels[Loopno] > 1) then
  begin
    MainView.CelBitmap[Loopno,MainView.NumCels[Loopno]-1].Free;
    MainView.NumCels[Loopno] := MainView.NumCels[Loopno] - 1;
    DeleteLastCel := True;
  end;
end;

{*************************************************************}
function TViewEdit.InsertCel(Loopno,Celno:byte) : boolean;
{*************************************************************}
var CurCel : byte;
begin
  InsertCel := False;
  if (Loopno <= MainView.NumLoops - 1) and (Celno <= MainView.NumCels[Loopno] - 1)
    and (MainView.NumCels[Loopno] < MaxCels) then
  begin
    AppendCel(Loopno);
    for CurCel := MainView.NumCels[Loopno] - 2 downto Celno do
    begin
      CopyCel(Loopno,CurCel,Loopno,CurCel+1);
    end;
    ClearCel(Loopno,Celno);
    InsertCel := True;
  end;
end;

{*************************************************************}
function TViewEdit.DeleteCel(Loopno,Celno:byte) : boolean;
{*************************************************************}
var CurCel : byte;
begin
  DeleteCel := False;
  if (Loopno <= MainView.NumLoops - 1) and (MainView.NumCels[Loopno] > 1) then
  begin
    for CurCel := Celno to MainView.NumCels[Loopno] - 2 do
      if not CopyCel(Loopno,CurCel+1,Loopno,CurCel) then
      begin
        ShowMessage('FATAL ERROR: CopyCel failed.');
        Halt;
      end;
    if not DeleteLastCel(Loopno) then
    begin
      ShowMessage('FATAL ERROR: DeleteLastCel failed.');
      Halt;
    end;
    DeleteCel := True;
  end;
end;

{*************************************************************}
function TViewEdit.CopyCel(SourceLoop,SourceCel,TargetLoop,TargetCel:byte) : boolean;
{*************************************************************}
begin
  CopyCel := False;
  if (TargetCel <= MaxCels - 1) and (TargetCel <= MainView.NumCels[TargetLoop])
     and (SourceLoop <= MainView.NumLoops - 1) and (SourceCel <= MainView.NumCels[SourceLoop] - 1) then
  begin
    MainView.CelWidth[TargetLoop,TargetCel] := MainView.CelWidth[SourceLoop,SourceCel];
    MainView.CelHeight[TargetLoop,TargetCel] := MainView.CelHeight[SourceLoop,SourceCel];
    MainView.CelTransCol[TargetLoop,TargetCel] := MainView.CelTransCol[SourceLoop,SourceCel];
    if TargetCel = MainView.NumCels[TargetLoop] then
    begin
      MainView.NumCels[TargetLoop] := MainView.NumCels[TargetLoop] + 1;
      MainView.CelBitmap[TargetLoop,TargetCel] := TBitmap.Create;
      FixPalette(MainView.CelBitmap[TargetLoop,TargetCel]);
    end;
    MainView.CelBitmap[TargetLoop,TargetCel].Assign(MainView.CelBitmap[SourceLoop,SourceCel]);
    CopyCel := True;
  end;
end;

{*************************************************************}
function TViewEdit.CopyLoop(SourceLoop,TargetLoop:byte;MirrorTarget:boolean) : boolean;
{*************************************************************}
var CurCel : byte;
begin
  CopyLoop := False;
  if (TargetLoop <= MaxLoops - 1) and (TargetLoop <= MainView.NumLoops) then
  begin
    if TargetLoop = MainView.NumLoops then AppendLoop;
    if not ClearLoop(TargetLoop) then
    begin
      ShowMessage('FATAL ERROR: ClearLoop failed.');
      Halt;
    end;
    for CurCel := 0 to MainView.NumCels[SourceLoop] - 1 do
    begin
      if not CopyCel(SourceLoop,CurCel,TargetLoop,CurCel) then
      begin
        ShowMessage('FATAL ERROR: CopyCel failed.');
        Halt;
      end;
      if MirrorTarget then MirrorBitmap(MainView.CelBitmap[TargetLoop,CurCel]);
    end;
    MainView.MaxWidth[TargetLoop] := MainView.MaxWidth[SourceLoop];
    MainView.MaxHeight[TargetLoop] := MainView.MaxHeight[SourceLoop];
    MainView.NumCels[TargetLoop] := MainView.NumCels[SourceLoop];
    CopyLoop := True;
  end;
end;

{*************************************************************}
function TViewEdit.DeleteLastLoop : boolean;
{*************************************************************}
var CurCel : byte;

begin
  DeleteLastLoop := False;
  if MainView.NumLoops > 1 then
  begin
    UnMirrorLoop(MainView.NumLoops-1);
    for CurCel := 0 to MainView.NumCels[MainView.NumLoops-1] - 1 do
      MainView.CelBitmap[MainView.NumLoops-1,CurCel].Free;
    MainView.NumLoops := MainView.NumLoops - 1;
    DeleteLastLoop := True;
  end;
end;

{*************************************************************}
function TViewEdit.InsertLoop(Loopno:byte) : boolean;
{*************************************************************}
var CurLoop : byte;
    BUMirrorLoop : array[0..MaxLoops-1] of boolean;
    BUMirrorLoopNum : array[0..MaxLoops-1] of byte;

begin
  InsertLoop := False;
  if (Loopno <= MainView.NumLoops - 1) and (MainView.NumLoops < MaxLoops) then
  begin
    for CurLoop := 0 to MainView.NumLoops-1 do
    begin
      if MainView.MirrorLoopNum[CurLoop] >= Loopno then
        MainView.MirrorLoopNum[CurLoop] := MainView.MirrorLoopNum[CurLoop]+1;
      BUMirrorLoop[CurLoop] := MainView.MirrorLoop[CurLoop];
      BUMirrorLoopNum[CurLoop] := MainView.MirrorLoopNum[CurLoop];
    end;
    AppendLoop;
    for CurLoop := MainView.NumLoops - 2 downto Loopno do
      CopyLoop(CurLoop,CurLoop+1,False);
    ClearLoop(Loopno);
    for CurLoop := Loopno + 1 to MainView.NumLoops - 1 do
    begin
      MainView.MirrorLoop[CurLoop] := BUMirrorLoop[CurLoop-1];
      MainView.MirrorLoopNum[CurLoop] := BUMirrorLoopNum[CurLoop-1];
    end;
    InsertLoop := True;
  end;
end;

{*************************************************************}
function TViewEdit.DeleteLoop(Loopno:byte) : boolean;
{*************************************************************}
var CurLoop : byte;
    BUMirrorLoop : array[0..MaxLoops-1] of boolean;
    BUMirrorLoopNum : array[0..MaxLoops-1] of byte;
begin
  DeleteLoop := False;
  if Loopno <= MainView.Numloops - 1 then
    if MainView.NumLoops > 1 then
    begin
      UnMirrorLoop(Loopno);
      for CurLoop := 0 to MainView.NumLoops-1 do
      begin
        if MainView.MirrorLoopNum[CurLoop] >= Loopno + 1 then
          MainView.MirrorLoopNum[CurLoop] := MainView.MirrorLoopNum[CurLoop] - 1;
        BUMirrorLoop[CurLoop] := MainView.MirrorLoop[CurLoop];
        BUMirrorLoopNum[CurLoop] := MainView.MirrorLoopNum[CurLoop];
      end;
      for CurLoop := Loopno to MainView.NumLoops - 2 do
        if not CopyLoop(CurLoop+1,CurLoop,False) then
        begin
          ShowMessage('FATAL ERROR: CopyLoop failed.');
          Halt;
        end;
      if not DeleteLastLoop then
      begin
        ShowMessage('FATAL ERROR: DeleteLastLoop failed.');
        Halt;
      end;
      for CurLoop := Loopno to MainView.NumLoops - 1 do
      begin
        MainView.MirrorLoop[CurLoop] := BUMirrorLoop[CurLoop+1];
        MainView.MirrorLoopNum[CurLoop] := BUMirrorLoopNum[CurLoop+1];
      end;
      DeleteLoop := True;
    end;
end;































{*************************************************************}
                      {Component Events}

{*************************************************************}
procedure TViewEdit.FormCreate(Sender: TObject);
{*************************************************************}
begin
  ZoominBut.Glyph.TransparentMode := tmAuto;
  WidthAdjust.Max := MaxCelWidth;
  HeightAdjust.Max := MaxCelHeight;
  ViewLoaded := False;
  UndoBitmap := TBitmap.Create;
  FixPalette(UndoBitmap);
  MainView.Description := TStringList.Create;
end;

{*************************************************************}
procedure TViewEdit.FormClose(Sender: TObject; var Action: TCloseAction);
{*************************************************************}
var CurLoop : byte;
    CurCel  : byte;
begin
  if MainView.NumLoops > 0 then
  for CurLoop := 0 to MainView.NumLoops - 1 do
    for CurCel := 0 to MainView.NumCels[CurLoop] - 1 do
      MainView.CelBitmap[CurLoop,CurCel].Free;
  ViewEditWinUsed[ViewEditorNum] := false;
  UndoBitmap.Free;
  MainView.Description.Free;
  action := caFree;
end;

{*************************************************************}
procedure TViewEdit.LoopAdjustClick(Sender: TObject; Button: TUDBtnType);
{*************************************************************}
var s1 : string;
begin
  if ViewLoaded then
  begin
    SaveExistingCel;
    SelectedLoop := LoopAdjust.Position;
    SelectedCel := 0;
    DisplayView(SelectedLoop,SelectedCel);
  end;
end;

{*************************************************************}
procedure TViewEdit.CelAdjustClick(Sender: TObject; Button: TUDBtnType);
{*************************************************************}
begin
  if ViewLoaded then
  begin
    SaveExistingCel;
    SelectedCel := CelAdjust.Position;
    DisplayView(SelectedLoop,SelectedCel);
  end;
end;

{*************************************************************}
procedure TViewEdit.ZoomInButClick(Sender: TObject);
{*************************************************************}
begin
  if ScaleFactor < 10 then
  begin
    ScaleFactor := ScaleFactor + 1;
    ViewImage.Width := Round(ViewImage.Picture.Bitmap.Width*2*ScaleFactor);
    ViewImage.Height := Round(ViewImage.Picture.Bitmap.Height*ScaleFactor);
    ViewImage.Top := MainView.MaxHeight[SelectedLoop]*ScaleFactor - ViewImage.Height;
    if MainView.MirrorLoop[SelectedLoop] and (MainView.MirrorLoopNum[SelectedLoop]<SelectedLoop) then
      ViewImage.Left := MainView.MaxWidth[SelectedLoop]*ScaleFactor*2-ViewImage.Width
    else if ViewImage.Left > 0 then ViewImage.Left := 0;
    MouseIsDown := False;
  end;
end;

{*************************************************************}
procedure TViewEdit.ZoomOutButClick(Sender: TObject);
{*************************************************************}
begin
  if ScaleFactor > 1 then
  begin
    ScaleFactor := ScaleFactor - 1;
    ViewImage.Width := Round(ViewImage.Picture.Bitmap.Width*2*ScaleFactor);
    ViewImage.Height := Round(ViewImage.Picture.Bitmap.Height*ScaleFactor);
    ViewImage.Top := MainView.MaxHeight[SelectedLoop]*ScaleFactor - ViewImage.Height;
    if MainView.MirrorLoop[SelectedLoop] and (MainView.MirrorLoopNum[SelectedLoop]<SelectedLoop) then
      ViewImage.Left := MainView.MaxWidth[SelectedLoop]*ScaleFactor*2-ViewImage.Width
    else if ViewImage.Left > 0 then ViewImage.Left := 0;
    MouseIsDown := False;
  end;
end;

{*************************************************************}
procedure TViewEdit.FormResize(Sender: TObject);
{*************************************************************}
begin
  Panel2.Height := ClientHeight - Panel2.Top;
  PaletteImage.Top := ClientHeight - PaletteImage.Height;
  if Panel2.Width + PaletteImage.Width < ClientWidth then
    PaletteImage.Left := Panel2.Width + ((ClientWidth - Panel2.Width) div 2) - (PaletteImage.Width div 2)
  else PaletteImage.Left := Panel2.Width;
  ViewPanel.Width := ClientWidth - ViewPanel.Left*2+Panel1.Width;
  VIewPanel.Height := ClientHeight - PaletteImage.Height - MirrorPanel.Height - ViewPanel.Top*2;
  MirrorPanel.Width := ClientWidth - MirrorPanel.Left;
  MirrorPanel.Top := PaletteImage.Top - MirrorPanel.Height;
end;

{*************************************************************}
procedure TViewEdit.DescriptionCheckBoxClick(Sender: TObject);
{*************************************************************}
begin
  if DescriptionCheckBox.Checked then DescriptionEditBut.Enabled := true
  else
  DescriptionEditBut.Enabled := false;
end;

{*************************************************************}
procedure TViewEdit.DescriptionEditButClick(Sender: TObject);
{*************************************************************}
var Description : string;
begin
  SaveExistingCel;
  ViewEditDescription.ViewImage.Picture.Bitmap.Assign(MainView.CelBitmap[0,0]);
  ViewEditDescription.ViewImage.Width := MainView.CelWidth[0,0]*4;
  ViewEditDescription.ViewImage.Height := MainView.CelHeight[0,0]*2;
  ViewEditDescription.ViewImage.Left := (ViewEditDescription.ClientWidth div 2) - (ViewEditDescription.ViewImage.Width div 2);
  ViewEditDescription.DescriptionEdit.Top := ViewEditDescription.ViewImage.Height + ViewEditDescription.ViewImage.Top*2;
  ViewEditDescription.DescriptionEdit.Height := ViewEditDescription.ClientHeight - ViewEditDescription.DescriptionEdit.Top - ViewEditDescription.ButtonPanel.Height - ViewEditDescription.ViewImage.Top*2;
  ViewEditDescription.ButtonPanel.Top := ViewEditDescription.ClientHeight - ViewEditDescription.ButtonPanel.Height - ViewEditDescription.ViewImage.Top;
  ViewEditDescription.DescriptionEdit.Lines.Assign(MainView.Description);
  ViewEditDescription.ViewEditorNum := ViewEditorNum;
  ViewEditDescription.ShowModal;
  ActiveControl := Panel1;
end;

{*************************************************************}
procedure TViewEdit.FormShow(Sender: TObject);
{*************************************************************}
var ColNum : byte;
    TempBitmap : TBitmap;
begin
  DrawRadio.Checked := True;
  FillRadio.Checked := False;
  SelectedTool := DrawTool;
  MouseIsDown := False;
{  PaletteImage.Picture.Bitmap.Assign(BaseBitmap);
  PaletteImage.Picture.Bitmap.Palette := CreateEGAPalette;
  SelectPalette(PaletteImage.Picture.Bitmap.Canvas.Handle, EGAPalHandle, True);}

  TempBitmap := TBitmap.Create;
  TempBitmap.Assign(BaseBitmap);
  TempBitmap.Palette := CreateEGAPalette;
  SelectPalette(TempBitmap.Canvas.Handle, EGAPalHandle, True);
  TempBitmap.Width := 240;
  TempBitmap.Height := 40;
  PaletteImage.Picture.Bitmap.Assign(TempBitmap);
  TempBitmap.Free;        
  for ColNum := 0 to 7 do
  begin
    PaletteImage.Canvas.Brush.Color := EGACOL[ColNum];
    PaletteImage.Canvas.FillRect(Rect(ColNum*30,0,ColNum*30+30,20));
  end;
  for ColNum := 8 to 15 do
  begin
    PaletteImage.Canvas.Brush.Color := EGACOL[ColNum];
    PaletteImage.Canvas.FillRect(Rect((ColNum-8)*30,20,(ColNum-8)*30+30,40));
  end;
  PaletteImage.Canvas.Font.Style := [fsBold];
  PaletteImage.Canvas.Brush.Color := clWhite;
  PaletteImage.Canvas.Font.Color := clBlack;
  PaletteImage.Canvas.TextOut(214,24,'L');
  PaletteImage.Canvas.Brush.Color := clBlack;
  PaletteImage.Canvas.Font.Color := clWhite;
  PaletteImage.Canvas.TextOut(19,4,'R');
  LCol := 15;
  RCol := 0;
  LastLoopStatusDisplayed := $FF;
end;

{*************************************************************}
procedure TViewEdit.DrawRadioClick(Sender: TObject);
{*************************************************************}
begin
  SelectedTool := DrawTool;
  MouseIsDown := False;
end;

{*************************************************************}
procedure TViewEdit.FillRadioClick(Sender: TObject);
{*************************************************************}
begin
  SelectedTool := FillTool;
  MouseIsDown := False;
end;

{*************************************************************}
procedure TViewEdit.ViewImageMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
{*************************************************************}

var  ViewX, ViewY    : integer;
     DrawCol : byte;

begin
  if Button = mbLeft then LeftButPressed := True
  else LeftButPressed := False;
  if LeftButPressed then DrawCol := LCol
  else DrawCol := RCol;
  ViewX := X div (2*ScaleFactor);
  ViewY := Y div ScaleFactor;
  SaveUndo(Draw);
  if SelectedTool = DrawTool then
  begin
    ViewImage.Canvas.Pen.Color := EGACOL[DrawCol];
    ViewImage.Canvas.MoveTo(ViewX,ViewY);
    ViewImage.Canvas.Pixels[ViewX,ViewY] := EGACOL[DrawCol];
    OldViewX := ViewX;
    OldViewY := ViewY;
  end {if SelectedTool = Draw}
  else if SelectedTool = FillTool then
  begin
    ViewImage.Canvas.Brush.Color := EGACOL[DrawCol];
    ViewImage.Canvas.FloodFill(ViewX,ViewY,ViewImage.Canvas.Pixels[ViewX,VIewY],fsSurface);
  end;
  MouseIsDown := True;
end;

{*************************************************************}
procedure TViewEdit.ViewImageMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
{*************************************************************}

var  ViewX, ViewY    : integer;
     DrawCol : byte;

begin
  if MouseIsDown then
  begin
    if LeftButPressed then DrawCol := LCol
    else DrawCol := RCol;
    ViewX := X div (2*ScaleFactor);
    ViewY := Y div ScaleFactor;
    if (OldViewX <> ViewX) or (OldViewY <> ViewY) then
    begin
      if SelectedTool = DrawTool then
      begin
        ViewImage.Canvas.Pen.Color := EGACOL[DrawCol];
        ViewImage.Canvas.LineTo(ViewX,ViewY);
        ViewImage.Canvas.Pixels[ViewX,ViewY] := EGACOL[DrawCol];
      end; {if SelectedTool = Draw}
    end; {if (OldViewX <> ViewX) or (OldViewY <> ViewY)}
  end;
end;

{*************************************************************}
procedure TViewEdit.ViewImageMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
{*************************************************************}
begin
  MouseIsDown := False;
  OldViewX := 255;
  OldViewY := 255;
end;


{*************************************************************}
procedure TViewEdit.PaletteImageMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
{*************************************************************}

var OldCol : byte;
begin
{  PaletteImage.Canvas.Font.Color := clWhite;}
  if Button = mbLeft then
  begin
    PaletteImage.Canvas.Brush.Color := EGACOL[LCol];
    if LCol < 8 then PaletteImage.Canvas.FillRect(Rect(LCol*30,0,LCol*30+15,20))
    else PaletteImage.Canvas.FillRect(Rect((LCol-8)*30,20,(LCol-8)*30+15,40));
    if y < 20 then
    begin
      LCol := x div 30;
      if LCol < 9 then PaletteImage.Canvas.Font.Color := clWhite
      else PaletteImage.Canvas.Font.Color := clBlack;
      PaletteImage.Canvas.Brush.Color := EGACOL[LCol];
      PaletteImage.Canvas.TextOut(LCol*30+4,4,'L');
    end
    else
    begin
      LCol := x div 30 + 8;
      if LCol < 9 then PaletteImage.Canvas.Font.Color := clWhite
      else PaletteImage.Canvas.Font.Color := clBlack;
      PaletteImage.Canvas.Brush.Color := EGACOL[LCol];
      PaletteImage.Canvas.TextOut((LCol-8)*30+4,24,'L');
    end;
  end {if Button = mbLeft}
  else
  begin
    PaletteImage.Canvas.Brush.Color := EGACOL[RCol];
    if RCol < 8 then PaletteImage.Canvas.FillRect(Rect(RCol*30+15,0,RCol*30+30,20))
    else PaletteImage.Canvas.FillRect(Rect((RCol-8)*30+15,20,(RCol-8)*30+30,40));
    if y < 20 then
    begin
      RCol := x div 30;
      if RCol < 9 then PaletteImage.Canvas.Font.Color := clWhite
      else PaletteImage.Canvas.Font.Color := clBlack;
      PaletteImage.Canvas.Brush.Color := EGACOL[RCol];
      PaletteImage.Canvas.TextOut(RCol*30+18,4,'R');
    end
    else
    begin
      RCol := x div 30 + 8;
      if RCol < 9 then PaletteImage.Canvas.Font.Color := clWhite
      else PaletteImage.Canvas.Font.Color := clBlack;
      PaletteImage.Canvas.Brush.Color := EGACOL[RCol];
      PaletteImage.Canvas.TextOut((RCol-8)*30+18,24,'R');
    end;
  end; {if not Button = mbLeft}
end;

{*************************************************************}
procedure TViewEdit.ViewCloseClick(Sender: TObject);
{*************************************************************}
begin
  Close;
end;

{*************************************************************}
procedure TViewEdit.LoopNextClick(Sender: TObject);
{*************************************************************}
begin
  if SelectedLoop < MainView.NumLoops - 1 then
  begin
    SaveExistingCel;
    SelectedLoop := SelectedLoop + 1;
    SelectedCel := 0;
    LoopAdjust.Position := SelectedLoop;
    MouseIsDown := False;
    DisplayView(SelectedLoop,SelectedCel);
  end;
end;

{*************************************************************}
procedure TViewEdit.LoopPrevClick(Sender: TObject);
{*************************************************************}
begin
  if SelectedLoop > 0 then
  begin
    SaveExistingCel;
    SelectedLoop := SelectedLoop - 1;
    SelectedCel := 0;
    LoopAdjust.Position := SelectedLoop;
    MouseIsDown := False;
    DisplayView(SelectedLoop,SelectedCel);
  end;
end;

{*************************************************************}
procedure TViewEdit.LoopFirstClick(Sender: TObject);
{*************************************************************}
begin
  if SelectedLoop > 0 then
  begin
    SaveExistingCel;
    SelectedLoop := 0;
    SelectedCel := 0;
    LoopAdjust.Position := SelectedLoop;
    MouseIsDown := False;
    DisplayView(SelectedLoop,SelectedCel);
  end;
end;

{*************************************************************}
procedure TViewEdit.LoopLastClick(Sender: TObject);
{*************************************************************}
begin
  if SelectedLoop < MainView.NumLoops - 1 then
  begin
    SaveExistingCel;
    SelectedLoop := MainView.NumLoops - 1;
    SelectedCel := 0;
    LoopAdjust.Position := SelectedLoop;
    MouseIsDown := False;
    DisplayView(SelectedLoop,SelectedCel);
  end;
end;

{*************************************************************}
procedure TViewEdit.CelNextClick(Sender: TObject);
{*************************************************************}
begin
  if SelectedCel < MainView.NumCels[SelectedLoop] - 1 then
  begin
    SaveExistingCel;
    SelectedCel := SelectedCel + 1;
    LoopAdjust.Position := SelectedCel;
    MouseIsDown := False;
    DisplayView(SelectedLoop,SelectedCel);
  end;
end;

{*************************************************************}
procedure TViewEdit.CelPrevClick(Sender: TObject);
{*************************************************************}
begin
  if SelectedCel > 0 then
  begin
    SaveExistingCel;
    SelectedCel := SelectedCel - 1;
    LoopAdjust.Position := SelectedCel;
    MouseIsDown := False;
    DisplayView(SelectedLoop,SelectedCel);
  end;
end;

{*************************************************************}
procedure TViewEdit.CelFirstClick(Sender: TObject);
{*************************************************************}
begin
  if SelectedCel > 0 then
  begin
    SaveExistingCel;
    SelectedCel := 0;
    LoopAdjust.Position := SelectedCel;
    MouseIsDown := False;
    DisplayView(SelectedLoop,SelectedCel);
  end;
end;

{*************************************************************}
procedure TViewEdit.CelLastClick(Sender: TObject);
{*************************************************************}
begin
  if SelectedCel < MainView.NumCels[SelectedLoop] - 1 then
  begin
    SaveExistingCel;
    SelectedCel := MainView.NumCels[SelectedLoop] - 1;
    LoopAdjust.Position := SelectedCel;
    MouseIsDown := False;
    DisplayView(SelectedLoop,SelectedCel);
  end;
end;

{*************************************************************}
procedure TViewEdit.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
{*************************************************************}
begin
  if (ActiveControl <> WidthDisplay) and (ActiveControl <> HeightDisplay) then
  begin  {keys such as 9 on the numeric keypad seem to trigger off other key events}
    if Shift=[ssAlt] then
    begin
      if Upcase(chr(Key)) = 'W' then WidthDisplay.SetFocus
      else if Upcase(chr(Key)) = 'H' then HeightDisplay.SetFocus;
    end
    else if Upcase(chr(Key)) = 'Q' then LoopPrevClick(Application)
    else if Upcase(chr(Key)) = 'W' then LoopNextClick(Application)
    else if Upcase(chr(Key)) = 'A' then CelPrevClick(Application)
    else if Upcase(chr(Key)) = 'S' then CelNextClick(Application)
    else if Upcase(chr(Key)) = 'Z' then ZoomOutButClick(Application)
    else if Upcase(chr(Key)) = 'X' then ZoomInButClick(Application)
    else if Upcase(chr(Key)) = 'T' then TransSetButClick(Application)
    else if Upcase(chr(Key)) = 'I' then CelUpButClick(Application)
    else if Upcase(chr(Key)) = 'K' then CelDownButClick(Application)
    else if Upcase(chr(Key)) = 'J' then CelLeftButClick(Application)
    else if Upcase(chr(Key)) = 'L' then CelRightButClick(Application)
    else if Upcase(chr(Key)) = 'D' then
    begin
      DrawRadio.Checked := True;
      SelectedTool := DrawTool;
      MouseIsDown := False;
    end
    else if Upcase(chr(Key)) = 'F' then
    begin
      FillRadio.Checked := True;
      SelectedTool := FillTool;
      MouseIsDown := False;
    end
  end;
end;

{*************************************************************}
procedure TViewEdit.EditUndoClick(Sender: TObject);
{*************************************************************}
begin
  if Active then
  begin
    SelectedCel := UndoCelno;
    SelectedLoop := UndoLoopno;
    LoopAdjust.Position := SelectedLoop;
    CelAdjust.Position := SelectedCel;
    LoopDisplay.Caption := IntToStr(SelectedLoop) + '/' + IntToStr(MainView.NumLoops-1);
    CelDisplay.Caption := IntToStr(SelectedCel) + '/' + IntToStr(MainView.NumCels[SelectedLoop]-1);
    ViewImage.Picture.Bitmap.Assign(UndoBitmap);
    ViewImage.Width := UndoWidth*ScaleFactor*2;
    ViewImage.Height := UndoHeight*ScaleFactor;
    ReCalculateMaxHW;
    ViewImage.Top := MainView.MaxHeight[SelectedLoop]*ScaleFactor - ViewImage.Height;
    if MainView.MirrorLoop[SelectedLoop] and (MainView.MirrorLoopNum[SelectedLoop]<SelectedLoop) then
      ViewImage.Left := MainView.MaxWidth[SelectedLoop]*ScaleFactor*2-ViewImage.Width
    else if ViewImage.Left > 0 then ViewImage.Left := 0;
    UpdateStatus(SelectedLoop,SelectedCel);
    EditUndo.Enabled := False;
    MouseIsDown := False;
  end;
end;

{*************************************************************}
procedure TViewEdit.TransSetButClick(Sender: TObject);
{*************************************************************}
begin
  TransCol := LCol;
  TransColDisplay.Color := EGACOL[TransCol];
  ActiveControl := Panel1;
  ViewModified := True;
end;

{*************************************************************}
procedure TViewEdit.WidthAdjustClick(Sender: TObject; Button: TUDBtnType);
{*************************************************************}
begin
  if (not (UndoType = SizeChange)) or (not EditUndo.Enabled) then
    SaveUndo(SizeChange);
  ViewImage.Canvas.Brush.Color := EGACol[TransCol];
  ViewImage.Picture.Bitmap.Width := WidthAdjust.Position;
  ViewImage.Width := ViewImage.Picture.Bitmap.Width*ScaleFactor*2;
  WidthDisplay.Text := IntToStr(WidthAdjust.Position);
  ReCalculateMaxHW;
  if MainView.MirrorLoop[SelectedLoop] and (MainView.MirrorLoopNum[SelectedLoop]<SelectedLoop) then
    ViewImage.Left := MainView.MaxWidth[SelectedLoop]*ScaleFactor*2-ViewImage.Width
  else if ViewImage.Left > 0 then ViewImage.Left := 0;
{  if MainView.Loop1Mirror0 then
  begin
    if SelectedLoop = 0 then CopyLoop(0,1,True)
    else if SelectedLoop = 1 then CopyLoop(1,0,True)
  end;}
end;

{*************************************************************}
procedure TViewEdit.HeightAdjustClick(Sender: TObject; Button: TUDBtnType);
{*************************************************************}
begin
  if (not (UndoType = SizeChange)) or (not EditUndo.Enabled) then
    SaveUndo(SizeChange);
  ViewImage.Canvas.Brush.Color := EGACol[TransCol];
  ViewImage.Picture.Bitmap.Height := HeightAdjust.Position;
  ViewImage.Height := ViewImage.Picture.Bitmap.Height*ScaleFactor;
  HeightDisplay.Text := IntToStr(HeightAdjust.Position);
  ReCalculateMaxHW;
  ViewImage.Top := MainView.MaxHeight[SelectedLoop]*ScaleFactor - ViewImage.Height;
{  if MainView.Loop1Mirror0 then
  begin
    if SelectedLoop = 0 then CopyLoop(0,1,True)
    else if SelectedLoop = 1 then CopyLoop(1,0,True)
  end;}
end;

{*************************************************************}
procedure TViewEdit.WidthDisplayExit(Sender: TObject);
{*************************************************************}
var WidthInt : integer;
    Code     : integer;
begin
  Val(WidthDisplay.Text,WidthInt,Code);
  if (Code <> 0) or (WidthInt < 1) or (WidthInt > MaxCelWidth) then
  begin
    ShowMessage('Invalid width value. Enter a number from 1 to '+IntToStr(MaxCelWidth)+'.');
    WidthDisplay.Text := IntToStr(ViewImage.Picture.Bitmap.Width);
    WidthDisplay.SetFocus;
    WidthDisplay.SelectAll;
  end
  else
  begin
    if WidthInt <> ViewImage.Picture.Bitmap.Width then
    begin
      if (not (UndoType = SizeChange)) or (not EditUndo.Enabled) then
        SaveUndo(SizeChange);
      ViewImage.Canvas.Brush.Color := EGACOL[MainView.CelTransCol[SelectedLoop,SelectedCel]];
      WidthAdjust.Position := WidthInt;
      ViewImage.Picture.Bitmap.Width := WidthAdjust.Position;
      ViewImage.Width := ViewImage.Picture.Bitmap.Width*ScaleFactor*2;
      WidthDisplay.Text := IntToStr(WidthAdjust.Position);
      ReCalculateMaxHW;
      if MainView.MirrorLoop[SelectedLoop] and (MainView.MirrorLoopNum[SelectedLoop]<SelectedLoop) then
        ViewImage.Left := MainView.MaxWidth[SelectedLoop]*ScaleFactor*2-ViewImage.Width
      else if ViewImage.Left > 0 then ViewImage.Left := 0;
{      if MainView.Loop1Mirror0 then
      begin
        if SelectedLoop = 0 then CopyLoop(0,1,True)
        else if SelectedLoop = 1 then CopyLoop(1,0,True)
      end;}
    end;
{    Panel1.SetFocus;}
  end;
end;

{*************************************************************}
procedure TViewEdit.WidthDisplayKeyPress(Sender: TObject; var Key: Char);
{*************************************************************}
begin
  if ord(Key) = 27 then {Esc pressed}
  begin
    WidthDisplay.Text := IntToStr(ViewImage.Picture.Bitmap.Width);
    Panel1.SetFocus;
  end;
  if ord(Key) = 13 then {Enter pressed}
  begin
    WidthDisplayExit(Application);
    Panel1.SetFocus;
  end;
end;

{*************************************************************}
procedure TViewEdit.WidthDisplayEnter(Sender: TObject);
{*************************************************************}
begin
  WidthDisplay.SelectAll;
end;

{*************************************************************}
procedure TViewEdit.WidthDisplayClick(Sender: TObject);
{*************************************************************}
begin
  WidthDisplay.SelectAll;
end;

{*************************************************************}
procedure TViewEdit.HeightDisplayExit(Sender: TObject);
{*************************************************************}
var HeightInt : integer;
    Code      : integer;

begin
  Val(HeightDisplay.Text,HeightInt,Code);
  if (Code <> 0) or (HeightInt < 1) or (HeightInt > MaxCelHeight) then
  begin
    ShowMessage('Invalid height value. Enter a number from 1 to '+IntToStr(MaxCelHeight)+'.');
    HeightDisplay.Text := IntToStr(ViewImage.Picture.Bitmap.Height);
    HeightDisplay.SetFocus;
    HeightDisplay.SelectAll;
  end
  else
  begin
    if HeightInt <> ViewImage.Picture.Bitmap.Height then
    begin
      if (not (UndoType = SizeChange)) or (not EditUndo.Enabled) then
        SaveUndo(SizeChange);
      ViewImage.Canvas.Brush.Color := EGACOL[MainView.CelTransCol[SelectedLoop,SelectedCel]];
      HeightAdjust.Position := HeightInt;
      ViewImage.Picture.Bitmap.Height := HeightAdjust.Position;
      ViewImage.Height := ViewImage.Picture.Bitmap.Height*ScaleFactor;
      HeightDisplay.Text := IntToStr(HeightAdjust.Position);
      ReCalculateMaxHW;
      ViewImage.Top := MainView.MaxHeight[SelectedLoop]*ScaleFactor - ViewImage.Height;
{      if MainView.Loop1Mirror0 then
      begin
        if SelectedLoop = 0 then CopyLoop(0,1,True)
        else if SelectedLoop = 1 then CopyLoop(1,0,True)
      end;}
    end;
{    Panel1.SetFocus;}
  end;
end;

{*************************************************************}
procedure TViewEdit.HeightDisplayKeyPress(Sender: TObject; var Key: Char);
{*************************************************************}
begin
  if ord(Key) = 27 then {Esc pressed}
    HeightDisplay.Text := IntToStr(ViewImage.Picture.Bitmap.Height);
  if ord(Key) = 13 then {Enter pressed}
    HeightDisplayExit(Application);
end;

{*************************************************************}
procedure TViewEdit.HeightDisplayEnter(Sender: TObject);
{*************************************************************}
begin
  HeightDisplay.SelectAll;
end;

{*************************************************************}
procedure TViewEdit.HeightDisplayClick(Sender: TObject);
{*************************************************************}
begin
  HeightDisplay.SelectAll;
end;

{*************************************************************}
procedure TViewEdit.CelInsertClick(Sender: TObject);
{*************************************************************}
var CurCel : byte;
    NewCelWidth : byte;
    NewCelHeight : byte;
    NewCelTransCol : byte;
    NewCelMirror : boolean;
    NewCelRefPoint : TRefPoint;

begin
  if MainView.NumCels[SelectedLoop] < MaxCels then
  begin
    SaveExistingCel;
    InsertCel(SelectedLoop,SelectedCel);
    if MainView.MirrorLoop[SelectedLoop] then InsertCel(MainView.MirrorLoopNum[SelectedLoop],SelectedCel);
    DisplayView(SelectedLoop,SelectedCel);
    EditUndo.Enabled := False;
    ViewModified := True;
  end
  else ShowMessage('Maximum of '+IntToStr(MaxCels)+' cels already reached in this loop.');
end;

{*************************************************************}
procedure TViewEdit.CelAppendClick(Sender: TObject);
{*************************************************************}
begin
  if MainView.NumCels[SelectedLoop] < MaxCels then
  begin
    SaveExistingCel;
    AppendCel(SelectedLoop);
    if MainView.MirrorLoop[SelectedLoop] then AppendCel(MainView.MirrorLoopNum[SelectedLoop]);
    SelectedCel := MainView.NumCels[SelectedLoop]-1;
    DisplayView(SelectedLoop,SelectedCel);
    EditUndo.Enabled := False;
    ViewModified := True;
  end
  else ShowMessage('Maximum of '+IntToStr(MaxCels)+' cels already reached in this loop.');
end;

{*************************************************************}
procedure TViewEdit.CelDeleteClick(Sender: TObject);
{*************************************************************}
begin
  if MainView.NumCels[SelectedLoop] > 1 then
  begin
    SaveExistingCel;
    DeleteCel(SelectedLoop,SelectedCel);
    if MainView.MirrorLoop[SelectedLoop] then DeleteCel(MainView.MirrorLoopNum[SelectedLoop],SelectedCel);
    if SelectedCel > MainView.NumCels[SelectedLoop] - 1 then
      SelectedCel := MainView.NumCels[SelectedLoop] - 1;
    DisplayView(SelectedLoop,SelectedCel);
    EditUndo.Enabled := False;
    ViewModified := True;
  end
  else ShowMessage('You can not delete the last cel in a loop.');
end;

{*************************************************************}
procedure TViewEdit.CelClearClick(Sender: TObject);
{*************************************************************}
begin
  SaveExistingCel;
  ClearCel(SelectedLoop,SelectedCel);
  if MainView.MirrorLoop[SelectedLoop] then ClearCel(MainView.MirrorLoopNum[SelectedLoop],SelectedCel);
  DisplayView(SelectedLoop,SelectedCel);
  EditUndo.Enabled := False;
  ViewModified := True;
end;

{*************************************************************}
procedure TViewEdit.LoopInsertClick(Sender: TObject);
{*************************************************************}
var CurLoop        : byte;
    CurCel         : byte;
    NewCelWidth : byte;
    NewCelHeight : byte;
    NewCelTransCol : byte;

begin
  if MainView.NumLoops < MaxLoops then
  begin
    SaveExistingCel;
    InsertLoop(SelectedLoop);
    SelectedCel := 0;
    DisplayView(SelectedLoop,SelectedCel);
    EditUndo.Enabled := False;
    ViewModified := True;
  end {if MainView.NumLoops < MaxLoops}
  else ShowMessage('Maximum of '+IntToStr(MaxLoops)+' loops already reached.');
end;


{*************************************************************}
procedure TViewEdit.LoopAppendClick(Sender: TObject);
{*************************************************************}
begin
  if MainView.NumLoops < MaxLoops then
  begin
    SaveExistingCel;
    AppendLoop;
    SelectedLoop := MainView.NumLoops-1;
    SelectedCel := 0;
    DisplayView(SelectedLoop,SelectedCel);
    EditUndo.Enabled := False;
    ViewModified := True;
  end {if MainView.NumLoops < MaxLoops}
  else ShowMessage('Maximum of '+IntToStr(MaxLoops)+' loops already reached.');
end;

{*************************************************************}
procedure TViewEdit.LoopDeleteClick(Sender: TObject);
{*************************************************************}
var CurLoop : byte;
    CurCel  : byte;
begin
  if MainView.NumLoops > 1 then
  begin
    SaveExistingCel;
    DeleteLoop(SelectedLoop);
    if SelectedLoop > MainView.NumLoops - 1 then SelectedLoop := MainView.NumLoops-1;
    SelectedCel := 0;
    DisplayView(SelectedLoop,SelectedCel);
    EditUndo.Enabled := False;
    ViewModified := True;
  end {if MainView.NumLoops > 0}
  else ShowMessage('You can not delete the last loop.');
end;

{*************************************************************}
procedure TViewEdit.LoopClearClick(Sender: TObject);
{*************************************************************}
var CurCel : byte;

begin
  SaveExistingCel;
  ClearLoop(SelectedLoop);
  if MainView.MirrorLoop[SelectedLoop] then CopyLoop(SelectedLoop,MainView.MirrorLoopNum[SelectedLoop],True);
  SelectedCel := 0;
  DisplayView(SelectedLoop,SelectedCel);
  EditUndo.Enabled := False;
  ViewModified := True;
end;

{*************************************************************}
procedure TViewEdit.EditCopyCelClick(Sender: TObject);
{*************************************************************}
var TempBitmap : TBitmap;
begin
{  if Active then
  begin}
  TempBitmap := TBitmap.Create;
  FixPalette(TempBitmap);
  TempBitmap.Width := ViewImage.Picture.Bitmap.Width*2;
  TempBitmap.Height := ViewImage.Picture.Bitmap.Height;
  TempBitmap.Canvas.StretchDraw(Rect(0,0,TempBitmap.Width,TempBitmap.Height),ViewImage.Picture.Bitmap);
  Clipboard.Assign(TempBitmap);
  TempBitmap.Free;
{  end;}
end;

{*************************************************************}
procedure TViewEdit.EditPasteCelClick(Sender: TObject);
{*************************************************************}
{red = 1}
{green = 2}
{blue = 3}
type TRGBCol   = array[1..3] of byte;

const Colname : array[1..3] of string[8] = ('Red','Green','Blue');

var TempBitmap            : TBitmap;
    CurX, CurY            : byte;
    ThisPixelRGB          : TRGBCol;
    EGARGB                : array[0..15] of TRGBCol;
    CurCol                : byte;
    CurEGACol             : byte;

  function GetRGBCol(Col24Bit:TColor) : TRGBCol;
  {returns the red, green & blue components of a TColor}
  var TempCol : longint;
  begin
    TempCol := Col24Bit mod $1000000;
    GetRGBCol[1] := TempCol mod $100;               {RED}
    GetRGBCol[2] := (TempCol div $100) mod $100;    {GREEN}
    GetRGBCol[3] := TempCol div $10000;             {BLUE}
  end;

  function GetClosestEGACol(Col24Bit:TColor) : byte;
  var Diff                  : byte;
      Closest               : array[1..3] of byte;
      NumClosestCols        : array[1..3] of byte;
      ClosestCol            : array[1..3,1..16] of byte;
      CurCol                : byte;
      CurEGACol             : byte;
      CurClosestEntry       : byte;
      GotClosestCol         : array[1..3] of boolean;
      lc : byte;
      gce : byte;
  begin
    for CurCol := 1 to 3 do
    begin
      Closest[CurCol] := 255;
      NumClosestCols[CurCol] := 0;
    end;
    ThisPixelRGB := GetRGBCol(Col24Bit);
    for CurEGACol := 0 to 15 do
    begin
      for CurCol := 1 to 3 do
      begin
        Diff := abs(ThisPixelRGB[CurCol]-EGARGB[CurEGACol][CurCol]);
        if Diff < Closest[CurCol] then
        begin
          NumClosestCols[CurCol] := 1;
          ClosestCol[CurCol,1] := CurEGACol;
          Closest[CurCol] := Diff;
{          ShowMessage('['+IntToStr(CurX)+','+IntToStr(CurY)+']/'+ColName[CurCol]+':'+IntToStr(CurEGACol));}
        end
        else if Diff = Closest[CurCol] then
        begin
          NumClosestCols[CurCol] := NumClosestCols[CurCol]+1;
          ClosestCol[CurCol,NumClosestCols[CurCol]] := CurEGACol;
{          ShowMessage('['+IntToStr(CurX)+','+IntToStr(CurY)+']/'+ColName[CurCol]+':'+IntToStr(CurEGACol));}
        end;
      end;
    end;
    GetClosestEGACol := 0;
    for CurEGACol := 0 to 15 do
    begin
      for CurCol := 1 to 3 do
      begin
        GotClosestCol[CurCol] := False;
        for CurClosestEntry := 1 to NumClosestCols[CurCol] do
          if ClosestCol[CurCol,CurClosestEntry] = CurEGACol then GotClosestCol[CurCol] := True;
      end;
      if GotClosestCol[1] and GotClosestCol[2] and GotClosestCol[3] then GetClosestEGACol := CurEGACol;
    end;
  end;

begin
  if Clipboard.HasFormat(CF_Bitmap) then
  begin
    TempBitmap := TBitmap.Create;
    FixPalette(TempBitmap);
    TempBitmap.Assign(Clipboard);
    if ((TempBitmap.Width div 2)>MaxCelWidth) or (TempBitmap.Height>MaxCelHeight) then
      ShowMessage('Error: Image in clipboard is too big!')
    else
    begin
      for CurCol := 0 to 15 do
        EGARGB[CurCol] := GetRGBCol(EGACOL[CurCol]);
      ViewImage.Picture.Bitmap.Width := TempBitmap.Width div 2;
      ViewImage.Picture.Bitmap.Height := TempBitmap.Height;
      ViewImage.Width := ViewImage.Picture.Bitmap.Width*ScaleFactor*2;
      ViewImage.Height := ViewImage.Picture.Bitmap.Height*ScaleFactor;
      WidthDisplay.Text := IntToStr(ViewImage.Picture.Bitmap.Width);
      HeightDisplay.Text := IntToStr(ViewImage.Picture.Bitmap.Height);
      WidthAdjust.Position := ViewImage.Picture.Bitmap.Width;
      HeightAdjust.Position := ViewImage.Picture.Bitmap.Height;
      LoadingGauge.MaxValue := TempBitmap.Height;
      LoadingGauge.Progress := 0;
      LoadingGauge.Visible := True;
      for CurY := 0 to TempBitmap.Height - 1 do
      begin
        for CurX := 0 to (TempBitmap.Width div 2) - 1 do
        begin
          ViewImage.Canvas.Pixels[CurX,CurY] := EGACOL[GetClosestEGACol(TempBitmap.Canvas.Pixels[CurX*2,CurY])];
        end;
      end;
      LoadingGauge.Visible := False;
    end;
    ReCalculateMaxHW;
    ViewImage.Top := MainView.MaxHeight[SelectedLoop]*ScaleFactor - ViewImage.Height;
    TempBitmap.Free;
  end;
end;

{*************************************************************}
procedure TViewEdit.Loop1MirrorCheck1Click(Sender: TObject);
{*************************************************************}
begin
{  if ViewLoaded then
  begin
    if Loop1MirrorCheck.Checked and (MainView.NumLoops > 1)then
    begin
      CopyLoop(0,1,True);
      MainView.Loop1Mirror0 := True;
      SelectedLoop := 1;
      SelectedCel := 0;
      DisplayView(SelectedLoop,SelectedCel);
    end
    else
    begin
      SaveExistingCel;
      MainView.Loop1Mirror0 := False;
    end;
  end;}
end;

{*************************************************************}
procedure TViewEdit.ViewNewClick(Sender: TObject);
{*************************************************************}
var SaveMessageResult : integer;
    AllowNew : boolean;

begin
  AllowNew := True;
  if ViewModified then
  begin
    if LoadedFromGame then
    begin
      SaveMessageResult := Application.MessageBox('Do you want to save changes to the current view? (view will be saved to game)','View Editor',MB_YESNOCANCEL);
      if SaveMessageResult = IDYES then ViewSaveToGameClick(Sender)
      else if SaveMessageResult = IDCANCEL then AllowNew := False;
    end
    else
    begin
      SaveMessageResult := Application.MessageBox('Do you want to save changes to the current view?  (view will be saved to a file)','View Editor',MB_YESNOCANCEL);
      if SaveMessageResult = IDYES then ViewSaveToFileClick(Sender)
      else if SaveMessageResult = IDCANCEL then AllowNew := False;
    end;
  end;

  if AllowNew then
    NewView;
end;

{*************************************************************}
procedure TViewEdit.CelUpButClick(Sender: TObject);
{*************************************************************}
begin
  if (not (UndoType = MoveCel)) or (not EditUndo.Enabled) then
    SaveUndo(MoveCel);
  ViewImage.Canvas.CopyRect(Rect(0,0,ViewImage.Picture.Bitmap.Width,ViewImage.Picture.Bitmap.Height-1),ViewImage.Canvas,Rect(0,1,ViewImage.Picture.Bitmap.Width,ViewImage.Picture.Bitmap.Height));
  ViewImage.Canvas.Brush.Color := EGACol[TransCol];
  ViewImage.Canvas.FillRect(Rect(0,ViewImage.Picture.Bitmap.Height-1,ViewImage.Picture.Bitmap.Width,ViewImage.Picture.Bitmap.Height));
end;

{*************************************************************}
procedure TViewEdit.CelDownButClick(Sender: TObject);
{*************************************************************}
begin
  if (not (UndoType = MoveCel)) or (not EditUndo.Enabled) then
    SaveUndo(MoveCel);
  ViewImage.Canvas.CopyRect(Rect(0,1,ViewImage.Picture.Bitmap.Width,ViewImage.Picture.Bitmap.Height),ViewImage.Canvas,Rect(0,0,ViewImage.Picture.Bitmap.Width,ViewImage.Picture.Bitmap.Height-1));
  ViewImage.Canvas.Brush.Color := EGACol[TransCol];
  ViewImage.Canvas.FillRect(Rect(0,0,ViewImage.Picture.Bitmap.Width,1));
end;

{*************************************************************}
procedure TViewEdit.CelLeftButClick(Sender: TObject);
{*************************************************************}
begin
  if (not (UndoType = MoveCel)) or (not EditUndo.Enabled) then
    SaveUndo(MoveCel);
  ViewImage.Canvas.CopyRect(Rect(0,0,ViewImage.Picture.Bitmap.Width-1,ViewImage.Picture.Bitmap.Height),ViewImage.Canvas,Rect(1,0,ViewImage.Picture.Bitmap.Width,ViewImage.Picture.Bitmap.Height));
  ViewImage.Canvas.Brush.Color := EGACol[TransCol];
  ViewImage.Canvas.FillRect(Rect(ViewImage.Picture.Bitmap.Width-1,0,ViewImage.Picture.Bitmap.Width,ViewImage.Picture.Bitmap.Height));
end;

{*************************************************************}
procedure TViewEdit.CelRightButClick(Sender: TObject);
{*************************************************************}
begin
  if (not (UndoType = MoveCel)) or (not EditUndo.Enabled) then
    SaveUndo(MoveCel);
  ViewImage.Canvas.CopyRect(Rect(1,0,ViewImage.Picture.Bitmap.Width,ViewImage.Picture.Bitmap.Height),ViewImage.Canvas,Rect(0,0,ViewImage.Picture.Bitmap.Width-1,ViewImage.Picture.Bitmap.Height));
  ViewImage.Canvas.Brush.Color := EGACol[TransCol];
  ViewImage.Canvas.FillRect(Rect(0,0,1,ViewImage.Picture.Bitmap.Height));
end;

{*************************************************************}
procedure TViewEdit.CelFlipHClick(Sender: TObject);
{*************************************************************}
begin
  SaveUndo(Draw);
  ViewImage.Canvas.StretchDraw(Rect(ViewImage.Picture.Bitmap.Width-1,0,-1,ViewImage.Picture.Bitmap.Height),ViewImage.Picture.Bitmap);
end;

{*************************************************************}
procedure TViewEdit.CelFlipVClick(Sender: TObject);
{*************************************************************}
begin
  SaveUndo(Draw);
  ViewImage.Canvas.StretchDraw(Rect(0,ViewImage.Picture.Bitmap.Height-1,ViewImage.Picture.Bitmap.Width,-1),ViewImage.Picture.Bitmap);
  ViewModified := True;
end;

{*************************************************************}
procedure TViewEdit.ViewLoadFromFileClick(Sender: TObject);
{*************************************************************}
var LoadingFile : file;
    LoadedResource : TResource;
    SaveMessageResult : integer;
    AllowLoad : boolean;

begin
  AllowLoad := True;
  if ViewModified then
  begin
    if LoadedFromGame then
    begin
      SaveMessageResult := Application.MessageBox('Do you want to save changes to the current view? (view will be saved to game)','View Editor',MB_YESNOCANCEL);
      if SaveMessageResult = IDYES then ViewSaveToGameClick(Sender)
      else if SaveMessageResult = IDCANCEL then AllowLoad := False;
    end
    else
    begin
      SaveMessageResult := Application.MessageBox('Do you want to save changes to the current view?  (view will be saved to a file)','View Editor',MB_YESNOCANCEL);
      if SaveMessageResult = IDYES then ViewSaveToFileClick(Sender)
      else if SaveMessageResult = IDCANCEL then AllowLoad := False;
    end;
  end;

  if AllowLoad then
    if OpenDialog1.Execute then
    begin
      AssignFile(LoadingFile,OpenDialog1.Filename);
      {$I-}
      Reset(LoadingFile,1);
      {$I+}
      if IOResult <> 0 then
        ShowMessage('Error: Could not open file '+OpenDialog1.Filename)
      else
      begin
        if FileSize(LoadingFile) > MaxResourceSize then
        begin
          ShowMessage('Error: File too big (>'+IntToStr(MaxResourceSize)+' bytes)');
          CloseFile(LoadingFile);
        end
        else
        begin
          LoadedResource.Size := FileSize(LoadingFile);
          GetMem(LoadedResource.Data,LoadedResource.Size);
          BlockRead(LoadingFile,LoadedResource.Data^,LoadedResource.Size);
          CloseFile(LoadingFile);
          LoadView(LoadedResource);  {loadview frees the memory after it has used it.}
          LoadedFromGame := False;
          ViewNum := 0;
          Caption := 'View Editor - '+ExtractFilename(OpenDialog1.Filename);
        end;
      end;
    end;
end;

{*************************************************************}
procedure TViewEdit.ViewSaveToFileClick(Sender: TObject);
{*************************************************************}
var SavedResource : TResource;
    ResourceFile : file;
begin
  SaveExistingCel;  {-- Important!}
  if (SaveDialog1.Filename = '') and (LoadedFromGame) then
    SaveDialog1.Filename := ResourceName(View,Viewnum);
  if SaveDialog1.Execute then
  begin
    SavedResource := SaveView;
    if SavedResource.Size > 0 then
    begin
      AssignFile(ResourceFile,SaveDialog1.Filename);
      {$I-}
      Rewrite(ResourceFile,1);
      {$I+}
      if IOResult <> 0 then
        ShowMessage('Error: Could not write to file '+SaveDialog1.Filename)
      else
      begin
        BlockWrite(ResourceFile,SavedResource.Data^,SavedResource.Size);
        CloseFile(ResourceFile);
        ViewModified := False;
        EditUndo.Enabled := False;
      end;
      FreeMem(SavedResource.Data,SavedResource.Size);
    end; {if SavedResource.Size > 0}
  end; {if SaveDialog1.Execute}
end;

{*************************************************************}
procedure TViewEdit.ViewSaveToGameAsClick(Sender: TObject);
{*************************************************************}
begin
  GetResourceNum.ViewEditorNum := ViewEditorNum;
  GetResourceNum.WindowFunction := ViewEditSaveAsResource;
  if LoadedFromGame then
  begin
    GetResourceNum.NumberEdit.Text := IntToStr(ViewNum);
    GetResourceNum.OKBut.Enabled := True;
  end
  else
  begin
    GetResourceNum.NumberEdit.Text := '';
    GetResourceNum.OKBut.Enabled := False;
  end;
  GetResourceNum.ShowModal;
end;

{*************************************************************}
procedure TViewEdit.SaveAsResource(ResNum:byte);
{*************************************************************}
var SavedResource : TResource;
begin
  SavedResource := SaveView;
  if SavedResource.Size > 0 then
  begin
    if (ResourceIndex(VIEW,Resnum)=-1) or ((ResourceIndex(VIEW,ResNUm)>-1)
      and AskYesNo('Confirm Save','Resource '+ResourceName(VIEW,ResNum)+' already exists. Replace it?')) then
    begin
      SaveExistingCel;  {-- Important!}
      AddResource(SavedResource,VIEW,ResNum); {Add resource will free the memory for SavedResource}
      LoadedFromGame := True;
      ViewNum := ResNum;
      ViewEditWin[ViewEditorNum].Caption := 'View Editor - ' + ResourceName(View,ViewNum);
      ViewModified := False;
      EditUndo.Enabled := False;
    end;
  end;
end;

{*************************************************************}
procedure TViewEdit.ViewSaveToGameClick(Sender: TObject);
{*************************************************************}
var SavedResource : TResource;
begin
  SaveExistingCel;  {-- Important!}
  if not LoadedFromGame then ViewSaveToGameAsClick(Application)
  else
  begin
    SavedResource := SaveView;
    if SavedResource.Size > 0 then
      AddResource(SavedResource,VIEW,Viewnum); {Add resource will free the memory for SavedResource}
    ViewModified := False;
    EditUndo.Enabled := False;
  end;
end;

{*************************************************************}
procedure TViewEdit.LoopMirrorSelectComboChange(Sender: TObject);
{*************************************************************}
var Loopno : byte;
begin
  EditUndo.Enabled := False;
  Loopno := LoopMirrorSelectLoopno[LoopMirrorSelectCombo.ItemIndex];
  if Loopno = $FF then {'no other loop'}
  begin
    UnMirrorLoop(SelectedLoop);
    UpdateStatus(SelectedLoop,SelectedCel);
  end
  else if Loopno <= MainView.NumLoops - 1 then
  begin
    UnMirrorLoop(SelectedLoop);
    UnMirrorLoop(Loopno);
    CopyLoop(Loopno,SelectedLoop,True);
    MainView.MirrorLoop[SelectedLoop] := True;
    MainView.MirrorLoopNum[SelectedLoop] := Loopno;
    MainView.MirrorLoop[Loopno] := True;
    MainView.MirrorLoopNum[Loopno] := SelectedLoop;
    SelectedCel := 0;
    DisplayView(SelectedLoop,SelectedCel);
  end;
end;

{*************************************************************}
procedure TViewEdit.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
{*************************************************************}
begin
  CanClose := AskClose;
end;


end.
