/*
 * Copyright (c) 2002 by Ravi Iyengar [ravi.i@softhome.net]
 * Released under the GNU General Public License
 * See LICENSE for details.
 */

// ChannelsDlg.cpp : implementation file
//

#include "stdafx.h"
#include "Soundbox.h"
#include "ChannelsDlg.h"
#include "SoundResource.h"

const int CChannelsDlg::FLAG_CHECK_IDS[8] = {IDC_FLAG0, IDC_FLAG1,
	IDC_FLAG2, IDC_FLAG3, IDC_FLAG4, IDC_FLAG5, IDC_FLAG6, IDC_FLAG7};

// CChannelsDlg dialog

IMPLEMENT_DYNAMIC(CChannelsDlg, CDialog)
CChannelsDlg::CChannelsDlg(SoundResource *sound, CWnd* pParent /*=NULL*/)
	: CDialog(CChannelsDlg::IDD, pParent), mSound(sound), mCurChannel(-1)
{
	mChannelsListFont.CreateFont((HIWORD(GetDialogBaseUnits()) * 5) / 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, "Courier New");
}

CChannelsDlg::~CChannelsDlg()
{
}

void CChannelsDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
}

void CChannelsDlg::SaveSettings(int channel)
{
	int flags = 0;
	for (int i = 0; i < 8; ++i) {
		CButton *btn = (CButton*)GetDlgItem(FLAG_CHECK_IDS[i]);
		if (btn->GetCheck() == BST_CHECKED) {
			flags |= (1 << i);
		}
	}

	CEdit *edit = (CEdit*)GetDlgItem(IDC_INITIALVOICESEDIT);
	CString str;
	edit->GetWindowText(str);
	int voices = atoi(str);

	if (voices < 0) voices = 0;
	if (voices > 255) voices = 255;

	mSound->SetChannelPlayFlag(channel, flags);
	mSound->SetChannelVoices(channel, voices);
}

void CChannelsDlg::LoadSettings(int channel)
{
	int i;
	int flags = mSound->GetChannelPlayFlag(channel);
	int voices = mSound->GetChannelVoices(channel);

	for (i = 0; i < 8; ++i) {
		CButton *btn = (CButton*)GetDlgItem(FLAG_CHECK_IDS[i]);
		btn->SetCheck((flags & (1 << i)) ? BST_CHECKED : BST_UNCHECKED);
	}

	char txt[32];
	sprintf(txt, "%d", voices);
	CEdit *edit = (CEdit*)GetDlgItem(IDC_INITIALVOICESEDIT);
	edit->SetWindowText(txt);

	CListBox *controlList = (CListBox*)GetDlgItem(IDC_CONTROLLIST);
	controlList->ResetContent();
	mIndex.RemoveAll();
	int n = mSound->GetNumEvents();
	for (i = 0; i < n; ++i) {
		sound_event event = mSound->GetEvent(i);
		if ((event.channel == channel) &&
			(event.status == sound_event::CONTROL_CHANGE) &&
			(event.param1 == 0x4B)) {

			sprintf(txt, "%-6d %d", event.absoluteTime, event.param2);
			controlList->AddString(txt);
			mIndex.Add(i);
		}
	}

	CButton *delBtn = (CButton*)GetDlgItem(IDC_DELBTN);
	delBtn->EnableWindow(false);
}


BEGIN_MESSAGE_MAP(CChannelsDlg, CDialog)
	ON_WM_CTLCOLOR()
	ON_LBN_SELCHANGE(IDC_CHANNELSLIST, OnChannelsListSelChange)
	ON_LBN_SELCHANGE(IDC_CONTROLLIST, OnControlListSelChange)
	ON_BN_CLICKED(IDC_DELBTN, OnDelBtnClicked)
	ON_BN_CLICKED(IDC_ADDBTN, OnAddBtnClicked)
END_MESSAGE_MAP()


// CChannelsDlg message handlers

BOOL CChannelsDlg::OnInitDialog()
{
	CDialog::OnInitDialog();
	int i;
	char txt[4];

	CListBox *channelsList = (CListBox*)GetDlgItem(IDC_CHANNELSLIST);
	channelsList->SetFont(&mChannelsListFont);
	for (i = 0; i < 16; ++i) {
		sprintf(txt, "%2d", i);
		channelsList->AddString(txt);
	}

	CButton *flagBtn;
	for (i = 0; i < 8; ++i) {
		flagBtn = (CButton*)GetDlgItem(FLAG_CHECK_IDS[i]);
		flagBtn->SetWindowText(CSoundboxApp::FILTER_NAMES[i]);
	}

	CListBox *controlList = (CListBox*)GetDlgItem(IDC_CONTROLLIST);
	controlList->SetFont(&CSoundboxApp::mListFont);

	CSpinButtonCtrl *spin = (CSpinButtonCtrl*)GetDlgItem(IDC_INITIALVOICESSPIN);
	spin->SetRange(0, 255);
	spin = (CSpinButtonCtrl*)GetDlgItem(IDC_TIMESPIN);
	spin->SetRange(0, mSound->GetTotalTime());
	spin = (CSpinButtonCtrl*)GetDlgItem(IDC_VALUESPIN);
	spin->SetRange(0, 127);

	channelsList->SetCurSel(0);
	mCurChannel = 0;
	LoadSettings(0);

	return TRUE;
}

HBRUSH CChannelsDlg::OnCtlColor(CDC* pDC, CWnd* pWnd, UINT nCtlColor)
{
	HBRUSH hbr = CDialog::OnCtlColor(pDC, pWnd, nCtlColor);

	int id = pWnd->GetDlgCtrlID();
	for (int i = 0; i < 8; ++i) {
		if (id == FLAG_CHECK_IDS[i]) {
			pDC->SetTextColor(CSoundboxApp::FILTER_COLORS_DARK[i]);
		}
	}

	return hbr;
}

void CChannelsDlg::OnChannelsListSelChange()
{
	if (mCurChannel != -1) SaveSettings(mCurChannel);
	mCurChannel = ((CListBox*)GetDlgItem(IDC_CHANNELSLIST))->GetCurSel();
	LoadSettings(mCurChannel);
}

void CChannelsDlg::OnControlListSelChange()
{
	CButton *delBtn = (CButton*)GetDlgItem(IDC_DELBTN);
	delBtn->EnableWindow();
}

void CChannelsDlg::OnOK()
{
	if (mCurChannel != -1) SaveSettings(mCurChannel);
	CDialog::OnOK();
}

void CChannelsDlg::OnDelBtnClicked()
{
	// remove the event and its entry from the list
	CListBox *listBox = (CListBox*)GetDlgItem(IDC_CONTROLLIST);
	int localIndex = listBox->GetCurSel();
	int soundIndex = mIndex.GetAt(localIndex);
	mSound->DeleteEvent(soundIndex);
	listBox->DeleteString(localIndex);

	// fix the index
	mIndex.RemoveAt(localIndex);
	int n = (int)mIndex.GetCount();
	for (int i = localIndex; i < n; ++i) {
		mIndex.SetAt(i, mIndex.GetAt(i) - 1);
	}

	// select another item from the list
	if (n > localIndex) {
		listBox->SetCurSel(localIndex);
	} else if (localIndex > 0) {
		listBox->SetCurSel(localIndex - 1);
	} else {
		CButton *delBtn = (CButton*)GetDlgItem(IDC_DELBTN);
 		delBtn->EnableWindow(false);
	}
}

void CChannelsDlg::OnAddBtnClicked()
{
	CEdit *edit = (CEdit*)GetDlgItem(IDC_TIMEEDIT);
	CString str;
	edit->GetWindowText(str);
	int time = atoi(str);

	edit = (CEdit*)GetDlgItem(IDC_VALUEEDIT);
	edit->GetWindowText(str);
	int value = atoi(str);

	// verify a valid time
	if (time < 0) {
		MessageBox("Time must be greater than or equal to 0");
		return;
	}
	if (time > mSound->GetTotalTime()) {
		MessageBox("Time cannot extend past end of soung");
		return;
	}

	// verify a valid value
	if (value < 0) {
		MessageBox("Number of voices must be greater than or equal to 0");
		return;
	}
	if (value > 127) {
		MessageBox("Number of voices must be less than 128");
		return;
	}

	// add the event
	sound_event event;
	event.absoluteTime = time;
	event.status = sound_event::CONTROL_CHANGE;
	event.channel = mCurChannel;
	event.param1 = 0x4B;
	event.param2 = value;
	int soundIndex = mSound->InsertEvent(event);

	// fix the index
	int localIndex = 0;
	int n = (int)mIndex.GetCount();
	while ((localIndex < n) && (mIndex.GetAt(localIndex) < soundIndex)) ++localIndex;
	mIndex.InsertAt(localIndex, soundIndex);
	for (int i = localIndex + 1; i < n; ++i) {
		mIndex.SetAt(i, mIndex.GetAt(i) + 1);
	}

	// add the point to the list box
	char txt[16];
	sprintf(txt, "%-6d %d", time, value);
	CListBox *listBox = (CListBox*)GetDlgItem(IDC_CONTROLLIST);
	listBox->InsertString(localIndex, txt);
}
