/*
 * Copyright (c) 2002 by Ravi Iyengar [ravi.i@softhome.net]
 * Released under the GNU General Public License
 * See LICENSE for details.
 */

#include "StdAfx.h"
#include "datastream.h"
#include "exception.h"

const int DataStream::BEG = std::ios_base::beg;
const int DataStream::CUR = std::ios_base::cur;
const int DataStream::END = std::ios_base::end;

DataStream::DataStream(std::iostream *stream)
: mStream(stream)
{
}

DataStream::~DataStream(void)
{
}

/*
 * Return the next char in the stream without removing it from the stream.
 */
unsigned char DataStream::PeekChar()
{
	if (mStream->eof()) throw EofException("Unexpected end of file");
	return mStream->peek();
}

/*
 * Remove and return the next char in the stream.
 */
unsigned char DataStream::ReadChar()
{
	if (mStream->eof()) throw EofException("Unexpected end of file");
	unsigned char rv;
	mStream->read((char*)&rv, 1);
	return rv;
}

/*
 * Remove and return the next short int from the stream.
 */
unsigned short DataStream::ReadShort()
{
	unsigned short rv = ReadChar();
	rv |= (ReadChar() << 8);
	return rv;
}

/*
 * Remove and return the next short int from the stream in big-endian order.
 */
unsigned short DataStream::ReadBigShort()
{
	unsigned short rv = ReadChar();
	rv = (rv << 8) | ReadChar();
	return rv;
}

/*
 * Remove and return the next three bytes from the stream in big-endian order.
 * This is a type used in MIDI files.
 */
unsigned long DataStream::ReadBigTri()
{
	unsigned long rv = ReadChar();
	rv = (rv << 8) | ReadChar();
	rv = (rv << 8) | ReadChar();
	return rv;
}

/*
 * Remove and return the next long int from the stream in big-endian oder.
 */
unsigned long DataStream::ReadBigLong()
{
	unsigned long rv = ReadChar();
	rv = (rv << 8) | ReadChar();
	rv = (rv << 8) | ReadChar();
	rv = (rv << 8) | ReadChar();
	return rv;
}

/*
 * Remove and return the next variable-length value from the stream. This is a
 * type used in MIDI files.
 */
unsigned long DataStream::ReadVariableLength()
{
	unsigned long rv = 0;
	int i = 0;
	while ((i < 3) && (PeekChar() & 0x80)) {
		rv = (rv << 7) | (ReadChar() & 0x7F);
		++i;
	}

	if (PeekChar() & 0x80) throw DataException("Variable length value is too long");
	rv = (rv << 7) | ReadChar();
	return rv;
}

/*
 * Insert a character into the stream.
 */
void DataStream::WriteChar(unsigned char data)
{
	mStream->write((char*)&data, 1);
}

/*
 * Insert a short int into the stream.
 */
void DataStream::WriteShort(unsigned short data)
{
	mStream->write((char*)&data, 2);
}

/*
 * Tell the position of the stream's get pointer.
 */
int DataStream::Tellg()
{
	return mStream->tellg();
}

/*
 * Seek the stream's get pointer.
 */
void DataStream::Seekg(int where, int how)
{
	mStream->seekg(where, how);
}

/*
 * Fill a buffer with raw bytes from the stream.
 */
void DataStream::ReadBuffer(unsigned char *data, int length)
{
	mStream->read((char*)data, length);
}
