/*
 * Copyright (c) 2002 by Ravi Iyengar [ravi.i@softhome.net]
 * Released under the GNU General Public License
 * See LICENSE for details.
 */

#include "StdAfx.h"
#include "DataStream.h"
#include "Exception.h"
#include "SoundPlayer.h"
#include "MidiPlayer.h"
#include "MtGmPlayer.h"
#include "PlayerManager.h"
#include <fstream>

PlayerManager::PlayerManager()
: mResourcePath("")
{
}

void PlayerManager::SetResourcePath(const CString &path)
{
	mResourcePath = path;
}

SoundPlayer* PlayerManager::CreateAndInit(HWND listener, int index)
{
	SoundPlayer *rv;
	switch (index) {
		case 0:   // MIDI mapper
			rv = new MidiPlayer(listener);
			rv->Initialize();
			return rv;

		case 1:   // MIDI mapper with FreeSCI MT-32 -> GM conversion
			rv = new MtGmPlayer(listener);
			InitWithPatch(rv, 1);
			return rv;
	}

	ASSERT("Unknown player index");
	return 0;
}

void PlayerManager::InitWithPatch(SoundPlayer *player, int patchNum)
{
	bool havePatch = false;
	std::fstream inStream;

	// first, try loading the patch from the current working directory
	if (mResourcePath != "") {
		char patchPath[1024];
		sprintf(patchPath, "%s\\PATCH.%03d", mResourcePath.GetBuffer(), patchNum);
		inStream.open(patchPath, std::ios_base::in | std::ios_base::binary);
		havePatch = inStream.good();
	}

	// if that didn't work, ask the user where it is
	if (!havePatch) {
		char txt[16];
		sprintf(txt, "PATCH.%03d", patchNum);
		CFileDialog openDlg(true, 0, txt, OFN_FILEMUSTEXIST | OFN_HIDEREADONLY);
		if (openDlg.DoModal() == IDOK) {
			CString file = openDlg.GetPathName();
			inStream.clear();
			inStream.open(file, std::ios_base::in | std::ios_base::binary);
			havePatch = inStream.good();
		}
	}

	// unable to load the patch, so throw an exception
	if (!havePatch) {
		delete player;   // prevent memory leak
		char txt[32];
		sprintf(txt, "Unable to load PATCH.%03d", patchNum);
		throw DeviceException(txt);
	}

	DataStream dataStream(&inStream);
	try {
		player->Initialize(&dataStream);
	} catch (std::runtime_error &e) {
		delete player;   // prevent memory leak
		throw e;   // pass the exception along
	}
}
