/*
 * Copyright (c) 2002 by Ravi Iyengar [ravi.i@softhome.net]
 * Released under the GNU General Public License
 * See LICENSE for details.
 */

#include "StdAfx.h"
#include "soundresource.h"
#include "SoundPlayer.h"
#include "Exception.h"
#include <mmsystem.h>


SoundPlayer::SoundPlayer(HWND listener)
: mListener(listener), mError(false), mErrorMessage("")
{
}

SoundPlayer::~SoundPlayer()
{
}

// Determine if an error occured in the play thread
bool SoundPlayer::IsError()
{
	return mError;
}

// Get the error message if there is one
std::string SoundPlayer::GetError()
{
	return mErrorMessage;
}


//-----------------------------------------------------------------------------


RealtimePlayer::RealtimePlayer(HWND listener)
: SoundPlayer(listener), mInitialized(false), mPlaying(false),
  mThreadHandle(0), mThreadId(0), mSound(0), mFilter(0)
{
}

RealtimePlayer::~RealtimePlayer()
{
	if (mInitialized) Cleanup();
}

// Initialize the device
void RealtimePlayer::Initialize(DataStream *patch)
{
	if (mInitialized) throw DeviceException("Player already initialized");
	OpenDevice(patch);
	mInitialized = true;
}

// Shutdown the device
void RealtimePlayer::Cleanup()
{
	if (!mInitialized) throw DeviceException("Player is not initialized");
	if (mPlaying) Stop();
	if (mThreadHandle) CleanupThread();
	CloseDevice();
	mInitialized = false;
}

// Begin playing a sound resource
void RealtimePlayer::Start(SoundResource *sound, int filter)
{
	if (!mInitialized) throw DeviceException("Player is not initialized");
	if (mPlaying) throw DeviceException("Player is already in use");
	if (mThreadHandle) CleanupThread();

	mPleaseStop = false;
	mSound = sound;
	mFilter = filter;
	mThreadHandle = ::CreateThread(0, 0, StaticThreadProc, this, 0, &mThreadId);
	if (mThreadHandle == 0) throw DeviceException("Could not create play thread");
}

// Stop playing the loaded sound resource
void RealtimePlayer::Stop()
{
	if (!mInitialized) throw DeviceException("Player is not initialized");
	if (!mPlaying) throw DeviceException("Player is not playing");
	mPleaseStop = true;
	WaitForSingleObject(mThreadHandle, INFINITE);
	CleanupThread();
}

// Close the thread handle so it can be freed by Windows
void RealtimePlayer::CleanupThread()
{
	CloseHandle(mThreadHandle);
	mThreadHandle = 0;
	mThreadId = 0;
}

// Thread function, just delegates to an instance method
DWORD WINAPI RealtimePlayer::StaticThreadProc(LPVOID lpvSelf)
{
	RealtimePlayer *self = (RealtimePlayer*)lpvSelf;
	self->ThreadProc();
	return 0;
}

// Play thread, iterates through sound playing music
void RealtimePlayer::ThreadProc()
{
	mPlaying = true;
	::PostMessage(mListener, SBM_PLAYSTARTED, 0, 0);
	::PostMessage(mListener, SBM_SOUNDSIGNAL, 0, 0);
	::PostMessage(mListener, SBM_SOUNDTICK, 0, 0);

	int i;
	bool channelEnabled[16];

	// load the sound header
	for (i = 0; i < 16; ++i) {
		if (mSound->GetChannelPlayFlag(i) & mFilter) {
			channelEnabled[i] = true;
			MapChannelVoices(i, mSound->GetChannelVoices(i));
		} else {
			channelEnabled[i] = false;
		}
	}

	int cumulativeCue = 127;
	bool keepGoing = true;
	int soundTime = 0;
	int loopPoint = -1;
	int loopTime;
	int numEvents = mSound->GetNumEvents();   // just so we don't repeat this call over and over
	for (i = 0; keepGoing && (i < numEvents); ++i) {
		if (mPleaseStop) {
			keepGoing = false;
			continue;
		}

		sound_event event = mSound->GetEvent(i);

		// wait for deltaTime ticks
		int deltaTime = event.absoluteTime - soundTime;
		if (deltaTime > 0) {
			int milis = deltaTime * 100 / 6;
			HANDLE timerEvent = CreateEvent(0, false, false, "tick timer");
			int timerId = timeSetEvent(milis, 0, (LPTIMECALLBACK)timerEvent, 0, TIME_ONESHOT | TIME_CALLBACK_EVENT_SET);

			bool keepWaiting = true;
			int tickShift = 0;
			while (keepWaiting) {
				// for long delta times, we want to wake ourselves up periodically to
				//   send a tick message to the listener and make sure we aren't supposed
				//   to be stopping
				DWORD rv = WaitForSingleObject(timerEvent, 17);
				if (mPleaseStop || (rv == WAIT_OBJECT_0)) {
					keepWaiting = false;
				} else if (rv == WAIT_TIMEOUT) {
					++tickShift;
					::PostMessage(mListener, SBM_SOUNDTICK, soundTime + tickShift, 0);
				}
			}
			
			timeKillEvent(timerId);
			CloseHandle(timerEvent);
			soundTime = event.absoluteTime;
			if (!mPleaseStop) ::PostMessage(mListener, SBM_SOUNDTICK, soundTime, 0);
		}

		// stop if we're supposed to
		if (mPleaseStop) continue;

		switch (event.status) {
			case sound_event::CONTROL_CHANGE:
				if (event.param1 == 0x4B) {
					// voice conrol
					MapChannelVoices(event.channel, event.param2);
				} else if (event.param1 == 0x60) {
					// cumulative cue
					cumulativeCue += event.param2;
					::PostMessage(mListener, SBM_SOUNDSIGNAL, cumulativeCue, 0);
				} else {
					// normal control change, let the subclass handle it
					if (channelEnabled[event.channel]) HandleEvent(event);
				}
				break;

			case sound_event::PATCH_CHANGE:
				if (event.channel == 15) {
					if (event.param1 == 127) {
						// loop point set
						loopPoint = i;
						loopTime = soundTime;
					} else {
						// direct cue
						::PostMessage(mListener, SBM_SOUNDSIGNAL, event.param1, 0);
					}
				} else {
					// normal patch change, let the subclass handle it
					if (channelEnabled[event.channel]) HandleEvent(event);
				}
				break;

			case sound_event::STOP:
				if (loopPoint == -1) {
					// no loop point, so stop playing
					keepGoing = false;
				} else {
					// restart from the loop point
					soundTime = loopTime;
					i = loopPoint;
				}
				break;

			default:
				if (channelEnabled[event.channel]) HandleEvent(event);
		}
	}

	::PostMessage(mListener, SBM_PLAYSTOPPED, 0, 0);
	::PostMessage(mListener, SBM_SOUNDSIGNAL, -1, 0);
	mPlaying = false;
}
