/* FUNCTION list 	---	---	---	---	---	---	---

*/

/* BASE headers	---	---	---	---	---	---	--- */


/* LIBRARY headers	---	---	---	---	---	---	--- */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <assert.h>

/* OTHER headers	---	---	---	---	---	---	--- */

#include "types.h"
#include "endian.h"
#include "list.h"
#include "param.h"
#include "common.h"

/* PROTOTYPES	---	---	---	---	---	---	--- */

struct ch_struct
{
	char type;
	int wanted_id;
	LIST *nlist;
};
typedef struct ch_struct CH;
	
struct note_struct
{
	u8 type;
	int duration;
	int freq_count;
	u8 atten;
};
typedef struct note_struct NOTE;


/* VARIABLES	---	---	---	---	---	---	--- */


/* CODE	---	---	---	---	---	---	---	--- */


//~ 0 = not set
//~ n = abcdef note
//~ a = agi freq_count
//~ f = actual frequency
//~ w = agi white noise
//~ p = agi periodic noise

//~ the default type when printing out notes
#define TYPE_DEFAULT 'a'

#define LINE_SIZE 0x400


u8 *in_name = 0;
u8 *out_name = 0;
int f_verbose = 0;

PARAM scr_param[] =
{
	{'i', "input", 0, -1, P_DATA},	// input scr file
	{'o', "output", 0, -1, P_DATA},	// output snd file
	{'h', "help", 0, -1, P_FLAG},	// need help
	{'v', "verbose", 0, -1, P_FLAG},	// want more crap on the screen?
	{0, 0, 0, -1, P_DATA},	// default file..  input?
};


void param_check()
{
	// if help is set.. print help.. exit
	if ((scr_param[2].flag == 1) ||
		(scr_param[0].data == 0) ||
		(scr_param[1].data == 0) )
	{
		if (scr_param[2].flag != 1)
		{
			if (scr_param[0].data == 0) 
				printf("%s(): input file not defined\n\n", 
					__PRETTY_FUNCTION__);
			if (scr_param[1].data == 0) 
				printf("%s(): output file not defined\n\n", 
					__PRETTY_FUNCTION__);
		}
		
		printf("scr2agi %s by Nick Sonneveld\n", TOOL_VER);
		printf("convert agi sound scripts (.ass) to agi sound objects (.aso)\n");
		printf("(no.. i don't think the names are funny.)\n");
		printf("\n");
		printf("-i, -input = input script name\n");
		printf("-o, -output = output object name\n");
		printf("-h, -help = print the help message\n");
		printf("-v, -verbose = print interesting messages that correspond with the current activity.\n");
		exit(0);
	}
	
	// check other flags.. if not set..  set to 1 or 0
	f_verbose = (scr_param[3].flag == 1);
	
	// check input
	// help prints out help if input and output aren't set.. but print error messages if 
	// help doesn't do this in another version
	
	in_name = scr_param[0].data;
	out_name = scr_param[1].data;
	
	// check default
	if (scr_param[4].data != 0)
	{
		printf("what do you want me to do with this: \"%s\"?\n", scr_param[4].data);
	}
}


LIST *script_read(FILE *scr)
{
	int freq_count;
	int duration;
	u8 atten;
	
	char type = 0;
	int i;
	int line_number;
	
	LIST *ch_list = 0;
	NOTE *n_cur = 0;
	CH *ch_cur = 0;
	
	u8 line[LINE_SIZE];
	u8 *token, *running;
	
	assert(scr != NULL);
	
	line_number = 0;
	
	while (fgets(line, LINE_SIZE, scr) != NULL)
	{
		line_number++;
		// check for comment
		// pfft  comments.. who uses comments
		// the source should be so obvious that you don't need to write comments
		// only silly people would bother writing comments.
		if (line[0] == '#')
			continue;

		// check if it's a next channel
		if (!memicmp(line, "tone", 4) ||
			!memicmp(line, "noise", 5) )
		{
			// open new channel
			if (ch_list == 0)
				ch_list = list_new(sizeof(CH));
			
			ch_cur = list_add(ch_list);
			
			if (!memicmp(line, "tone", 4))
				ch_cur->type = 't';
			else
				ch_cur->type = 'n';
			ch_cur->wanted_id = -1;
			ch_cur->nlist = 0;
			
			type = 0;
			continue;	// we've done all we can from this line
		}

		// READING IN A NOTE LINE
		
		// if channel is open
		if (ch_cur != 0)
		{
			// check for number of items
			i=0;
			token = line;
			
			while ((token=strchr(token, ',')) != NULL)
			{
				i++;
				token++;
			}
			
			token = strtok_r(line, ",", &running);
			switch (i)
			{
				case 3:
					// READING IN THE TYPE
					// read the first character that isn't a ' '
					while ( (*token != 0) && (*token == ' ') )
						token++;
					type = *token;

					// make sure it's a valid type
					if (strchr("afwp", type) == 0)
					{
						printf("%s(): invalid note type '%c' @ %d\n", 
							__PRETTY_FUNCTION__, type, line_number);
						exit(-1);
					}
					token = strtok_r(0, ",", &running);

					// no break.. once we've read the type.. 
					//we continue onto reading the other parameters
				case 2:
					assert(token != NULL);
					// token should be pointing at the freq/note
					if (type == 0)
					{
						printf("warning.. using default type '%c' @ %d", TYPE_DEFAULT, line_number);
						type = TYPE_DEFAULT;
					}

					// read frequency
					switch (type)
					{
						case 'a':
							if (ch_cur->type != 't')
							{
								printf("%s(): invalid type used for that channel type @ %d\n", 
									__PRETTY_FUNCTION__, line_number);
								exit(-1);
							}
							freq_count = (int)(atof(token) + 0.5);
							break;
						case 'f':
							if (ch_cur->type != 't')
							{
								printf("%s(): invalid type used for that channel type @ %d\n", 
									__PRETTY_FUNCTION__, line_number);
								exit(-1);
							}
							freq_count = (int)(111860.0 / atof(token) + 0.5);
							break;
						case 'w':
						case 'p':
							if (ch_cur->type != 'n')
							{
								printf("%s(): invalid type used for that channel type @ %d\n", 
									__PRETTY_FUNCTION__, line_number);
								exit(-1);
							}
							freq_count = (int)(atof(token) + 0.5);
							break;
						default:
							printf("%s(): fatal type error @ %d\n", 
								__PRETTY_FUNCTION__, line_number);
							exit(-1);
					}
					token = strtok_r(0, ",", &running);
					
					// read atten
					atten = (u8)(atof(token) + 0.5);
					token = strtok_r(0, ",", &running);
					
					// read duration
					duration = (int)(atof(token) + 0.5);
					
					// add new note to list
					if (ch_cur->nlist == 0)
						ch_cur->nlist = list_new(sizeof(NOTE));
					n_cur = list_add(ch_cur->nlist);
					
					// fill in details
					if ((type == 'w') || (type == 'p') )
						n_cur->type = type;
					else
						n_cur->type = 'a';
					n_cur->duration = duration;
					n_cur->freq_count = freq_count;
					n_cur->atten = atten;
					
					#warning todo: warnings for notes that dont do anything.. duration 0
					break;
					
				default:
					// wrong number of items
					printf("%s(): ignoring line @ %d\n", 
						__PRETTY_FUNCTION__, line_number);
					break;
			}
		}
	}
	
	return ch_list;
}


u8 handles_used = 0x1F;	// 3 bits.. 3 chans

int handle_new(int wanted)
{
	int handle = 1;
	u8 mask = 0x80;
	
	if (wanted)
	{
		mask >>= wanted - 1;
		if (handles_used&mask)
			mask = 0;
	}
	else
	{
		while ( (mask) && (handles_used&mask) )
		{
			mask >>= 1;
			handle++;
		}
	}
	
	// if mask == 0 then handles_uses is untouched.
	handles_used |= mask;
	
	return (mask) ? handle : 0;
}

// note: we should make noise channels that are defined as 4 have higher preference that just
// any noise channel.. but I dunno.. can't be bothered.

u8 reg_add[] = 
{
	0x80, 0x90, 0xA0, 0xB0, 0xC0, 0xD0, 0xE0, 0xF0
};


// to do: copy the 4 channels out of the list and put them in an array
void agi_snd_write(FILE *out, LIST *ch_list)
{
	u8 noise_avail = 1;
	CH *ch;
	NOTE *nh;

	u8 header[8];
	u8 item[5];
	int offset;
	int i;
	
	handles_used = 0x1F;

	// find id's already used
	// make sure only one of each id exists
	// assign only one channel for noise
	ch = list_element_head(ch_list);
	while (ch != NULL)
	{
		if (ch->type == 'n') // noise
			ch->wanted_id = noise_avail ? 4 : -1;
		else if (ch->wanted_id > 0)
		{
			ch->wanted_id = handle_new(ch->wanted_id);
			if (ch->wanted_id <= 0)	// if it's already taken.. give it a undefined.
				ch->wanted_id = -1;
		}
		ch = node_next(ch);
	}
	
	// assign numbers for the channels that aren't defined
	// noise should have been defined before if one actually exists.
	ch = list_element_head(ch_list);
	while (ch != NULL)
	{
		if ((ch->wanted_id <= 0) && (ch->type != 'n'))
		{
			ch->wanted_id = handle_new(0);
			if (ch->wanted_id <= 0)	// no room.. sorry
				ch->wanted_id = -1;	// we have to keep on going to set the rest of the
													// chans to -1
		}
		ch = node_next(ch);
	}
	
	// write header
	offset = 0;
	offset += sizeof(header);

	store_le_16(header, offset);

	i = 1;
	while (i < 4)
	{
		ch = list_element_head(ch_list);
		while ((ch != NULL) && (ch->wanted_id != i) )
			ch = node_next(ch);
		
		if (ch)
			offset += 5*list_length(ch->nlist);  // 5 for each item, 
		
		offset += 2; //2 for the 0xffff
		i ++;
		
		store_le_16(header + i*2 - 2, offset);
	}
	
	fwrite(header, sizeof(header), 1, out);
	
	// write channels
	for (i=0; i<4; i++)
	{
		ch = list_element_head(ch_list);
		while ((ch != NULL) && (ch->wanted_id != (i+1)) )
			ch = node_next(ch);
		
		if (ch)
		{
			nh = list_element_head(ch->nlist);
			while (nh != NULL)
			{
				// 1-2 duration
				store_le_16(item, nh->duration);
				// 3-4 freq
				if (i!=3)
				{
					item[2] = (nh->freq_count >> 4) & 0x3F;
					item[3] = reg_add[i*2] |(nh->freq_count & 0x0F);
				}
				else
				{
					item[2] = 0;
					item[3] = reg_add[i*2] |
								((nh->type == 'w')?0x04:0) |
								(nh->freq_count&0x3);
				}
				// 5 atten
				item[4] = reg_add[i*2 + 1] | (nh->atten & 0xF);
				fwrite(item, 5, 1, out);
				nh = node_next(nh);
			}
		}
		
		// end of channel.
		store_le_16(header, 0xFFFF);
		fwrite(header, sizeof(u16), 1, out);
	}
	
	
}


int main (int argc, char **argv)
{
	LIST *ch_list = 0;
	
	FILE *scr;
	FILE *out;

	param_parse(argc, argv, 
			sizeof(scr_param)/sizeof(PARAM), scr_param);
	param_check(sizeof(scr_param)/sizeof(PARAM), scr_param);
	
	if (f_verbose)
		printf("%s(): opening input file \"%s\"\n", 
			__PRETTY_FUNCTION__, in_name);
	
	// open input
	if ((scr=fopen(in_name, "r")) == NULL)
	{
		printf("%s(): unable to open input\n", __PRETTY_FUNCTION__);
		exit(-1);
	}
	
	if (f_verbose)
		printf("%s(): parsing input file \"%s\"\n", 
			__PRETTY_FUNCTION__, in_name);
	
	ch_list = script_read(scr);
	
	if (fclose(scr) != 0)
		printf("%s(): warning.  could not close input script\n",
		__PRETTY_FUNCTION__);
	scr=0;
	
	if (f_verbose)
		printf("%s(): opening output file \"%s\"\n", 
			__PRETTY_FUNCTION__, out_name);
	
	// open output
	if ((out=fopen(out_name, "wb")) == NULL)
	{
		printf("%s(): unable to open output\n", __PRETTY_FUNCTION__);
		exit(-1);
	}
	
	if (f_verbose)
		printf("%s(): writing to output file \"%s\"\n", 
			__PRETTY_FUNCTION__, out_name);
	
	agi_snd_write(out, ch_list);
	
	// close ouput
	if (fclose(out) != 0)
		printf("%s(): warning.  could not close output\n", __PRETTY_FUNCTION__);
	out = 0;
	
	return 0;
}