{
    LogDecode.pas - Logic decoder/decompiler
    Copyright (C) 1997-1999 Peter Kelly <peter@area51.org.au>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
}

unit LogDecode;

interface

uses Classes, SysUtils, Dialogs, Defines;

function DecodeLogic(LogicResource:TResource;OutputText:TStringList) : boolean;

implementation

uses AGICommands, LogEdit1, WordsEditMain1, objed;

{******************************************************************************}
function DecodeLogic(LogicResource:TResource;OutputText:TStringList) : boolean;
{******************************************************************************}
const MaxBlockDepth = 12;
      MaxLabels = 255;
      ShowArgTypes = True;
//      ShowAllMessages = True;
//      ShowAllElsesAsGotos = False;
//      ShowSpecialSyntax = True;  // shows v30 >= 4 type syntax for some commands
      SpecialSyntaxType = 1;   // 0 for v30 = v30 + 4;, 1 for v30 += 4;
      ShowNonExistingValues = True;  // Uses the number of an object, word or message instead of the text if it does not exist
                                     // If false, an error message is generated when a "non-existing" value is encountered
      MaxLineLength = 80;


var ResPos : word;
    CurByte : byte;
    ThisLine : string;
    IndentPos : byte;
    ThisCommand : byte;
    DoGoto : boolean;
    ErrorOccured : boolean;
    CurChar : word;

    ShowAllMessages : boolean;
    ShowAllElsesAsGotos : boolean;
    ShowSpecialSyntax : boolean;  // shows v30 >= 4 type syntax for some commands

    MessageSectionStart : word;
    Messages : TStringList;
    MessageUsed : array[0..255] of boolean;
    MessageExists : array[0..255] of boolean;

    BlockDepth : byte;
    BlockEnd : array[0..MaxBlockDepth] of word;
    BlockIsIf : array[0..MaxBlockDepth] of boolean;
    BlockLength : array[0..MaxBlockDepth] of word;
    TempBlockLength : word;
    CurBlock : word;

    CurArg : byte;
    ArgsStart : word;

    LabelIndex : TResource;
    LabelLoc : word;
    NumLabels : word;


    Words : TWordList;
    CurWordGroup : word;

    OBJECTFile : TOBJECTFile;
    OBJECTFileEncrypted : boolean;
    InvObjects : TStringList;
    NumInvObjects : word;
    CurInvObject : word;

  {************************}
  function ReadByte : byte;
  {************************}
  begin
    if ResPos < LogicResource.Size then
    begin
      ReadByte := LogicResource.Data^[ResPos];
      ResPos := ResPos + 1;
    end
    else
      ReadByte := 0;
  end;

  {************************}
  function ReadLSMSWord : word;
  {************************}
  begin
    ReadLSMSWord := ReadByte + ReadByte*256;
  end;

  {************************}
  procedure ShowError(Msg:string);
  {************************}
  begin
    ShowMessage('Error decoding logic (position 0x'+IntToHex(ResPos,4)+'): '+Msg);
//    OutputText.Add(ThisLine);
//    OutputText.Add('******ERROR: '+Msg+'*************');
    ErrorOccured := True;
  end;

  {*************************************************************}
  function GetWordGroupIndex(Groupnum:word) : integer;
  {*************************************************************}
  var CurIndex : word;
  begin
    GetWordGroupIndex := -1;
    if Words.NumGroups > 0 then
    begin
      for CurIndex := 0 to Words.NumGroups - 1 do
        if Words.WordGroup[CurIndex].GroupNum = GroupNum then GetWordGroupIndex := CurIndex;
    end;
  end;

  {************************}
  procedure ReadMessages;
  {************************}
  var MessageSectionEnd : word;
      MessageStart : array[1..255] of word;
      CurMessage : byte;
      EncryptionStart : word;
      ThisMessage : string;
      FinishedReadingMessage : boolean;
      CurByte : byte;
      NumMessages : byte;

      function ReadEncByte : byte;
      begin
        ReadEncByte := ReadByte XOR ord(EncryptionKey[(ResPos-EncryptionStart+10) mod 11 + 1]);
      end;

  // NOTE: There is no message 0 (this is not supported by the file format).
  begin
    Messages := TStringList.Create;
    for CurMessage := 0 to 255 do
    begin
      Messages.Add('');
      MessageExists[CurMessage] := False;
      MessageUsed[CurMessage] := False;
    end;
    ResPos := MessageSectionStart;
    NumMessages := ReadByte;
    if NumMessages > 0 then
    begin
      MessageSectionEnd := ReadLSMSWord + MessageSectionStart;
      for CurMessage := 1 to NumMessages do
        MessageStart[CurMessage] := ReadLSMSWord;
      EncryptionStart := ResPos;
      for CurMessage := 1 to NumMessages do
      begin
        if MessageStart[CurMessage] > 0 then
        begin
          ThisMessage := '';
          ResPos := MessageSectionStart + MessageStart[CurMessage] + 1;
          FinishedReadingMessage := False;
          repeat
          begin
            CurByte := ReadEncByte;
            if (CurByte = 0) or (ResPos >= LogicResource.Size) then
              FinishedReadingMessage := True
            else
            begin
              if CurByte = $0A then ThisMessage := ThisMessage + '\n'
              else if CurByte = $22 then ThisMessage := ThisMessage + '\"'
              else if CurByte = $5C then ThisMessage := ThisMessage + '\\'
              else ThisMessage := ThisMessage + chr(CurByte);
            end;
          end; until FinishedReadingMessage;
          Messages[CurMessage] := ThisMessage;
          MessageExists[CurMessage] := True;
        end;
      end;
    end;
  end;

  {************************}
  procedure DisplayMessages;
  {************************}
  var CurMessage : byte;
  begin
    if ShowAllMessages then
    begin
//    OutputText.Add('// Messages (all messages are here)');
    OutputText.Add('// Messages');
    for CurMessage := 1 to 255 do
      if MessageExists[CurMessage] then
      OutputText.Add('#message '+IntToStr(CurMessage)+' "'+Messages[CurMessage]+'"');
    end
    else
    begin
//      OutputText.Add('// Messages (only those not used elsewhere in the logic are here)');
      OutputText.Add('// Messages');
      for CurMessage := 1 to 255 do
        if MessageExists[CurMessage] and (not MessageUsed[CurMessage]) then
        OutputText.Add('#message '+IntToStr(CurMessage)+' "'+Messages[CurMessage]+'"');
    end;
  end;


  {************************}
  procedure FindLabels;
  {************************}
  var I : integer;
      CurBlock : integer;

    procedure FindLabels_ReadIfs;
    var
        CurArg : byte;
        IfFinished : boolean;
        NumSaidArgs : byte;
    begin
      IfFinished := False;
      repeat
      begin
        CurByte := ReadByte;
        if CurByte = $FC then CurByte := ReadByte;
        if CurByte = $FC then CurByte := ReadByte;  // we may have 2 $FCs in a row, e.g. (a || b) && (c || d)
        if CurByte = $FD then CurByte := ReadByte;
        if (CurByte > 0) and (CurByte <= NumTestCommands) then
        begin
          ThisCommand := CurByte;
          if ThisCommand = 14 then // said command
          begin
            NumSaidArgs := ReadByte;
            ResPos := ResPos + NumSaidArgs*2;
          end
          else ResPos := ResPos + TestCommand[ThisCommand].NumArgs;
        end //if (CurByte > 0) and (CurByte <= NumTestCommands)
        else if CurByte = $FF then
        begin
          if BlockDepth >= MaxBlockDepth - 1 then ShowError('Too many nested blocks ('+IntToStr(BlockDepth)+')')
          else
          begin
            BlockDepth := BlockDepth + 1;
            BlockIsIf[BlockDepth] := True;
            BlockLength[BlockDepth] := ReadLSMSWord;
            if BlockLength[BlockDepth] = 0 then ShowError('Encountered command block of length 0.');
            BlockEnd[BlockDepth] := BlockLength[BlockDepth] + ResPos;
            if BlockEnd[BlockDepth] > BlockEnd[BlockDepth-1] then ShowError('Block to long ('+IntToStr(BlockEnd[BlockDepth]-BlockEnd[BlockDepth-1])+' bytes longer than rest of previous block)');
          end;
          IfFinished := True;
        end  // if CurByte = $FF
        else ShowError('Unknown test command ('+IntToStr(CurByte)+')');
      end; until IfFinished or ErrorOccured;
    end;

  begin
    LabelIndex.Size := LogicResource.Size;
    GetMem(LabelIndex.Data,LabelIndex.Size);
    FillChar(LabelIndex.Data^,LabelIndex.Size,0);
    BlockDepth := 0;
    NumLabels := 0;
    repeat
    begin
      for CurBlock := BlockDepth downto 1 do
        if BlockEnd[CurBlock] <= ResPos then BlockDepth := BlockDepth - 1;
      CurByte := ReadByte;
      if CurByte = $FF then FindLabels_ReadIfs
      else
      if CurByte <= NumAGICommands then
        ResPos := ResPos + AGICommand[CurByte].NumArgs
      else if CurByte = $FE then
      begin
        DoGoto := False;
        TempBlockLength := ReadLSMSWord;
         if (BlockEnd[BlockDepth] = ResPos) and (BlockIsIf[BlockDepth]) and (BlockDepth > 0)
            and (not ShowAllElsesAsGotos) then
        // else
         begin
          BlockIsIf[BlockDepth] := False;
          if (TempBlockLength + ResPos > BlockEnd[BlockDepth-1]) or (TempBlockLength > $8000)
            or (BlockLength[BlockDepth] <= 3) then
            DoGoto := True
          else
          begin
            BlockLength[BlockDepth] := TempBlockLength;
            BlockEnd[BlockDepth] := BlockLength[BlockDepth] + ResPos;
          end;
        end else DoGoto := True;
        // goto
        if DoGoto then
        begin
          LabelLoc := TempBlockLength + ResPos;
          if LabelLoc > LabelIndex.Size - 1 then ShowError('Label past end of logic (0x'+IntToHex(LabelLoc,4)+')')
          else
          begin
            if LabelIndex.Data^[LabelLoc] = 0 then
            begin
              NumLabels := NumLabels + 1;
              LabelIndex.Data^[LabelLoc] := NumLabels;
            end;
          end;
        end;
      end
      else
      begin
        ShowError('Unknown action command ('+IntToStr(CurByte)+')');
      end;
    end; until (ResPos >= MessageSectionStart) or ErrorOccured;
  end;

  {************************}
  function AddArg(CommandIsIf:boolean;Arg,ArgType:byte) : string;
  {************************}
  var ThisMessage : string;
      NumCharsToDisplay : word;
  begin
    if ShowArgTypes then
    begin
      case ArgType of
        atMsg: begin
                 if MessageExists[Arg] then
                 begin
                   ThisMessage := Messages[Arg];
                   repeat
                   begin
                     if Length(ThisLine) + Length(ThisMessage) > MaxLineLength then
                     begin
                       NumCharsToDisplay := MaxLineLength-Length(ThisLine);
                       repeat Dec(NumCharsToDisplay) until (NumCharsToDisplay = 0) or (ThisMessage[NumCharsToDisplay]=' ');
                       if NumCharsToDisplay <= 1 then NumCharsToDisplay := MaxLineLength-Length(ThisLine);
                       ThisLine := ThisLine + '"' + Copy(ThisMessage,1,NumCharsToDisplay) + '"';
                       ThisMessage := Copy(ThisMessage,NumCharsToDisplay+1,9999);
                       OutputText.Add(ThisLine);
                       if ArgsStart >= MaxLineLength - 20 then ArgsStart := MaxLineLength - 20;
                       ThisLine := MultStr(' ',ArgsStart);
                     end
                     else
                     begin
                       ThisLine := ThisLine + '"' + ThisMessage + '"';
                       ThisMessage := '';
                     end;
                   end; until ThisMessage = '';
                 end
                 else if ShowNonExistingValues then
                   ThisLine := ThisLine + ArgTypePrefix[atMsg] + IntToStr(Arg)
                 else ShowError('Unknown message ('+IntToStr(Arg)+')');
                 MessageUsed[Arg] := True;
               end;
        atIObj: begin
                  if Arg <= NumInvObjects - 1 then
                    ThisLine := ThisLine + '"' + InvObjects[Arg] + '"'
                  else if ShowNonExistingValues then
                    ThisLine := ThisLine + ArgTypePrefix[atIObj] + IntToStr(Arg)
                  else ShowError('Unknown inventory item ('+IntToStr(Arg)+')');
                end;
        else ThisLine := ThisLine + ArgTypePrefix[ArgType] + IntToStr(Arg);
      end;
    end
    else ThisLine := ThisLine + IntToStr(Arg);
  end;

  {************************}
  procedure ReadIfs;
  {************************}
  var OROn : boolean;
      NOTOn : boolean;
      FirstCommand : boolean;
      CurArg : byte;
      IfFinished : boolean;
      NumSaidArgs : byte;
      ThisWordGroupIndex : integer;
      ThisWordGroupNum : word;



      procedure AddSpecialIFSyntaxCommand;
      begin
        case ThisCommand of
        1: begin  // equaln
             ThisLine := ThisLine + 'v' + IntToStr(ReadByte);
             if NOTOn then ThisLine := ThisLine + ' != '
             else ThisLine := ThisLine + ' == ';
             ThisLine := ThisLIne + IntToStr(ReadByte);
           end;
        2: begin  // equalv
             ThisLine := ThisLine + 'v' + IntToStr(ReadByte);
             if NOTOn then ThisLine := ThisLine + ' != v'
             else ThisLine := ThisLine + ' == v';
             ThisLine := ThisLine + IntToStr(ReadByte);
           end;
        3: begin  // lessn
             ThisLine := ThisLine + 'v' + IntToStr(ReadByte);
             if NOTOn then ThisLine := ThisLine + ' >= '
             else ThisLine := ThisLine + ' < ';
             ThisLine := ThisLIne + IntToStr(ReadByte);
           end;
        4: begin  // lessv
             ThisLine := ThisLine + 'v' + IntToStr(ReadByte);
             if NOTOn then ThisLine := ThisLine + ' >= v'
             else ThisLine := ThisLine + ' < v';
             ThisLine := ThisLIne + IntToStr(ReadByte);
           end;
        5: begin  // greatern
             ThisLine := ThisLine + 'v' + IntToStr(ReadByte);
             if NOTOn then ThisLine := ThisLine + ' <= '
             else ThisLine := ThisLine + ' > ';
             ThisLine := ThisLIne + IntToStr(ReadByte);
           end;
        6: begin  // greaterv
             ThisLine := ThisLine + 'v' + IntToStr(ReadByte);
             if NOTOn then ThisLine := ThisLine + ' <= v'
             else ThisLine := ThisLine + ' > v';
             ThisLine := ThisLIne + IntToStr(ReadByte);
           end;
        end;
      end;

  begin
    IfFinished := False;
    FirstCommand := True;
    OROn := False;
    ThisLine := MultStr('  ',BlockDepth)+'if (';
    repeat
    begin
      NOTOn := False;
      CurByte := ReadByte;
      if CurByte = $FC then
      begin
        OROn := not OROn;
        if OROn then
        begin
          if not FirstCommand then
          begin
            ThisLine := ThisLine + ' &&';
            OutputText.Add(ThisLine);
            ThisLine := MultStr('  ',BlockDepth)+'    ';
            FirstCommand := True;
          end;
          ThisLine := ThisLine + '(';
        end
        else ThisLine := ThisLine + ')';
        CurByte := ReadByte;
      end;
      if (CurByte = $FC) and (not OROn) then  // we may have 2 $FCs in a row, e.g. (a || b) && (c || d)
      begin
        ThisLine := ThisLine + ' &&';
        OutputText.Add(ThisLine);
        ThisLine := MultStr('  ',BlockDepth)+'    ';
        FirstCommand := True;
        ThisLine := ThisLine + '(';
        OROn := True;
        CurByte := ReadByte;
      end;
      if CurByte = $FD then   // NOT
      begin
        NOTOn := True;
        CurByte := ReadByte;
      end;  //if CurByte = $FD
      if (CurByte > 0) and (CurByte <= NumTestCommands) then
      begin
        if not FirstCommand then
        begin
          if OROn then ThisLine := ThisLine + ' ||'
          else ThisLine := ThisLine + ' &&';
          OutputText.Add(ThisLine);
          ThisLine := MultStr('  ',BlockDepth)+'    ';
        end;
        ThisCommand := CurByte;
        if ShowSpecialSyntax and (ThisCommand in [1..6]) then AddSpecialIFSyntaxCommand
        else
        begin
          if NOTOn then ThisLine := ThisLine + '!';
          ThisLine := ThisLine + TestCommand[ThisCommand].Name + '(';
          ArgsStart := Length(ThisLine);
          if ThisCommand = 14 then // said command
          begin
            NumSaidArgs := ReadByte;
            for CurArg := 1 to NumSaidArgs do
            begin
              ThisWordGroupNum := ReadLSMSWord;
              ThisWordGroupIndex := GetWordGroupIndex(ThisWordGroupNum);
              if ThisWordGroupIndex < 0 then
              begin
                if ShowNonExistingValues then ThisLine := ThisLine + IntToStr(ThisWordGroupNum)
                else ShowError('Unknown word group ('+IntToStr(ThisWordGroupNum)+').')
              end
              else ThisLine := ThisLine + '"' + Words.WordGroup[ThisWordGroupIndex].Words[0] + '"';
              if CurArg < NumSaidArgs then
                ThisLine := ThisLine + ',';
            end;
          end
          else
          begin
            for CurArg := 1 to TestCommand[ThisCommand].NumArgs do
            begin
              CurByte := ReadByte;
              AddArg(True,CurByte,TestCommand[ThisCommand].argTypes[CurArg]);
              if CurArg < TestCommand[ThisCommand].Numargs then
                ThisLine := ThisLine + ',';
            end; // for CurArg := 1 to TestCommand[ThisCommand].NumArgs
          end; // if ThisCommand <> 14
          ThisLine := ThisLine + ')';
        end;
        FirstCommand := False;
      end //if (CurByte > 0) and (CurByte <= NumTestCommands)
      else if CurByte = $FF then
      begin
        ThisLine := ThisLine + ') {';
        if BlockDepth >= MaxBlockDepth - 1 then ShowError('Too many nested blocks ('+IntToStr(BlockDepth)+')')
        else
        begin
          BlockDepth := BlockDepth + 1;
          BlockIsIf[BlockDepth] := True;
          BlockLength[BlockDepth] := ReadLSMSWord;
          BlockEnd[BlockDepth] := BlockLength[BlockDepth] + ResPos;
//          ThisLine := ThisLine + '   // block length = '+IntToStr(BlockEnd[BlockDepth]-ResPos);
          if BlockEnd[BlockDepth] > BlockEnd[BlockDepth-1] then ShowError('Block to long ('+IntToStr(BlockEnd[BlockDepth]-BlockEnd[BlockDepth-1])+' bytes longer than rest of previous block)');
        end;
        OutputText.Add(ThisLine);
        ThisLine := MultStr('  ',BlockDepth);
        IfFinished := True;
      end  // if CurByte = $FF
      else ShowError('Unknown test command ('+IntToStr(CurByte)+')');
    end; until IfFinished or ErrorOccured;
  end;


  {************************}
  procedure AddSpecialSyntaxCommand;
  {************************}
  var arg1 : byte;
  begin
    arg1 := Readbyte;
    case ThisCommand of
    // increment
    $01: ThisLine := ThisLine+'v'+IntToStr(arg1)+'++';
    // decrement
    $02: ThisLine := ThisLine+'v'+IntToStr(arg1)+'--';
    // assignn
    $03: ThisLine := ThisLine+'v'+IntToStr(arg1)+' = '+IntToStr(ReadByte);
    // assignv
    $04: ThisLine := ThisLine+'v'+IntToStr(arg1)+' = v'+IntToStr(ReadByte);
    // addn
    $05: if SpecialSyntaxType = 0 then
           ThisLine := ThisLine+'v'+IntToStr(arg1)+' = v'+IntToStr(arg1)+' + '+IntToStr(ReadByte)
         else ThisLine := ThisLine+'v'+IntToStr(arg1)+' += '+IntToStr(ReadByte);
    // addv
    $06: if SpecialSyntaxType = 0 then
           ThisLine := ThisLine+'v'+IntToStr(arg1)+' = v'+IntToStr(arg1)+' + v'+IntToStr(ReadByte)
         else ThisLine := ThisLine+'v'+IntToStr(arg1)+' += v'+IntToStr(ReadByte);
    // subn
    $07: if SpecialSyntaxType = 0 then
           ThisLine := ThisLine+'v'+IntToStr(arg1)+' = v'+IntToStr(arg1)+' - '+IntToStr(ReadByte)
         else ThisLine := ThisLine+'v'+IntToStr(arg1)+' -= '+IntToStr(ReadByte);
    // subv
    $08: if SpecialSyntaxType = 0 then
           ThisLine := ThisLine+'v'+IntToStr(arg1)+' = v'+IntToStr(arg1)+' - v'+IntToStr(ReadByte)
         else ThisLine := ThisLine+'v'+IntToStr(arg1)+' -= v'+IntToStr(ReadByte);
    // lindirectv
    $09: ThisLine := ThisLine+'*v'+IntToStr(arg1)+' = v'+IntToStr(ReadByte);
    // rindirect
    $0A: ThisLine := ThisLine+'v'+IntToStr(arg1)+' = *v'+IntToStr(ReadByte);
    // lindirectn
    $0B: ThisLine := ThisLine+'*v'+IntToStr(arg1)+' = '+IntToStr(ReadByte);
    // mul.n
    $A5: if SpecialSyntaxType = 0 then
           ThisLine := ThisLine+'v'+IntToStr(arg1)+' = v'+IntToStr(arg1)+' * '+IntToStr(ReadByte)
         else ThisLine := ThisLine+'v'+IntToStr(arg1)+' *= '+IntToStr(ReadByte);
    // mul.v
    $A6: if SpecialSyntaxType = 0 then
           ThisLine := ThisLine+'v'+IntToStr(arg1)+' = v'+IntToStr(arg1)+' * v'+IntToStr(ReadByte)
         else ThisLine := ThisLine+'v'+IntToStr(arg1)+' *= v'+IntToStr(ReadByte);
    // div.n
    $A7: if SpecialSyntaxType = 0 then
           ThisLine := ThisLine+'v'+IntToStr(arg1)+' = v'+IntToStr(arg1)+' / '+IntToStr(ReadByte)
         else ThisLine := ThisLine+'v'+IntToStr(arg1)+' /= '+IntToStr(ReadByte);
    // div.v
    $A8: if SpecialSyntaxType = 0 then
           ThisLine := ThisLine+'v'+IntToStr(arg1)+' = v'+IntToStr(arg1)+' / v'+IntToStr(ReadByte)
         else ThisLine := ThisLine+'v'+IntToStr(arg1)+' /= v'+IntToStr(ReadByte);
    end;
  end;

  {************************}
  procedure AddBlockEnds;
  {************************}
  var CurBlock : word;
  begin
    for CurBlock := BlockDepth downto 1 do
    begin
      if BlockEnd[CurBlock] <= ResPos then
      begin
        OutputText.Add(MultStr('  ',CurBlock-1)+'}');
        BlockDepth := BlockDepth - 1;
      end;
    end;
  end;

begin
  DecodeLogic := False;
  ShowAllMessages := Settings.LogicEditor.ShowAllMessages;
  ShowAllElsesAsGotos := Settings.LogicEditor.ShowAllElsesAsGotos;
  ShowSpecialSyntax := Settings.LogicEditor.ShowSpecialSyntax;
  if LogicResource.Size > 0 then
  begin
    Words := ReadWORDSTOKFile(GameDir+'WORDS.TOK');
    if Words.Exists then  // ReadWORDSTOKFile displays an error message if necessary, so we don't need to here.
    begin
      OBJECTFileEncrypted := False;
      OBJECTFile := ReadOBJECTFile(GameDir+'OBJECT',OBJECTFileEncrypted);
      if OBJECTFile.Exists then
      begin
        InvObjects := TStringList.Create;
        InvObjects.Assign(OBJECTFile.ItemNames);
        OBJECTFile.ItemNames.Free;
        NumInvObjects := InvObjects.Count;
        for CurInvObject := 0 to NumInvObjects - 1 do
        begin
          InvObjects[CurInvObject] := LowerCase(InvObjects[CurInvObject]);  // words already in lower case in file so we don't need to convert them
          if Length(InvObjects[CurInvObject]) > 0 then
            for CurChar := Length(InvObjects[CurInvObject]) downto 1 do
            if InvObjects[CurInvObject][CurChar] = '"' then
              InvObjects[CurInvObject] := Copy(InvObjects[CurInvObject],1,CurChar-1) + '\"' + Copy(InvObjects[CurInvObject],CurChar+1,255);
        end;

        ResPos := 0;
        OutputText.Clear;
        MessageSectionStart := ReadLSMSWord + 2;
        if MessageSectionStart > LogicResource.Size - 1 then
        begin
          ShowMessage('Error: Message section start (0x'+IntToHex(MessageSectionStart,2)+') is beyond end of resource.');
        end
        else
        begin
          ReadMessages;
          ResPos := 2;
          ErrorOccured := False;
          BlockEnd[0] := MessageSectionStart;
          BlockIsIf[0] := False;
          FindLabels;
          BlockDepth := 0;
          ResPos := 2;
          repeat
          begin
            AddBlockEnds;
            if LabelIndex.Data^[ResPos] > 0 then
              OutputText.Add('Label'+IntToStr(LabelIndex.Data^[ResPos])+':');
            CurByte := ReadByte;
            if CurByte = $FF then ReadIfs
            else
            if CurByte <= NumAGICommands then
            begin
              ThisCommand := CurByte;
              ThisLine := MultStr('  ',BlockDepth);
              if ShowSpecialSyntax and (ThisCommand in [$01..$0B,$A5..$A8]) then AddSpecialSyntaxCommand
              else
              begin
                ThisLine := ThisLine + AGICommand[ThisCommand].Name + '(';
                ArgsStart := Length(ThisLine);
                IndentPos := Length(ThisLine);
                for CurArg := 1 to AGICommand[ThisCommand].NumArgs do
                begin
                  CurByte := ReadByte;
                  AddArg(False,CurByte,AGICommand[ThisCommand].argTypes[CurArg]);
                  if CurArg < AGICommand[ThisCommand].Numargs then
                    ThisLine := ThisLine + ',';
                end;
                ThisLine := ThisLine + ')';
              end;
              ThisLine := ThisLine + ';';
              OutputText.Add(ThisLine);
           end
           else if CurByte = $FE then
           begin
             DoGoto := False;
             TempBlockLength := ReadLSMSWord;
             if (BlockEnd[BlockDepth] = ResPos) and (BlockIsIf[BlockDepth]) and (BlockDepth > 0)
                and (not ShowAllElsesAsGotos) then
             // else
             begin
               BlockIsIf[BlockDepth] := False;
               if (TempBlockLength + ResPos > BlockEnd[BlockDepth-1]) or (TempBlockLength > $8000)
                 or (BlockLength[BlockDepth] <= 3) then
                 DoGoto := True
               else
               begin
                 OutputText.Add(MultStr('  ',BlockDepth-1)+'}');
                 OutputText.Add(MultStr('  ',BlockDepth-1)+'else {');
                 BlockLength[BlockDepth] := TempBlockLength;
                 BlockEnd[BlockDepth] := BlockLength[BlockDepth] + ResPos;
               end;
             end else DoGoto := True;
             // goto
             if DoGoto then
             begin
               LabelLoc := TempBlockLength + ResPos;
               if LabelLoc > LabelIndex.Size - 1 then ShowError('Label past end of logic (0x'+IntToHex(LabelLoc,4)+')')
               else
               begin
                 OutputText.Add(MultStr('  ',BlockDepth)+'goto(Label'+IntToStr(LabelIndex.Data^[LabelLoc])+');');
               end;
             end;
           end
           else
           begin
             ShowError('Unknown action command ('+IntToStr(CurByte)+')');
           end;
          end; until (ResPos >= MessageSectionStart) or ErrorOccured;
          if not ErrorOccured then AddBlockEnds;
          FreeMem(LabelIndex.Data,LabelIndex.Size);
          OutputText.Add('');
          DisplayMessages;
          Messages.Free;
          DecodeLogic := not ErrorOccured;
        end;  // if MessageSectionStart <= LogicResource.Size - 1
        if Words.NumGroups > 0 then
          for CurWordGroup := 0 to Words.NumGroups - 1 do
            Words.WordGroup[CurWordGroup].Words.Free;
        InvObjects.Free;
      end;  // if OBJECTFile.Exists
    end;  // if Words.Exists
    FreeMem(LogicResource.Data, LogicResource.Size);
  end;  // if LogicResource.Size > 0
end;

end.
