{
    WordsEditMain1.pas - WORDS.TOK editor
    Copyright (C) 1997-1999 Peter Kelly <peter@area51.org.au>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
}

unit WordsEditMain1;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, Menus, defines;

type
  TWordsEditMain = class(TForm)
    MainMenu1: TMainMenu;
    FileMenu: TMenuItem;
    FileNew: TMenuItem;
    FileOpen: TMenuItem;
    FileSave: TMenuItem;
    FileSaveAs: TMenuItem;
    N1: TMenuItem;
    FileClose: TMenuItem;
    Panel1: TPanel;
    WordGroupPanel: TPanel;
    Label1: TLabel;
    WordGroupList: TListBox;
    WordPanel: TPanel;
    WordGroupNumDisplay: TLabel;
    WordList: TListBox;
    Splitter1: TSplitter;
    WordsMenu: TMenuItem;
    WordsAddWG: TMenuItem;
    WordsDeleteWG: TMenuItem;
    N2: TMenuItem;
    WordsAddW: TMenuItem;
    Deleteword1: TMenuItem;
    N3: TMenuItem;
    WordsCountWG: TMenuItem;
    WordsCountW: TMenuItem;
    WordsFind: TMenuItem;
    WordEdit: TEdit;
    ButPanel: TPanel;
    AddGroupBut: TButton;
    DeleteGroupBut: TButton;
    AddWordBut: TButton;
    DeleteWordBut: TButton;
    FindBut: TButton;
    FindDialog1: TFindDialog;
    ChangeGroupNumBut: TButton;
    Changegroupnumber1: TMenuItem;
    OpenDialog1: TOpenDialog;
    SaveDialog1: TSaveDialog;
    FileMerge: TMenuItem;
    procedure FormResize(Sender: TObject);
    procedure Splitter1Moved(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure WordGroupListClick(Sender: TObject);
    procedure WordListClick(Sender: TObject);
    procedure WordEditKeyPress(Sender: TObject; var Key: Char);
    procedure WordEditExit(Sender: TObject);
    procedure FindButClick(Sender: TObject);
    procedure FindDialog1Find(Sender: TObject);
    procedure DeleteWordButClick(Sender: TObject);
    procedure WordListKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure AddWordButClick(Sender: TObject);
    procedure DeleteGroupButClick(Sender: TObject);
    procedure WordGroupListKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure AddGroupButClick(Sender: TObject);
    procedure FindDialog1Close(Sender: TObject);
    procedure WordsAddWGClick(Sender: TObject);
    procedure WordsDeleteWGClick(Sender: TObject);
    procedure WordsAddWClick(Sender: TObject);
    procedure Deleteword1Click(Sender: TObject);
    procedure WordsFindClick(Sender: TObject);
    procedure WordsCountWGClick(Sender: TObject);
    procedure WordsCountWClick(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure ChangeGroupNumButClick(Sender: TObject);
    procedure FileOpenClick(Sender: TObject);
    procedure FileNewClick(Sender: TObject);
    procedure FileSaveAsClick(Sender: TObject);
    procedure FileMergeClick(Sender: TObject);
    procedure FileCloseClick(Sender: TObject);
    procedure FileSaveClick(Sender: TObject);
    procedure Changegroupnumber1Click(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
    CurrentFilename : String;
    FileModified : boolean;
    function AskClose : boolean;
    procedure NewFile;
    function OpenFile(Filename:string) : boolean;
    function MergeFile(Filename:string) : boolean;
    function SaveFile(Filename:string) : boolean;
    function GroupExists(GroupNum:word) : boolean;
    procedure AddGroup(GroupNum:word);
    function DeleteGroup(GroupIndex:word) : boolean;
    procedure ChangeGroupNum(OldGroupIndex,NewGroupNum:word);
    procedure UpdateGroupList(GroupIndex:word);
  end;

var
  WordsEditMain: TWordsEditMain;

const MaxGroupNum = 65535;
const MaxWordGroups = 10000;

type TWordGroup = record
                    Words : TStringList;
                    GroupNum : word;
                  end;

     TWordList  = record
                    WordGroup : array[0..MaxWordGroups-1] of TWordGroup;
                    NumGroups : word;
                    Exists    : boolean;
                  end;


function ReadWORDSTOKFile(FileName:string) : TWordList;

implementation

uses WordsEditGetGroupNum1, WordsEditSortDisplay1, WordsEditMerge1;

{$R *.DFM}

const
      WordSeparator : string = ' | ';

var WordGroup : array[0..MaxWordGroups-1] of TWordGroup;
    NumGroups : word;

{*************************************************************}
function TWordsEditMain.AskClose : boolean;
{*************************************************************}
var SaveMessageResult : integer;
begin
  AskClose := True;
  if FileModified then
  begin
    SaveMessageResult := Application.MessageBox('Do you want to save changes before closing?','WORDS.TOK Editor',MB_YESNOCANCEL);
    if SaveMessageResult = IDYES then FileSaveClick(WordsEditMain)
    else if SaveMessageResult = IDCANCEL then AskClose := False;
    if (SaveMessageResult = IDYES) or (SaveMessageResult = IDNO) then FileModified := False;
  end;
end;

{*************************************************************}
function GroupName(GroupIndex:word) : string;
{*************************************************************}
var Name : string;
    CurWord : word;
begin
  Name := '';
  if NumGroups <= 0 then Name := 'Error: No word groups!'
  else
  if WordGroup[GroupIndex].Words.Count > 0 then
    for CurWord := 0 to WordGroup[GroupIndex].Words.Count - 1 do
      if CurWord < WordGroup[GroupIndex].Words.Count - 1 then
        Name := Name + WordGroup[GroupIndex].Words[CurWord] + WordSeparator
      else Name := Name + WordGroup[GroupIndex].Words[CurWord];
  GroupName := Name;
end;

{*************************************************************}
function GroupIndexOfWord(TheWord:string) : integer;
{*************************************************************}
{returns the group index of the group containing the specified word}
var FoundWord : boolean;
    CurGroupIndex : word;
begin
  GroupIndexOfWord := -1;
  if NumGroups > 0 then
  begin
    FoundWord := False;
    CurGroupIndex := 0;
    repeat
    begin
      if WordGroup[CurGroupIndex].Words.IndexOf(TheWord) >= 0 then
      begin
        GroupIndexOfWord := CurGroupIndex;
        FoundWord := True;
      end;
      CurGroupIndex := CurGroupIndex + 1;
    end; until FoundWord or (CurGroupIndex > NumGroups - 1);
  end;
end;

{*************************************************************}
function GetGroupIndex(Groupnum:word) : integer;
{*************************************************************}
var CurIndex : word;
begin
  GetGroupIndex := -1;
  if NumGroups > 0 then
  begin
   for CurIndex := 0 to NumGroups - 1 do
      if WordGroup[CurIndex].GroupNum = GroupNum then GetGroupIndex := CurIndex;
  end;
end;

{*************************************************************}
function TWordsEditMain.GroupExists(GroupNum:word) : boolean;
{*************************************************************}
var CurGroupIndex : word;
begin
  GroupExists := False;
  if NumGroups > 0 then
    for CurGroupIndex := 0 to NumGroups - 1 do
    begin
      if WordGroup[CurGroupIndex].GroupNum = GroupNum then
        GroupExists := True;
    end;
end;














{*************************************************************}
procedure ClearFile;
{*************************************************************}
var CurGroupIndex : word;
begin
  if NumGroups > 0 then
    for CurGroupIndex := 0 to NumGroups - 1 do
      WordGroup[CurGroupIndex].Words.Free;
  NumGroups := 0;
end;

{*************************************************************}
procedure TWordsEditMain.UpdateGroupList(GroupIndex:word);
{*************************************************************}
begin
  if GroupIndex <= NumGroups - 1 then
  begin
    WordGroupList.Items[GroupIndex] := IntToStr(WordGroup[GroupIndex].GroupNum) + '. ' + GroupName(GroupIndex);
  end;
end;

{*************************************************************}
procedure TWordsEditMain.AddGroup(GroupNum:word);
{*************************************************************}
var CurGroupIndex : word;
    InsertPosition : word;
    InsertPositionFound : boolean;

begin
  if not GroupExists(GroupNum) then {WordsEditGetGroupNum1 already checks this but we check again just to be safe, in case the other procedure is changed}
  begin
    if NumGroups >= MaxWordGroups then
      ShowMessage('Error: Could not add group - maximum number of supported groups '+IntToStr(MaxWordGroups)+' already reached.')
    else
    begin
      CurGroupIndex := 0;
      InsertPositionFound := False;
      repeat
      begin
        if WordGroup[CurGroupIndex].GroupNum > GroupNum then
        begin
          InsertPosition := CurGroupIndex;
          InsertPositionFound := True;
        end
        else CurGroupIndex := CurGroupIndex + 1;
      end until InsertPositionFound or (CurGroupIndex > NumGroups - 1);
      if CurGroupIndex > NumGroups - 1 then InsertPosition := NumGroups;
      WordGroup[NumGroups].Words := TStringList.Create;
      if NumGroups > 0 then
        for CurGroupIndex := NumGroups - 1 downto InsertPosition do
        begin
          WordGroup[CurGroupIndex+1].GroupNum := WordGroup[CurGroupIndex].GroupNum;
          WordGroup[CurGroupIndex+1].Words.Assign(WordGroup[CurGroupIndex].Words);
        end;
      NumGroups := NumGroups + 1;
      WordGroup[InsertPosition].GroupNum := GroupNum;
      WordGroup[InsertPosition].Words.Clear;
      WordGroupList.Items.Insert(InsertPosition,IntToStr(WordGroup[InsertPosition].GroupNum)+'. '+GroupName(InsertPosition));
      WordGroupList.ItemIndex := InsertPosition;
      FileModified := True;
      WordGroupList.OnClick(Application);
    end;
  end;
end;

{*************************************************************}
function TWordsEditMain.DeleteGroup(GroupIndex:word) : boolean;
{*************************************************************}
var CurGroupIndex : word;
begin
  DeleteGroup := False;
  if GroupIndex <= NumGroups - 1 then
  begin
    if NumGroups > 1 then
      for CurGroupIndex := GroupIndex to NumGroups - 2 do
      begin
        WordGroup[CurGroupIndex].GroupNum := WordGroup[CurGroupIndex+1].GroupNum;
        WordGroup[CurGroupIndex].Words.Assign(WordGroup[CurGroupIndex+1].Words);
      end;
    WordGroup[NumGroups-1].Words.Free;
    NumGroups := NumGroups - 1;
    WordGroupList.Items.Delete(GroupIndex);
    DeleteGroup := True;
    FileModified := True;
  end;
end;

{*************************************************************}
procedure TWordsEditMain.ChangeGroupNum(OldGroupIndex,NewGroupNum:word);
{*************************************************************}
var TempWords              : TStringList;
    CurGroupIndex          : word;
    InsertPosition         : word;
    InsertPositionFound    : boolean;
begin
  if not GroupExists(NewGroupNum) then
  begin
    TempWords := TStringList.Create;
    TempWords.Assign(WordGroup[OldGroupIndex].Words);
    DeleteGroup(OldGroupIndex);
    CurGroupIndex := 0;
    InsertPositionFound := False;
    repeat
    begin
      if WordGroup[CurGroupIndex].GroupNum > NewGroupNum then
      begin
        InsertPosition := CurGroupIndex;
        InsertPositionFound := True;
      end
      else CurGroupIndex := CurGroupIndex + 1;
    end until InsertPositionFound or (CurGroupIndex > NumGroups - 1);
    if CurGroupIndex > NumGroups - 1 then InsertPosition := NumGroups;
    WordGroup[NumGroups].Words := TStringList.Create;
    if NumGroups > 0 then
      for CurGroupIndex := NumGroups - 1 downto InsertPosition do
      begin
        WordGroup[CurGroupIndex+1].GroupNum := WordGroup[CurGroupIndex].GroupNum;
        WordGroup[CurGroupIndex+1].Words.Assign(WordGroup[CurGroupIndex].Words);
      end;
    NumGroups := NumGroups + 1;
    WordGroup[InsertPosition].GroupNum := NewGroupNum;
    WordGroup[InsertPosition].Words.Assign(TempWords);
    WordGroupList.Items.Insert(InsertPosition,IntToStr(WordGroup[InsertPosition].GroupNum)+'. '+GroupName(InsertPosition));
    WordGroupList.ItemIndex := InsertPosition;
    WordGroupList.OnClick(Application);
    TempWords.Free;
    FileModified := True;
  end;
end;

{*************************************************************}
procedure TWordsEditMain.NewFile;
{*************************************************************}
var CurGroupIndex : word;
begin
  ClearFile;
  CurrentFilename := '';
  WordsEditMain.Caption := 'WORDS.TOK Editor';
  WordGroupList.Items.Clear;
  WordList.Items.Clear;
  WordGroupNumDisplay.Caption := 'Word group';
  WordEdit.Text := '';
  WordEdit.Enabled := False;
  NumGroups := 3;
  WordGroup[0].GroupNum := 0;
  WordGroup[0].Words := TStringList.Create;
  WordGroup[0].Words.Add('a');
  WordGroup[1].GroupNum := 1;
  WordGroup[1].Words := TStringList.Create;
  WordGroup[1].Words.Add('anyword');
  WordGroup[2].GroupNum := 9999;
  WordGroup[2].Words := TStringList.Create;
  WordGroup[2].Words.Add('rol');
  for CurGroupIndex := 0 to 2 do
    WordGroupList.Items.Add(IntToStr(WordGroup[CurGroupIndex].GroupNum)+'. '+GroupName(CurGroupIndex));
  AddWordBut.Enabled := False;
  DeleteWordBut.Enabled := False;
  DeleteGroupBut.Enabled := False;
  ChangeGroupNumBut.Enabled := False;
  FileModified := False;
end;

{*************************************************************}
function ReadFile(Filename:string) : TWordList;
// NOTE: The logic decoder and compiler both rely on this function
{*************************************************************}
var WordsFile        : file;
    WordsData        : TResource;
    ErrorOccured     : boolean;
    ResPos           : word;
    CurWord          : string;
    PrevWord         : string;
    CurByte          : byte;
    EndOfFileReached : boolean;
    GroupNum         : word;
    GroupIndex       : integer;
    CharsFromPrevWord : byte;
    CurIndex         : word;
    CurAddGroup      : word;
    LowestRemainingGroup : integer;
    SecondLowestRemainingGroup : integer;
    SecondLowestRemainingGroupIndex : word;
    GroupAddOrder    : array[0..MaxWordGroups-1] of word;

  {The following two variables store the words read from the file.}
  {They are the same as the main WordGroup and NumGroups variables but are}
  {used when reading in the file so if an error occurs the user can continue}
  {using the editor with the previous file.}
    New_WordGroup : array[0..MaxWordGroups-1] of TWordGroup;
    New_NumGroups : word;

    function ReadByte : byte;
    begin
      if ResPos < WordsData.Size then
      begin
        ReadByte := WordsData.Data^[ResPos];
        ResPos := ResPos + 1;
      end
      else
      begin
        ReadByte := 0;
        EndOfFileReached := true;
      end;
    end;

    function ReadMSLSWord : word;
    var MSbyte, LSbyte : byte;
    begin
      MSbyte := ReadByte;
      LSbyte := ReadByte;
      ReadMSLSWord := MSbyte*256 + LSbyte;
    end;

    function GetNew_GroupIndex(Groupnum:word) : integer;
    var CurIndex : word;
    begin
      GetNew_GroupIndex := -1;
      if New_NumGroups > 0 then
      begin
        for CurIndex := 0 to New_NumGroups - 1 do
          if New_WordGroup[CurIndex].GroupNum = GroupNum then GetNew_GroupIndex := CurIndex;
      end;
    end;

begin
  ReadFile.Exists := False;
  ErrorOccured := False;
  EndOfFileReached := False;
  if not FileExists(Filename) then
    ShowMessage('Error! File "'+Filename+'" does not exist.')
  else
  begin
    AssignFile(WordsFile,Filename);
    {$I-}
    Reset(WordsFile,1);
    {$I+}
    if IOResult <> 0 then
      ShowMessage('Error! Could not open file "'+Filename+'". Make sure you are not running the game and do not have the file open in another program.')
    else if FileSize(WordsFile) > MaxResourceSize then
      ShowMessage('Error! File is too big (>'+IntToStr(MaxResourceSize)+' bytes.')
    else
    begin
      WordsData.Size := FileSize(WordsFile);
      GetMem(WordsData.Data,WordsData.Size);
      BlockRead(WordsFile,WordsData.Data^,WordsData.Size);
      CloseFile(WordsFile);
      New_NumGroups := 0;
      ResPos := 0;
      ResPos := ReadMSLSWord; {start of words section}
      PrevWord := '';
      repeat
      begin
        CurWord := '';
        CharsFromPrevWord := ReadByte;
        if CharsFromPrevWord > Length(PrevWord) then CharsFromPrevWord := Length(PrevWord);
        if CharsFromPrevWord > 0 then CurWord := Copy(PrevWord,1,CharsFromPrevWord);
        repeat
        begin
          CurByte := ReadByte;
          if CurByte < $80 then CurWord := CurWord + chr(CurByte XOR $7F);
        end; until (CurByte >= $80) or EndOfFileReached;
        {we must check for end of file, otherwise if the file is invalid the}
        {program may read indefinetly.}
        if EndOfFileReached then
        begin
          ShowMessage('Error! Invalid WORDS.TOK file.');
          ErrorOccured := True;
        end {if EndOfFileReached}
        else
        begin
          CurWord := CurWord + chr($7F XOR (CurByte-$80));
          CurWord := LowerCase(CurWord); {convert any upper-case characters to lower-case}
                                         {(just in case)}
          Groupnum := ReadMSLSWord;
          if CurWord <> PrevWord then  {this word different to previous, so add it}
                                       {in this way, no duplicates are added}
          begin
            GroupIndex := GetNew_GroupIndex(Groupnum);
            if GroupIndex >= 0 then {group exists}
              New_WordGroup[GroupIndex].Words.Add(CurWord)
            else {group doesn't exist - create new one}
            begin
              if NumGroups >= MaxWordGroups then
              begin
                ShowMessage('Error! Too many groups');
                ErrorOccured := True;
              end {if NumGroups >= MaxWordGroups}
              else
              begin
                New_NumGroups := New_NumGroups + 1;
                New_WordGroup[New_NumGroups-1].GroupNum := GroupNum;
                New_WordGroup[New_NumGroups-1].Words := TStringList.Create;
                New_WordGroup[New_NumGroups-1].Words.Clear;
                New_WordGroup[New_NumGroups-1].Words.Add(CurWord);
              end; {if NumGroups < MaxWordGroups}
            end; {group doesn't exist - create new one}
            PrevWord := CurWord;
          end; {if CurWord <> PrevWord}
        end; {if not EndOfFileReached}
        CurByte := ReadByte;
        if (CurByte = 0) and (ResPos >= WordsData.Size-1) then EndOfFileReached := True
        else ResPos := ResPos - 1;
      end; until EndOfFileReached;
      FreeMem(WordsData.Data,WordsData.Size);
      if not ErrorOccured then
      begin
        LowestRemainingGroup := -1;
        for CurAddGroup := 0 to New_NumGroups - 1 do
        begin
          SecondLowestRemainingGroup := 65536;
          for CurIndex := 0 to New_NumGroups - 1 do
            if (New_WordGroup[CurIndex].GroupNum < SecondLowestRemainingGroup)
            and (New_WordGroup[CurIndex].GroupNum > LowestRemainingGroup) then
            begin
              SecondLowestRemainingGroup := New_WordGroup[CurIndex].GroupNum;
              SecondLowestRemainingGroupIndex := CurIndex;
            end;
          GroupAddOrder[CurAddGroup] := SecondLowestRemainingGroupIndex;
          LowestRemainingGroup := SecondLowestRemainingGroup;
        end;

        ReadFile.NumGroups := New_NumGroups;
        for CurIndex := 0 to New_NumGroups - 1 do
        begin
          ReadFile.WordGroup[CurIndex].GroupNum := New_WordGroup[GroupAddOrder[CurIndex]].GroupNum;
          ReadFile.WordGroup[CurIndex].Words := TStringList.Create;
          ReadFile.WordGroup[CurIndex].Words.Assign(New_WordGroup[GroupAddOrder[CurIndex]].Words);
          New_WordGroup[GroupAddOrder[CurIndex]].Words.Free;
        end;
        ReadFile.Exists := True;
      end; {if not ErrorOccured}
    end; {if (IOResult = 0) and (FileSize(WordsFile) <= MaxResourceSize)}
  end; {if FileExists(Filename)}
end;



{*************************************************************}
function ReadWORDSTOKFile(FileName:string) : TWordList;
// This function is just so other components, e.g. logic editor
// can read the WORDS.TOK file.
{*************************************************************}
begin
  ReadWORDSTOKFile := ReadFile(Filename);
end;


{*************************************************************}
function TWordsEditMain.OpenFile(Filename:string) : boolean;
{*************************************************************}
var NewWordList : TWordList;
    CurIndex    : word;
    CurWord     : word;
begin
  OpenFile := False;
  NewWordList := ReadFile(Filename);
  if NewWordList.Exists then
  begin
    CurrentFilename := Filename;
    WordsEditMain.Caption := 'WORDS.TOK Editor - '+CurrentFilename;
    WordGroupList.Items.Clear;
    NumGroups := NewWordList.NumGroups;
    for CurIndex := 0 to NewWordList.NumGroups - 1 do
    begin
      WordGroup[CurIndex].GroupNum := NewWordList.WordGroup[CurIndex].GroupNum;
      WordGroup[CurIndex].Words := TStringList.Create;
      WordGroup[CurIndex].Words.Assign(NewWordList.WordGroup[CurIndex].Words);
      NewWordList.WordGroup[CurIndex].Words.Free;
      WordGroupList.Items.Add(IntToStr(WordGroup[CurIndex].GroupNum)+'. '+GroupName(CurIndex));
    end;
    WordGroupNumDisplay.Caption := 'Word group';
    WordList.Items.Clear;
    WordEdit.Text := '';
    AddWordBut.Enabled := False;
    DeleteWordBut.Enabled := False;
    WordGroupList.ItemIndex := -1;
    DeleteGroupBut.Enabled := False;
    ChangeGroupNumBut.Enabled := False;
    FileModified := False;
    OpenFile := True;
  end;
end;

{*************************************************************}
function TWordsEditMain.MergeFile(Filename:string) : boolean;
{*************************************************************}
var NewWordList    : TWordList;
    CurIndex       : word;
    CurWord        : word;
    GroupIndex     : word;
    NumWordsAdded  : word;
    ThisWord       : string;
    WhatToDoWithExistingWords : (AlwaysReplace,NeverReplace,AskUser);
    GroupIndexOfExistingWord : integer;

  procedure DeleteWord(TheWord:string);
  var CurIndex : word;
      CurWord  : word;
  begin
    if NumGroups > 0 then
    for CurIndex := 0 to NumGroups - 1 do
      if WordGroup[CurIndex].Words.Count > 0 then
      begin
        CurWord := 0;
        repeat
          if WordGroup[CurIndex].Words[CurWord] = TheWord then
            WordGroup[CurIndex].Words.Delete(CurWord)
          else CurWord := CurWord + 1;
        until CurWord > WordGroup[CurIndex].Words.Count - 1;
      end;
  end;

  procedure DeleteWordFromGroup(GroupIndex:word;TheWord:string);
  var CurWord  : word;
  begin
    if (GroupIndex <= NumGroups - 1) and (WordGroup[GroupIndex].Words.Count > 0) then
      if WordGroup[GroupIndex].Words.Count > 0 then
      begin
        CurWord := 0;
        repeat
          if WordGroup[GroupIndex].Words[CurWord] = TheWord then
            WordGroup[GroupIndex].Words.Delete(CurWord)
          else CurWord := CurWord + 1;
        until CurWord >= WordGroup[GroupIndex].Words.Count;
      end;
  end;

//  function WordExists(TheWord:string) : boolean;
//  var CurIndex : word;
//      CurWord  : word;
//  begin
//    WordExists := False;
//    if NumGroups > 0 then
//    CurIndex := 0;
//    repeat
//    begin
//      if WordGroup[CurIndex].Words.Count > 0 then
//      begin
//        CurWord := 0;
//        repeat
//        begin
//          if WordGroup[CurIndex].Words[CurWord] = TheWord then
//            WordExists := True;
//          CurWord := CurWord + 1;
//        end;
//        until (CurWord >= WordGroup[CurIndex].Words.Count) or WordExists;
//      end;
//      CurIndex := CurIndex + 1;
//    end; until (CurIndex > NumGroups - 1) or WordExists;
//  end;

  function InsertWordGroup(GroupNum:word) : boolean;
  var CurGroupIndex : word;
      InsertPosition : word;
      InsertPositionFound : boolean;
  begin
    InsertWordGroup := False;
    if NumGroups >= MaxWordGroups then
      ShowMessage('Error! Too many groups (max '+IntToStr(MaxWordGroups)+').')
    else
    begin
      CurGroupIndex := 0;
      InsertPositionFound := False;
      repeat
      begin
        if WordGroup[CurGroupIndex].GroupNum > GroupNum then
        begin
          InsertPosition := CurGroupIndex;
          InsertPositionFound := True;
        end
        else CurGroupIndex := CurGroupIndex + 1;
      end until InsertPositionFound or (CurGroupIndex > NumGroups - 1);
      if CurGroupIndex > NumGroups - 1 then InsertPosition := NumGroups;
      WordGroup[NumGroups].Words := TStringList.Create;
      if NumGroups > 0 then
        for CurGroupIndex := NumGroups - 1 downto InsertPosition do
        begin
          WordGroup[CurGroupIndex+1].GroupNum := WordGroup[CurGroupIndex].GroupNum;
          WordGroup[CurGroupIndex+1].Words.Assign(WordGroup[CurGroupIndex].Words);
        end;
      NumGroups := NumGroups + 1;
      WordGroup[InsertPosition].GroupNum := GroupNum;
      WordGroup[InsertPosition].Words.Clear;
      WordGroupList.Items.Insert(InsertPosition,IntToStr(WordGroup[InsertPosition].GroupNum)+'. ');
      InsertWordGroup := True;
    end; {if NumGroups < MaxWordGroups}
  end;

  function OKToReplaceWord(TheWord:string;OldGroupNum,NewGroupNum:integer) : boolean;
  var AskResult : integer;
  begin
    if WhatToDoWithExistingWords = AlwaysReplace then
      OKToReplaceWord := True
    else if WhatToDoWithExistingWords = NeverReplace then
      OKToReplaceWord := False
    else if OldGroupNum = NewGroupNum then
      OKToReplaceWord := True
    else
    begin
      WordsEditMerge.Label1.Caption := 'The word "'+TheWord+'" already exists in group '+IntToStr(OldGroupNum)+' of the currently open file.';
      WordsEditMerge.Label2.Caption := 'Do you wish to replace it with the occurance in the merge file (group '+IntToStr(NewGroupNum)+')?';
      AskResult := WordsEditMerge.ShowModal;
      if (AskResult = mrYes) or (AskResult = mrYesToAll) then OKToReplaceWord := True
      else OKToReplaceWord := False;
      if AskResult = mrYesToAll then WhatToDoWithExistingWords := AlwaysReplace;
      if AskResult = mrNoToAll then WhatToDoWithExistingWords := NeverReplace;
    end;
  end;

begin
  WhatToDoWithExistingWords := AskUser;
  MergeFile := False;


  NewWordList := ReadFile(Filename);
  if NewWordList.Exists then
  begin
    NumWordsAdded := 0;
    if NewWordList.NumGroups > 0 then
    begin
      WordsEditSortDisplay.Gauge1.MaxValue := NewWordList.NumGroups - 1;
      WordsEditSortDisplay.Gauge1.Progress := 0;
      WordsEditSortDisplay.Left := WordsEditMain.Left + (WordsEditMain.Width div 2) - (WordsEditSortDisplay.Width div 2);
      WordsEditSortDisplay.Top := WordsEditMain.Top + (WordsEditMain.Height div 2) - (WordsEditSortDisplay.Height div 2);
      WordsEditSortDisplay.Label1.Caption := 'Merging....';
      WordsEditSortDisplay.Show;
      WordsEditSortDisplay.Repaint;

      for CurIndex := 0 to NewWordList.NumGroups - 1 do
      begin
        if not GroupExists(NewWordList.WordGroup[CurIndex].GroupNum) then
          InsertWordGroup(NewWordList.WordGroup[CurIndex].GroupNum);
        GroupIndex := GetGroupIndex(NewWordList.WordGroup[CurIndex].GroupNum);
        if (GroupIndex >= 0) and (NewWordList.WordGroup[CurIndex].Words.Count > 0) then
        begin
          for CurWord := 0 to NewWordList.WordGroup[CurIndex].Words.Count - 1 do
          begin
            GroupIndexOfExistingWord := GroupIndexOfWord(NewWordList.WordGroup[CurIndex].Words[CurWord]);
            if (GroupIndexOfExistingWord < 0) or ((GroupIndexOfExistingWord >= 0) and
             (OKToReplaceWord(NewWordList.WordGroup[CurIndex].Words[CurWord],WordGroup[GroupIndexOfExistingWord].GroupNum,WordGroup[GroupIndex].GroupNum))) then
            begin
              DeleteWord(NewWordList.WordGroup[CurIndex].Words[CurWord]);
              WordGroup[GroupIndex].Words.Add(NewWordList.WordGroup[CurIndex].Words[CurWord]);
              NumWordsAdded := NumWordsAdded + 1;
            end;
          end; {for CurWord := 0 to NewWordList.WordGroup[CurIndex].Words.Count}
          WordsEditSortDisplay.Gauge1.Progress := WordsEditSortDisplay.Gauge1.Progress + 1;
        end; {if (GroupIndex >= 0) and (NewWordList.WordGroup[CurIndex].Words.Count > 0)}
        NewWordList.WordGroup[CurIndex].Words.Free;
      end; {for CurIndex := 0 to NewWordList.NumGroups - 1}

      for CurIndex := 0 to NewWordList.NumGroups - 1 do
      begin
        WordGroup[CurIndex].Words.Sort;
        UpdateGroupList(CurIndex);
      end;
      WordsEditSortDisplay.Close;
      ShowMessage('Merge completed. A total of '+IntToStr(NumWordsAdded)+' words were added.');
      FileModified := True;
      MergeFile := True;
    end; {if NewWordList.NumGroups > 0}
  end; {if NewWordList.Exists}
end;

{*************************************************************}
function TWordsEditMain.SaveFile(Filename:string) : boolean;
{*************************************************************}
var WordsData            : TResource;
    WordsFile            : file of byte;
    AllWords             : TStringList;

    NumWords             : integer;
    ContinueWithSave     : boolean;
    NumEmptyWordGroups   : word;

    CurGroupIndex        : word;
    CurWord              : integer;
    CurByte              : byte;
    CurChar              : word;

    ThisWord             : string;
    ThisGroupNum         : word;
    PrevWord             : string;
    CharsFromPrevWord    : byte;
    FinishedComparison   : boolean;
    FirstLetter          : char;
    FirstLetterLoc       : word;
    CurFirstLetter       : word;
    LetterLoc            : array[1..26] of word;


    function GetWord(inword:string) : string;
    {returns the part of a string before the last ' '}
    var curpos : word;
    begin
      curpos := Length(inword)+1;
      repeat curpos := curpos - 1
      until (copy(inword,curpos,1) = ' ') or (curpos = 1);
      if curpos > 1 then GetWord := copy(inword,1,curpos-1)
      else GetWord := ' ';
    end;

    function GetGroupNum(inword:string) : word;
    {returns the number after the last ' ' in a string}
    var curpos : word;
        GroupNum : integer;
        code : integer;
    begin
      curpos := Length(inword)+1;
      repeat curpos := curpos - 1
      until (copy(inword,curpos,1) = ' ') or (curpos = 1);
      if curpos < Length(inword) then
      begin
        Val(copy(inword,curpos+1,Length(inword)-curpos),GroupNum,code);
        if code = 0 then GetGroupNum := GroupNum
        else GetGroupNum := 0;
      end
      else GetGroupNum := 0;
    end;

begin
  ContinueWithSave := True;
  if NumGroups = 0 then
  begin
    ShowMessage('Error: Could not save the file as there are no word groups.');
    ContinueWithSave := False;
  end
  else
  begin
    NumEmptyWordGroups := 0;
    for CurGroupIndex := 0 to NumGroups - 1 do
    begin
      if WordGroup[CurGroupIndex].Words.Count = 0 then NumEmptyWordGroups := NumEmptyWordGroups + 1;
    end;
    if NumEmptyWordGroups > 0 then
      ShowMessage('Warning: There are '+IntToStr(NumEmptyWordGroups)+' empty word groups. These will not be saved.');
  end; {if NumGroups > 0}
  if ContinueWithSave then
  begin
    AssignFile(WordsFile,Filename);
    {$I-}
    Rewrite(WordsFile);
    {$I+}
   if IOResult <> 0 then
       ShowMessage('Error writing to file "'+Filename+'".')
    else
    begin
      AllWords := TStringList.Create;
      for CurGroupIndex := 0 to Numgroups - 1 do
      if WordGroup[CurGroupIndex].Words.Count > 0 then
      for CurWord := 0 to WordGroup[CurGroupIndex].Words.Count - 1 do
      begin
        if Length(WordGroup[CurGroupIndex].Words[CurWord]) > 0 then
           AllWords.Add(WordGroup[CurGroupIndex].Words[CurWord]+' '+IntToStr(WordGroup[CurGroupIndex].GroupNum))
        else AllWords.Add(' ');
      end;
      AllWords.Sort;
      CurByte := 0;
      for CurChar := 0 to 51 do
        Write(WordsFile,CurByte);
      for CurFirstLetter := 1 to 26 do
        LetterLoc[CurFirstLetter] := 0;
      FirstLetter := 'a';
      LetterLoc[1] := FileSize(WordsFile);
      for CurWord := 0 to AllWords.Count - 1 do
      begin
        ThisWord := GetWord(AllWords[CurWord]);
        if (ThisWord[1] <> FirstLetter)
        and (ord(ThisWord[1])>=97) and (ord(ThisWord[1])<=122) then
        begin
          FirstLetter := ThisWord[1];
          LetterLoc[ord(FirstLetter)-96] := FileSize(WordsFile);
        end;
        ThisGroupNum := GetGroupNum(AllWords[CurWord]);
        {work out # chars from prev word}
        CharsFromPrevWord := 0;
        CurChar := 1;
        FinishedComparison := False;
        repeat
          if (CurChar <= Length(ThisWord)) and (Length(PrevWord) >= CurChar) and (PrevWord[CurChar] = ThisWord[CurChar]) then
            CharsFromPrevWord := CharsFromPrevWord + 1
          else FinishedComparison := True;
          CurChar := CurChar + 1;
        until FinishedComparison;
        if CharsFromPrevWord >= Length(ThisWord) then
          CharsFromPrevWord := Length(ThisWord) - 1;
        {write # chars from prev word}
        Write(WordsFile,CharsFromPrevWord);
        {write word}
        PrevWord := ThisWord;
        ThisWord := Copy(ThisWord,CharsFromPrevWord+1,Length(ThisWord)-CharsFromPrevWord);
        if Length(ThisWord) > 1 then
          for CurChar := 1 to Length(ThisWord)-1 do
          begin
            CurByte := $7F XOR ord(ThisWord[CurChar]);
            Write(WordsFile,CurByte);
          end;
        CurByte := $80+($7F XOR ord(ThisWord[Length(ThisWord)]));
        Write(WordsFile,CurByte);
        {write group number}
        CurByte := ThisGroupNum div 256;
        Write(WordsFile,CurByte);
        CurByte := ThisGroupNum mod 256;
        Write(WordsFile,CurByte);
      end;
      CurByte := 0;
      Write(WordsFile,CurByte);
      Seek(WordsFile,0);
      for CurFirstLetter := 1 to 26 do
      begin
        CurByte := LetterLoc[CurFirstLetter] div 256;
        Write(WordsFile,CurByte);
        CurByte := LetterLoc[CurFirstLetter] mod 256;
        Write(WordsFile,CurByte);
      end;
      AllWords.Free;
      CloseFile(WordsFile);
      FileModified := False;
    end; {if IOResult <> 0}
  end; {if ContinueWithSave}
end;

{*************************************************************}
procedure TWordsEditMain.FormResize(Sender: TObject);
{*************************************************************}
begin
  ButPanel.Top := ClientHeight - ButPanel.Height;
  Panel1.Width := ClientWidth;
  Panel1.Height := ClientHeight - ButPanel.Height;
  WordEdit.Top := WordPanel.ClientHeight - WordEdit.Height;
  WordList.Height := WordPanel.ClientHeight - WordGroupNumDisplay.Height - WordEdit.Height - Splitter1.Width;
  WordList.Width := WordPanel.ClientWidth;
  WordEdit.Width := WordPanel.ClientWidth;
end;

{*************************************************************}
procedure TWordsEditMain.Splitter1Moved(Sender: TObject);
{*************************************************************}
begin
  WordList.Width := WordPanel.ClientWidth;
  WordEdit.Width := WordPanel.ClientWidth;
end;

{*************************************************************}
procedure TWordsEditMain.FormShow(Sender: TObject);
{*************************************************************}
begin
  WordGroupList.ItemIndex := -1;
  WordList.Clear;
  WordEdit.Text := '';
  DeleteGroupBut.Enabled := False;
  AddWordBut.Enabled := False;
  DeleteWordBut.Enabled := False;
end;

{*************************************************************}
procedure TWordsEditMain.FormCreate(Sender: TObject);
{*************************************************************}
begin
  NumGroups := 0;
end;

{*************************************************************}
procedure TWordsEditMain.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
{*************************************************************}
begin
  CanClose := AskClose;
end;

{*************************************************************}
procedure TWordsEditMain.FormClose(Sender: TObject; var Action: TCloseAction);
{*************************************************************}
begin
  Action := caFree;
end;

{*************************************************************}
procedure TWordsEditMain.WordGroupListClick(Sender: TObject);
{*************************************************************}
var CurWord : word;
begin
  if WordGroupList.ItemIndex >= 0 then
  begin
    WordGroup[WordGroupList.ItemIndex].Words.Sort;
    WordList.Items.Assign(WordGroup[WordGroupList.ItemIndex].Words);
    WordGroupNumDisplay.Caption := 'Word group '+IntToStr(WordGroup[WordGroupList.ItemIndex].GroupNum);
    WordEdit.Text := '';
    DeleteGroupBut.Enabled := True;
    WordList.ItemIndex := -1;
    AddWordBut.Enabled := True;
    DeleteWordBut.Enabled := False;
    WordEdit.Enabled := False;
    ChangeGroupNumBut.Enabled := True;
  end
  else
  begin
    DeleteGroupBut.Enabled := False;
    WordList.Items.Clear;
    WordEdit.Text := '';
    WordGroupNumDisplay.Caption := 'Word group';
    AddWordBut.Enabled := False;
    DeleteWordBut.Enabled := False;
    WordEdit.Enabled := False;
    ChangeGroupNumBut.Enabled := False;
  end;
end;

{*************************************************************}
procedure TWordsEditMain.WordListClick(Sender: TObject);
{*************************************************************}
begin
  if WordList.ItemIndex >= 0 then
  begin
    WordEdit.Text := WordGroup[WordGroupList.ItemIndex].Words[WordList.ItemIndex];
    WordEdit.Enabled := True;
    DeleteWordBut.Enabled := True;
  end
  else
  begin
    WordEdit.Text := '';
    WordEdit.Enabled := False;
    DeleteWordBut.Enabled := False;
  end;
end;

{*************************************************************}
procedure TWordsEditMain.WordEditKeyPress(Sender: TObject; var Key: Char);
{*************************************************************}
begin
  if ord(Key) = 13 then    {ENTER}
    WordEditExit(Sender)
  else if ord(Key) = 27 then {ESC}
  begin
    WordEdit.Text := WordList.Items[WordList.ItemIndex];
    WordList.SetFocus;
  end;
end;

{*************************************************************}
procedure TWordsEditMain.WordEditExit(Sender: TObject);
{*************************************************************}
var GroupIndexOfThisWord : integer;

  function CleanUpWord(TheWord:string) : string;
  var CurChar : word;
  begin
    TheWord := LowerCase(TheWord);
    CurChar := 1;
    repeat
    begin
      if not (TheWord[CurChar] in ['0'..'9','a'..'z',' ']) then
        TheWord := Copy(TheWord,1,CurChar-1) + Copy(TheWord,CurChar+1,Length(TheWord)-CurChar)
      else CurChar := CurChar + 1;
    end; until CurChar > Length(TheWord) - 1;
    CleanUpWord := TheWord;
  end;

begin
  if (WordGroupList.ItemIndex >= 0) and (WordList.ItemIndex >= 0) then
  begin
    WordEdit.Text := CleanUpWord(WordEdit.Text);
    if WordEdit.Text = '' then
    begin
      WordGroup[WordGroupList.ItemIndex].Words.Delete(WordList.ItemIndex);
      WordList.Items.Assign(WordGroup[WordGroupList.ItemIndex].Words);
      UpdateGroupList(WordGroupList.ItemIndex);
      DeleteWordBut.Enabled := False;
      WordEdit.Enabled := False;
    end {if WordEdit.Text = ''}
    else
    begin
      if (WordList.Items.IndexOf(WordEdit.Text) >= 0)
         and (WordList.ItemIndex <> WordList.Items.IndexOf(WordEdit.Text)) then
      begin {another occurance of this word already exists in this group}
        WordGroup[WordGroupList.ItemIndex].Words.Delete(WordList.ItemIndex);
        WordList.Items.Delete(WordList.ItemIndex);
        WordList.ItemIndex := WordList.Items.IndexOf(WordEdit.Text);
      end   {another occurance of this word already exists in this group}
      else
      begin
        GroupIndexOfThisWord := GroupIndexOfWord(WordEdit.Text);
        if (GroupIndexOfThisWord >= 0) and (GroupIndexOfThisWord <> WordGroupList.ItemIndex) then
        begin  {another occurance of this word already exists in another group}
          if AskYesNo('Remove duplicate word?','This word already exists (in group '+IntToStr(WordGroup[GroupIndexOfThisWord].GroupNum)+'). Do you wish to remove this occurance and add it to this group?') then
          begin
            WordGroup[GroupIndexOfThisWord].Words.Delete(WordGroup[GroupIndexOfThisWord].Words.IndexOf(WordEdit.Text));
            UpdateGroupList(GroupIndexOfThisWord);
            WordGroup[WordGroupList.ItemIndex].Words[WordList.ItemIndex] := WordEdit.Text;
            WordGroup[WordGroupList.ItemIndex].Words.Sort;
            WordList.Items.Assign(WordGroup[WordGroupList.ItemIndex].Words);
            UpdateGroupList(WordGroupList.ItemIndex);
            WordList.ItemIndex := WordList.Items.IndexOf(WordEdit.Text);
          end
          else
          begin
            WordGroup[WordGroupList.ItemIndex].Words.Delete(WordList.ItemIndex);
            WordList.Items.Assign(WordGroup[WordGroupList.ItemIndex].Words);
            UpdateGroupList(WordGroupList.ItemIndex);
            WordEdit.Text := '';
            DeleteWordBut.Enabled := False;
            WordEdit.Enabled := False;
          end;
        end    {another occurance of this word already exists in another group}
        else
        begin  {there are no other occurances of this word}
          WordGroup[WordGroupList.ItemIndex].Words[WordList.ItemIndex] := WordEdit.Text;
          WordGroup[WordGroupList.ItemIndex].Words.Sort;
          WordList.Items.Assign(WordGroup[WordGroupList.ItemIndex].Words);
          UpdateGroupList(WordGroupList.ItemIndex);
          WordList.ItemIndex := WordList.Items.IndexOf(WordEdit.Text);
        end;   {there are no other occurances of this word}
      end;
    end; {if not (WordEdit.Text = '')}
    FileModified := True;
  end;
end;

{*************************************************************}
procedure TWordsEditMain.FindButClick(Sender: TObject);
{*************************************************************}
begin
  FindDialog1.Execute;
end;

{*************************************************************}
procedure TWordsEditMain.FindDialog1Find(Sender: TObject);
{*************************************************************}
type TFindDirection = (Up, Down);

var FoundText : boolean;
    StartGroup : word;
    StartWord : word;
    CurGroup : integer;
    CurWord : integer;
    CurChar : word;
    FoundGroupNo : word;
    FoundWordNo : word;
    FindTextCapital : string;
    WordCapital : string;
    FindDirection : TFindDirection;

  procedure NextWord;
  begin
    if FindDirection = Down then
    begin
      CurWord := CurWord + 1;
      if CurWord > WordGroup[CurGroup].Words.Count - 1 then
      begin
        CurGroup := CurGroup + 1;
        if CurGroup > NumGroups - 1 then CurGroup := 0;
        CurWord := 0;
      end;
    end
    else
    begin
      CurWord := CurWord - 1;
      if CurWord < 0 then
      begin
        CurGroup := CurGroup - 1;
        if CurGroup < 0 then CurGroup := NumGroups - 1;
        if WordGroup[CurGroup].Words.Count > 0 then
          CurWord := WordGroup[CurGroup].Words.Count - 1
        else CurWord := 0;
      end;
    end;
  end;

begin
  FoundText := False;
  if NumGroups > 0 then
  begin
    if frDown in FindDialog1.Options then FindDirection := Down
    else FindDirection := Up;
    FindTextCapital := UpperCase(FindDialog1.FindText);
    if WordGroupList.ItemIndex >= 0 then
    begin
      StartGroup := WordGroupList.ItemIndex;
      if WordList.ItemIndex >= 0 then StartWord := WordList.ItemIndex
      else StartWord := 0;
    end
    else
    begin
      StartGroup := 0;
      StartWord := 0;
    end;
    CurGroup := StartGroup;
    CurWord := StartWord;
    NextWord;
    repeat
    begin
      if WordGroup[CurGroup].Words.Count > 0 then
      begin
        WordCapital := UpperCase(WordGroup[CurGroup].Words[CurWord]);
        if Length(FindTextCapital) <= Length(WordCapital) then
        for CurChar := 1 to Length(WordCapital) - Length(FindTextCapital) + 1 do
          if UpperCase(FindTextCapital) = UpperCase(Copy(WordCapital,CurChar,Length(FindTextCapital))) then
            if not FoundText then
            begin
              FoundText := True;
              FoundGroupNo := CurGroup;
              FoundWordNo := CurWord;
            end; {if not FoundText}
      end; {if WordGroup[CurGroup].Words.Count > 0}
{      else CurWord := 0;}
      NextWord;
    end; until FoundText or ((CurGroup = StartGroup) and (CurWord = StartWord));
  end; {if NumGroups > 0}
  if not FoundText then
    ShowMessage('Cannot find "'+FindDialog1.FindText+'"')
  else
  begin
    WordGroupList.ItemIndex := FoundGroupNo;
    WordGroupList.OnClick(Sender);
    WordList.ItemIndex := FoundWordNo;
    WordList.OnClick(Sender);
  end;
end;

{*************************************************************}
procedure TWordsEditMain.DeleteWordButClick(Sender: TObject);
{*************************************************************}
var OldItemIndex : word;
begin
  if WordList.ItemIndex < 0 then ShowMessage('Error: No word selected!')
  else
  begin
    if (WordGroupList.ItemIndex >= 0) then
    if WordList.ItemIndex <= WordGroup[WordGroupList.ItemIndex].Words.Count - 1 then
    begin
      OldItemIndex := WordList.ItemIndex;
      WordGroup[WordGroupList.ItemIndex].Words.Delete(WordList.ItemIndex);
      WordList.Items.Delete(WordList.ItemIndex);
      if OldItemIndex <= WordList.Items.Count - 1 then
        WordList.ItemIndex := OldItemIndex
      else if WordList.Items.Count > 0 then
        WordList.ItemIndex := WordList.Items.Count - 1;
      WordList.OnClick(Sender);
      UpdateGroupList(WordGroupList.ItemIndex);
      FileModified := True;
    end; {if WordList.ItemIndex <= WordGroup[WordGroupList.ItemIndex].Words.Count - 1}
  end; {if (WordList.ItemIndex >= 0)}
end;

{*************************************************************}
procedure TWordsEditMain.WordListKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
{*************************************************************}
begin
  if (WordList.ItemIndex >= 0) and (Key = 46) then
    DeleteWordBut.Click
  else if Key = 45 then AddWordBut.Click;
end;

{*************************************************************}
procedure TWordsEditMain.AddWordButClick(Sender: TObject);
{*************************************************************}
var NewWordName : string;
    NewWordNum  : word;
begin
  if WordGroupList.ItemIndex < 0 then
    ShowMessage('Error: No group selected!')
  else
  begin
    NewWordName := 'new word';
    NewWordNum := 1;
    if GroupIndexOfWord(NewWordName) >= 0 then
    begin
      repeat
        NewWordNum := NewWordNum + 1;
      until GroupIndexOfWord('new word '+IntToStr(NewWordNum)) = -1;
      NewWordName := 'new word '+IntToStr(NewWordNum);
    end;
    WordGroup[WordGroupList.ItemIndex].Words.Add(NewWordName);
    WordGroup[WordGroupList.ItemIndex].Words.Sort;
    WordList.Items.Assign(WordGroup[WordGroupList.ItemIndex].Words);
    WordList.ItemIndex := WordList.Items.IndexOf(NewWordName);
    UpdateGroupList(WordGroupList.ItemIndex);
    WordEdit.Enabled := True;
    WordEdit.Text := NewWordName;
    WordEdit.SelectAll;
    WordEdit.SetFocus;
    DeleteWordBut.Enabled := True;
    FileModified := True;
  end;
end;

{*************************************************************}
procedure TWordsEditMain.DeleteGroupButClick(Sender: TObject);
{*************************************************************}
var OldItemIndex : word;
begin
  if WordGroupList.ItemIndex >= 0 then
  begin
    OldItemIndex := WordGroupList.ItemIndex;
    DeleteGroup(WordGroupList.ItemIndex);
    if OldItemIndex <= WordGroupList.Items.Count - 1 then
    begin
      WordGroupList.ItemIndex := OldItemIndex;
    end
    else if WordGroupList.Items.Count > 0 then
    begin
      WordGroupList.ItemIndex := WordGroupList.Items.Count - 1;
    end;
    WordGroupList.OnClick(Sender);
  end {if WordGroupList.ItemIndex >= 0}
  else ShowMessage('Error: No group selected!');
end;

{*************************************************************}
procedure TWordsEditMain.WordGroupListKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
{*************************************************************}
begin
  if (WordGroupList.ItemIndex >= 0) and (Key = 46) then
    DeleteGroupBut.Click
  else if Key = 45 then
    AddGroupBut.Click;
end;

{*************************************************************}
procedure TWordsEditMain.AddGroupButClick(Sender: TObject);
{*************************************************************}
begin
  WordsEditGetGroupNum.WindowFunction := WFAddGroup;
  WordsEditGetGroupNum.NumberEdit.Text := '';
  WordsEditGetGroupNum.ShowModal;
end;

procedure TWordsEditMain.FindDialog1Close(Sender: TObject);
begin

end;

{*************************************************************}
procedure TWordsEditMain.WordsAddWGClick(Sender: TObject);
{*************************************************************}
begin
  AddGroupBut.Click
end;

{*************************************************************}
procedure TWordsEditMain.WordsDeleteWGClick(Sender: TObject);
{*************************************************************}
begin
  DeleteGroupBut.Click
end;

{*************************************************************}
procedure TWordsEditMain.WordsAddWClick(Sender: TObject);
{*************************************************************}
begin
  AddWordBut.Click;
end;

{*************************************************************}
procedure TWordsEditMain.Deleteword1Click(Sender: TObject);
{*************************************************************}
begin
  DeleteWordBut.Click
end;

{*************************************************************}
procedure TWordsEditMain.WordsFindClick(Sender: TObject);
{*************************************************************}
begin
  FindBut.Click;
end;

{*************************************************************}
procedure TWordsEditMain.WordsCountWGClick(Sender: TObject);
{*************************************************************}
begin
  ShowMessage('There are '+IntToStr(NumGroups)+' word groups.');
end;

{*************************************************************}
procedure TWordsEditMain.WordsCountWClick(Sender: TObject);
{*************************************************************}
var CurGroupIndex : word;
    TotalCount : integer;
begin
  TotalCount := 0;
  if NumGroups > 0 then
    for CurGroupIndex := 0 to NumGroups - 1 do
      TotalCount := TotalCount + WordGroup[CurGroupIndex].Words.Count;
  ShowMessage('There are '+IntToStr(TotalCount)+' words.');
end;

{*************************************************************}
procedure TWordsEditMain.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
{*************************************************************}
begin
  if (Key = ord('F')) and (Shift=[ssCtrl]) then FindBut.Click;
end;

{*************************************************************}
procedure TWordsEditMain.ChangeGroupNumButClick(Sender: TObject);
{*************************************************************}
begin
  if WordGroupList.ItemIndex >= 0 then
  begin
    WordsEditGetGroupNum.WindowFunction := WFChangeGroup;
    WordsEditGetGroupNum.OldGroupIndex := WordGroupList.ItemIndex;
    WordsEditGetGroupNum.NumberEdit.Text := IntToStr(WordGroup[WordGroupList.ItemIndex].GroupNum);
    WordsEditGetGroupNum.ShowModal;
  end
  else ShowMessage('Error: No group selected!');
end;

{*************************************************************}
procedure TWordsEditMain.FileOpenClick(Sender: TObject);
{*************************************************************}
var SaveMessageResult : integer;
    AllowOpen : boolean;
begin
  AllowOpen := True;
  if FileModified then
  begin
    SaveMessageResult := Application.MessageBox('Do you want to save changes to the current file?','WORDS.TOK Editor',MB_YESNOCANCEL);
    if SaveMessageResult = IDYES then FileSaveClick(Sender)
    else if SaveMessageResult = IDCANCEL then AllowOpen := False;
  end;
  if AllowOpen then
    if OpenDialog1.Execute then
    begin
      OpenFile(OpenDialog1.Filename);
    end;
end;

{*************************************************************}
procedure TWordsEditMain.FileNewClick(Sender: TObject);
{*************************************************************}
var SaveMessageResult : integer;
    AllowNew : boolean;
begin
  AllowNew := True;
  if FileModified then
  begin
    SaveMessageResult := Application.MessageBox('Do you want to save changes to the current file?','WORDS.TOK Editor',MB_YESNOCANCEL);
    if SaveMessageResult = IDYES then FileSaveClick(Sender)
    else if SaveMessageResult = IDCANCEL then AllowNew := False;
  end;
  if AllowNew then
    NewFile;
end;

{*************************************************************}
procedure TWordsEditMain.FileSaveAsClick(Sender: TObject);
{*************************************************************}
begin
  WordEdit.OnExit(Sender);
  if SaveDialog1.Execute then
  begin
    SaveFile(SaveDialog1.Filename);
  end;
end;

{*************************************************************}
procedure TWordsEditMain.FileMergeClick(Sender: TObject);
{*************************************************************}
var MergeMethod : integer;
begin
  if OpenDialog1.Execute then
    MergeFile(OpenDialog1.Filename);
end;

{*************************************************************}
procedure TWordsEditMain.FileCloseClick(Sender: TObject);
{*************************************************************}
begin
  Close;
end;

{*************************************************************}
procedure TWordsEditMain.FileSaveClick(Sender: TObject);
{*************************************************************}
begin
  WordEdit.OnExit(Sender);
  if CurrentFileName = '' then FileSaveAsClick(Sender)
  else
    SaveFile(CurrentFilename);
end;

{*************************************************************}
procedure TWordsEditMain.Changegroupnumber1Click(Sender: TObject);
{*************************************************************}
begin
  ChangeGroupNumBut.Click;
end;

end.
