{
    ObjEd.pas - OBJECT Editor
    Copyright (C) 1997-1999 Peter Kelly <peter@area51.org.au>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
}

unit objed;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Menus, ComCtrls, ExtCtrls;

type
  TObjEditMain = class(TForm)
    ItemList: TListBox;
    MainMenu1: TMainMenu;
    File1: TMenuItem;
    FileClose: TMenuItem;
    N1: TMenuItem;
    FileSaveAsFile: TMenuItem;
    FileSave: TMenuItem;
    FileOpen: TMenuItem;
    FileNew: TMenuItem;
    OpenDialog: TOpenDialog;
    OptionsMenu: TMenuItem;
    OptionsEncryptedFile: TMenuItem;
    ItemNameEdit: TEdit;
    Panel1: TPanel;
    AddBut: TButton;
    DeleteBut: TButton;
    Label1: TLabel;
    RoomEdit: TEdit;
    RoomChange: TUpDown;
    SaveDialog1: TSaveDialog;
    function AskClose : boolean;
    procedure ReadButClick(Sender: TObject);
    procedure FileCloseClick(Sender: TObject);
    procedure InitialiseFile(Sender: TObject);
    procedure FileNewClick(Sender: TObject);
    procedure FileOpenClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure ItemListClick(Sender: TObject);
    procedure DeleteButClick(Sender: TObject);
    procedure AddButClick(Sender: TObject);
    procedure OptionsEncryptedFileClick(Sender: TObject);
    procedure ItemNameEditChange(Sender: TObject);
    procedure RoomChangeClick(Sender: TObject; Button: TUDBtnType);
    procedure RoomEditExit(Sender: TObject);
    procedure RoomEditKeyPress(Sender: TObject; var Key: Char);
    procedure RoomEditChange(Sender: TObject);
    procedure ItemListKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FileSaveClick(Sender: TObject);
    procedure FileSaveAsFileClick(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
  private
    { Private declarations }
    procedure ClearFile;
    function SaveFile(ObjectFilename:string) : boolean;
  public
    { Public declarations }
    CurrentFilename : String;
    ObjectModified : boolean;
    procedure NewFile;
    function ReadFile(ObjectFilename:string) : boolean;
  end;

  TObjFileData = array[0..10240] of byte;
    {10k array used to store decrypted object file}
    {data allocated using GetMem, so it won't take up the whole 10k}



const EncryptionKey : string = 'Avis Durgan';
      MaxItems      = 256;

type TOBJECTFile = record
                     ItemNames : TStringList;
                     RoomNum : array[0..maxItems-1] of byte;
                     MaxScreenObjects : byte;
                     NumItems : byte;
                     Exists : boolean;
                   end;

var
  ObjEditMain: TObjEditMain;
  ItemNames : TStringList;
  RoomNum : array[0..MaxItems-1] of byte;
  MaxScreenObjects : byte;

function ReadOBJECTFile(ObjectFilename:string;var FileIsEncrypted:boolean) : TOBJECTFile;

implementation

{$R *.DFM}

uses defines, main1;

var VertBorder : word;

{*************************************************************}
function TObjEditMain.AskClose : boolean;
{*************************************************************}
var SaveMessageResult : integer;
begin
  AskClose := True;
  if ObjectModified then
  begin
    SaveMessageResult := Application.MessageBox('Do you want to save changes before closing?','Object Editor',MB_YESNOCANCEL);
    if SaveMessageResult = IDYES then FileSaveClick(ObjEditMain)
    else if SaveMessageResult = IDCANCEL then AskClose := False;
    if (SaveMessageResult = IDYES) or (SaveMessageResult = IDNO) then ObjectModified := False;
  end;
end;

{*************************************************************}
procedure TObjEditMain.ClearFile;
{*************************************************************}
begin
  DeleteBut.Enabled := False;
  ItemList.Items.Clear;
  ItemNames.Clear;
  ItemNameEdit.Text := '';
  MaxScreenObjects := 16;
  ObjectModified := False;
  RoomEdit.Text := '';
end;

{*************************************************************}
procedure TObjEditMain.NewFile;
{*************************************************************}
begin
  ClearFile;
  ItemNames.Add('?');
  ItemNameEdit.Text := '';
  ItemList.Items.Add('0. ?');
  CurrentFilename := '';
  ObjEditmain.Caption := 'Object Editor';
end;

{*************************************************************}
function TObjEditMain.SaveFile(ObjectFilename:string) : boolean;
{*************************************************************}
var ObjectFileSize : word;  {size of object file - maximum needed, final size may be less due to multiplle}
                            {objects named '?' all pointing to the same place}
    CurrentItem : word;
    CurrentChar : word;     {current character in name}
    EncryptedObjectFile : ^TObjFileData;
    DecryptedObjectFile : ^TObjFileData;

    ObjectFilePos : word;
    lsbyte, msbyte : byte;
    ItemNamesStart : word;
    ObjectFile : file;
    curbyte : byte;
    curbytenum : word;

begin
  SaveFile := False;
  ObjectFileSize := ItemNames.Count*3 + 5;  {3 bytes for each index entry}
                                            {3 bytes for header, 2 for '?' object}
  for CurrentItem := 01to ItemNames.Count  do
    if ItemNames[CurrentItem-1]<>'?' then ObjectFileSize := ObjectFileSize + Length(ItemNames[CurrentItem-1]) + 1;

  GetMem(DecryptedObjectFile,ObjectFileSize);

  {create data}
  ItemNamesStart := ItemNames.Count*3 + 3;
  msbyte := (ItemNamesStart-3) div 256;
  lsbyte := (ItemNamesStart-3) mod 256;
  DecryptedObjectFile[0] := lsbyte;
  DecryptedObjectFile[1] := msbyte;
  DecryptedObjectFile[2] := MaxScreenObjects;
  DecryptedObjectFile[3] := lsbyte;
  DecryptedObjectFile[4] := msbyte;
  DecryptedObjectFile[5] := 0;
  DecryptedObjectFile[ItemNamesStart] := ord('?');
  DecryptedObjectFile[ItemNamesStart + 1] := 0;
  ObjectFilePos := ItemNamesStart + 2;
  for CurrentItem := 1 to ItemNames.Count do
  begin
    if ItemNames[CurrentItem-1] = '?' then
    begin
      DecryptedObjectFile^[CurrentItem*3] := DecryptedObjectFile^[0];
      DecryptedObjectFile^[CurrentItem*3+1] := DecryptedObjectFile^[1];
      DecryptedObjectFile^[CurrentItem*3+2] := RoomNum[CurrentItem-1];
    end
    else
    begin
      msbyte := (ObjectFilePos-3) div 256;
      lsbyte := (ObjectFilePos-3) mod 256;
      DecryptedObjectFile[CurrentItem*3] := lsbyte;;
      DecryptedObjectFile[CurrentItem*3+1] := msbyte;
      DecryptedObjectFile[CurrentItem*3+2] := RoomNum[CurrentItem-1];
      for CurrentChar := 1 to Length(ItemNames[CurrentItem-1]) do
      begin
        DecryptedObjectFile[ObjectFilePos] := ord(ItemNames[CurrentItem-1][CurrentChar]);
        ObjectFilePos := ObjectFilePos + 1;
      end;
      DecryptedObjectFile[ObjectFilePos] := 0;
      ObjectFilePos := ObjectFilePos + 1;
    end;
  end;
  {end create data}


  {encrypt and write file}
  AssignFile(ObjectFile,ObjectFilename);
  {$I-}
  Rewrite(ObjectFile,1);
  {$I+}
  if IOResult <> 0 then
    ShowMessage('Error writing to file.');
  begin
    if ObjEditMain.OptionsEncryptedFile.Checked then
    begin
      GetMem(EncryptedObjectFile,ObjectFileSize);
      for curbytenum := 0 to ObjectFileSize -1 do
      begin
        curbyte := DecryptedObjectFile^[curbytenum];
        EncryptedObjectFile^[curbytenum] := curbyte XOR ord(EncryptionKey[(curbytenum mod 11) + 1]); {get decrypted byte}
      end;
      Blockwrite(ObjectFile,EncryptedObjectFile^,ObjectFileSize);
      FreeMem(EncryptedObjectFile,ObjectFileSize);
    end
    else
      Blockwrite(ObjectFile,DecryptedObjectFile^,ObjectFileSize);
    CloseFile(ObjectFile);
    SaveFile := True;
    ObjectModified := False;
  end;

{finish off}
  FreeMem(DecryptedObjectFile,ObjectFileSize);
end;

{*************************************************************}
function ReadOBJECTFile(ObjectFilename:string;var FileIsEncrypted:boolean) : TOBJECTFile;
{*************************************************************}

const NoError : byte = 0;
      InvalidFileError : byte = 1;
      ZeroItems : byte = 2;

var curbyte : byte;
    curbytestr : string;

    EncryptedObjectFile : TResource;
    DecryptedObjectFile1 : TResource;

    CurrentItem : word;    {current item name being read}
    lsbyte,msbyte : byte;
    ItemNamesStart : word;   {start of item names}
    ThisNameStart : word;  {start of current item name}
    NamePos : word; {position in current item name}
    ThisItemName : string;

    ErrorType : word;
    ObjectFile : file;
//    ObjectFileSize : word;

    NewItemNames : TStringList;
    NewRoomNum : array[0..MaxItems-1] of byte;
    NewMaxScreenObjects : byte;

    ThisOBJECTFile : TOBJECTFile;

    function GetItems(ObjectData:TResource) : boolean;
    var GetItemsErrorType : (NoError,InvalidFileError);
    begin
      GetItemsErrorType := NoError;
      CurrentItem := 0;
      lsbyte := ObjectData.Data^[0];
      msbyte := ObjectData.Data^[1];
      ItemNamesStart := msbyte * 256 + lsbyte + 3;
      NewMaxScreenObjects := ObjectData.Data^[2];
      repeat {for each name}
      begin
        lsbyte := ObjectData.Data^[3+CurrentItem*3];
        msbyte := ObjectData.Data^[3+CurrentItem*3+1];
        ThisNameStart := msbyte * 256 + lsbyte + 3;
        NewRoomNum[CurrentItem] := ObjectData.Data^[3+CurrentItem*3+2];
        NamePos := ThisNameStart;
        if NamePos > ObjectData.Size then
            GetItemsErrorType := InvalidFileError; {object name past end of file}
        if GetItemsErrorType = NoError then
        begin
          ThisItemName := '';
          repeat
            begin
          if ObjectData.Data^[NamePos] > 0 then
            begin
              ThisItemName := ThisItemName + chr(ObjectData.Data^[NamePos]);
              NamePos := Namepos + 1;
            end;
          end; until (ObjectData.Data^[NamePos] = 0) or (NamePos >= ObjectData.Size);
          NewItemNames.Add(ThisItemName);
          CurrentItem := CurrentItem + 1;
        end;
      end; until ((CurrentItem+1)*3 >= ItemNamesStart) or (CurrentItem >= MaxItems) or (GetItemsErrorType <> NoError);
      if GetItemsErrorType = NoError then GetItems := True
      else GetItems := False;
    end;

    procedure XORData(TheData:TResource);
    var curbytenum : word;
    begin
      for curbytenum := 0 to TheData.Size-1 do
        TheData.Data^[curbytenum] := TheData.Data^[curbytenum] XOR ord(EncryptionKey[(curbytenum mod 11) + 1]); {get decrypted byte}
    end;

begin
  ReadOBJECTFile.Exists := False;
  if not FileExists(ObjectFilename) then
    ShowMessage('Error! File "'+ObjectFilename+'" does not exist.')
  else
  begin
    AssignFile(ObjectFile,ObjectFilename);
    {$I-}
    Reset(ObjectFile,1);
    {$I+}
    if IOResult <> 0 then
      ShowMessage('Error! Could not open file "'+ObjectFilename+'". Make sure you are not running the game and do not have the file open in another program.')
    else
    begin
      ErrorType := NoError;
      NewItemNames := TStringList.Create;
      NewItemNames.Clear;
      DecryptedObjectFile1.Size := FileSize(ObjectFile);
      GetMem(DecryptedObjectFile1.Data,DecryptedObjectFile1.Size);
      BlockRead(ObjectFile,DecryptedObjectFile1.Data^,DecryptedObjectFile1.Size);
      CloseFile(ObjectFile);

      if FileIsEncrypted then
        XORData(DecryptedObjectFile1);
      if not GetItems(DecryptedObjectFile1) then
      begin
        XORData(DecryptedObjectFile1);
        FileIsEncrypted := not FileIsEncrypted;
        if not GetItems(DecryptedObjectFile1) then
        begin
          FileIsEncrypted := not FileIsEncrypted;
          ErrorType := InvalidFileError;
        end;
      end;
      FreeMem(DecryptedObjectFile1.Data,DecryptedObjectFile1.Size);
      if NewItemNames.Count = 0 then
      begin
        ShowMessage('Error! 0 objects in file.');
        ErrorType := ZeroItems;
      end;
      if ErrorType = InvalidFileError then ShowMessage('Error! Invalid OBJECT file.')
      else if ErrorType = NoError then
      begin
        ThisOBJECTFile.Exists := True;
        ThisOBJECTFile.MaxScreenObjects := NewMaxScreenObjects;
        ThisOBJECTFile.ItemNames := TStringList.Create;
        for CurrentItem := 0 to NewItemNames.Count-1 do
          ThisOBJECTFile.RoomNum[CurrentItem] := NewRoomNum[CurrentItem];
        ThisOBJECTFile.ItemNames.Assign(NewItemNames);
        NewItemNames.Free;
        ReadOBJECTFile := ThisOBJECTFile;
      end;
    end; {if IOResult = 0}
  end; {if not FileExists(ObjectFilename)}
end;




{*************************************************************}
function TObjEditMain.ReadFile(ObjectFilename:string) : boolean;
{*************************************************************}
var ThisOBJECTFile : TOBJECTFile;
    CurItem : word;
    FileIsEncrypted : boolean;
begin
  ReadFile := False;
  FileIsEncrypted := OptionsEncryptedFile.Checked;
  ThisOBJECTFile := ReadOBJECTFile(ObjectFilename,FileIsEncrypted);
  if ThisOBJECTFile.Exists then
  begin
    OptionsEncryptedFile.Checked := FileIsEncrypted;
    ClearFile;
    MaxScreenObjects := ThisOBJECTFile.MaxScreenObjects;
    ItemNames.Assign(ThisOBJECTFile.ItemNames);
    ThisOBJECTFile.ItemNames.Free;
    if ItemNames.Count > 0 then
    for CurItem := 0 to ItemNames.Count - 1 do
    begin
      RoomNum[CurItem] := ThisOBJECTFile.RoomNum[CurItem];
      ObjEditMain.ItemList.Items.Add(IntToStr(CurItem)+'. '+ItemNames[CurItem]);
    end;
    CurrentFilename := ObjectFilename;
    ObjEditmain.Caption := 'Object Editor - '+ObjectFilename;
    ReadFile := True;
    ObjectModified := False;
  end;
end;




                {Component events}


{*************************************************************}
procedure TObjEditMain.ReadButClick(Sender: TObject);
{*************************************************************}
begin
{  ReadFile;}
end;

{*************************************************************}
procedure TObjEditMain.FileCloseClick(Sender: TObject);
{*************************************************************}
begin
  Close;
end;

{*************************************************************}
procedure TObjEditMain.InitialiseFile(Sender: TObject);
{*************************************************************}
begin
  ItemNames := TStringList.Create;
  NewFile;

  VertBorder := ItemNameEdit.Top - ItemList.Height;
  ClientWidth := ItemList.Width;
  ClientHeight := ItemList.Height + ItemNameEdit.Height + Panel1.Height + VertBorder*2;
end;

{*************************************************************}
procedure TObjEditMain.FileNewClick(Sender: TObject);
{*************************************************************}
var SaveMessageResult : integer;
    AllowNew : boolean;
begin
  AllowNew := True;
  if ObjectModified then
  begin
    SaveMessageResult := Application.MessageBox('Do you want to save changes to the current file?','Object Editor',MB_YESNOCANCEL);
    if SaveMessageResult = IDYES then FileSaveClick(Sender)
    else if SaveMessageResult = IDCANCEL then AllowNew := False;
  end;
  if AllowNew then
    NewFile;
end;

{*************************************************************}
procedure TObjEditMain.FileOpenClick(Sender: TObject);
{*************************************************************}
var SaveMessageResult : integer;
    AllowOpen : boolean;
begin
  AllowOpen := True;
  if ObjectModified then
  begin
    SaveMessageResult := Application.MessageBox('Do you want to save changes to the current file?','Object Editor',MB_YESNOCANCEL);
    if SaveMessageResult = IDYES then FileSaveClick(Sender)
    else if SaveMessageResult = IDCANCEL then AllowOpen := False;
  end;
  if AllowOpen then
  begin
    OpenDialog.Execute;
    if Length(OpenDialog.Filename) > 0 then
    begin
      if not ReadFile(OpenDialog.Filename) then Close;
    end;
  end;
end;

{*************************************************************}
procedure TObjEditMain.FormResize(Sender: TObject);
{*************************************************************}
begin
  Panel1.Top := ClientHeight - Panel1.Height - VertBorder;
  Panel1.Left := (ClientWidth div 2) - (Panel1.Width div 2);
  ItemNameEdit.Top := ClientHeight - Panel1.Height - VertBorder*2 - ItemNameEdit.Height;
  ItemNameEdit.Width := ClientWidth;
  ItemList.Height := ItemNameEdit.Top - VertBorder;
  ItemList.Width := ClientWidth;
end;

{*************************************************************}
procedure TObjEditMain.ItemListClick(Sender: TObject);
{*************************************************************}
begin
  DeleteBut.Enabled := True;
  ItemNameEdit.Text := ItemNames[ItemList.ItemIndex];
  RoomChange.Position := RoomNum[ItemList.ItemIndex];
  RoomEdit.Text := IntToStr(RoomNum[ItemList.ItemIndex]);
end;

{*************************************************************}
procedure TObjEditMain.DeleteButClick(Sender: TObject);
{*************************************************************}
begin
  if ItemList.ItemIndex >= 0 then
  begin
    if (ItemList.ItemIndex+1<ItemNames.Count) or (ItemList.ItemIndex=0) then
    begin
      ItemNames[ItemList.ItemIndex] := '?';
      ItemList.Items[ItemList.ItemIndex] := IntToStr(ItemList.ItemIndex) + '. ?';
      ItemNameEdit.Text := '?';
    end
    else if (ItemList.ItemIndex+1=ItemNames.Count) then
    begin
      ItemNames.Delete(ItemNames.Count-1);
      ItemList.Items.Delete(ItemList.Items.Count-1);
      ItemList.ItemIndex := ItemList.Items.Count - 1;
    end;
    ObjectModified := True;
  end; {if ItemList.ItemIndex >= 0}
end;

{*************************************************************}
procedure TObjEditMain.AddButClick(Sender: TObject);
{*************************************************************}
begin
  if ItemNames.Count >= MaxItems then
    Showmessage('Maximum number of objects ('+IntToStr(MaxItems)+') already reached.')
  else
  begin
    ItemNameEdit.Text := '?';
    ItemNames.Add(ItemNameEdit.Text);
    ItemList.Items.Add(IntToStr(ItemNames.Count-1)+'. '+ItemNameEdit.Text);
    ItemList.ItemIndex := ItemList.Items.Count-1;
    if ItemNameEdit.Text = '' then ItemNameEdit.Text := '?';
    ItemNameEdit.SetFocus;
    ObjectModified := True;
  end;
end;

{*************************************************************}
procedure TObjEditMain.OptionsEncryptedFileClick(Sender: TObject);
{*************************************************************}
begin
  if OptionsEncryptedFile.Checked then OptionsEncryptedFile.Checked := False
  else OptionsEncryptedFile.Checked := True;
end;

{*************************************************************}
procedure TObjEditMain.ItemNameEditChange(Sender: TObject);
{*************************************************************}
begin
  if (ItemList.ItemIndex >= 0) then
    if ItemNameEdit.Text<>ItemNames[ItemList.ItemIndex] then
    begin
      if ItemNameEdit.Text = '' then
      begin
        ItemList.Items[ItemList.ItemIndex] := IntToStr(ItemList.ItemIndex) + '. ?';
        ItemNames[ItemList.ItemIndex] := '?';
      end
      else
      begin
        ItemList.Items[ItemList.ItemIndex] := IntToStr(ItemList.ItemIndex) + '. ' + ItemNameEdit.Text;
        ItemNames[ItemList.ItemIndex] := ItemNameEdit.Text;
      end;
      ObjectModified := True;
    end;
end;

{*************************************************************}
procedure TObjEditMain.RoomChangeClick(Sender: TObject;
  Button: TUDBtnType);
{*************************************************************}
begin
  if ItemList.ItemIndex >= 0 then
  begin
    RoomNum[ItemList.ItemIndex] := RoomChange.Position;
    RoomEdit.Text := IntToStr(RoomChange.Position);
    ObjectModified := True;
  end;
end;

{*************************************************************}
procedure TObjEditMain.RoomEditExit(Sender: TObject);
{*************************************************************}
begin
  if ItemList.ItemIndex >= 0 then
  begin
    RoomEdit.Text := IntToStr(RoomNum[ItemList.ItemIndex])
  end
  else RoomEdit.Text := '';
end;

{*************************************************************}
procedure TObjEditMain.RoomEditKeyPress(Sender: TObject; var Key: Char);
{*************************************************************}
begin
  if ItemList.ItemIndex >= 0 then
  begin
    if ord(Key) = 27 then {Esc pressed}
    RoomEdit.Text := IntToStr(RoomNum[ItemList.ItemIndex]);
  end;
end;


{*************************************************************}
procedure TObjEditMain.RoomEditChange(Sender: TObject);
{*************************************************************}
var NewRoomNum : integer;
    code : integer;
begin
  if (ItemList.ItemIndex >= 0) then
  begin
    Val(RoomEdit.Text,NewRoomNum,code);
    if (code = 0) and (NewRoomNum >= 0) and (NewRoomNum <= 255) then
    begin
      if RoomNum[ItemList.ItemIndex] <> NewRoomNum then
      begin
        RoomNum[ItemList.ItemIndex] := NewRoomNum;
        ObjectModified := True;
      end;
      RoomChange.Position := NewRoomNum;
    end; {if (code = 0) and (NewRoomNum >= 0) and (NewRoomNum <= 255)}
  end; {if (ItemList.ItemIndex >= 0)}
end;

{*************************************************************}
procedure TObjEditMain.ItemListKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
{*************************************************************}
begin
  if (ItemList.ItemIndex >= 0) and (Key = 46) then
    DeleteBut.Click;
end;

{*************************************************************}
procedure TObjEditMain.FileSaveClick(Sender: TObject);
{*************************************************************}
begin
  if CurrentFilename = '' then FileSaveAsFileClick(Sender)
  else SaveFile(CurrentFilename);
end;

{*************************************************************}
procedure TObjEditMain.FileSaveAsFileClick(Sender: TObject);
{*************************************************************}
begin
  if SaveDialog1.Execute then
  begin
    if SaveFile(SaveDialog1.Filename) then
    begin
      CurrentFilename := SaveDialog1.Filename;
      ObjEditMain.Caption := 'Object Editor - '+CurrentFilename;
    end;
  end;
end;

{*************************************************************}
procedure TObjEditMain.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
{*************************************************************}
begin
  CanClose := AskClose;
end;

procedure TObjEditMain.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
  Action := caFree;
end;

end.
