/*

Noptec Vector
Copyright (C) 2000 Erik Agsjo (erik.agsjo@noptec.com)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

//
// Test implementation of a bitmap vectorizing algorithm,
// specially developed to create Sierra AGI compatible
// vector images. (160 x 168 x 4bpp)
//
// The program reads, resizes a BMP, TGA or PCX image
// to 160 x 168, maps it to the EGA palette and writes
// the result to a Sierra AGI picture resource file.
//

#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <allegro.h>

#include "vector.h"

#define VERSION "1.4"

// This is used to shrink the generated .exe a bit:
BEGIN_GFX_DRIVER_LIST
GFX_DRIVER_VGA
END_GFX_DRIVER_LIST


// Global options as set by commandline
struct _options {
  char *infile;
  char *outfile;
  char *priofile;
  int show;
  int step;
};


static _options g_options = {
  NULL, // infile
  NULL,	// outfile
  NULL,	// priofile
  0,	// show
  0,	// step
};


// Generate a VGA palette entry triplet from a 0-255 triplet
#define VGACOL(r, g, b) {(r) >> 2, (g) >> 2, (b) >> 2}


// The 16 standard EGA colors
PALETTE EGA_PAL = {
  VGACOL(0, 0, 0),
  VGACOL(0, 0, 159),
  VGACOL(0, 159, 0),
  VGACOL(0, 159, 159),
  VGACOL(159, 0, 0),
  VGACOL(127, 0, 159),
  VGACOL(159, 79, 0),
  VGACOL(159, 159, 159),
  VGACOL(79, 79, 79),
  VGACOL(79, 79, 255),
  VGACOL(0, 255, 79),
  VGACOL(79, 255, 255),
  VGACOL(255, 79, 79),
  VGACOL(255, 79, 255),
  VGACOL(255, 255, 79),
  VGACOL(255, 255, 255)
};

void init_palette() {
  for(int i=16; i < 256; i++){
    EGA_PAL[i].r = EGA_PAL[i].g = EGA_PAL[i].b = 0;
  }
}

void usage(char *prog){
  printf(
	 "Bitmap vectorizing tool v%s, Copyright (c) Noptec / Erik Agsjo 2000\n"
	 "This is free software and comes with ABSOLUTELY NO WARRANTY; see\n"
	 "the GNU General Public License version 2 or later for details.\n\n"
	 "Reads PCX, BMP or TGA bitmaps, writes Sierra AGI picture files\n\n"
	 "Usage: %s [options] <input> <output>\n"
	 "Options:\n"
	 "\t-p <prio>\tUse given file for priority map\n"
	 "\t-v\t\tView result on screen\n"
	 "\t-w\t\tWait for keypress between different shapes (ESC skips)\n"
	 "\t-h\t\tShow this message\n",
	 VERSION, get_filename(prog));
}

void cmdline(int argc, char **argv) {
  int c;
  char *prog = argv[0];
  opterr = 0;
  while((c = getopt(argc, argv, "p:vwh")) != -1) {
    switch(c) {
    case 'v':
      g_options.show = 1;
      break;
    case 'w':
      g_options.step = 1;
      break;
    case 'p':
      g_options.priofile = strdup(optarg);
      break;
    case 'h':
    case '?':
      usage(prog);
      exit(-1);
    }
  }
  if(argv && argv[optind] && argv[optind + 1]) {
    g_options.infile = argv[optind];
    g_options.outfile = argv[optind + 1];
  }else{
    usage(prog);
    exit(-1);
  }
}

extern "C" double floor(double);
extern "C" double ceil(double);

int round(float aNumber, float dirn) {
  if(dirn < 0)
    return int((aNumber - floor(aNumber) <= 0.501)? floor(aNumber) : ceil(aNumber));
  return int((aNumber - floor(aNumber) < 0.499)? floor(aNumber) : ceil(aNumber));
}

#define pset(x, y) putpixel(bmp, x, y, c)

void sline(BITMAP *bmp, int x1, int y1, int x2, int y2, int c) {
  int height, width;
  float x, y, addX, addY;

  height = (y2 - y1);
  width = (x2 - x1);
  addX = (height == 0 ? height : float(width / abs(height)));
  addY = (width == 0 ? width : float(height / abs(width)));

  if (abs(width) > abs(height)) {
    y = y1;
    addX = (width == 0 ? 0 : (width / abs(width)));
    for (x = x1; x != x2; x += addX) {
      pset(round(x, addX), round(y, addY));
      y += addY;
    }
    pset(x2,y2);
  } else {
    x = x1;
    addY = (height == 0 ? 0 : (height / abs(height)));
    for (y = y1; y != y2; y += addY) {
      pset(round(x, addX), round(y, addY));
      x += addX;
    }
    pset(x2, y2);
  }

}

void write_agi_stream(list<PixelStripeList*> *areas, FILE *output, bool prio){
  int color = -1;

  // Disable picture draw if in prio mode and v.v.
  fputc(prio ? 0xF1 : 0xF3, output);

  for(list<PixelStripeList*>::iterator pai = areas->begin();
      pai != areas->end(); pai++) {

    list<Pixel*> *lines = (*pai)->getLines();

    if(!lines->empty()) {
      int thisColor = lines->front()->getPixel();
      if(color != thisColor) {
	color = thisColor;
	fputc(prio ? 0xF2 : 0xF0, output);
	fputc(color, output);
      }

      bool shortMode = false;
      Pixel *p0 = NULL;
      for(list<Pixel*>::iterator pi = lines->begin(); pi != lines->end(); pi++) {
	if(p0 == NULL) {
	  p0 = *pi;
	  continue;
	}
	Pixel *p1 = *pi;
	if(abs(p1->getX() - p0->getX()) > 7 ||
	   abs(p1->getY() - p0->getY()) > 7 ) {
	  shortMode = false;
	  break;
	}
      }

      p0 = NULL;

      for(list<Pixel*>::iterator pi = lines->begin(); pi != lines->end(); pi++) {
	if(shortMode) {
	  if(p0 == NULL) {
	    p0 = *pi;
	    fputc(0XF7, output);
	    fputc(p0->getX(), output);
	    fputc(p0->getY(), output);
	    continue;
	  }else if((*pi)->getX() == -1) {
	    p0 = NULL;
	    continue;
	  }
	  Pixel *p1 = *pi;
	  int xdiff = p1->getX() - p0->getX();
	  int ydiff = p1->getY() - p0->getY();
	  if(xdiff != 0 || ydiff != 0)
	    fputc( (xdiff < 0 ? 1 << 7 : 0) | (abs(xdiff) << 4) |
		   (ydiff < 0 ? 1 << 3 : 0) | abs(ydiff), output);
	  p0 = p1;
	}else{
	  if(p0 == NULL) {
	    fputc(0xF6, output);
	  }else if((*pi)->getX() == -1) {
	    pi++;
	    if(pi == lines->end()) break;
	    fputc(0xF6, output);
	  }
	  p0 = *pi;
	  fputc(p0->getX(), output);
	  fputc(p0->getY(), output);
	}
      }
    }
  }

  color = -1;

  for(list<PixelStripeList*>::iterator pai = areas->begin();
      pai != areas->end(); pai++) {

    list<Pixel*> *fills = (*pai)->getFills();
    if(fills->empty()) continue;
    int thisColor = fills->front()->getPixel();
    if(color != thisColor) {
      color = thisColor;
      fputc(prio ? 0xF2 : 0xF0, output);
      fputc(color, output);
      fputc(0xF8, output);
    }

    for(list<Pixel*>::iterator pi = fills->begin(); pi != fills->end(); pi++) {
      fputc((*pi)->getX(), output);
      fputc((*pi)->getY(), output);
    }
  }
}

BITMAP *fix_bitmap(BITMAP *rawinput, PALETTE rawpalette){
  //
  // PLEASE REWRITE ME!
  //
  // The following (up to "fix black pixels") should be rewritten
  // for better structure. It should be divided between the tasks
  // cropping/resizing and colormapping.
  //

  // remap loaded 8-bit images to the EGA palette
  // NOTE: hicolor images will be automatically remapped
  // during blitting later.
  if(bitmap_color_depth(rawinput) == 8) {
    for(int y = 0; y < rawinput->h; y++)
      for(int x = 0; x < rawinput->w; x++) {
	RGB pcol = rawpalette[getpixel(rawinput, x, y)];
	putpixel(rawinput, x, y,
		 bestfit_color(EGA_PAL, pcol.r, pcol.g, pcol.b));
      }
  }

  // Create and clear the source bitmap
  // This is the bitmap that will be scanned later
  BITMAP *source = create_bitmap_ex(8, 160, 168);
  clear(source);

  // Do resize and blit result to the source bitmap

  if(bitmap_color_depth(rawinput) != 8) {
    BITMAP *resizebimp = create_bitmap_ex(8, rawinput->w, rawinput->h);
    blit(rawinput, resizebimp, 0, 0, 0, 0, rawinput->w, rawinput->h);
    destroy_bitmap(rawinput);
    rawinput = resizebimp;
  }

  stretch_blit(rawinput, source,
	       0, 0, rawinput->w, rawinput->h,
	       0, 0, source->w, source->h);
  destroy_bitmap(rawinput);

  // Fix black pixels > 15
  for(int y = 0; y < source->h; y++)
    for(int x = 0; x < source->w; x++) {
      if(getpixel(source, x, y) > 15) {
	putpixel(source, x, y, 0);
      }
    }

  return source;
}

int main(char argc, char **argv) {
  FILE *sink;

  // help silly user
  if(argc < 2) {
    usage(argv[0]);
    exit(-1);
  }

  // parse commandline getopt style
  cmdline(argc, argv);

  // initialize allegro, fill end of EGA_PAL with black
  allegro_init();
  set_color_depth(8);
  set_color_conversion(COLORCONV_NONE);
  init_palette();
  select_palette(EGA_PAL);
  install_keyboard();

  // read input image
  PALETTE inputpalette;
  BITMAP *rawinput = load_bitmap(g_options.infile, inputpalette);
  if(rawinput == NULL){
    fprintf(stderr, "Failed to load bitmap \"%s\"\n", g_options.infile);
    exit(-1);
  }

  BITMAP *fixedinput = fix_bitmap(rawinput, inputpalette);

  // read prio file
  PALETTE priopalette;
  BITMAP *rawprio;
  BITMAP *fixedprio = NULL;
  if(g_options.priofile != NULL){
    rawprio = load_bitmap(g_options.priofile, priopalette);
    if(rawprio == NULL){
      fprintf(stderr, "Failed to load prio bitmap \"%s\"\n",
	      g_options.priofile);
      exit(-1);
    }
    fixedprio = fix_bitmap(rawprio, priopalette);
  }

  // Open output file
  sink = fopen(g_options.outfile, "wb");
  if(sink == NULL) {
    fprintf(stderr, "Failed to create output \"%s\"\n", g_options.outfile);
    exit(-1);
  }

  // create a Vectorizer for the image, scan and get the result
  Vectorizer pic(fixedinput, 15);
  pic.scan();
  list<PixelStripeList*> *picareas = pic.getAreas();
  write_agi_stream(picareas, sink, false);

  if(fixedprio != NULL){
    Vectorizer prio(fixedprio, 4);
    prio.scan();
    list<PixelStripeList*> *prioareas = prio.getAreas();
    write_agi_stream(prioareas, sink, true);
  }

  // write "end of PIC" to output stream
  fputc(0xFF, sink);

  if(g_options.show) {
    set_gfx_mode(GFX_VGA, 320, 200, 320, 200);
    clear_to_color(screen, 0);
    BITMAP *view = create_bitmap_ex(8, 160, 168);
    clear_to_color(view, 15);
    BITMAP *text = create_sub_bitmap(screen, 0, 168, 320, 32);

    if(g_options.step) {
      textout(text, font, "Press any key to show next step,", 1, 1, 15);
      textout(text, font, "<esc> skips all.", 1, 10, 15);
    }

    for(list<PixelStripeList*>::iterator pai = picareas->begin();
	pai != picareas->end(); pai++) {

      list<Pixel*> *lines = (*pai)->getLines();

      Pixel *p0 = NULL;
      Pixel *p1 = NULL;
      int dx, dy;
      for(list<Pixel*>::iterator pi = lines->begin(); pi != lines->end(); pi++) {
	if(p0 == NULL) {
	  p0 = *pi;
	  putpixel(view, p0->getX(), p0->getY(), p0->getPixel());
	  pi++;
	  if(pi == lines->end()) break;
	}

	p1 = *pi;

	if(p1->getX() == -1) {
	  p0 = NULL;
	  continue;
	}

	dx = p1->getX();
	dy = p1->getY();
	sline(view, p0->getX(), p0->getY(), p1->getX(), p1->getY(), p0->getPixel());
	p0 = p1;
      }
      if(p0 != NULL && p1 != NULL && p1->getX() != -1 && p0->getX() != -1)
	sline(view, p0->getX(), p0->getY(), p1->getX(), p1->getY(), p0->getPixel());

      if(g_options.step) {
	stretch_blit(view, screen, 0, 0, 160, 168, 0, 0, 320, 168);
	if(readkey() >> 8 == KEY_ESC) g_options.step = false;
      }
    }

    for(list<PixelStripeList*>::iterator pai = picareas->begin();
	pai != picareas->end(); pai++) {
      list<Pixel*> *fills = (*pai)->getFills();
      for(list<Pixel*>::iterator pi = fills->begin(); pi != fills->end(); pi++) {
	floodfill(view, (*pi)->getX(), (*pi)->getY(), (*pi)->getPixel());
      }
      if(g_options.step) {
	stretch_blit(view, screen, 0, 0, 160, 168, 0, 0, 320, 168);
	if(readkey() >> 8 == KEY_ESC) g_options.step = false;
      }
    }
    stretch_blit(view, screen, 0, 0, 160, 168, 0, 0, 320, 168);
    clear(text);
    textout(text, font, "Press any key to exit.", 1, 1, 15);
    readkey();
    set_gfx_mode(GFX_TEXT, 25, 80, 25, 80);
  }
  fclose(sink);
}


