/*

Noptec Vector
Copyright (C) 2000 Erik Agsjo (erik.agsjo@noptec.com)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#ifndef VECTOR_H
#define VECTOR_H

#include <list>

class PixelStripeList;

//
// PixelMap class
//  Defines a map to keep track of which PixelStripeList a
//  certain pixel belongs to.
//
class PixelMap {
 private:
  int width;
  PixelStripeList **map;
 public:
  PixelMap(int, int);
  ~PixelMap(void);
  void setList(int, int, PixelStripeList*);
  PixelStripeList *getList(int, int);
};

class PixelStripe;

//
// Pixel class
//  Has ordinary pixel properties (x, y and pixel (the color value))
//  as well as the PixelStripe that it belongs to.
//
class Pixel {
 private:
  int x, y, pixel;
  PixelStripe *stripe;
 public:
  Pixel(int, int, int, PixelStripe*);
  ~Pixel(void);
  int getX(void);
  int getY(void);
  int getPixel(void);
  PixelStripe *getStripe(void);
  double distance(Pixel*);
};

//
// PixelStripe class
//  A PixelStripe is a horizontal continuous line of monocolored
//  pixels.
//
//  None of the overlap related stuff is currently used.
//
class PixelStripe {
 private:
  int row;
  int startcol, endcol;
  int pixel;
  int overlap;
 public:
  PixelStripe(int, int, int);
  ~PixelStripe(void);
  void range(int);
  int getStartCol(void);
  int getEndCol(void);
  int getRow(void);
  int getPixel(void);
  void increaseOverlap(void);
  int getOverlap(void);
  void shrink(void);
  int size(void);
  static bool lessThanOrEqual(PixelStripe*, PixelStripe*);
};

//
// PixelStripeList class
//  A PixelStripeList is a list of PixelStripe objects forming
//  a monocolored area of the scanned image.
//
//  This is the workhorse of the conversion process
//
class PixelStripeList {
 private:
  list<PixelStripe*> stripes;
  PixelMap *map;
  BITMAP *bitmap;
  bool filled;
  list<Pixel*> *fills;
  list<Pixel*> *lines;
  int color;
  int fill;
 public:
  PixelStripeList(PixelMap*, BITMAP*, int);
  ~PixelStripeList(void);
  void add(PixelStripe*);
  void merge(PixelStripeList*);
  void report(void);
  void convert(void);
  void setFilled(bool);
  bool getFilled(void);
  list<Pixel*> *getLines(void);
  list<Pixel*> *getFills(void);
  list<PixelStripe*> &getStripes(void);
  void addFill(int, int, int);
  void optimize(void);
  static bool PixelStripeList::colorSort(PixelStripeList *, PixelStripeList *);
  PixelMap *getMap(void);
};

//
// PixelAreas class
//  Contains all the PixelStripeList object that make up
//  all the different areas of the image.
//
class PixelAreas {
 private:
  list<PixelStripeList*> *areas;
  int fillcolor;
 public:
  PixelAreas(int);
  ~PixelAreas(void);
  void add(PixelStripeList*);
  void remove(PixelStripeList*);
  void report(void);
  void convert(void);
  list<PixelStripeList*> *getAreas(void);
};

class Vectorizer {
 private:
  BITMAP *source;
  PixelMap* map;
  PixelAreas* lol;
  int fill;

  void scanRow(int);
 public:
  Vectorizer(BITMAP *, int);
  ~Vectorizer(void);
  void scan();
  void report(void);
  list<PixelStripeList*> *getAreas(void);
};

#endif
