
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                         *
 * Module Name : BCMUTIL.C                                                 *
 *                                                                         *
 * Description : This is the software interface for the low level DOS      *
 *               functions.
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * BICOM (R) BCMUTIL.C  Version 1.00                                       *
 * Copyright (c) BICOM 1990, 1991     All rights reserved.                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                          REVISION HISTORY                               *
 *-------------------------------------------------------------------------*
 *  Date    New Rev.                    Changes                            *
 *                                                                         *
 * 11/24/92  1.01     Revised DosCreateFile to return handle               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#include "bcmlib.h"


/*
   Variable Naming
        p-   means pointer
        h-   means handle
        Reg  means register variable

   ex: phFile : pointer to a file handle
       pRegCX : pointer to a register CX variable
 */



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm_OpenFile(pFileName, Mode)
 * DESCRIPTION : Open a file using the DOS file open function.
 *  PARAMETERS : pFileName : pointer to name of file as an ASCIIZ string
 *               Mode      : type of open
 *                             = 0 (READ)   : read mode
 *                             = 1 (RDWR)   : read write mode
 *                             = 2 (CREATE) : create mode
 *                             = 3 (APPEND) : append mode
 *     RETURNS : = DOS file handle if successful.
 *               = 0, if unsuccessful, and DOS error is returned in
 *                 public variable BCM_DosError.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm_OpenFile(unsigned char *pFileName,
                 unsigned int Mode)
{
   unsigned int hFile;
   unsigned int Method;
   unsigned int HighPos, LowPos;

   switch (Mode)
   {
      case READ:    /* read mode */
         if (BCM_DosError = DosOpenFile(pFileName, &hFile, 0))
            return(0);
         
         return((int)hFile);

      case RDWR:    /* read write mode */
         if (BCM_DosError = DosOpenFile(pFileName, &hFile, 2))
            return(0);
         
         return((int)hFile);

      case CREATE:  /* create mode */
         if (BCM_DosError = DosCreateFile(pFileName, &hFile, 0))
            return(0);
         
         return((int)hFile);

      case APPEND:  /* append mode */
         if (BCM_DosError = DosOpenFile(pFileName, &hFile, 2))
            return(0);
         
         HighPos = 0;
         LowPos  = 0;
         Method  = 2;
         if (!(BCM_DosError = DosMoveFilePtr(hFile, Method, &HighPos, &LowPos)))
            return(0);
         
         return((int)hFile);
   }
   return(0);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   FUNCTION : Bcm_CloseFile(hFile)
 *DESCRIPTION : Close a file using the DOS file close function.
 * PARAMETERS : hFile : DOS file handle.
 *    RETURNS : Error code if unsuccessful.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int Bcm_CloseFile(unsigned int hFile)
{
   return((int)DosCloseFile(hFile));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : Bcm_MoveFilePtr(hFile, RelPos, Method)
 * DESCRIPTION : Move file pointer using the DOS file seek function.
 *  PARAMETERS : hFile  : DOS file handle
 *               RelPos : relative file position
 *               Method : seek mode
 *                         = 0 (MP_BEG) : beginning of file
 *                         = 1 (MP_CUR) : current file position
 *                         = 2 (MP_END) : end of file
 *     RETURNS : = actual file position if successful.
 *               = -1L if unsuccessful (see BCM_DosError for error code).
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
long int Bcm_MoveFilePtr(unsigned int hFile,
                          unsigned long int RelPos,
                          unsigned int Method)
{
   *pRegCX = (int)((RelPos>>16)&0xFFFF);
   *pRegDX = (int)(RelPos&0xFFFF);

   if ((ReturnCode = DosMoveFilePtr(hFile, Method, pRegCX, pRegDX)))
   {
      return((long int)-1);
   }

   return((long int)((long int)*pRegDX<<16) + (long int)(*pRegCX));
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : DosOpenFile(pFileName, phFile, Mode)
 * DESCRIPTION : Opens an existing file or device. The pathname and filename
 *               are entered in the form of an ASCIIZ string. The file-access
 *               and sharing codes are specified as follows.
 *                 MODE DEFINITION:
 *                   bit : 7 6 5 4 3 2 1 0 
 *                         i - - - - - - -   Inheritence Flag
 *                         - s s s - - - -   Sharing Mode
 *                         - - - - r - - -   Reserved
 *                         - - - - - a a a   Access Code
 * 
 *                 Access Code:    0 = Read  Only
 *                                 1 = Write Only
 *                                 2 = Read/Write
 * 
 *                 Reserved:       (set to ZERO)
 * 
 *                 Sharing Mode:   0 = Compatible 
 *                                 1 = Deny Read/Write
 *                                 2 = Deny Write
 *                                 3 = Deny Read
 *                                 4 = Deny None Mode 
 * 
 *                 Note: This field only used if SHARE utility is installed
 *                       or running on a network.
 * 
 *                 Inheritance:    1 = Child process can use same handle as parent
 *                                 0 = Child process must open and get new handle
 *  PARAMETERS : pFileName = pointer to the ASCIIZ file name.
 *               phFile    = filehandle returned by DOS.
 *               Mode      = mode definition for opened file. (see above)
 *      RETURN : 0x00 = Ok
 *               0x02 = File Not Found
 *               0x03 = Path Not Found
 *               0x04 = No Handles Available
 *               0x05 = Access Denied
 *               0x0C = Invalid Access Code
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int DosOpenFile(unsigned char *pFileName,
                unsigned int  *phFile,
                unsigned int   Mode)
{
   unsigned int ReturnCode;

   ireg.h.ah = F_OPNF;
   ireg.h.al = (unsigned char)Mode;
   ireg.x.dx = FindOffset(pFileName);
   sr.ds     = FindSegment(pFileName);

   ReturnCode = intdosx(&ireg, &oreg, &sr);

   if (oreg.x.cflag)
      return(oreg.x.ax);
   
   *phFile = oreg.x.ax;
   return((int)0);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : DosCloseFile(hFile)
 * DESCRIPTION : Close a file using low level DOS functions.
 *  PARAMETERS : hFile : DOS filehandle.
 *     RETURNS : zero if successful.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int DosCloseFile(unsigned int hFile)
{
   unsigned int ReturnCode;

   ireg.h.ah = F_CLSF;
   ireg.x.bx = hFile;

   ReturnCode = intdosx(&ireg,&oreg,&sr);
   
   if (oreg.x.cflag)
      return(oreg.x.ax);
   
   return((int)0);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : DosMoveFilePtr(hFile, Method, pRegCX, pRegDX)
 * DESCRIPTION : Move file pointer to given position using the method indicated.
 *  PARAMETERS : hFile  : DOS filehandle of file to be operated on.
 *               Method : method in which to move file pointer.
 *               pRegCX : pointer to high order value of requested position.
 *               pRegDX : pointer to low order value of requested position.
 *      OUTPUT : pRegDX : points to high order value of new position.
 *               pRegCX : points to low order value of new position.
 *     RETURNS : DOS filehandle.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int DosMoveFilePtr(unsigned int  hFile,
                   unsigned int  Method,
                   unsigned int *pRegCX,
                   unsigned int *pRegDX)
{
   unsigned int ReturnCode;

   ireg.h.ah = F_SEEK;
   ireg.h.al = (unsigned char)Method;
   ireg.x.bx = hFile;
   ireg.x.cx = *pRegCX;
   ireg.x.dx = *pRegDX;

   ReturnCode = intdosx(&ireg, &oreg, &sr);

   if (oreg.x.cflag) 
      return(oreg.x.ax);
   
   *pRegCX = oreg.x.ax;
   *pRegDX = oreg.x.dx;

   return((int)0);
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    FUNCTION : DosCreateFile(pFileName, phFile, Mode)
 * DESCRIPTION : Create a file using low level DOS functions.
 *  PARAMETERS : pFileName : pointer to the ASCIIZ file name.
 *               phFile    : filehandle returned by DOS.
 *               Mode      : mode in which file is to be opened.
 *                 MODE DEFINITION:
 *                    bit : 7 6 5 4 3 2 1 0 
 *                          - - - - - - - r   Read Only
 *                          - - - - - - h -   Hidden 
 *                          - - - - - s - -   System
 *                          - - - - v - - -   Volume Label
 *                          - - - d - - - -   Subdirectory
 *                          - - a - - - - -   Archive
 *                          u u - - - - - -   Unused 
 *     RETURNS : DOS filehandle.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int DosCreateFile(unsigned char *pFileName,
                  unsigned int  *phFile,
                  unsigned int   Mode)
{
   unsigned int ReturnCode;

   ireg.h.ah = F_CREAT;
   ireg.x.cx = Mode;
   ireg.x.dx = FindOffset(pFileName);
   sr.ds     = FindSegment(pFileName);

   ReturnCode = intdosx(&ireg, &oreg, &sr);

   
   if (oreg.x.cflag)
      return(oreg.x.ax);
   
   *phFile = oreg.x.ax;

   return((int)0);
}



