/*
 * event.c -- scheduler for rmdp
 *
 * This file is part of
 *
 * rmdp -- Reliable Multicast data Distribution Protocol
 * 
 * (C) 1996-1998 Luigi Rizzo and Lorenzo Vicisano
 *     (luigi@iet.unipi.it, vicisano@cs.ucl.ac.uk)
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Luigi Rizzo,
 *      Lorenzo Vicisano and other contributors.
 * 4. Neither the name of the Authors nor the names of other contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
 * IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */

/*** 
 ***  

These routines implement an event loop. First, the event list must
be created using createevlist(). Timeouts can be registered with
settimer(). They always expire, and there is no way to remove them.
i/o descriptors can be inserted/deleted using insertchan() and
deletechan().

One function per timeout, and one function per channel, can be called.

 ***
 ***/

#include "event.h"
#include "time.h"

#undef TEST
#undef ONE_GO		/* define to make getevent return after each event */


static TIMIT *timit_free_list = NULL;
static int timit_free_count = 0 ;

void
free_timit(TIMIT *p)
{
    p->next = timit_free_list ;
    timit_free_list = p;
    bzero(p, sizeof (TIMIT) );
    timit_free_count++ ;
}

TIMIT *
new_timit()
{
    TIMIT *p;

    if (timit_free_list) {
	p = timit_free_list ;
	timit_free_list = p->next ;
	timit_free_count-- ;
    } else {
        p = (TIMIT *)calloc(1, sizeof(TIMIT)) ;
	if (p == NULL) {
	    perror("new_timit: calloc");
	    exit(1);
	}
    }
    return p ;
}

EVLIST *
createevlist()
{
    int i;
    EVLIST *e;

    if (NULL==(e=(EVLIST *)calloc(1, sizeof(EVLIST)))) {
	perror("createevlist: calloc");
	exit(1);
    }

    for (i=0; i<MAX_CHAN; i++) {
	e->io_chan[i]= -1;
    }
    e->io_ch_num = 0;
    e->timevent = NULL;
    FD_ZERO(&(e->in_s));
    FD_ZERO(&(e->out_s));
    FD_ZERO(&(e->err_s));
    timestamp_t(&(e->lasttime));
    e->late = 0;
    return(e);
}

int
getevent(EVLIST *e, int blocking)
{
    struct timeval thistime, interval, *timp;
    ui32 past;
    TIMIT *tim;
    int id,
	max_chan,
	i, j, ret;

again:
    id = EV_NO_PENDING ;
    /* see how much time has past since last update */
    timestamp_t(&thistime);
    past = diff_t(thistime, e->lasttime) ;

    DEB(fprintf(stderr, "# %u usec past\n", past));

    tim = e->timevent;
    DEB( { TIMIT *ti = e->timevent;
	    fprintf(stderr, "X");
	    for ( ; ti != NULL ; ti = ti->next )
		fprintf(stderr, "--->[%u] (%u)|", ti->delta, ti->id);
	    fprintf(stderr, "\n");
	} );

    if (tim != NULL) {
	if (tim->delta <= past) {
	    e->late = past - tim->delta;  /* how late was this event ? */
	    if (e->late > 1000) { /* more than 1ms late... */
		DEB(fprintf(stderr, "--- we are %u usec late with ev. %d\n",
			e->late, tim->id));
	    }
	    e->timevent = tim->next;	/* first adj the event list
				       (which could be acceded) */
	    if (tim->t_f)
		tim->t_f(tim->t_p);	/* execute the action */

	    past -= tim->delta;		/* update virtual time */
#ifdef NODEF /* we don't free the arguments */
	    free(tim->t_p);
#endif
	    id = tim->id;
	    free_timit(tim);
	    tim = e->timevent;
	    e->lasttime = thistime ;
#ifdef ONE_GO
	    return(id);
#endif
	    goto again;
	} else
	    e->late = past ;
    }

    /* set timeout to next event */
    timp = &interval;
    if (e->timevent != NULL) {
	ui32 next = e->timevent->delta - e->late;
	interval.tv_sec = next/1000000;
	interval.tv_usec = next%1000000;
	DEB(fprintf(stderr, "# timeout in %u usec\n", next));
    } else {
	interval = e->default_timeout ;
	if (blocking && e->default_timeout.tv_sec==0
		&& e->default_timeout.tv_usec==0) timp = NULL;
    }

    FD_ZERO(&(e->in_s));
    max_chan =  -1 ;
    for (j=0; j< e->io_ch_num; j++)
	if ( (i = e->io_chan[j]) >= 0 ) {
	    if (i > max_chan) max_chan = i ;
	    FD_SET( i, &(e->in_s));
	    DEB(fprintf(stderr, "selecting with %d\n", i ));
	}
    if (e->io_ch_num) {
	ret=select(1+max_chan, &(e->in_s), &(e->out_s), &(e->err_s), timp);
	DEB(fprintf(stderr, "select returning...\n"))
	if (ret < 0) {		/* signal... */
	    fprintf(stderr,
		"getevent SIGNAL during poll, shouldn't happen...\n");
	    return EV_GOTSIGNAL ;
	} else if (ret == 0) {	/* timeout */
	    DEB(fprintf(stderr,"timeout, ev %x blocking %d\n",
		e->timevent, blocking));
	    if (e->timevent != NULL)
		goto again;
	    else
		return EV_DEFTIMEOUT ;
	} else {			/* descriptor ready */
	    for (i=0; !(e->io_chan[i]>=0
		&& FD_ISSET(e->io_chan[i], &(e->in_s)))
		&& i<MAX_CHAN; i++);
	    assert((e->io_chan[i]>=0 && FD_ISSET(e->io_chan[i], &(e->in_s))));
	    e->selected = e->io_chan[i];
	    if (e->io_f[i])
		e->io_f[i](e->io_p[i]);
	    id = e->io_id[i];
	}
    } else if (id == EV_NO_PENDING)
	id = EV_TERMINATE;

    return(id);
}

void
set_default_timeout(EVLIST *e, struct timeval *tv)
{
    if (tv)
	e->default_timeout = *tv;
    else
	e->default_timeout.tv_sec = e->default_timeout.tv_usec = 0;
}
	
void
insertchan(EVLIST *e, int chan, int id, int (*f)(void *), void *p)
{
    int i;

    DEB(fprintf(stderr, "insertchan: inserting chan %d\n",
	chan))
    /***
     *** check this...
     ***/
    for (i=0; e->io_chan[i]!=chan && e->io_chan[i]>=0 && i<MAX_CHAN; i++);
    assert(e->io_chan[i]<0 || e->io_chan[i]==chan);
    if (e->io_chan[i] != chan)
	(e->io_ch_num)++;
    e->io_chan[i]=chan;
    e->io_id[i]=id;
    e->io_f[i] = f;
    e->io_p[i] = p;
}

void
deletechan(EVLIST *e, int chan)
{
    int i;

    DEB(fprintf(stderr, "deletechan: deleting chan %d\n",
	chan))
    for (i=0; e->io_chan[i]!=chan && i<MAX_CHAN; i++);
    assert(e->io_chan[i]==chan);
    e->io_chan[i]=-1;
    (e->io_ch_num)--;
    
}

int
showselected (EVLIST *e)
{
    return(e->selected);
}

/*
 * arrange for a timer event to happen after delta usec
 *
 * `id' will be returned by getevent after the event execution,
 * avoid using id = EV_NO_PENDING .
 *
 * If `real' is set, the event will occurr delta usec
 * after the actual calling time. Otherwise, it is assumed that the
 * function has been called by the scheduler, and the event will
 * occur exactly delta usec. after the previous schedule
 * for the event. That allows not to accumulate errors.
 */
void
settimer(EVLIST *e, int id, ui32 delta, CALLBACK f, void *p, int real)
{
    TIMIT *a, *pr = NULL, *t = e->timevent;
    struct timeval thistime;

    a = new_timit() ;
    a->t_f = f;
    a->t_p = p;
    a->next = NULL;
    a->id = id;

    if (delta < e->late) {
	DEB(fprintf(stderr, "ARGH we are scheduling event"
		" %d %u usec in the past!!!\n",
		id, e->late-delta));
#ifdef NO
	delta = 0;
    } else {
	delta -= e->late;
#endif
    }

    if (real) {
	timestamp_t(&thistime);
	delta += diff_t( thistime, e->lasttime) ;
    }

    if (e->timevent == NULL) {
	a->delta = delta;
	e->timevent = a;
	return;
    }
	
    while(t!= NULL && delta > t->delta) {
	delta -= t->delta;
	pr = t;
	t = t->next;
    }

    a->delta = delta;
    a->next = t;
    if (t != NULL)
	(t->delta) -= delta;
    if (t == e->timevent) {
	e->timevent = a;
    } else {
	pr->next = a;
    }
}


#ifdef TEST

int my_f(void *p)
{
   struct timeval time;
   timestamp_t(&time);

   printf("%d at %u\n", *(int *)p, time.tv_usec+(time.tv_sec%1000)*1000000);
   *(int *)p = *(int *)p +1;
   if (*(int *)p < 1000)
	settimer(e, (*(int *)p)+1, 10000, my_f, p, 0);
}

/*
 * arrange for a timer event to happen every delta msec
 */
int main()
{
  int cont =0;
  EVLIST *e = createevlist();

  settimer(e, 1, 10000, my_f, (void *)&cont, 1);
  while (getevent(e, 0)!=EV_NO_PENDING)
  	;
}

#endif /* TEST */
