/*
 * net.c -- socket I/O -- This is a stripped down version of the
 * net.cc module from vic, with the following (original) copyright:
 *
 * Copyright (c) 1993-1994 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and the Network Research Group at
 *      Lawrence Berkeley Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "rmdp.h"

/***
 ***
   We need the following routines:
   - open a multicast socket for send.
       (receiver: send request; sender: send data)
   - open a multicast socket for receive
       (receiver: get request; sender: get data)
   - open a unicast socket for send
       (sender: send reply)
   - open a unicast socket for receive
       (receiver: get reply)
 ***
 ***/

void
nonblock(int fd)
{
#ifdef WIN32
    u_long flag = 1;

    if (ioctlsocket(fd, FIONBIO, &flag) == -1) {
	fprintf(stderr, "ioctlsocket: FIONBIO: %lu\n", GetLastError());
	exit(1);
    }
#else 
    int flags = fcntl(fd, F_GETFL, 0);
#if defined(hpux) || defined(__hpux)
    flags |= O_NONBLOCK;
#else
    flags |= O_NONBLOCK|O_NDELAY;
#endif
    if (fcntl(fd, F_SETFL, flags) == -1) {
	perror("fcntl: F_SETFL");
    }
#endif
}    

int
openssock(n32 addr, n16 port, int ttl)
{
    int fd;
    struct sockaddr_in sin;

    fprintf(stderr,"--- openssock addr %s port %d\n",
	inet_ntoa(*(struct in_addr *)&addr), ntohs(port) );
    fd = socket(AF_INET, SOCK_DGRAM, 0);
    if (fd < 0) {
	perror("socket");
	exit(1);
    }
    nonblock(fd);
#ifdef WIN32
    memset((char *)&sin, 0, sizeof(sin));
    sin.sin_family = AF_INET;
    sin.sin_port = 0;
    sin.sin_addr.s_addr = INADDR_ANY;
    if (bind(fd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
	perror("bind");
	exit(1);
    }       
#endif          
    memset((char *)&sin, 0, sizeof(sin));
    sin.sin_family = AF_INET;
    sin.sin_port = port;
    sin.sin_addr.s_addr = addr;
    if (connect(fd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
	perror("connect");
	exit(1);
    }
    if (IN_CLASSD(ntohl(addr))) {
#ifdef IP_ADD_MEMBERSHIP
#ifdef WIN32
	u_int t;
#else
	u_char t;
#endif

	/*** set the multicast TTL ***/
	t = (ttl > 255) ? 255 : (ttl < 0) ? 0 : ttl;
	if (setsockopt(fd, IPPROTO_IP, IP_MULTICAST_TTL,
		   (char*)&t, sizeof(t)) < 0) {
	    perror("IP_MULTICAST_TTL");
	    exit(1);
	}
#else
	fprintf(stderr,
	    "not compiled with support for IP multicast\n"
	    "you must specify a unicast destination\n");
	exit(1);
#endif
    }
    return fd ;
}

int
openrsock(n32 addr, n16 *port)
{
    int fd;
    struct sockaddr_in sin;
    int on = 1;
                
fprintf(stderr,"--- openrsock addr %s port %d\n",
    inet_ntoa(*(struct in_addr *)&addr), ntohs(*port) );
    fd = socket(AF_INET, SOCK_DGRAM, 0);
    if (fd < 0) {
	perror("socket");
	exit(1);
    }
    nonblock(fd);
    if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR,(char *)&on, sizeof(on)) < 0) {
	perror("SO_REUSEADDR");
    }
#ifdef SO_REUSEPORT
    on = 1;
    if (setsockopt(fd, SOL_SOCKET, SO_REUSEPORT, (char *)&on, sizeof(on)) < 0) {
	perror("SO_REUSEPORT");
	exit(1);
    }
#endif
    memset((char *)&sin, 0, sizeof(sin));
    sin.sin_family = AF_INET;
    sin.sin_port = *port;
#ifdef IP_ADD_MEMBERSHIP
    if (IN_CLASSD(ntohl(addr))) {
	struct ip_mreq mr;

	/*
	 * Try to bind the multicast address as the socket
	 * dest address.  On many systems this won't work
	 * so fall back to a destination of INADDR_ANY if
	 * the first bind fails.
	 */
#if 0
	sin.sin_addr.s_addr = addr;
	if (bind(fd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
	    fprintf(stderr, "first bind failed, try INADDR_ANY\n");
	    sin.sin_addr.s_addr = INADDR_ANY;
	    if (bind(fd, (struct sockaddr*)&sin, sizeof(sin)) < 0) {
		perror("bind");
		exit(1);
	    }
	}
#else
	sin.sin_addr.s_addr = INADDR_ANY;
	if (bind(fd, (struct sockaddr*)&sin, sizeof(sin)) < 0) {
	    perror("bind");
	    exit(1);
	}
#endif
	/*
	 * XXX This is bogus multicast setup that really
	 * shouldn't have to be done (group membership should be
	 * implicit in the IP class D address, route should contain
	 * ttl & no loopback flag, etc.).  Steve Deering has promised
	 * to fix this for the 4.4bsd release.  We're all waiting
	 * with bated breath.
	 */

	mr.imr_multiaddr.s_addr = addr;
	mr.imr_interface.s_addr = INADDR_ANY;
	if (setsockopt(fd, IPPROTO_IP, IP_ADD_MEMBERSHIP,
		       (char *)&mr, sizeof(mr)) < 0) {
	    perror("IP_ADD_MEMBERSHIP");
	    return (fd);
	}
    } else
#endif
    {
	/*
	 * bind the local port to this socket.  If that
	 * fails, another app probably has the addresses bound so
	 * just exit.
	 */
	sin.sin_addr.s_addr = addr ;
	if (bind(fd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		perror("bind");
	    exit(1);
	}
    }
    if (*port == 0) {
	memset((char *)&sin, 0, sizeof(sin));
	on = sizeof(sin);
        getsockname(fd, (struct sockaddr *)&sin, &on) ;
	*port = sin.sin_port ;
    }
    fprintf(stderr, "--- port bound to %s : %d\n",
	inet_ntoa(sin.sin_addr), ntohs(*port) );
    return fd ;
}

void
localname(struct sockaddr_in * p, int ssock_)
{
    int len = sizeof(*p);   
    memset((char *)p, 0, sizeof(*p));
    p->sin_family = AF_INET;
    if (getsockname(ssock_, (struct sockaddr *)p, &len) < 0) {
	perror("getsockname"); 
	p->sin_addr.s_addr = 0;
	p->sin_port = 0;
    }
    if (p->sin_addr.s_addr == 0) { 
	char hostname[80];
	struct hostent *hp;

	if (gethostname(hostname, sizeof(hostname)) >= 0) {
	    if ((hp = gethostbyname(hostname)) >= 0) {
		p->sin_addr.s_addr = ((struct in_addr *)hp->h_addr)->s_addr;
	    }
	}
    }
    fprintf(stderr, "localname addr is %s\n",
	inet_ntoa(p->sin_addr) );
}
/*** end of file ***/
