/*
 (c) 2014 Dmitry Grinberg - http://dmitry.gr - me@dmitry.gr
  Free for non-commercial use only
*/

#include "bool.h"
#include <avr/wdt.h>
#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <util/delay.h>
#include <stdint.h>
#include <avr/sleep.h>
#include "ufat.h"
#include "sd.h"


#define MAX_FILE_FRAGMENTS		8

static uint32_t gFragStart[MAX_FILE_FRAGMENTS] = {0,};
static uint32_t gFragLen[MAX_FILE_FRAGMENTS] = {0,};
static volatile uint8_t gSamples[256];
static volatile uint8_t gSampleReadPos = 0;

Boolean ufatExtRead(uint32_t sector, uint16_t offset, uint8_t len, uint8_t* buf)
{

	static uint32_t curSec = 0xFFFFFFFFUL;
	static uint16_t curPos = 0;

	if(sector != curSec || offset < curPos){

		if(curSec != 0xFFFFFFFFUL){

			while(curPos++ != 512) sdSpiByteFast(0xFF);	//fast forward to sector end
			sdSecReadStop();
			curSec = 0xFFFFFFFFUL;
		}
		if(!sdReadStart(curSec = sector)) return false;
		curPos = 0;
	}

	while(curPos != offset){	//skip to where we're needed
		curPos++;
		sdSpiByteFast(0xFF);
	}

	curPos += len;
	while(len--) *buf++ = sdSpiByteFast(0xFF);
	
	if(curPos == 512){
		sdSecReadStop();
		curSec = 0xFFFFFFFFUL;
	}

	return true;
}

void ufatExtReadTerminate(void){

	//finish reading whatever sector we're reading so that we can issue commands to the SD card
	ufatExtRead(0, 512, 0, NULL);
}

static void init()
{
	
	//wdt
	{		
		cli();
		wdt_reset();
		wdt_disable();
	}
	
	//ports
	{
		MCUCR |= 0x40;
		ACSR = 0x80;	//disable comparator
		DIDR0 = 0;
	}
}

static uint8_t eeRead(uint8_t addr)
{
	while(EECR & (1<<EEPE));
	EEAR = addr;
	EECR |= 0x01;
	return EEDR;
}

static  void eeWrite(uint8_t addr, uint8_t data)
{
	while(EECR & (1<<EEPE));
	EECR = 0;
	EEAR = addr;
	EEDR = data;
	
	EECR |= 0x04;
	EECR |= 0x02;
}

static uint16_t rnd(void)
{
	uint32_t seed = 0;
	uint16_t ret;
	uint8_t i;
	
	for (i = 0; i < 4; i++)
		seed = (seed << 8) | eeRead(i);
	
	seed *= 1664525UL;
	seed += 1013904223UL;
	
	ret = seed >> 16;
	
	for (i = 0; i < 4; i++, seed >>= 8)
		eeWrite(i, seed);
	
	return ret;
}

static void  __attribute__((noreturn)) bedtime(void)
{
	cli();
	
	set_sleep_mode(2);	//powerdown mode
	
	while(1)
	{
		sleep_enable();
		sleep_bod_disable();
		sleep_cpu();
	}
}

void fatal(uint8_t n)
{
	uint8_t i;
	uint16_t j;
	
	//make sue speaker is on
	DDRB |= 1 << 4;
	
	for (i = 0; i < n; i++) {
	
		for (j = 0; j < 300; j++){
			PINB = 1 << 4;
			_delay_ms(3);
		}
		_delay_ms(500);
	}
	
	bedtime();
}

ISR(TIM0_COMPA_vect )
{
	OCR1B = gSamples[gSampleReadPos++];
}

static void audioOn(void)
{
	//enable analog output
	PLLCSR = 0x07;	//fast mode
	TCCR1 = 0x01;	//timer1 on, no PWM on pin A, clock is PCK
	GTCCR = 0x60;	//pwm on pin B, pin notB disconnected
	DDRB |= 1 << 4;
	OCR1B = 0;
	
	//enable sample timer
	// we can either get it exacty right (32KHz) at cost of 2x as many interrupts since we
	// have no clk/2 prescaler which we'd need to count to 500, or we can get it almost right
	// and be 0.8% slower or faster. For now we choose the 0.8% slower version
	TCCR0A = 0x02;	//CTC mode
	TCCR0B = 0x02;	//clk/8 prescaler
	OCR0A = 63;
	TIMSK = 0x10;	//int on match with A
	
	sei();
}

static void audioOff(void)
{
	//turn off sample timer
	TCCR0B = 0;
	TIMSK = 0;
	
	//output an analong zero and wait for output to settle
	OCR1B = 0;
	_delay_ms(1);
	
	//turn off analog timer and pins
	TCCR1 = 0;
	GTCCR = 0;	//pwm off
	
	//pin low
	DDRB |= 1 << 4;
	PORTB &=~ (1 << 4);
}

static void play(void)
{
	Boolean ret;
	uint32_t sec, firstSec, numSec;
	uint8_t secListIdx = 0, writePos = 1;

	while(1){

		if (secListIdx == MAX_FILE_FRAGMENTS)
			break;
			
		firstSec = gFragStart[secListIdx];
		numSec = gFragLen[secListIdx];
		secListIdx++;
		if(!firstSec && !numSec) break;

		ret = sdReadStart(firstSec);
		if(!ret) fatal(7);
	
		//we just play the WAV header as is...who cares it is too short to notice

		for(sec = 0; sec < numSec; sec++){
			uint16_t i;

			if(sec)
				sdNextSec();

			for (i = 0; i < 512; i++)
			{
				while (writePos == gSampleReadPos);	//wait
				
				gSamples[writePos++] = sdSpiByteFast(0xFF);
			}
		}
		sdSecReadStop();
	}
}

int __attribute__((noreturn)) main()
{
	uint16_t n, i, id, numFiles = 0;
	char name[11];
	uint8_t j, flags;
	uint32_t sz, sec;
	Boolean ret;
	
	init();
	audioOff();
	_delay_ms(100);
	
	ret = sdInit();
	if(!ret) fatal(1);

	ufatInit();
	ret = ufatMount();
	if(!ret) fatal(2);
	
	i = 0;
	while(ufatGetNthFile(i, name, &sz, &flags, &id)){
		i++;
		if(flags & (UFAT_FLAG_DIR | UFAT_FLAG_HIDDEN)) continue;		//skip dirs and hidden files
		if(name[8] !='W' || name[9] != 'A' || name[10] != 'V') continue;	//skip non-WAV files
		numFiles++;
	}
	
	if(!numFiles) fatal(3);	//no files

	n = rnd() % numFiles;	//pick a random file
	
	i = 0;
	while(ufatGetNthFile(i, name, &sz, &flags, &id)){
		i++;
		if(flags & (UFAT_FLAG_DIR | UFAT_FLAG_HIDDEN)) continue;				//skip dirs and hidden files
		if(name[8] !='W' || name[9] != 'A' || name[10] != 'V') continue;	//skip non-WAV files
		if(!n--) goto found;
	}

	fatal(4);
found:

	if(!ufatOpen(id)) fatal(5);

	j = 0;
	while(ufatGetNextSectorRange(&sec, &sz) && j < MAX_FILE_FRAGMENTS){
		gFragStart[j] = sec;
		gFragLen[j] = sz;
		j++;
	}
	ufatExtReadTerminate();

	if (!j)
		fatal(6);

	audioOn();
	play();
	audioOff();
	
	bedtime();
}



