#include "nRF.h"

//state transitions
//init: start -> off
//pwOn: off -> sby or sby -> off
//doRx: sby -> rx
//doRx: rx -> sby
//doTx: sby -> tx -> sby 



static void delay_ms(uint8_t ms){

	uint8_t i;

	while(ms--) for(i = 0; i < 5; i++) nrfDelay200us();
}

void nrfInit(uint8_t channel, uint8_t power){

	nrfCE(false);

	nrfSetReg(NRF_REG_EN_AA,		0b00000011);				//auto-aknowledgements on both pipes
	nrfSetReg(NRF_REG_EN_RXADDR,		0b00000010);				//only receive on pipe 1
	nrfSetReg(NRF_REG_SETUP_AW,		0b00000010);				//use 4-byte addresses
	nrfSetReg(NRF_REG_SETUP_RETR,		0b00110000);				//auto-retransmit every millisecond
	nrfSetReg(NRF_REG_RF_CH,		channel);				//set channel
	nrfSetReg(NRF_REG_RF_SETUP,		0b00100000 | (power << 1));		//250 kbps at maximum transmit power
	nrfSetReg(NRF_REG_DYNPD,		0b00000011);				//dynamic payload lengths on pipe 0 and 1
	nrfSetReg(NRF_REG_FEATURE,		0b00000100);				//Dynamic payload length enabled
	nrfSetReg(NRF_REG_CONFIG,		0b01111110);				//powered down in TX mode with interrupts masked, 16-bit crc
	nrfCmd(CMD_FLUSH_TX);
	nrfCmd(CMD_FLUSH_RX);

	delay_ms(10);
}


void nrfSetPowerState(bool pwron){

	uint8_t t, reg;

	t = reg = nrfGetReg(NRF_REG_CONFIG);
	if(pwron) reg |= 2;
	else reg = (reg &~ 2);

	if(t == reg) return;	// no state change? do nothing

	if(!pwron){
		nrfCE(false);
		delay_ms(1);
	}
	nrfSetReg(NRF_REG_CONFIG, reg);
	if(pwron) delay_ms(5);
}

void nrfRxEnable(bool rxon){

	uint8_t t, reg;

	t = reg = nrfGetReg(NRF_REG_CONFIG);
	if(rxon) reg |= 1;
	else reg &=~ 1;
	
	if(reg == t) return;	// no state change? do nothing
	if(!(t & 2)) return;	//invalid transition

	nrfCE(false);
	nrfSetReg(NRF_REG_CONFIG, reg);
	if(rxon){
		nrfCE(true);
	}
}

void nrfSetAddr(uint8_t which, uint32_t addrv){

	uint8_t addr[4];
	
	addr[0] = addrv;
	addr[1] = addrv >> 8;
	addr[2] = addrv >> 16;
	addr[3] = addrv >> 24;
	

	if(which == NRF_ADDR_MINE){		//where i listen to to receive

		nrfSetRegEx(NRF_REG_RX_ADDR_P1, addr, 4);
	}
	else{							//where i send to

		nrfSetRegEx(NRF_REG_RX_ADDR_P0, addr, 4);
		nrfSetRegEx(NRF_REG_TX_ADDR, addr, 4);
	}
}

bool nrfSend(const void* dataP, uint8_t len, uint8_t retries){

	const uint8_t* data = dataP;
	uint8_t reg;

	reg = nrfGetReg(NRF_REG_CONFIG);	//enable transmit
	if(!(reg & 2)) return false;	//invalid transition

	if(retries > 15) retries = 15;

	nrfCE(false);
	reg &=~ 1;
	nrfSetReg(NRF_REG_CONFIG, reg);
	nrfSetReg(NRF_REG_SETUP_RETR, (nrfGetReg(NRF_REG_SETUP_RETR) & 0xF0) | retries);
	nrfSetReg(NRF_REG_RF_CH, nrfGetReg(NRF_REG_RF_CH));	//clear PLOS_CNT
	nrfSetReg(NRF_REG_EN_RXADDR, nrfGetReg(NRF_REG_EN_RXADDR) | 1);

	nrfSetReg(NRF_REG_STATUS, 0x30);	//clear "data sent" and MAX_RT status bit
	nrfCmd(CMD_FLUSH_TX);
	
	nrfCS(true);
	nrfSpiByte(CMD_W_TX_PAYLOAD);
	while(len--) nrfSpiByte(*data++);
	nrfCS(false);

	nrfCE(true);
	nrfDelay200us();
	nrfCE(false);
	nrfSetReg(NRF_REG_CONFIG, nrfGetReg(NRF_REG_CONFIG) &~ 0x30);
	while(!(nrfCmd(CMD_NOP) & 0x30)) nrfWaitForIrq();	//wait for "data sent" or max-retransmits status bit is set by hardware
	nrfSetReg(NRF_REG_CONFIG, nrfGetReg(NRF_REG_CONFIG) | 0x30);
	nrfSetReg(NRF_REG_STATUS, (reg = nrfCmd(CMD_NOP)) | 0x70);
	nrfSetReg(NRF_REG_EN_RXADDR, nrfGetReg(NRF_REG_EN_RXADDR) &~ 1);
	return !!(reg & 0x20);
}

static uint8_t nrfGetRxLen(void)
{
	uint8_t len, sta;

	nrfCS(true);
	sta = nrfSpiByte(CMD_R_RX_PL_WID);
	len = nrfSpiByte(0xFF);
	nrfCS(false);

	return len;
}

uint8_t nrfRecv(void* dataP, uint8_t len){	//return num bytes read

	uint8_t v, i = 0;
	uint8_t *data = dataP;

	if(!(nrfCmd(CMD_NOP) & 0x40)) return 0;		//no data to return;
	v = (nrfCmd(CMD_NOP) >> 1) & 0x07;		//which pipe?
	if(v != 1) return 0;				//we only support pipe1 for now

	v = nrfGetRxLen();
	
	if (v){
		nrfCS(true);
		nrfSpiByte(CMD_R_RX_PAYLOAD);
		for(i = 0; i < v && i < len; i++) *data++ = nrfSpiByte(0xFF);
		for(;v > len; v--) nrfSpiByte(0xFF);	//RX the rest
		nrfCS(false);
	}
	
	return i;
}

void nrfSetReg(uint8_t reg, uint8_t val){

	nrfSetRegEx(reg, &val, 1);
}

uint8_t nrfGetReg(uint8_t reg){

	uint8_t val;

	nrfGetRegEx(reg, &val, 1);

	return val;
}

uint8_t nrfCmd(uint8_t cmd){

	uint8_t status;

	nrfCS(true);
	status = nrfSpiByte(cmd);
	nrfCS(false);

	return status;
}

void nrfSetRegEx(uint8_t reg, const uint8_t* data, uint8_t len){

	nrfCS(true);
	nrfSpiByte(CMD_W_REGISTER | reg);
	while(len--) nrfSpiByte(*data++);
	nrfCS(false);
}

void nrfGetRegEx(uint8_t reg, uint8_t* data, uint8_t len){

	nrfCS(true);
	nrfSpiByte(CMD_R_REGISTER | reg);
	while(len--) *data++ = nrfSpiByte(0xff);
	nrfCS(false);
}