#include <stdint.h>
#include <stdarg.h>
#include "mke.h"
#include "leds.h"
#include "time.h"


static uint64_t mRndState[2];	//seed cannot be zero

static uint32_t umod8(uint8_t num8, uint8_t denom8)
{
	uint32_t i, num = num8, denom = denom8;
	
	denom <<= 9;
	do {
		denom >>= 1;
		
		if (num >= denom)
			num -= denom;
		
	} while (denom != (uint32_t)denom8);
	
	return num;
}

uint32_t rnd32(void)//xorshift128plus
{
	uint64_t ret, x, y;
	
	x = mRndState[0];
	y = mRndState[1];
	
	mRndState[0] = y;
	x ^= x << 23; // a
	mRndState[1] = x ^ y ^ (x >> 17) ^ (y >> 26); // b, c
	ret = mRndState[1] + y;

	
	//ret is the 64-bit random number generated
	
	return ((uint32_t)(ret >> 32)) ^ (uint32_t)ret;
}

static void hwrInit(void)
{
	//watchdog off
	{
		//set window and timeout (docs say we must write these regs)
		WDOG->WIN = 0x0101;
		WDOG->TOVAL = 0xffff;
		
		//configure CS2 and finally turn the shit off (but allow later updates)
		WDOG->CS2 = 0x41;
		WDOG->CS1 = 0x20;
	}
	
	//setup clocking
	{
		SIM->CLKDIV = 0x01000000;	//bus is cpu / 2, FTM is cpu/1
		FTMRE->FCLKDIV = 0x17;		//set flash voltage
		ICS->C2 = 0x00;				//BDIV=0 cpu is FLL / 1 = 48MHz
	}
	
	//NMI off
	SIM->SOPT &=~ 2ul;
	
	//ints on
	asm volatile("cpsie i");
}

static void diamonds(uint64_t timeout)
{
	while (timeGet() < timeout) {
		uint32_t i, t;
		
		for (i = 0; i < NUM_LEDS; i++) {
			t = (((uint32_t)ledGet(i)) * 240) >> 8;
			if (!t)
				t = 1;
			ledSet(i, t);
		}
		
		//same as the following, but faster: ((uint8_t)rnd32()) % NUM_LEDS;
		i = (uint8_t)rnd32();
		i -= ((i * 838861) >> 24) * 20;
		
		t = 128 + (rnd32() & 127);
		
		if (ledGet(i) < t / 2)
			ledSet(i, t);
		
		ledSwap();
	}
}

static void comet(uint64_t timeout)
{
	bool otherway = rnd32() & 1;
	
	while (timeGet() < timeout) {
		uint32_t i, j;
		
		for (j = 0; j < NUM_LEDS; j++) {
			for (i = 0; i < NUM_LEDS; i++)
				ledSet(i, (((uint32_t)ledGet(i)) * 180) >> 8);
			
			ledSet(otherway ? NUM_LEDS - 1 - j : j, 255);
			
			//60fps is good for delays
			for (i = 0; i < 5; i++)
				ledSwap();
		}	
	}
}

static void beats(uint64_t timeout)
{
	while (timeGet() < timeout) {
		uint32_t v, i;
		
		for (i = 0; i < NUM_LEDS; i++)
			ledSet(i, v = (((uint32_t)ledGet(i)) * 251) >> 8);
		
		if (!v)
			for (i = 0; i < NUM_LEDS; i++)
				ledSet(i, 255);
		ledSwap();
	}
}

static void vegas(uint64_t timeout)
{
	while (timeGet() < timeout) {
		int32_t i, j;
		
		for (i = 0; i < 3; i++) {
			
			ledClear();
			for (j = i; j < NUM_LEDS; j += 3)
				ledSet(j, 255);
				
			for (j = i - 1; j < NUM_LEDS; j += 3)
				if (j >= 0)
					ledSet(j, 16);
		
			for (j = 0; j < 10; j++)
				ledSwap();
		}
	}
}

static void builder(uint64_t timeout)
{
	while(timeGet() < timeout) {
		uint32_t built = 0, pos = NUM_LEDS, i;
		
		//build
		while (built != NUM_LEDS) {
			
			for (i = 0; i < built; i++)
				ledSet(i, 255);
			
			for (; i < NUM_LEDS; i++)
				ledSet(i, (((uint32_t)ledGet(i)) * 160) >> 8);
			
			if (pos-- == built) {
				pos = NUM_LEDS - 1;
				ledSet(built++, 255);
			}
			ledSet(pos, 32);
			
			for (i = 0; i < 1 + (1 + 3 * built) / 4; i++)
				ledSwap();
		}
		
		//unbuild
		pos = 0;
		while (built) {
			
			for (i = 0; i < NUM_LEDS - built; i++)
				ledSet(i, (((uint32_t)ledGet(i)) * 160) >> 8);
			
			for (; i < NUM_LEDS; i++)
				ledSet(i, 255);
			
			if (pos-- == 0) {
				pos = NUM_LEDS - built;
				built--;
			}
			ledSet(pos, 32);
			
			for (i = 0; i < 1 + (1 + built) / 2; i++)
				ledSwap();
		}
	}
}

//all of these return distance squared
static int32_t distVert(int32_t x, int32_t y, int32_t pos)
{
	int32_t ret = y - pos;
	
	return ret * ret;
}

static int32_t distHoriz(int32_t x, int32_t y, int32_t pos)
{
	int32_t ret = x - pos;
	
	return ret * ret;
}

static int32_t distDiagPosSlp(int32_t x, int32_t y, int32_t pos)	//distance between (x, y) and a line with slope of -1 passing through (-pos, pos)
{
	int32_t ret = -x + y - 2 * pos;
	
	return ret * ret / 2;
}

static int32_t distDiagNegSlp(int32_t x, int32_t y, int32_t pos)	//distance between (x, y) and a line with slope of -1 passing through (-pos, -pos)
{
	int32_t ret = x + y + 2 * pos;
	
	return ret * ret / 2;
}

static void waves(uint64_t timeout)
{
	const int32_t overshootInitial = 20, briFade = 252, briMin = 1;
	
	int32_t minX = 127, minY = 127, maxX = -127, maxY = -127, minD, maxD, maxDist, briFade64, mul, overshoot = overshootInitial, mulSquaredInverseShift;
	int32_t (*distF)(int32_t x, int32_t y, int32_t pos);
	uint32_t i;
	
	//find the limits
	for (i = 0; i < NUM_LEDS; i++) {
		if (mLedsPosX[i] < minX)
			minX = mLedsPosX[i];
		if (mLedsPosX[i] > maxX)
			maxX = mLedsPosX[i];
		if (mLedsPosY[i] < minY)
			minY = mLedsPosY[i];
		if (mLedsPosY[i] > maxY)
			maxY = mLedsPosY[i];
	}
	
	i = 255;
	maxDist = 0;
	while (i) {
		maxDist++;
		i = (i * briFade) >> 8;
	}
	
	briFade64 = 256;
	for (i = 0; i < 64; i++)
		briFade64 = (briFade64 * briFade) >> 8;
	
	minD = minX < minY ? minX : minY;
	maxD = maxX > maxY ? maxX : maxY;
	
	while (timeGet() < timeout) {
		
		int32_t moveDir = 1, pos, sta, sto;
		bool hadAnyOn = false, done = false;
		
		//figure out the kind of move we'll be doing
		i = rnd32();
		switch(i & 3) {
			case 0:	//diagonal between bottom left and top right
				distF = distDiagPosSlp;
				sta = minD;
				sto = maxD;
				mul = 4;
				mulSquaredInverseShift = 4;	//== log_2(mul * mul)
				break;
			case 1: //diagonal between bottom right and top left
				distF = distDiagNegSlp;
				sta = minD;
				sto = maxD;
				mul = 4;
				mulSquaredInverseShift = 4;	//== log_2(mul * mul)
				break;
			case 2:	//vertical
				distF = distVert;
				sta = minY;
				sto = maxY;
				mul = 2;
				mulSquaredInverseShift = 2;	//== log_2(mul * mul)
				break;
			case 3:	//horizontal
				distF = distHoriz;
				sta = minX;
				sto = maxY;
				mul = 2;
				mulSquaredInverseShift = 2;	//== log_2(mul * mul)
				break;
		}
		
		sta *= mul;
		sto *= mul;
		overshoot = overshootInitial * mul;
		
		sta -= overshoot;
		sto += overshoot;
		
		//invert as needed
		if (i & 4) {
			int32_t t;
			
			moveDir = -1;
			t = sta;
			sta = sto;
			sto = t;
		}
		
		//move
		for (pos = sta; pos != sto && !done; pos += moveDir) {
			
			bool anyOn = false;
			
			for (i = 0; i < NUM_LEDS; i++) {
				
				int32_t bri = 255, dist = distF(mLedsPosX[i] * mul, mLedsPosY[i] * mul, pos) >> mulSquaredInverseShift;
						
				if (dist > maxDist)
					bri = 0;
				else {
					
					while(dist >= 64) {
						dist -= 64;
						bri = (bri * briFade64) >> 8;
					}
					
					while (dist--) {
						bri = (bri * briFade) >> 8;
					}
				}
				
				if (bri < briMin)
					bri = briMin;
				else
					anyOn = true;
				
				ledSet(i, bri);
			}
			
			if (!anyOn){
				if (!hadAnyOn)
					continue;
				else
					done = true;
			}
			hadAnyOn = anyOn;
			
			ledSwap();
		}
	}
}


typedef void (*EffectType)(uint64_t timeout);

static const EffectType mEffects[] = {comet, diamonds, beats, vegas, builder, waves}; 

void main(void)
{
	uint32_t mEffectsUsed[(sizeof(mEffects) / sizeof(*mEffects) + 31) / 32] = {0,};		//for random effect selection
	uint32_t i, jk, numLeft;
	
	hwrInit();
	ledInit();
	timeInit();
	
	mRndState[0] = *(uint64_t*)&SIM->UUIDL;	//use uuid's lower 64 bits as rnd seed
	
	//playlist-shuffle style randomness - each effect will be done once before any are repeated
	while(1) {
		
		EffectType ef;
		
		numLeft = sizeof(mEffects) / sizeof(*mEffects);
		while(numLeft) {
			
			i = umod8(rnd32(), numLeft);
			jk = 0;
			
			while(1) {
				while ((mEffectsUsed[jk / 32] >> (jk % 32)) & 1)
					jk++;
				
				ef = mEffects[jk];
				
				if (!i--)
					break;
				
				jk++;	//skip it
			}
			
			mEffectsUsed[jk / 32] |= 1 << (jk % 32);
			
			ef(timeGet() + TICK_RATE * 30);
			numLeft--;
		}
		
		for (i = 0; i < sizeof(mEffectsUsed) / sizeof(*mEffectsUsed); i++)
			mEffectsUsed[i] = 0;
	}
}

