#define	BUILDING_APP_OR_LIB		1
#include <PalmOS.h>
#include <SlotDrvrLib.h>
#include <ExpansionMgr.h>
#include <SerialMgrOld.h>
#include <VFSMgr.h>
#include <stddef.h>
#include "sd.h"


#define MY_CRID					'PpSs'


struct Globals {

	//library/slot
	UInt16 refCnt, slotRef;
	
	//SD card/reader
	struct SD sd;
};

static void slotPrvOpenAlarm(UInt16 almProcCmd, SysAlarmTriggeredParamType *paramP)
{
	struct Globals *g = (struct Globals*)paramP->ref;
	union SdFlags sdf;

	//configure it as two 1MB 8bit addresses, enabled
	*(volatile UInt16*)0xfffff112 = 0x01e7;		//6 wait states
	sdCardInit(&g->sd);

	sdf.value = sdGetFlags(&g->sd);
	if (sdf.inited)
		(void)ExpCardInserted(g->slotRef);
}

Err SlotOpen(UInt16 slotLibRefNum)
{
	SysLibTblEntryPtr entry = SysLibTblEntry(slotLibRefNum);
	struct Globals *g = (struct Globals*)entry->globalsP;
	Err e = errNone;
	
	if (!g) {
			
		entry->globalsP = g = (struct Globals*)MemChunkNew(0, sizeof(struct Globals), memNewChunkFlagNonMovable);

		MemSet(g, sizeof(struct Globals), 0);
		
		if (errNone != ExpSlotRegister(slotLibRefNum, &g->slotRef))
			SysFatalAlert("Cannot init slot");

		e = AlmSetProcAlarm(slotPrvOpenAlarm, (UInt32)g, TimGetSeconds());
	}
	
	g->refCnt++;
	
	return e;
}

static struct Globals* __attribute__((const)) slotPrvGetGlobals(UInt16 slotLibRefNum)
{
	SysLibTblEntryPtr entry = SysLibTblEntry(slotLibRefNum);
	struct Globals *g = (struct Globals*)entry->globalsP;
	
	if (!g || !g->refCnt)
		SysFatalAlert("NOT OPEN");
	
	return g;
}

Err SlotClose(UInt16 slotLibRefNum)
{
	SysLibTblEntryPtr entry = SysLibTblEntry(slotLibRefNum);
	struct Globals *g = slotPrvGetGlobals(slotLibRefNum);
		
	if (--g->refCnt)
		return expErrStillOpen;
	
	ExpSlotUnregister(g->slotRef);
	MemChunkFree(g);
	entry->globalsP = NULL;
	
	return errNone;
}


Err SlotSleep(UInt16 slotLibRefNum)
{
	struct Globals *g = slotPrvGetGlobals(slotLibRefNum);
	
	//todo
	
	return errNone;
}

static void slotPrvWakeAlarm(UInt16 almProcCmd, SysAlarmTriggeredParamType *paramP)
{
	struct Globals *g = (struct Globals*)paramP->ref;
	union SdFlags sdf = {.value = sdGetFlags(&g->sd), };
	Boolean havCard, hadCard = sdf.inited;
	UInt32 prevSnum, newSnum;
	UInt16 prevOid, newOid;
	UInt8 prevMid, newMid;

	if (hadCard) {
		//save current card identification if we had a car
		sdGetInfo(&g->sd, &prevMid, &prevOid, &prevSnum);

		//maybe the card is still in?
		if (sdPostSleepCidReread(&g->sd)) {

			//verify identification
			sdGetInfo(&g->sd, &newMid, &newOid, &newSnum);

			//if same - we're done
			if (newMid == prevMid && newOid == prevOid && newSnum == prevSnum)
				return;
		}
	}

	//try to init new card
	*(volatile UInt16*)0xfffff112 = 0x01e7;		//6 wait states
	
	sdCardInit(&g->sd);

	sdf.value = sdGetFlags(&g->sd);

	if (!sdf.inited && hadCard)  {

		//card removed and no new card
		(void)ExpCardRemoved(g->slotRef);
		return;
	}

	if (sdf.inited && hadCard) {
		//had card and have new card. maybe same card?
		//verify identification
		sdGetInfo(&g->sd, &newMid, &newOid, &newSnum);
		//if same - we're done
		if (newMid == prevMid && newOid == prevOid && newSnum == prevSnum)
			return;
		//not same card - eject old one
		(void)ExpCardRemoved(g->slotRef);
	}

	if (sdf.inited) {	//new card
		(void)ExpCardInserted(g->slotRef);
		WinDrawChars("P", 1, 0, 0);
	}
	return;
}

Err SlotWake(UInt16 slotLibRefNum)
{
	struct Globals *g = slotPrvGetGlobals(slotLibRefNum);

	//XXX: wake is triggered many things are ready, so we use a procedural alarm to delay our resumption a bit.
	//Procedural alarms are available in PalmOS 3.2 or newer

	return AlmSetProcAlarm(slotPrvWakeAlarm, (UInt32)g, TimGetSeconds());
}

Err SlotReturnNoError(UInt16 slotLibRefNum)
{
	struct Globals *g = slotPrvGetGlobals(slotLibRefNum);
	
	return errNone;
}

Err SlotReturnUnsupportedOp(UInt16 slotLibRefNum)
{
	struct Globals *g = slotPrvGetGlobals(slotLibRefNum);
	
	return expErrUnsupportedOperation;
}

UInt32 SlotLibAPIVersion(UInt16 slotLibRefNum)
{
	return slotDrvrAPIVersion;
}

Err SlotCustomControl(UInt16 slotLibRefNum, UInt32 apiCreator, UInt16 apiSelector, void *valueP, UInt16 *valueLenP)
{
	struct Globals *g = slotPrvGetGlobals(slotLibRefNum);
	
	return expErrUnsupportedOperation;
}

struct {
	UInt8 manuf;
	UInt16 oem;
	char name[18];
} const static sdManufs[] = {
	{1, 'PA', "Panasonic"},
	{2, 'TM', "Toshiba"},
	{3, 'SD', "Sandisk"},
	{3, 'PT', "Sandisk"},
	{6, 'RK', "Fuji"},
	{9, 'AP', "ATP"},
	{17, 'D ', "PNY"},
	{25, 'DY', "Integral Platinum"},
	{48, 'PR', "Integral Platinum"},
	{27, 'SM', "Samsung"},
	{28, 'SV',"Transcend"},
	{116, 'JE',"Transcend"},
	{116, 'J\\',"Transcend"},
	{29, 'AD', "Corsair"},
	{29, 'AF', "AData"},
	{39, 'PH', "Phison"},
	{40, 'BE', "Lexar/PNY"},
	{49, 'SP', "Silicon Power"},
	{65, '42', "Kingston"},
	{130, 'JT', "Sony"},
};

Err SlotCardInfo(UInt16 slotLibRefNum, UInt16 slotRefNum, ExpCardInfoType *infoP)
{
	struct Globals *g = slotPrvGetGlobals(slotLibRefNum);
	const char *manuf = NULL;
	union SdFlags flags;
	UInt8 divShift, mid;
	UInt32 numKb, snum;
	UInt16 oid;
	char letter;
	
	flags.value = sdGetFlags(&g->sd);

	if (!flags.inited)
		return expErrCardNotPresent;
	
	numKb = sdGetNumSecs(&g->sd) / 2;
	sdGetInfo(&g->sd, &mid, &oid, &snum);

	infoP->capabilityFlags = expCapabilityHasStorage | (flags.RO ? expCapabilityReadOnly : 0);
	
	if (flags.SD) {
		
		UInt8 i;
		
		for (i = 0; i < sizeof(sdManufs) / sizeof(*sdManufs); i++) {
			if (mid == sdManufs[i].manuf && oid == sdManufs[i].oem) {
				manuf = sdManufs[i].name;
				break;
			}
		}
	}
	else {

		switch (mid) {
			case 1:		manuf = "Infineon";			break;
			case 2:		manuf = "Sandisk";			break;
			case 6:		manuf = "Hitachi";			break;
			case 22:	manuf = "Mobipocket eBook";	break;
			case 44:	manuf = "PNY";				break;
		}
	}
	
	if (manuf)
		StrCopy(infoP->manufacturerStr, manuf);
	else if (flags.SD)
		StrPrintF(infoP->manufacturerStr, "?: 's-%d-%04x'", mid, oid);
	else
		StrPrintF(infoP->manufacturerStr, "?: 'm-%d'", mid);
	
	if (numKb >= (1UL << (divShift = 31)))
		letter = 'T';
	else if (numKb >= (1UL << (divShift = 21)))
		letter = 'G';
	else if (numKb >= (1UL << (divShift = 11)))
		letter = 'M';
	else {
		divShift = 1;
		letter ='K';
	}
	
	if (numKb & ((1UL << divShift) - 1))
		StrPrintF(infoP->deviceClassStr, "%lu.%lu%c ", numKb >> divShift, (numKb & ((1UL << divShift) - 1)) * 10 >> divShift, letter);
	else
		StrPrintF(infoP->deviceClassStr, "%lu%c ", numKb >> divShift, letter);
	
	if (flags.SD) {
		
		if (!flags.v2)
			StrCat(infoP->deviceClassStr, "SDv1");
		else if (flags.HC)
			StrCat(infoP->deviceClassStr, "SDxC");
		else
			StrCat(infoP->deviceClassStr, "SDv2");
	}
	else {
		
		if (!flags.v2)
			StrCat(infoP->deviceClassStr, "MMC");
		else if (flags.HC)
			StrCat(infoP->deviceClassStr, "HS-MMC");
		else
			StrCat(infoP->deviceClassStr, "MMCplus");
	}
	
	StrCopy(infoP->productStr, "Storage");
	StrIToA(infoP->deviceUniqueIDStr, snum);
	
	return errNone;
}

Err SlotMediaType(UInt16 slotLibRefNum, UInt16 slotRefNum, UInt32 *mediaTypeP)
{
	*mediaTypeP = expMediaType_SecureDigital;
	return errNone;
}

Boolean SlotCardIsFilesystemSupported(UInt16 slotLibRefNum, UInt16 slotRefNum, UInt32 filesystemType)
{
	return filesystemType == vfsFilesystemType_VFAT || filesystemType == vfsFilesystemType_FAT;
}

Err SlotCardMetrics(UInt16 slotLibRefNum, UInt16 slotRefNum, CardMetricsPtr metricsV2)
{
	struct Globals *g = slotPrvGetGlobals(slotLibRefNum);
	union SdFlags flags;
	
	flags.value = sdGetFlags(&g->sd);
	if (!flags.inited)
		return expErrCardNotPresent;

	MemSet(metricsV2, sizeof(*metricsV2), 0);
	metricsV2->totalSectors = sdGetNumSecs(&g->sd);
	metricsV2->bytesPerSector = slotSectorSize;
	metricsV2->partitionSize = metricsV2->totalSectors - 1;	//tell it to not use an MBR
	metricsV2->sectorsPerBlock = 1;							//cannot be zero or VFS gets sad in PrvGetBestChunkSize()
		
	return errNone;
}

Err SlotCardSectorRead(UInt16 slotLibRefNum, UInt16 slotRefNum, UInt32 sectorNumber, UInt8 *bufferP, UInt32 *numSectorsP)
{
	struct Globals *g = slotPrvGetGlobals(slotLibRefNum);
	union SdFlags flags;
	UInt32 i;
	
	flags.value = sdGetFlags(&g->sd);
	if (!flags.inited) {
		SysFatalAlert("err1\n");
		return expErrCardNotPresent;
	}
	
	if (!numSectorsP || !bufferP)
		return sysErrParamErr;
	
	if (sectorNumber >= sdGetNumSecs(&g->sd))
		return expErrCardBadSector;

	if (!*numSectorsP) {

		//nothing to do
		return errNone;
	}
	else if (*numSectorsP == 1) {

		if (sdSecRead(&g->sd, sectorNumber, bufferP))
			return errNone;
		else {
			*numSectorsP = 0;
			SysFatalAlert("single read");
			return expErrCardBadSector;
		}
	}
	else {		//multiple

		UInt32 doneSec = 0, cardAvailSec = sdGetNumSecs(&g->sd) - sectorNumber, numReqedSecs = *numSectorsP;

		if (numReqedSecs > cardAvailSec)
			numReqedSecs = cardAvailSec;

		if (!sdReadStart(&g->sd, sectorNumber, numReqedSecs)) {

			*numSectorsP = 0;
			SysFatalAlert("read start");
			return expErrCardBadSector;
		}

		for (i = 0; i < numReqedSecs; i++) {

			if (!sdReadNext(&g->sd, bufferP))
				break;

			bufferP += slotSectorSize;
		}

		(void)sdReadStop(&g->sd);

		*numSectorsP = i;
		if(!i)
			SysFatalAlert("read complete");
		return i ? errNone : expErrCardBadSector;
	}
}

Err SlotCardSectorWrite(UInt16 slotLibRefNum, UInt16 slotRefNum, UInt32 sectorNumber, UInt8 *bufferP, UInt32 *numSectorsP)
{
	static const uint8_t __attribute__((aligned(4))) zeroes[slotSectorSize];
	struct Globals *g = slotPrvGetGlobals(slotLibRefNum);
	union SdFlags flags;
	UInt32 i;
	
	flags.value = sdGetFlags(&g->sd);
	if (!flags.inited) {
		SysFatalAlert("err1\n");
		return expErrCardNotPresent;
	}
	
	if (flags.RO)
		return expErrCardReadOnly;
	
	if (!numSectorsP)
		return sysErrParamErr;
	
	if (sectorNumber >= sdGetNumSecs(&g->sd))
		return expErrCardBadSector;

	if (!*numSectorsP) {

		//nothing to do
		return errNone;
	}
	else if (*numSectorsP == 1) {

		if (sdSecWrite(&g->sd, sectorNumber, bufferP ? bufferP : zeroes))
			return errNone;
		else {
			*numSectorsP = 0;
			SysFatalAlert("single write");
			return expErrCardBadSector;
		}
	}
	else {		//multiple

		UInt32 doneSec = 0, cardAvailSec = sdGetNumSecs(&g->sd) - sectorNumber, numReqedSecs = *numSectorsP;

		if (numReqedSecs > cardAvailSec)
			numReqedSecs = cardAvailSec;

		if (!sdWriteStart(&g->sd, sectorNumber, numReqedSecs)) {

			*numSectorsP = 0;
			SysFatalAlert("write start");
			return expErrCardBadSector;
		}

		for (i = 0; i < numReqedSecs; i++) {

			if (!sdWriteNext(&g->sd, bufferP ? bufferP : zeroes))
				break;

			if (bufferP)
				bufferP += slotSectorSize;
		}

		(void)sdWriteStop(&g->sd);

		*numSectorsP = i;
		if(!i)
			SysFatalAlert("write complete");

		return i ? errNone : expErrCardBadSector;
	}
}

Err SlotCardPresent(UInt16 slotLibRefNum, UInt16 slotRefNum)
{
	struct Globals *g = slotPrvGetGlobals(slotLibRefNum);
	union SdFlags flags;
	

	flags.value = sdGetFlags(&g->sd);
	return flags.inited ? errNone : expErrCardNotPresent;
}

static void SerialSDdispatchTable(void)
{
	asm volatile(
		"dc.w	116									\n\t"
		"dc.w	40									\n\t"
		"dc.w	44									\n\t"
		"dc.w	48									\n\t"
		"dc.w	52									\n\t"
		"dc.w	56									\n\t"
		"dc.w	60									\n\t"
		"dc.w	64									\n\t"
		"dc.w	68									\n\t"
		"dc.w	72									\n\t"
		"dc.w	76									\n\t"
		"dc.w	80									\n\t"
		"dc.w	84									\n\t"
		"dc.w	88									\n\t"
		"dc.w	92									\n\t"
		"dc.w	96									\n\t"
		"dc.w	100									\n\t"
		"dc.w	104									\n\t"
		"dc.w	108									\n\t"
		"dc.w	112									\n\t"
		"jmp	SlotOpen(%pc)						\n\t"
		"jmp	SlotClose(%pc)						\n\t"
		"jmp	SlotSleep(%pc)						\n\t"
		"jmp	SlotWake(%pc)						\n\t"
		"jmp	SlotLibAPIVersion(%pc)				\n\t"
		"jmp	SlotCustomControl(%pc)				\n\t"
		"jmp	SlotCardPresent(%pc)				\n\t"
		"jmp	SlotCardInfo(%pc)					\n\t"
		"jmp	SlotMediaType(%pc)					\n\t"	//was: SlotCardMediaType
		"jmp	SlotCardIsFilesystemSupported(%pc)	\n\t"
		"jmp	SlotCardMetrics(%pc)				\n\t"
		"jmp	SlotReturnNoError(%pc)				\n\t"	//was: SlotCardLowLevelFormat
		"jmp	SlotCardSectorRead(%pc)				\n\t"
		"jmp	SlotCardSectorWrite(%pc)			\n\t"
		"jmp	SlotReturnNoError(%pc)				\n\t"//was: SlotPowerCheck
		"jmp	SlotMediaType(%pc)					\n\t"
		"jmp	SlotReturnUnsupportedOp(%pc)		\n\t"	//was: SlotCardReserve
		"jmp	SlotReturnUnsupportedOp(%pc)		\n\t"	//was: SlotCardRelease
		"jmp	SlotReturnUnsupportedOp(%pc)		\n\t"	//was: SlotCardGetSerialPort
		".ascii   \"SpringSD\\0\"					\n\t"
	);
}

Err __Startup__(UInt16 refNum, SysLibTblEntryType *entryP)
{
	entryP->dispatchTblP = (void**)&SerialSDdispatchTable;
	entryP->globalsP = NULL;
	return 0;
}



void* memset(void *dst, int val, size_t len) {
	MemSet(dst, len, val);
	return dst;
}

void* memcpy(void *dst, const void *src, size_t len) {
	MemMove(dst, src, len);
	return dst;
}



