#include <PalmOS.h>
#include <stdbool.h>
#include <stdint.h>
#include "sd.h"

#define printf(...)	do{} while(0)

#define VERBOSE				0


#define FLAG_TIMEOUT		0x80
#define FLAG_PARAM_ERR		0x40
#define FLAG_ADDR_ERR		0x20
#define FLAG_ERZ_SEQ_ERR	0x10
#define FLAG_CMD_CRC_ERR	0x08
#define FLAG_ILLEGAL_CMD	0x04
#define FLAG_ERZ_RST		0x02
#define FLAG_IN_IDLE_MODE	0x01


#define ADDR_PIN_NCS		17
#define ADDR_PIN_MOSI		8		//sme code depdnds on this
#define CS_SIZE				(0x100000ul)

#define NCS_BASE			0x28000000ul
#define UNSELECTED_BASE		(NCS_BASE + (0ul << ADDR_PIN_NCS))
#define SELECTED_BASE		(NCS_BASE + (1ul << ADDR_PIN_NCS))
#define SELECTED_BASE_0		(SELECTED_BASE + (1ul << ADDR_PIN_MOSI))
#define SELECTED_BASE_1		(SELECTED_BASE + (0ul << ADDR_PIN_MOSI))
#define UNSELECTED_BASE_0	(UNSELECTED_BASE + (1ul << ADDR_PIN_MOSI))
#define UNSELECTED_BASE_1	(UNSELECTED_BASE + (0ul << ADDR_PIN_MOSI))
#define CLK_BASE			(NCS_BASE + CS_SIZE)


static void sdSpiByteUnselected(void)	//sends whatever current state is
{
	uint_fast8_t i;

	for (i = 0; i < 8; i++) {

		(void)*(volatile uint16_t*)CLK_BASE;
	}
}

static void sdSpiByteTxOnly(uint16_t val)
{
	uint_fast8_t i;

	val = ~val;

	for (i = 0; i < 8; i++) {

		val <<= 1;
		(void)*(volatile uint16_t*)(SELECTED_BASE + (val & 0x100));
		(void)*(volatile uint16_t*)CLK_BASE;
	}
}

static uint8_t sdSpiByteRxOnly(void)		//sends 0xff
{
	uint_fast8_t i, val = 0;

	(void)*(volatile uint16_t*)SELECTED_BASE_1;
	for (i = 0; i < 8; i++) {

		val = val * 2 + (1 & *(volatile uint16_t*)CLK_BASE);
	}
	return val;
}

static uint8_t sdSpiByte(uint16_t val)
{
	uint_fast8_t i;

	val = ~val;

	for (i = 0; i < 8; i++) {

		val <<= 1;
		(void)*(volatile uint16_t*)(SELECTED_BASE + (val & 0x100));
		val += (1 & *(volatile uint16_t*)CLK_BASE);
	}
	
	return val;
}

static void sdChipSelect(void)
{
	(void)*(volatile uint16_t*)SELECTED_BASE;
}

static void sdChipDeselect(void)
{
	(void)*(volatile uint16_t*)UNSELECTED_BASE_1;
	sdSpiByteUnselected();
}

static uint_fast8_t sdCrc7(uint8_t* chr, uint_fast8_t cnt, uint_fast8_t crc)
{
	static const uint8_t mCrcTab7[] = {		//generated from the iterative func :)
		0x00, 0x09, 0x12, 0x1b, 0x24, 0x2d, 0x36, 0x3f, 0x48, 0x41, 0x5a, 0x53, 0x6c, 0x65, 0x7e, 0x77,
		0x19, 0x10, 0x0b, 0x02, 0x3d, 0x34, 0x2f, 0x26, 0x51, 0x58, 0x43, 0x4a, 0x75, 0x7c, 0x67, 0x6e,
		0x32, 0x3b, 0x20, 0x29, 0x16, 0x1f, 0x04, 0x0d, 0x7a, 0x73, 0x68, 0x61, 0x5e, 0x57, 0x4c, 0x45,
		0x2b, 0x22, 0x39, 0x30, 0x0f, 0x06, 0x1d, 0x14, 0x63, 0x6a, 0x71, 0x78, 0x47, 0x4e, 0x55, 0x5c,
		0x64, 0x6d, 0x76, 0x7f, 0x40, 0x49, 0x52, 0x5b, 0x2c, 0x25, 0x3e, 0x37, 0x08, 0x01, 0x1a, 0x13,
		0x7d, 0x74, 0x6f, 0x66, 0x59, 0x50, 0x4b, 0x42, 0x35, 0x3c, 0x27, 0x2e, 0x11, 0x18, 0x03, 0x0a,
		0x56, 0x5f, 0x44, 0x4d, 0x72, 0x7b, 0x60, 0x69, 0x1e, 0x17, 0x0c, 0x05, 0x3a, 0x33, 0x28, 0x21,
		0x4f, 0x46, 0x5d, 0x54, 0x6b, 0x62, 0x79, 0x70, 0x07, 0x0e, 0x15, 0x1c, 0x23, 0x2a, 0x31, 0x38,
		0x41, 0x48, 0x53, 0x5a, 0x65, 0x6c, 0x77, 0x7e, 0x09, 0x00, 0x1b, 0x12, 0x2d, 0x24, 0x3f, 0x36,
		0x58, 0x51, 0x4a, 0x43, 0x7c, 0x75, 0x6e, 0x67, 0x10, 0x19, 0x02, 0x0b, 0x34, 0x3d, 0x26, 0x2f,
		0x73, 0x7a, 0x61, 0x68, 0x57, 0x5e, 0x45, 0x4c, 0x3b, 0x32, 0x29, 0x20, 0x1f, 0x16, 0x0d, 0x04,
		0x6a, 0x63, 0x78, 0x71, 0x4e, 0x47, 0x5c, 0x55, 0x22, 0x2b, 0x30, 0x39, 0x06, 0x0f, 0x14, 0x1d,
		0x25, 0x2c, 0x37, 0x3e, 0x01, 0x08, 0x13, 0x1a, 0x6d, 0x64, 0x7f, 0x76, 0x49, 0x40, 0x5b, 0x52,
		0x3c, 0x35, 0x2e, 0x27, 0x18, 0x11, 0x0a, 0x03, 0x74, 0x7d, 0x66, 0x6f, 0x50, 0x59, 0x42, 0x4b,
		0x17, 0x1e, 0x05, 0x0c, 0x33, 0x3a, 0x21, 0x28, 0x5f, 0x56, 0x4d, 0x44, 0x7b, 0x72, 0x69, 0x60,
		0x0e, 0x07, 0x1c, 0x15, 0x2a, 0x23, 0x38, 0x31, 0x46, 0x4f, 0x54, 0x5d, 0x62, 0x6b, 0x70, 0x79,
	};
	
	uint_fast8_t i;
	
	for (i = 0; i < cnt; i++)
		crc = mCrcTab7[crc * 2 + *chr++];
	
	return crc;
}

static void sdPrvSendCmd(uint8_t cmd, uint32_t param, bool crc)
{
	sdSpiByteTxOnly(0x40 | cmd);
	sdSpiByteTxOnly(param >> 24);
	sdSpiByteTxOnly(param >> 16);
	sdSpiByteTxOnly(param >> 8);
	sdSpiByteTxOnly(param);
	
	if (crc) {
		
		uint8_t send[] = {0x40 | cmd, param >> 24, param >> 16, param >> 8, param};
	
		sdSpiByteTxOnly(sdCrc7(send, sizeof(send), 0) * 2 + 1);
	}
}

static uint_fast8_t sdPrvSimpleCommand(uint_fast8_t cmd, uint32_t param, bool crc, bool end_cs)	//do a command, return R1 reply
{
	uint_fast8_t ret, i = 0;
	
	sdChipSelect();
	
	sdPrvSendCmd(cmd, param, crc);
	
	do {		//our max wait time is 128 byte clocks (1024 clock ticks)
		
		ret = sdSpiByteRxOnly();
		
	} while (i++ < 128 && (ret == 0xFF));
	
	if (end_cs)
		sdChipDeselect();
	
	return ret;
}

static uint_fast8_t sdPrvACMD(uint_fast8_t cmd, uint32_t param, bool crc, bool end_cs)
{
	uint_fast8_t ret;
	
	ret = sdPrvSimpleCommand(55, 0, crc, true);
	if (ret & (FLAG_TIMEOUT | FLAG_ILLEGAL_CMD))
		return ret;
	
	return sdPrvSimpleCommand(cmd, param, crc, end_cs);
}

static bool sdPrvCardInit(bool isSdCard, bool signalHighCapacityHost)
{
	uint32_t param, time = 0, resp;
	bool first = true;
	
	param = (signalHighCapacityHost ? 0x40000000 : 0x00000000);
	
	while (time++ < 1024) {
	
		//busy bit is top bit of OCR, which is top bit of resp[1]
		//busy bit at 1 means init complete
		//see pages 85 and 26
		
		resp = isSdCard ? sdPrvACMD(41, param, true, true) : sdPrvSimpleCommand(1, param, true, true);
		
		if (resp & FLAG_TIMEOUT)
			break;
		
		if (!first && !(resp & FLAG_IN_IDLE_MODE))
			return true;
		first = false;
	}
	
	return false;
}

static uint_fast8_t sdPrvReadData(struct SD *sd, uint8_t* data, uint_fast16_t sz)	//length must be divisible by 8, pointer must be halfword aligned
{
	uint32_t tries = 1000000;
	
	do {
		uint_fast8_t ret;
	
		ret = sdSpiByteRxOnly();
		if ((ret & 0xF0) == 0x00) {
			return ret;	//fail
		}
		if (ret == 0xFE)
			break;
		tries--;
	} while(tries);
	
	if (!tries) {
		return 0xFF;
	}

	if (((((uintptr_t)data) | sz) & 1) || !sd->fast) {	//slow case
		
		while (sz--)
			*data++ = sdSpiByteRxOnly();
		sdSpiByteTxOnly(0xFF);
		sdSpiByteTxOnly(0xFF);
	}
	else {								//fast case
		
		uint32_t dummy0, dummy1;

		(void)*(volatile uint16_t*)SELECTED_BASE_1;
		sz = sz / 2 - 1;
		asm volatile(
			"2:						\n\t"
			"	moveq  #0, %2		\n\t"
			
			".rept 8				\n\t"

			"	move.l (%4), %0		\n\t"
			"	and.l  %5, %0		\n\t"
			"	lsl.l  #2, %2		\n\t"
			"	or.l   %0, %2		\n\t"
			
			".endr					\n\t"

			"	move.w %2, %0		\n\t"	//lows
			"	swap   %2			\n\t"
			"	lsl.w  #1, %2		\n\t"	//highs
			"	or.w   %0, %2		\n\t"
			"	move.w %2, (%1)+	\n\t"
			"	dbra   %3, 2b		\n\t"
			:"=&d"(dummy0), "+a"(data), "=&d"(dummy1), "+d"(sz)
			:"a"(CLK_BASE), "d"(0x00010001)
			:"cc", "memory"
		);

		//crc
		sdSpiByteTxOnly(0xFF);
		sdSpiByteTxOnly(0xFF);
	}
	return 0;
}

static uint8_t __attribute__((noinline)) sdPrvWriteData(struct SD *sd, const uint8_t *data, uint_fast16_t sz, bool isMultiblock)
{
	uint_fast16_t tries = 10000;
	uint_fast8_t ret;
	
	sdSpiByteTxOnly(isMultiblock ? 0xFC : 0xFE);	//start block
	
	if (!sd->fast) {	//slow case

		while (sz--)
			sdSpiByteTxOnly(*data++);
	}
	else {

		uint32_t dummy0, dummy1;

		sz--;
		asm volatile(
			"1:							\n\t"
			
			"	moveq  #0, %0			\n\t"
			"	move.b (%1)+, %0		\n\t"
			"	not.b  %0				\n\t"

			".rept 8					\n\t"

			"	add.w  %0, %0			\n\t"
			"	and.w  %6, %0			\n\t"
			"	move.w (%4, %0.w), %3	\n\t"
			"	move.w (%5), %3			\n\t"
			
			".endr						\n\t"

			"	dbra   %2, 1b			\n\t"
			:"=&d"(dummy0), "+a"(data), "+d"(sz), "=&d"(dummy1)
			:"a"(SELECTED_BASE), "a"(CLK_BASE), "d"(0x1fe)
			:"cc", "memory"
		);
	}
	
	//crc
	sdSpiByteTxOnly(0);
	sdSpiByteTxOnly(0);

	//wait for a reply
	do {
		ret = sdSpiByteRxOnly();
		if ((ret & 0x11) == 0x01)
			return ret;
		tries--;
	} while(tries);

	return 0xFF;
}


static bool sdPrvPrgBusyWait(void)
{
	uint32_t tries = 100000;
	
	while (sdSpiByteRxOnly() != 0xFF) {
		
		if (!--tries)
			return false;	
	}
	
	return true;
}

static uint32_t sdPrvGetBits(const uint8_t* data, uint32_t numBytesInArray, int32_t startBit, uint32_t len)	//for CID and CSD data..
{
	uint32_t ret = 0;
	int32_t i;
	
	for (i = startBit + len - 1; i >= startBit; i--)
		ret = (ret << 1) + ((data[numBytesInArray - 1 - i / 8] >> (i % 8)) & 1);
	
	return ret;
}

static bool sdPrvGetAndParseCid(struct SD *sd, uint8_t respBuf[static 64])
{
	uint_fast8_t v;

	v = sdPrvSimpleCommand(10, 0, false, false);		//read CID
	if (v & FLAG_TIMEOUT)
		return false;
	v = sdPrvReadData(sd, respBuf, 16);
	sdChipDeselect();
	if (v)
		return false;
	
	//parse CID
	sd->mid = sdPrvGetBits(respBuf, 16, 120, 8);
	sd->oid = sdPrvGetBits(respBuf, 16, 104, sd->flags.SD ? 16 : 8);
	sd->snum = sdPrvGetBits(respBuf, 16, sd->flags.SD ? 24 : 16, 32);

	return true;
}

bool sdPostSleepCidReread(struct SD *sd)
{
	uint8_t respBuf[64];

	return sdPrvGetAndParseCid(sd, respBuf);
}

bool sdCardInit(struct SD *sd)
{
	uint_fast8_t i, v, j;
	uint8_t respBuf[64];
	uint32_t ocr = 0;
	uint64_t t;
	
	
	MemSet(sd, sizeof(*sd), 0);

	(void)*(volatile uint16_t*)UNSELECTED_BASE_1;
	for (i = 0; i < 16; i++)	//give card time to init with CS deasserted
		sdSpiByteUnselected();
	
	SysTaskDelay(1);
		
	for (i = 0; i < 16; i++) {	//try CMD0 a few times
		
		v = sdPrvSimpleCommand(0, 0, true, true);
		if (v == FLAG_IN_IDLE_MODE)
			break;
	}
	if (v & FLAG_TIMEOUT)
		return false;
	if (!(v & FLAG_IN_IDLE_MODE))
		return false;
	
	v = sdPrvSimpleCommand(8, 0x000001AAUL, true, false);	//try CMD8 to init SDHC cards
	if (v & FLAG_TIMEOUT) {
		sdChipDeselect();
		return false;
	}
	sd->flags.v2 = !(v & FLAG_ILLEGAL_CMD);
	if (sd->flags.v2) {
		//ignore the reply
		sdSpiByteTxOnly(0xff);
		sdSpiByteTxOnly(0xff);
		sdSpiByteTxOnly(0xff);
		sdSpiByteTxOnly(0xff);
	}
	sdChipDeselect();
	
	v = sdPrvSimpleCommand(55, 0, true, true);			//see if this is SD or MMC
	if (v & FLAG_TIMEOUT)
		return false;
	sd->flags.SD = !(v & FLAG_ILLEGAL_CMD);
	
	if (sd->flags.SD) {

		if (!sdPrvCardInit(true, true) && !sdPrvCardInit(true, false))
			return false;
	}
	else{
	
		if (!sdPrvCardInit(false, true) && !sdPrvCardInit(false, false))
			return false;
	}
	
	//check for HC
	v = sdPrvSimpleCommand(58, 0, false, false);
	if (v)
		return false;
	
	//read 32 bits of ocr
	for (v = 0; v < 4; v++)
		ocr = (ocr << 8) + sdSpiByteRxOnly();
	sdChipDeselect();
	sd->flags.HC = !!(ocr & 0x40000000);
	
	v = sdPrvSimpleCommand(59, 0, true, true);		//crc off
	if (v & FLAG_TIMEOUT)
		return false;

	v = sdPrvSimpleCommand(9, 0, false, false);		//read CSD
	if (v & FLAG_TIMEOUT)
		return false;
	
	v = sdPrvReadData(sd, respBuf, 16);
	sdChipDeselect();
	if (v)
		return false;
	
	if (VERBOSE) {
		printf("CSD:");
		for (i = 0; i < 16; i++)
			printf(" %02x", respBuf[i]);
		printf("\n\n");
	}
	
	if (VERBOSE)
		printf("CSD ver %u\n", sdPrvGetBits(respBuf, 16, 126, 2) + 1);

	//calc capacity
	if (sd->flags.SD && sdPrvGetBits(respBuf, 16, 126, 2)) {	//v2 SD csd
		
		sd->numSec = 1024 * (1 + sdPrvGetBits(respBuf, 16, 48, 22));
		
		if (VERBOSE)
			printf("c_size: 0x%08x\n", sdPrvGetBits(respBuf, 16, 48, 22));
	}
	else {
		
		uint32_t cSize = sdPrvGetBits(respBuf, 16, 62, 12);
		uint32_t cSizeMult = sdPrvGetBits(respBuf, 16, 47, 3);
		uint32_t readBlLen = sdPrvGetBits(respBuf, 16, 80, 4);
		uint32_t shiftBy = readBlLen + cSizeMult + 2;
		
		if (!sd->flags.SD && sdPrvGetBits(respBuf, 16, 126, 2) >= 2 && sdPrvGetBits(respBuf, 16, 122, 4) >= 4) {
			
			if (VERBOSE)
				printf("MMC+ over 4GB not supported due to lack of testing\n");
			
			return false;
		}
		
		if (shiftBy < 9) {	//fatal
			if (VERBOSE)
				printf("card shift size invalid\n");
			return false;
		}
		shiftBy -= 9;
		
		sd->numSec = (cSize + 1) << shiftBy;
	}
	
	sd->flags.RO = false;
	
	if (sdPrvGetBits(respBuf, 16, 12, 1)) {
		
		if (VERBOSE)
			printf("Card temporarily write protected\n");
		sd->flags.RO = true;
	}
	
	if (sdPrvGetBits(respBuf, 16, 13, 1)) {
		
		if (VERBOSE)
			printf("Card permanently write protected\n");
		sd->flags.RO = true;
	}
	
	if (VERBOSE)
		printf("CARD capacity: %u 512-byte blocks\n", sd->numSec);
	
	if (sd->flags.SD) {		//some SD-specific things
		
		v = sdPrvACMD(51, 0, false, false);
		if (v & FLAG_TIMEOUT)
			return false;
		
		v = sdPrvReadData(sd, respBuf, 8);
		sdChipDeselect();
		if (v)
			return false;
		
		if (VERBOSE) {
			printf("SCR:");
			for (i = 0; i < 8; i++)
				printf(" %02x", respBuf[i]);
			printf("\n\n");
		}
		
		if (!(respBuf[1] & 1)) {
			if (VERBOSE)
				printf("card claims to not support 1-bit mode...clearly SCR is wrong\n");
			return false;
		}
		
		sd->flags.cmd23supported = !!(respBuf[3] & 2);
		
		v = sdPrvACMD(13, 0, false, false);
		if (v & FLAG_TIMEOUT)
			return false;
		
		//that command sends a 2-byte reply, so consume the second byte
		(void)sdSpiByteRxOnly();
		
		v = sdPrvReadData(sd, respBuf, 64);
		sdChipDeselect();
		if (v)
			return false;
		
		if (VERBOSE) {
			printf("SD_STATUS:");
			for (i = 0; i < 64; i++)
				printf(" %02x", respBuf[i]);
			printf("\n\n");
		}
		
		sd->flags.hasDiscard = !!(respBuf[24] & 2);
	}
	
	//speed the clock up
	*(volatile UInt16*)0xfffff112 = 0x0187;		//no wait states
	sd->fast = true;
	
	//set block size
	v = sdPrvSimpleCommand(16, SD_BLOCK_SIZE, true, true);		//crc off
	if (v & FLAG_TIMEOUT)
		return false;
	
	if (!sdPrvGetAndParseCid(sd, respBuf))
		return false;

	sd->flags.inited = true;

	return true;
}

uint32_t sdGetNumSecs(struct SD *sd)
{
	return sd->numSec;
}

bool sdSecRead(struct SD *sd, uint32_t sec, uint8_t *dst)
{
	uint_fast8_t v;

	
	v = sdPrvSimpleCommand(17, sd->flags.HC ? sec : sec << 9, false, false);
	if (v & FLAG_TIMEOUT) {
		if (VERBOSE)
			printf("command timeout\n");
		return false;
	}
	
	v = sdPrvReadData(sd, dst, SD_BLOCK_SIZE);
	sdChipDeselect();
	
	return !v;
}

bool sdSecWrite(struct SD *sd, uint32_t sec, const uint8_t *src)
{
	bool busyDone;
	uint_fast8_t v;
	
	
	v = sdPrvSimpleCommand(24, sd->flags.HC ? sec : sec << 9, false, false);
	if (v & FLAG_TIMEOUT) {


		return false;
	}
	
	v = sdPrvWriteData(sd, src, SD_BLOCK_SIZE, false);
	busyDone = sdPrvPrgBusyWait();
	
	sdChipDeselect();
	
	return (v & 0x1f) == 0x05 && busyDone;
}

void sdGetInfo(struct SD *sd, uint8_t *midP, uint16_t *oidP, uint32_t *snumP)
{
	if (midP)
		*midP = sd->mid;
	
	if (oidP)
		*oidP = sd->oid;
	
	if (snumP)
		*snumP = sd->snum;
}

uint8_t sdGetFlags(struct SD *sd)
{
	return sd->flags.value;
}

bool sdTrim(struct SD *sd, uint32_t firstBlock, uint32_t numBlocks)
{
	//discard support is signalled via b313 in SD_STATUS (ACMD13)
	//seq is: CMD32, CMD33, CMD38
	
	uint32_t addrStart, addrEnd;
	uint8_t rsp;
	
	if (sd->flags.RO)
		return false;
	
	if (!sd->flags.hasDiscard)
		return true;
	
	if (!numBlocks)
		return true;
		
	addrStart = sd->flags.HC ? firstBlock : (firstBlock * SD_BLOCK_SIZE);
	addrEnd = sd->flags.HC ? (firstBlock + numBlocks - 1) : ((firstBlock + numBlocks - 1) * SD_BLOCK_SIZE);
	
	rsp = sdPrvSimpleCommand(32, addrStart, false, true);
	if (rsp)
		return false;

	rsp = sdPrvSimpleCommand(33, addrEnd, false, true);
	if (rsp)
		return false;
	
	rsp = sdPrvSimpleCommand(38, 1, false, true);
	if (rsp)
		return false;
	
	return true;
}

bool sdWriteStart(struct SD *sd, uint32_t sec, uint32_t numSec)
{

	uint8_t rsp;
	
	if (!numSec)
		numSec = 0xffffffff;
	else {
		uint_fast8_t resp;
		//SD: ACMD23 is required, 23-bit counter
		//MMC: CMD23 is required, 16 bit counter
		
		if (sd->flags.SD)
			rsp = sdPrvACMD(23, (numSec >= (1ul << 23)) ? ((1ul << 23) - 1) : numSec, false, true);
		else
			rsp = sdPrvSimpleCommand(23, (numSec >= (1ul << 16)) ? ((1ul << 16) - 1) : numSec, false, true);

		if (rsp)
			return false;
	}
	
	rsp = sdPrvSimpleCommand(25, sd->flags.HC ? sec : sec << 9, false, false);
	if (rsp) {
		sdChipDeselect();
		return false;
	}
	
	return true;
}

bool sdWriteNext(struct SD *sd, const uint8_t *src)
{
	uint_fast8_t v;
	bool busyDone;

	v = sdPrvWriteData(sd, src, SD_BLOCK_SIZE, true);
	busyDone = sdPrvPrgBusyWait();
	
	return (v & 0x1f) == 0x05 && busyDone;
}

bool sdWriteStop(struct SD* sd)
{
	bool ret;

	sdSpiByteTxOnly(0xFD);	//stop tran

	ret = sdPrvPrgBusyWait();

	sdChipDeselect();

	return ret;
}

bool sdReadStart(struct SD *sd, uint32_t sec, uint32_t numSec)
{
	uint_fast8_t rsp = 0;

	if (!numSec)
		numSec = 0xffffffff;
	else if (sd->flags.cmd23supported || !sd->flags.SD) {	//all MMC cards support CMD23, only some SD cards do
	
		uint32_t maxCounter = sd->flags.SD ? ((1ul << 23) - 1) : ((1ul << 16) - 1);
	
		rsp = sdPrvSimpleCommand(23, numSec > maxCounter ? maxCounter : numSec, false, false);
		if (rsp) {

			sdChipDeselect();
			return false;
		}
	}
	
	rsp = sdPrvSimpleCommand(18, sd->flags.HC ? sec : sec << 9, false, false);
	if (rsp) {
		
		sdChipDeselect();
		return false;
	}
	
	return true;
}

bool sdReadNext(struct SD *sd, uint8_t *dst)
{
	uint_fast8_t v = sdPrvReadData(sd, dst, SD_BLOCK_SIZE);

	if (!v) {
		sdChipDeselect();
		return false;
	}

	return true;
}

bool sdReadStop(struct SD *sd)
{
	uint_fast8_t i;

	(void)sdPrvSimpleCommand(12, 0, false, false);
	
	//some card keep sending data a few more cycles on SPI...ignore it
	for (i = 0; i < 8; i++)
		sdSpiByteTxOnly(0xff);
	sdChipDeselect();
	return true;
}

void sdSleep(struct SD *sd)
{
	//
}

bool sdWake(struct SD *sd);



