/*
**  VIA Technologies, Inc.              
**
**  Authors: AJ Jiang & Saten Shih & Yiping Chen, 
**
**  v2.05 Dec. 2001
**
**/

#include <sys/param.h>

#if __FreeBSD_version < 400000
#include "bpfilter.h"
#endif

#include <sys/systm.h>
#include <sys/sockio.h>
#include <sys/mbuf.h>
#include <sys/malloc.h>
#include <sys/kernel.h>
#include <sys/socket.h>

#include <net/if.h>
#include <net/if_arp.h>
#include <net/ethernet.h>
#include <net/if_dl.h>
#include <net/if_media.h>

#if (__FreeBSD_version < 400000 && NBPFILTER > 0) || __FreeBSD_version >= 400000
#include <net/bpf.h>
#endif

#if __FreeBSD_version >=400000 && __FreeBSD_version <410000
#include "opt_bdg.h"
#ifdef BRIDGE
#include <net/bridge.h>
#endif /* BRIDGE */
#endif

#include <vm/vm.h>              /* for vtophys */
#include <vm/pmap.h>            /* for vtophys */
#include <machine/clock.h>      /* for DELAY */
#include <machine/bus_pio.h>
#include <machine/bus_memio.h>
#include <machine/bus.h>

#if __FreeBSD_version >= 400000
#include <machine/resource.h>
#include <sys/bus.h>
#include <sys/rman.h>

#include <dev/mii/mii.h>
#include <dev/mii/miivar.h>
#include "miibus_if.h"
#endif

#include <pci/pcireg.h>
#include <pci/pcivar.h>

#define FET_USEIOSPACE
/*
#define DEBUG
*/

/* #define FET_BACKGROUND_AUTONEG */

#include <pci/if_fet.h>


/*
 * Various supported device vendors/types and their names.
 */
static struct fet_type fet_devs[] = {
	{ VENDORID, DEVICEID_3043,
		"VIA VT86C100A Fast Ethernet Adapter                         " },
	{ VENDORID, DEVICEID_3065,
		"VIA PCI 10/100Mb Fast Ethernet Adapter                      " },
	{ 0, 0, NULL }
};

#if __FreeBSD_version >= 400000
static int fet_probe		__P((device_t));
static int fet_attach		__P((device_t));
static int fet_detach		__P((device_t));
static int fet_newbuf		__P((struct fet_softc *,
					struct fet_chain_onefrag *,
					struct mbuf *));
/*
static void fet_tick		__P((void *));
*/
#elif __FreeBSD_version < 400000
static unsigned long fet_count = 0;
static const char *fet_probe	__P((pcici_t, pcidi_t));
static void fet_attach		__P((pcici_t, int));
static int fet_newbuf		__P((struct fet_softc *,
					struct fet_chain_onefrag *));
#endif

static int fet_encap		__P((struct fet_softc *, struct fet_chain *,
						struct mbuf * ));

static void fet_rxeof		__P((struct fet_softc *));
static void fet_rxeoc		__P((struct fet_softc *));
static void fet_txeof		__P((struct fet_softc *));
static void fet_txeoc		__P((struct fet_softc *));
static void flow_control_ability	__P((struct fet_softc *));
static void fet_intr		__P((void *));
static void fet_start		__P((struct ifnet *));
static int fet_ioctl		__P((struct ifnet *, u_long, caddr_t));
static void fet_init		__P((void *));
static void fet_stop		__P((struct fet_softc *));
static void fet_watchdog		__P((struct ifnet *));

#if __FreeBSD_version >= 400000
static void fet_shutdown		__P((device_t));
#elif __FreeBSD_version < 400000
static void fet_shutdown		__P((int, void *));
#endif

static int fet_ifmedia_upd	__P((struct ifnet *));
static void fet_ifmedia_sts	__P((struct ifnet *, struct ifmediareq *));
static void fet_link_change	__P((struct fet_softc *));

static unsigned fet_read_mii __P((struct fet_softc *, char));
static void fet_write_mii    __P((struct fet_softc *, char, unsigned ));

static int fet_query_auto    __P((struct fet_softc *));

#if __FreeBSD_version >= 400000
static int fet_miibus_readreg	__P((device_t, int, int));
static int fet_miibus_writereg	__P((device_t, int, int, int));
static void fet_miibus_statchg	__P((device_t));
static void fet_setcfg		__P((struct fet_softc *, int));
#endif

static u_int8_t fet_calchash	__P((u_int8_t *));
static void fet_setmulti		__P((struct fet_softc *));
static void fet_reset		__P((struct fet_softc *));
static int fet_list_rx_init	__P((struct fet_softc *));
static int fet_list_tx_init	__P((struct fet_softc *));
static int fet_safe_rx_off	__P((struct fet_softc *));

#if __FreeBSD_version >= 400000
#ifdef FET_USEIOSPACE
#define FET_RES			SYS_RES_IOPORT
#define FET_RID			FET_PCI_LOIO
#else
#define FET_RES			SYS_RES_MEMORY
#define FET_RID			FET_PCI_LOMEM
#endif

static device_method_t fet_methods[] = {
	/* Device interface */
	DEVMETHOD(device_probe,		fet_probe),
	DEVMETHOD(device_attach,	fet_attach),
	DEVMETHOD(device_detach, 	fet_detach),
	DEVMETHOD(device_shutdown,	fet_shutdown),

	/* bus interface */
	DEVMETHOD(bus_print_child,	bus_generic_print_child),
	DEVMETHOD(bus_driver_added,	bus_generic_driver_added),

	/* MII interface */
	DEVMETHOD(miibus_readreg,	fet_miibus_readreg),
	DEVMETHOD(miibus_writereg,	fet_miibus_writereg),
	DEVMETHOD(miibus_statchg,	fet_miibus_statchg),

	{ 0, 0 }
};

static driver_t fet_driver = {
	"fet",
	fet_methods,
	sizeof(struct fet_softc)
};

static devclass_t fet_devclass;

DRIVER_MODULE(if_fet, pci, fet_driver, fet_devclass, 0, 0);
DRIVER_MODULE(miibus, fet, miibus_driver, miibus_devclass, 0, 0);
#endif

#define FET_SETBIT(sc, reg, x)				\
	CSR_WRITE_1(sc, reg,				\
		CSR_READ_1(sc, reg) | x)

#define FET_CLRBIT(sc, reg, x)				\
	CSR_WRITE_1(sc, reg,				\
		CSR_READ_1(sc, reg) & ~x)

#define FET_SETBIT16(sc, reg, x)				\
	CSR_WRITE_2(sc, reg,				\
		CSR_READ_2(sc, reg) | x)

#define FET_CLRBIT16(sc, reg, x)				\
	CSR_WRITE_2(sc, reg,				\
		CSR_READ_2(sc, reg) & ~x)

#define FET_SETBIT32(sc, reg, x)				\
	CSR_WRITE_4(sc, reg,				\
		CSR_READ_4(sc, reg) | x)

#define FET_CLRBIT32(sc, reg, x)				\
	CSR_WRITE_4(sc, reg,				\
		CSR_READ_4(sc, reg) & ~x)

#define SIO_SET(x)					\
	CSR_WRITE_1(sc, FET_MIICR,			\
		CSR_READ_1(sc, FET_MIICR) | x)

#define SIO_CLR(x)					\
	CSR_WRITE_1(sc, FET_MIICR,			\
		CSR_READ_1(sc, FET_MIICR) & ~x)

#define FET_SET_MII(sc, reg, x)				\
	fet_write_mii(sc, reg, fet_read_mii (sc, reg)|x)

#define FET_CLR_MII(sc, reg, x)				\
	fet_write_mii(sc, reg, fet_read_mii (sc, reg)& ~x)	

static int fet_safe_rx_off(sc)
	struct fet_softc		*sc;
{
	u_int16_t	ww;
	
	FET_CLRBIT16(sc, FET_COMMAND, FET_CMD_RX_ON);
	DELAY(10000);
	
	for (ww = 0; ww < W_MAX_TIMEOUT; ww++)
		if (!(CSR_READ_2(sc, FET_COMMAND) & FET_CMD_RX_ON))
			break;
	if (ww == W_MAX_TIMEOUT) {
		if(sc->fet_chip_revid < REV_ID_VT3065_A)
			return FALSE;
		else {
			/* set MAC to internal loopback */
			FET_SETBIT(sc, FET_TXCFG, FET_TXCFG_LB0);

			/* enter FIFO test mode to issue RX reject */
			FET_SETBIT(sc, FET_GFTEST, 0x01);
			FET_SETBIT(sc, FET_RFTCMD, 0x08);	
			DELAY(10000);
			FET_SETBIT(sc, FET_GFTEST, 0x00);

			/* set MAC to normal */
			FET_CLRBIT(sc, FET_TXCFG, FET_TXCFG_LB0);

			return TRUE;
			}		
		}

	else return TRUE;
}

static unsigned fet_read_mii(sc, byMIIIndex)
	struct fet_softc		*sc;
	char	byMIIIndex;
{
	unsigned ReturnMII;
	char byMIIAdrbak;
	char byMIICRbak;
	char byMIItemp;
	int s;

	s = splimp();

	byMIIAdrbak = CSR_READ_1(sc, FET_MIIADDR);
	byMIICRbak = CSR_READ_1(sc, FET_MIICR);


	CSR_WRITE_1(sc, FET_MIICR, byMIICRbak & ~FET_MIICR_MAUTO);    

	if (sc->fet_chip_revid < REV_ID_VT3071_A)
		DELAY(1000);
	else
		while(!(CSR_READ_1(sc, FET_MIIADDR) & FET_MIIADDR_MIDLE));

	CSR_WRITE_1(sc, FET_MIIADDR, byMIIIndex);
	DELAY(1000);

	FET_SETBIT(sc, FET_MIICR, FET_MIICR_RCMD);

	byMIItemp = CSR_READ_1(sc, FET_MIICR);
	byMIItemp = byMIItemp & FET_MIICR_RCMD;


	while (byMIItemp != 0) {
		   byMIItemp = CSR_READ_1(sc, FET_MIICR);
		   byMIItemp = byMIItemp & FET_MIICR_RCMD;
		  }
	DELAY(1000);


	ReturnMII = CSR_READ_2(sc, FET_MIIDATA);

	CSR_WRITE_1(sc, FET_MIIADDR, byMIIAdrbak);
	CSR_WRITE_1(sc, FET_MIICR, byMIICRbak);
	DELAY(1000);


	(void)splx(s);
	return(ReturnMII);
}

static void fet_write_mii(sc, byMIISetByte, byMIISetData)
	struct fet_softc		*sc;
	char	byMIISetByte;
	unsigned 	byMIISetData;
{

	char byMIIAdrbak;
	char byMIICRbak;
	char byMIItemp;
	int  s;

	s = splimp();

	byMIIAdrbak = CSR_READ_1(sc, FET_MIIADDR);
	byMIICRbak= CSR_READ_1(sc, FET_MIICR);

	CSR_WRITE_1(sc, FET_MIICR, byMIICRbak & ~FET_MIICR_MAUTO);

	if(sc->fet_chip_revid < REV_ID_VT3071_A)
	    DELAY(1000);
	else
	   while(!(CSR_READ_1(sc, FET_MIIADDR) & FET_MIIADDR_MIDLE));

	CSR_WRITE_1(sc, FET_MIIADDR, byMIISetByte);
	DELAY(1000);

	CSR_WRITE_2(sc, FET_MIIDATA, byMIISetData);
	DELAY(1000);

	FET_SETBIT(sc, FET_MIICR, FET_MIICR_WCMD);
	byMIItemp = CSR_READ_1(sc, FET_MIICR);
	byMIItemp = byMIItemp & FET_MIICR_WCMD;

	while (byMIItemp != 0) {
		   byMIItemp = CSR_READ_1(sc, FET_MIICR);
		   byMIItemp = byMIItemp & FET_MIICR_WCMD;
		  }
	DELAY(1000);

	CSR_WRITE_1(sc, FET_MIIADDR, byMIIAdrbak);
	CSR_WRITE_1(sc, FET_MIICR, byMIICRbak|FET_MIICR_MAUTO);
	DELAY(1000);

	(void)splx(s);
}


/*
 * Query duplex mode
*/

static int fet_query_auto(sc)
	struct fet_softc		*sc;
{
	u_int16_t FDXFlag, MIIReturn;

	MIIReturn = fet_read_mii(sc, PHY_ANEXP);

	if ((MIIReturn & PHY_ANEXP_LPAUTOABLE) == 0) {
		 FDXFlag = 0;
		 return FDXFlag;
	   }

	MIIReturn = fet_read_mii(sc, PHY_LPAR);

	if ((MIIReturn & (PHY_LPAR_100BTXFULL | PHY_LPAR_10BTFULL)) == 0) {
		 FDXFlag = 0;
		 return FDXFlag;
	   }

	MIIReturn = fet_read_mii(sc, PHY_ANAR);

	if ((MIIReturn & (PHY_ANAR_100BTXFULL | PHY_ANAR_10BTFULL)) == 0) {
		 FDXFlag = 0;
		 return FDXFlag;
	   }

	FDXFlag = 1;
	return FDXFlag;
}

/* 
 * process link status change in ISR
 */
static void fet_link_change(sc)
	struct fet_softc		*sc;
{
	u_int8_t FDXFlag;
	

	if (CSR_READ_1(sc, FET_MIISR) & FET_MIISR_LNKFL) {		
		if (sc->fet_modeinfo == FET_AUTO) {
			if (sc->fet_chip_revid < REV_ID_VT3071_A)				
				FET_SET_MII(sc, PHY_BMCR, PHY_BMCR_RESET);
		}
	}
	else {
		if (sc->fet_modeinfo == FET_AUTO) {

			/* special treatment for VT3071B's LED */
			if(sc->fet_chip_revid == REV_ID_VT3071_B) {
				if (CSR_READ_1(sc, FET_MIISR) & FET_MIISR_SPEED)
					FET_CLRBIT(sc, FET_BCR0, FET_BCR0_LED100M);
				else
					FET_SETBIT(sc, FET_BCR0, FET_BCR0_LED100M);
			}
			FDXFlag = fet_query_auto(sc);
			if (FDXFlag == 1) {
				FET_SETBIT16(sc, FET_COMMAND, FET_CMD_FDX);       /* Set Chip Fullduplex mode */
				sc->fet_full_duplex = 1;
                        }
			else {
				FET_CLRBIT16(sc, FET_COMMAND, FET_CMD_FDX);      /* Set Chip Halfduplex mode */
				sc->fet_full_duplex = 0;
			}
		}
            /* if VT3106 and VT3065 */
            if (sc->fet_chip_revid >= 0x40)
                flow_control_ability (sc);
	}

}

#if __FreeBSD_version >= 400000

static int fet_miibus_readreg(dev, phy, reg)
	device_t		dev;
	int			phy, reg;
{
	struct fet_softc	*sc;
	u_int16_t		data;
	char 			byMIICRbak;

	sc = device_get_softc(dev);
	/* change PHY address */
	byMIICRbak= CSR_READ_1(sc, FET_MIICR);
	CSR_WRITE_1(sc, FET_MIICR, byMIICRbak & ~FET_MIICR_MAUTO);
	if(sc->fet_chip_revid < REV_ID_VT3071_A)
	    DELAY(10);
	else
	   while(!(CSR_READ_1(sc, FET_MIIADDR) & FET_MIIADDR_MIDLE));
	CSR_WRITE_1(sc, FET_PHYADDR, (CSR_READ_1(sc, FET_PHYADDR)& 0xe0)| phy);
	CSR_WRITE_1(sc, FET_MIICR, byMIICRbak|FET_MIICR_MAUTO);

	/* read data */
	data = fet_read_mii(sc,reg);

	return(data);
}

static int fet_miibus_writereg(dev, phy, reg, data)
	device_t		dev;
	u_int16_t		phy, reg, data;
{
	struct fet_softc		*sc;
	char 			byMIICRbak;
	
	sc = device_get_softc(dev);

	/* change PHY address */
	byMIICRbak= CSR_READ_1(sc, FET_MIICR);
	CSR_WRITE_1(sc, FET_MIICR, byMIICRbak & ~FET_MIICR_MAUTO);
	if(sc->fet_chip_revid < REV_ID_VT3071_A)
	    DELAY(10);
	else
	   while(!(CSR_READ_1(sc, FET_MIIADDR) & FET_MIIADDR_MIDLE));
	CSR_WRITE_1(sc, FET_PHYADDR, (CSR_READ_1(sc, FET_PHYADDR)& 0xe0)| phy);
	CSR_WRITE_1(sc, FET_MIICR, byMIICRbak|FET_MIICR_MAUTO);
	
	/* write data */
	FET_SET_MII(sc, reg, data);
	
	/* if reset the phy, delay some time until the phy stable and the autonegotiate be done*/
	if(reg==0 && (data&0x8000))
		DELAY(2300000);
	return(0);
}

static void fet_miibus_statchg(dev)
	device_t		dev;
{
	struct fet_softc		*sc;
	struct mii_data		*mii;

	sc = device_get_softc(dev);
	mii = device_get_softc(sc->fet_miibus);
	fet_setcfg(sc, mii->mii_media_active);

	return;
}

/*
 * In order to fiddle with the
 * 'full-duplex' and '100Mbps' bits in the netconfig register, we
 * first have to put the transmit and/or receive logic in the idle state.
 */
static void fet_setcfg(sc, media)
	struct fet_softc		*sc;
	int			media;
{
	int			restart = 0;

	if (CSR_READ_2(sc, FET_COMMAND) & (FET_CMD_TX_ON|FET_CMD_RX_ON)) {
		restart = 1;
		FET_CLRBIT16(sc, FET_COMMAND, (FET_CMD_TX_ON|FET_CMD_RX_ON));
	}

	if ((media & IFM_GMASK) == IFM_FDX) {
		FET_SETBIT16(sc, FET_COMMAND, FET_CMD_FDX);
		sc->fet_full_duplex = 1;
        }
	else {
		FET_CLRBIT16(sc, FET_COMMAND, FET_CMD_FDX);
		sc->fet_full_duplex = 0;
        }
	if (restart)
		FET_SETBIT16(sc, FET_COMMAND, FET_CMD_TX_ON|FET_CMD_RX_ON);

	return;
}

#endif

/*
 * Calculate CRC of a multicast group address, return the lower 6 bits.
 */
static u_int8_t fet_calchash(addr)
	u_int8_t		*addr;
{
	u_int32_t		crc, carry;
	int			i, j;
	u_int8_t		c;

	/* Compute CRC for the address value. */
	crc = 0xFFFFFFFF; /* initial value */

	for (i = 0; i < 6; i++) {
		c = *(addr + i);
		for (j = 0; j < 8; j++) {
			carry = ((crc & 0x80000000) ? 1 : 0) ^ (c & 0x01);
			crc <<= 1;
			c >>= 1;
			if (carry)
				crc = (crc ^ 0x04c11db6) | carry;
		}
	}

	/* return the filter bit position */
	return((crc >> 26) & 0x0000003F);
}

/*
 * Program the 64-bit multicast hash filter.
 */
static void fet_setmulti(sc)
	struct fet_softc		*sc;
{
	struct ifnet		*ifp;
	int			h = 0;
	u_int32_t		hashes[2] = { 0, 0 };
	struct ifmultiaddr	*ifma;
	u_int8_t		rxfilt;
	int			mcnt = 0;

	ifp = &sc->arpcom.ac_if;

	rxfilt = CSR_READ_1(sc, FET_RXCFG);

	if (ifp->if_flags & IFF_ALLMULTI || ifp->if_flags & IFF_PROMISC) {
		rxfilt |= FET_RXCFG_RX_MULTI;
		CSR_WRITE_1(sc, FET_RXCFG, rxfilt);
		CSR_WRITE_4(sc, FET_MAR0, 0xFFFFFFFF);
		CSR_WRITE_4(sc, FET_MAR1, 0xFFFFFFFF);
		return;
	}

	/* first, zot all the existing hash bits */
	CSR_WRITE_4(sc, FET_MAR0, 0);
	CSR_WRITE_4(sc, FET_MAR1, 0);

	/* now program new ones */
	for (ifma = ifp->if_multiaddrs.lh_first; ifma != NULL;
				ifma = ifma->ifma_link.le_next) {
		if (ifma->ifma_addr->sa_family != AF_LINK)
			continue;
		h = fet_calchash(LLADDR((struct sockaddr_dl *)ifma->ifma_addr));
		if (h < 32)
			hashes[0] |= (1 << h);
		else
			hashes[1] |= (1 << (h - 32));
		mcnt++;
	}

	if (mcnt)
		rxfilt |= FET_RXCFG_RX_MULTI;
	else
		rxfilt &= ~FET_RXCFG_RX_MULTI;

	CSR_WRITE_4(sc, FET_MAR0, hashes[0]);
	CSR_WRITE_4(sc, FET_MAR1, hashes[1]);
	CSR_WRITE_1(sc, FET_RXCFG, rxfilt);

	return;
}

static void fet_reset(sc)
	struct fet_softc		*sc;
{	
	u_int16_t	ww;

	FET_SETBIT16(sc, FET_COMMAND, FET_CMD_RESET);

	for (ww = 0; ww < W_MAX_TIMEOUT; ww++) {
		DELAY(10);
		if (!(CSR_READ_2(sc, FET_COMMAND) & FET_CMD_RESET))
			break;
	}
	if (ww == W_MAX_TIMEOUT) {
		if (sc->fet_chip_revid < REV_ID_VT3065_A)
			printf("fet%d: reset never completed!\n", sc->fet_unit);
		else
			/* turn on force reset */
			FET_SETBIT(sc, FET_MISC_CR1, FET_MISCCR1_FORSRST);
		}
			

	/* Wait a little while for the chip to get its brains in order. */
	DELAY(1000);	
			
	return;
}

/*
 * Probe for our chip. Check the PCI vendor and device
 * IDs against our list and return a device name if we find a match.
 */
#if __FreeBSD_version >= 400000
static int fet_probe(dev)
	device_t		dev;
{
	struct fet_type		*t;

	t = fet_devs;

	while(t->fet_name != NULL) {
		if ((pci_get_vendor(dev) == t->fet_vid) &&
		    (pci_get_device(dev) == t->fet_did)) {
			device_set_desc(dev, t->fet_name);
			return(0);
		}
		t++;
	}

	return(ENXIO);
}

#elif __FreeBSD_version < 400000

static const char *
fet_probe(config_id, device_id)
	pcici_t			config_id;
	pcidi_t			device_id;
{
	struct fet_type		*t;

	t = fet_devs;

	while(t->fet_name != NULL) {
		if ((device_id & 0xFFFF) == t->fet_vid &&
		    ((device_id >> 16) & 0xFFFF) == t->fet_did) {
			return(t->fet_name);
		}
		t++;
	}

	return(NULL);
}
#endif

/*
 * Attach the interface. Allocate softc structures, do ifmedia
 * setup and ethernet/BPF attach.
 */

#if __FreeBSD_version >= 400000

static int fet_attach(dev)
	device_t		dev;
{
	int			s, i;
	u_char			eaddr[ETHER_ADDR_LEN];
	u_int32_t		command, id;
	struct fet_softc		*sc;
	struct ifnet		*ifp;
	int 			LineSpeed, MIICRbak;
    	unsigned char 		byBCR1Value;
	int 			FDXFlag, AssignFlag;
	int			idx;
	int			unit, error = 0, rid;
	
	s = splimp();

	sc = device_get_softc(dev);
	unit = device_get_unit(dev);
	bzero(sc, sizeof(struct fet_softc *));

	/*
	 * Handle power management nonsense.
	 */

	command = pci_read_config(dev, FET_PCI_CAPID, 4) & 0x000000FF;	
	if (command == 0x01) {

		command = pci_read_config(dev, FET_PCI_PWRMGMTCTRL, 4);		
		if (command & FET_PSTATE_MASK) {
			u_int32_t		iobase, membase, irq;

			/* Save important PCI config data. */
			iobase = pci_read_config(dev, FET_PCI_LOIO, 4);
			membase = pci_read_config(dev, FET_PCI_LOMEM, 4);
			irq = pci_read_config(dev, FET_PCI_INTLINE, 4);

			/* Reset the power state. */
			printf("fet%d: chip is in D%d power mode "
			"-- setting to D0\n", unit, command & FET_PSTATE_MASK);
			command &= 0xFFFFFFFC;
			pci_write_config(dev, FET_PCI_PWRMGMTCTRL, command, 4);

			/* Restore PCI config data. */
			pci_write_config(dev, FET_PCI_LOIO, iobase, 4);
			pci_write_config(dev, FET_PCI_LOMEM, membase, 4);
			pci_write_config(dev, FET_PCI_INTLINE, irq, 4);
		}
	}

	/*
	 * Map control/status registers.
	 */
	command = pci_read_config(dev, PCI_COMMAND_STATUS_REG, 4);
	command |= (PCIM_CMD_PORTEN|PCIM_CMD_MEMEN|PCIM_CMD_BUSMASTEREN);
	pci_write_config(dev, PCI_COMMAND_STATUS_REG, command, 4);
	command = pci_read_config(dev, PCI_COMMAND_STATUS_REG, 4);
	
	sc->fet_chip_revid  = pci_read_config(dev, FET_PCI_REVID, 4) & 0x000000FF;
	id = pci_read_config(dev, FET_PCI_VENDOR_ID, 4);
	if (PCI_VENDORID(id) != VENDORID) {
		printf("fet%d: wrong vendor ID !\n", unit);
		goto fail;
	}

	if (!(PCI_CHIPID(id) == DEVICEID_3043 ||
				 PCI_CHIPID(id) == DEVICEID_3065)) {
		printf("fet%d: wrong device ID !\n", unit);
		goto fail;
	}

#ifdef FET_USEIOSPACE
	if (!(command & PCIM_CMD_PORTEN)) {
		printf("fet%d: failed to enable I/O ports!\n", unit);
		free(sc, M_DEVBUF);
		goto fail;
	}
#else
	if (!(command & PCIM_CMD_MEMEN)) {
		printf("fet%d: failed to enable memory mapping!\n", unit);
		goto fail;
	}
#endif
	rid = FET_RID;
	sc->fet_res = bus_alloc_resource(dev, FET_RES, &rid,
	    0, ~0, 1, RF_ACTIVE);

	if (sc->fet_res == NULL) {
		printf("fet%d: couldn't map ports/memory\n", unit);
		error = ENXIO;
		goto fail;
	}

	sc->fet_btag = rman_get_bustag(sc->fet_res);
	sc->fet_bhandle = rman_get_bushandle(sc->fet_res);

	/* Allocate interrupt */
	rid = 0;
	sc->fet_irq = bus_alloc_resource(dev, SYS_RES_IRQ, &rid, 0, ~0, 1,
	    RF_SHAREABLE | RF_ACTIVE);

	if (sc->fet_irq == NULL) {
		printf("fet%d: couldn't map interrupt\n", unit);
		bus_release_resource(dev, FET_RES, FET_RID, sc->fet_res);
		error = ENXIO;
		goto fail;
	}
	
	error = bus_setup_intr(dev, sc->fet_irq, INTR_TYPE_NET,
	    fet_intr, sc, &sc->fet_intrhand);

	if (error) {
		bus_release_resource(dev, SYS_RES_IRQ, 0, sc->fet_irq);
		bus_release_resource(dev, FET_RES, FET_RID, sc->fet_res);
		printf("fet%d: couldn't set up irq\n", unit);
		goto fail;
	}
	
	/* Reset the adapter. */
	fet_reset(sc);

        /* if vt3065 */    
        if (sc->fet_chip_revid >= REV_ID_VT3065_A ) { 
            /* clear sticky bit before reset & read ethernet address  */      
            CSR_WRITE_1(sc, FET_STICKYHW, CSR_READ_1(sc, FET_STICKYHW) & 0xFC);              
            /* disable force PME-enable */
            CSR_WRITE_1(sc, FET_WOLCGCLR, 0x80);
            /* disable power-event config bit */
            CSR_WRITE_1(sc, FET_WOLCRCLR, 0xFF);
            /* clear power status */
            CSR_WRITE_1(sc, FET_PWRCSRCLR, 0xFF);   
        }
    
    	/* issue AUTOLoad in EECSR to reload eeprom */
    
    	FET_SETBIT(sc, FET_EECSR, FET_EECSR_LOAD);

    	/* if vt3065 delay after reset */
    	if (sc->fet_chip_revid >= REV_ID_VT3065_A ) { 
           	DELAY(10000);
           	/* for 3065D, EEPROM reloaded will cause bit 0 in MAC_REG_CFGA turned on.
            	it makes MAC receive magic packet automatically. So, driver turn it off. */
           
           	CSR_WRITE_1(sc, FET_CFGA,CSR_READ_1(sc, FET_CFGA) & 0xFE);
    	}

        /* Turn on bit2 (MIION) in PCI configuration register 0x53 during initialization */
        pci_write_config(dev, FET_PCI_MODE, pci_read_config(dev, FET_PCI_MODE, 4)|(FET_MODE3_MIION<<24), 4);
	/*
	 * Get station address. The way the Rhine chips work,
	 * you're not allowed to directly access the EEPROM once
	 * they've been programmed a special way. Consequently,
	 * we need to read the node address from the PAR0 and PAR1
	 * registers.
	 */
	for (i = 0; i < ETHER_ADDR_LEN; i++)
		eaddr[i] = CSR_READ_1(sc, FET_PAR0 + i);

	/*
	 * A Rhine chip was detected. Inform the world.
	 */
	printf("fet%d: Ethernet address: %6D\n", unit, eaddr, ":");

	/* print driver version */
	printf("fet%d: if_fet.c: v2.05 12/09/2001 \n", unit);

	sc->fet_unit = unit;
	bcopy(eaddr, (char *)&sc->arpcom.ac_enaddr, ETHER_ADDR_LEN);
	
	sc->fet_ldata = contigmalloc(sizeof(struct fet_list_data), M_DEVBUF,
	    M_NOWAIT, 0, 0xffffffff, PAGE_SIZE, 0);

	if (sc->fet_ldata == NULL) {
		printf("fet%d: no memory for list buffers!\n", unit);
		bus_teardown_intr(dev, sc->fet_irq, sc->fet_intrhand);
		bus_release_resource(dev, SYS_RES_IRQ, 0, sc->fet_irq);
		bus_release_resource(dev, FET_RES, FET_RID, sc->fet_res);
		error = ENXIO;
		goto fail;
	}

	bzero(sc->fet_ldata, sizeof(struct fet_list_data));
	
	ifp = &sc->arpcom.ac_if;
	ifp->if_softc = sc;
	ifp->if_unit = unit;
	ifp->if_name = "fet";
	ifp->if_mtu = ETHERMTU;
	ifp->if_flags = IFF_BROADCAST | IFF_SIMPLEX | IFF_MULTICAST;
	ifp->if_ioctl = fet_ioctl;
	ifp->if_output = ether_output;
	ifp->if_start = fet_start;
	ifp->if_watchdog = fet_watchdog;
	ifp->if_init = fet_init;
	ifp->if_baudrate = 10000000;
	ifp->if_snd.ifq_maxlen = FET_TX_LIST_CNT - 1;

	/*
	 * Do MII setup. mii_phy_probe will probe phy and reset phy.
	 */
 
	if (mii_phy_probe(dev, &sc->fet_miibus,
	    fet_ifmedia_upd, fet_ifmedia_sts)) {
		printf("fet%d: MII without any phy!\n", sc->fet_unit);
		bus_teardown_intr(dev, sc->fet_irq, sc->fet_intrhand);
		bus_release_resource(dev, SYS_RES_IRQ, 0, sc->fet_irq);
		bus_release_resource(dev, FET_RES, FET_RID, sc->fet_res);
		contigfree(sc->fet_ldata,
		    sizeof(struct fet_list_data), M_DEVBUF);
		error = ENXIO;
		goto fail;
	}
	
	/*
	 * check if read the mode from EEPROM
	*/

	AssignFlag = 0;		/* initial value is 0 */

	if(CSR_READ_1(sc, FET_CFGC) & FET_CFGC_MEDEN){

		if(CSR_READ_1(sc, FET_BCR0) & FET_BCR0_MED2){
			byBCR1Value = CSR_READ_1(sc, FET_BCR1) & (FET_BCR1_MED0 | FET_BCR1_MED1);

		   if(byBCR1Value == 0x00){
				AssignFlag = 0;
				sc->fet_modeinfo = FET_AUTO;
			}
		}

		else{
			AssignFlag = 1;
			byBCR1Value = CSR_READ_1(sc, FET_BCR1) & (FET_BCR1_MED0 | FET_BCR1_MED1);

			if(byBCR1Value == (FET_BCR1_MED0 | FET_BCR1_MED1))
				sc->fet_modeinfo = FET_100_FDX;

			else if(byBCR1Value == FET_BCR1_MED1)
					sc->fet_modeinfo = FET_10_FDX;

			else if(byBCR1Value == FET_BCR1_MED0)
					sc->fet_modeinfo = FET_100_HDX;

			else if(byBCR1Value == 0x00)
					sc->fet_modeinfo = FET_10_HDX;
			}
		}

	if (AssignFlag){
		FET_CLR_MII(sc, PHY_BMCR, PHY_BMCR_AUTONEGENBL); /* turn off N-way */
		printf("fet%d: ",unit);
		if (sc->fet_modeinfo == FET_100_FDX || sc->fet_modeinfo == FET_100_HDX){
			FET_SET_MII(sc, PHY_BMCR, PHY_BMCR_SPEEDSEL); /* Speed 100 */
			printf("LineSpeed=100Mb,");
			sc->arpcom.ac_if.if_baudrate=100000000;
		   }
		else{
			FET_CLR_MII(sc, PHY_BMCR, PHY_BMCR_SPEEDSEL); /* Speed 10 */
			printf("LineSpeed=10Mb,");
		}
		if (sc->fet_modeinfo == FET_100_FDX || sc->fet_modeinfo == FET_10_FDX){
			FET_SET_MII(sc, PHY_BMCR, PHY_BMCR_DUPLEX);  /* write mii Fullduplex mode */
			FET_SETBIT16(sc, FET_COMMAND, FET_CMD_FDX);                /* Set Chip Fullduplex mode */
                        sc->fet_full_duplex = 1;
			printf("Fullduplex\n");
		}
		else{
			FET_CLR_MII(sc, PHY_BMCR, PHY_BMCR_DUPLEX); /* write miiHalfduplex mode */
			FET_CLRBIT16(sc, FET_COMMAND, FET_CMD_FDX);               /* Set Chip Fullduplex mode */
                        sc->fet_full_duplex = 0;
			printf("Halfduplex\n");
		}
	   }	/* end if (AssignFlag) */

	else {

		/* restart MII auto-negotiation*/
		FET_SET_MII(sc, PHY_BMCR, PHY_BMCR_AUTONEGRSTR);
		printf("Analyzing Media type of fet%d, this will take several seconds........",unit);
		DELAY(1000000);
		for (idx=0;idx<20;idx++)
		{
			DELAY(100000);
			if (fet_read_mii(sc, PHY_BMSR) & PHY_BMSR_AUTONEGCOMP)
				break;
		}
		printf("OK\n");

		/* query MII to know LineSpeed,duplex mode */
		LineSpeed = CSR_READ_1(sc, FET_MIISR) & FET_MIISR_SPEED;
		printf("fet%d: ",unit);

		if(LineSpeed==1)
			printf("Linespeed=10Mb,");
		else
		{
			printf("Linespeed=100Mb,");
			sc->arpcom.ac_if.if_baudrate=100000000;
		}
		if(sc->fet_chip_revid == REV_ID_VT3071_B){
			if (LineSpeed == 1)
				FET_CLRBIT(sc, FET_BCR0, FET_BCR0_LED100M);
			else
				FET_SETBIT(sc, FET_BCR0, FET_BCR0_LED100M);
			}

		FDXFlag = fet_query_auto(sc);
		if (FDXFlag == 1) {
		 	printf("Fullduplex\n");
		 	FET_SETBIT16(sc, FET_COMMAND, FET_CMD_FDX);       /* Set Chip Fullduplex mode */
			sc->fet_full_duplex = 1;
		}
		else {
		  	printf("Halfduplex\n");
		  	FET_CLRBIT16(sc, FET_COMMAND, FET_CMD_FDX);      /* Set Chip Halfduplex mode */
			sc->fet_full_duplex = 0;
		  	}
	 }
	    
	/* special treatment on LED for various phys */
	if (sc->fet_chip_revid < REV_ID_VT3071_A)
		/* for NS or Mison phys, turn on bit 1 in register 17h, no effect to DAVICOM phy */
		FET_SET_MII(sc,0x17,0x0002);

	else if (sc->fet_chip_revid < REV_ID_VT3065_A)
			 /* for ESI phys, turn on bit 7 in register 17h */
			 FET_SET_MII(sc,0x17,0x0080);

	 /* turn on MII link change */
	MIICRbak = CSR_READ_1(sc, FET_MIICR);
	FET_CLRBIT(sc, FET_MIICR, FET_MIICR_MAUTO);
	DELAY(1000);
	CSR_WRITE_1(sc, FET_MIIADDR, FET_MIIADDR_MSRCEN | FET_MIIADDR_MAD0);
	DELAY(1000);
		  
	CSR_WRITE_1(sc, FET_MIICR, MIICRbak | FET_MIICR_MAUTO);	

	callout_handle_init(&sc->fet_stat_ch);

	fet_init(sc);	
	fet_stop(sc);
	
#if __FreeBSD_version >= 400000 && __FreeBSD_version <410000
	/*
	 * Call MI attach routines.
	 */
	if_attach(ifp);
	ether_ifattach(ifp);
	bpfattach(ifp, DLT_EN10MB, sizeof(struct ether_header));

#elif __FreeBSD_version >=410000 && __FreeBSD_version < 500000	
	/*
	 * Call MI attach routine.
	 */
	ether_ifattach(ifp, ETHER_BPF_SUPPORTED);
#endif
	
fail:
	splx(s);
	return(error);
}

static int fet_detach(dev)
	device_t		dev;
{
	struct fet_softc	*sc;
	struct ifnet		*ifp;
	int			s;

	s = splimp();

	sc = device_get_softc(dev);
	ifp = &sc->arpcom.ac_if;

	fet_stop(sc);

#if __FreeBSD_version >= 400000 && __FreeBSD_version < 410000
	if_detach(ifp);
#elif __FreeBSD_version >=410000 && __FreeBSD_version < 500000
	ether_ifdetach(ifp, ETHER_BPF_SUPPORTED);
#endif

	bus_generic_detach(dev);
	device_delete_child(dev, sc->fet_miibus);

	bus_teardown_intr(dev, sc->fet_irq, sc->fet_intrhand);
	bus_release_resource(dev, SYS_RES_IRQ, 0, sc->fet_irq);
	bus_release_resource(dev, FET_RES, FET_RID, sc->fet_res);

	contigfree(sc->fet_ldata, sizeof(struct fet_list_data), M_DEVBUF);

	splx(s);

	return(0);
}
#elif __FreeBSD_version < 400000

static void
fet_attach(config_id, unit)
	pcici_t			config_id;
	int			unit;
{
	int			s, i;
#ifndef FET_USEIOSPACE
	vm_offset_t		pbase, vbase;
#endif
	u_char			eaddr[ETHER_ADDR_LEN];
	u_int32_t		command, id;
	struct fet_softc		*sc;
	struct ifnet		*ifp;
	unsigned int		round;
	caddr_t			roundptr;	
	u_int16_t		nmedia, defmedia;
	const int		*media;
	int 			LineSpeed, MIICRbak;
    unsigned char 	byBCR1Value;
	int 			FDXFlag, AssignFlag;
	int			idx;

	s = splimp();

	sc = malloc(sizeof(struct fet_softc), M_DEVBUF, M_NOWAIT);
	if (sc == NULL) {
		printf("fet%d: no memory for softc struct!\n", unit);
		return;
	}
	bzero(sc, sizeof(struct fet_softc));

	/*
	 * Handle power management nonsense.
	 */

	command = pci_conf_read(config_id, FET_PCI_CAPID) & 0x000000FF;
	if (command == 0x01) {

		command = pci_conf_read(config_id, FET_PCI_PWRMGMTCTRL);
		if (command & FET_PSTATE_MASK) {
			u_int32_t		iobase, membase, irq;

			/* Save important PCI config data. */
			iobase = pci_conf_read(config_id, FET_PCI_LOIO);
			membase = pci_conf_read(config_id, FET_PCI_LOMEM);
			irq = pci_conf_read(config_id, FET_PCI_INTLINE);

			/* Reset the power state. */
			printf("fet%d: chip is in D%d power mode "
			"-- setting to D0\n", unit, command & FET_PSTATE_MASK);
			command &= 0xFFFFFFFC;
			pci_conf_write(config_id, FET_PCI_PWRMGMTCTRL, command);

			/* Restore PCI config data. */
			pci_conf_write(config_id, FET_PCI_LOIO, iobase);
			pci_conf_write(config_id, FET_PCI_LOMEM, membase);
			pci_conf_write(config_id, FET_PCI_INTLINE, irq);
		}
	}

	/*
	 * Map control/status registers.
	 */
	command = pci_conf_read(config_id, PCI_COMMAND_STATUS_REG);
	command |= (PCIM_CMD_PORTEN|PCIM_CMD_MEMEN|PCIM_CMD_BUSMASTEREN);
	pci_conf_write(config_id, PCI_COMMAND_STATUS_REG, command);
	command = pci_conf_read(config_id, PCI_COMMAND_STATUS_REG);

	sc->fet_chip_revid  = pci_conf_read(config_id, FET_PCI_REVID) & 0x000000FF;
	id = pci_conf_read(config_id, FET_PCI_VENDOR_ID);
	if (PCI_VENDORID(id) != VENDORID) {
		printf("fet%d: wrong vendor ID !\n", unit);
		goto fail;
	}

	if (!(PCI_CHIPID(id) == DEVICEID_3043 ||
				 PCI_CHIPID(id) == DEVICEID_3065)) {
		printf("fet%d: wrong device ID !\n", unit);
		goto fail;
	}

#ifdef FET_USEIOSPACE
	if (!(command & PCIM_CMD_PORTEN)) {
		printf("fet%d: failed to enable I/O ports!\n", unit);
		free(sc, M_DEVBUF);
		goto fail;
	}

	if (!pci_map_port(config_id, FET_PCI_LOIO,
					(u_int16_t *)(&sc->fet_bhandle))) {
		printf ("fet%d: couldn't map ports\n", unit);
		goto fail;
	}
	sc->fet_btag = I386_BUS_SPACE_IO;
#else
	if (!(command & PCIM_CMD_MEMEN)) {
		printf("fet%d: failed to enable memory mapping!\n", unit);
		goto fail;
	}

	if (!pci_map_mem(config_id, FET_PCI_LOMEM, &vbase, &pbase)) {
		printf ("fet%d: couldn't map memory\n", unit);
		goto fail;
	}

	sc->fet_bhandle = vbase;
	sc->fet_btag = I386_BUS_SPACE_MEM;
#endif

	/* Allocate interrupt */
	if (!pci_map_int(config_id, fet_intr, sc, &net_imask)) {
		printf("fet%d: couldn't map interrupt\n", unit);
		goto fail;
	}

	/* Reset the adapter. */
	fet_reset(sc);

        /* if vt3065 */    
        if (sc->fet_chip_revid >= REV_ID_VT3065_A ) { 
            /* clear sticky bit before reset & read ethernet address  */      
            CSR_WRITE_1(sc, FET_STICKYHW, CSR_READ_1(sc, FET_STICKYHW) & 0xFC);              
            /* disable force PME-enable */
            CSR_WRITE_1(sc, FET_WOLCGCLR, 0x80);
            /* disable power-event config bit */
            CSR_WRITE_1(sc, FET_WOLCRCLR, 0xFF);
            /* clear power status */
            CSR_WRITE_1(sc, FET_PWRCSRCLR, 0xFF);   
        }
    
    	/* issue AUTOLoad in EECSR to reload eeprom */
    
    	FET_SETBIT(sc, FET_EECSR, FET_EECSR_LOAD);

    	/* if vt3065 delay after reset */
    	if (sc->fet_chip_revid >= REV_ID_VT3065_A ) { 
           	DELAY(10000);
           	/* for 3065D, EEPROM reloaded will cause bit 0 in MAC_REG_CFGA turned on.
            	it makes MAC receive magic packet automatically. So, driver turn it off. */
           
           	CSR_WRITE_1(sc, FET_CFGA,CSR_READ_1(sc, FET_CFGA) & 0xFE);
    	}

        /* Turn on bit2 (MIION) in PCI configuration register 0x53 during initialization */
        pci_conf_write(config_id, FET_PCI_MODE, pci_conf_read(config_id,FET_PCI_MODE)|(FET_MODE3_MIION<<24));
	/*
	 * Get station address. The way the Rhine chips work,
	 * you're not allowed to directly access the EEPROM once
	 * they've been programmed a special way. Consequently,
	 * we need to read the node address from the PAR0 and PAR1
	 * registers.
	 */
	for (i = 0; i < ETHER_ADDR_LEN; i++)
		eaddr[i] = CSR_READ_1(sc, FET_PAR0 + i);

	/*
	 * A Rhine chip was detected. Inform the world.
	 */
	printf("fet%d: Ethernet address: %6D\n", unit, eaddr, ":");

	/* print driver version */
	printf("fet%d: if_fet.c: v2.04 05/17/2001 \n", unit);

	sc->fet_unit = unit;
	bcopy(eaddr, (char *)&sc->arpcom.ac_enaddr, ETHER_ADDR_LEN);

	sc->fet_ldata_ptr = malloc(sizeof(struct fet_list_data) + 8,
				M_DEVBUF, M_NOWAIT);
	if (sc->fet_ldata_ptr == NULL) {
		free(sc, M_DEVBUF);
		printf("fet%d: no memory for list buffers!\n", unit);
		return;
	}

	sc->fet_ldata = (struct fet_list_data *)sc->fet_ldata_ptr;
	round = (unsigned int)sc->fet_ldata_ptr & 0xF;
	roundptr = sc->fet_ldata_ptr;
	for (i = 0; i < 8; i++) {
		if (round % 8) {
			round++;
			roundptr++;
		} else
			break;
	}
	sc->fet_ldata = (struct fet_list_data *)roundptr;
	bzero(sc->fet_ldata, sizeof(struct fet_list_data));

	ifp = &sc->arpcom.ac_if;
	ifp->if_softc = sc;
	ifp->if_unit = unit;
	ifp->if_name = "fet";
	ifp->if_mtu = ETHERMTU;
	ifp->if_flags = IFF_BROADCAST | IFF_SIMPLEX | IFF_MULTICAST;
	ifp->if_ioctl = fet_ioctl;
	ifp->if_output = ether_output;
	ifp->if_start = fet_start;
	ifp->if_watchdog = fet_watchdog;
	ifp->if_init = fet_init;
	ifp->if_baudrate = 10000000;
	ifp->if_snd.ifq_maxlen = FET_TX_LIST_CNT - 1;

	/*
	 * check if read the mode from EEPROM
	*/

	AssignFlag = 0;		/* initial value is 0 */

	if(CSR_READ_1(sc, FET_CFGC) & FET_CFGC_MEDEN){

		if(CSR_READ_1(sc, FET_BCR0) & FET_BCR0_MED2){
			byBCR1Value = CSR_READ_1(sc, FET_BCR1) & (FET_BCR1_MED0 | FET_BCR1_MED1);

		   if(byBCR1Value == 0x00){
				AssignFlag = 0;
				sc->fet_modeinfo = FET_AUTO;
			}
		}

		else{
			AssignFlag = 1;
			byBCR1Value = CSR_READ_1(sc, FET_BCR1) & (FET_BCR1_MED0 | FET_BCR1_MED1);

			if(byBCR1Value == (FET_BCR1_MED0 | FET_BCR1_MED1))
				sc->fet_modeinfo = FET_100_FDX;

			else if(byBCR1Value == FET_BCR1_MED1)
					sc->fet_modeinfo = FET_10_FDX;

			else if(byBCR1Value == FET_BCR1_MED0)
					sc->fet_modeinfo = FET_100_HDX;

			else if(byBCR1Value == 0x00)
					sc->fet_modeinfo = FET_10_HDX;
			}
		}

	if (AssignFlag){
		FET_CLR_MII(sc, PHY_BMCR, PHY_BMCR_AUTONEGENBL); /* turn off N-way */
		printf("fet%d: ",unit);
		if (sc->fet_modeinfo == FET_100_FDX || sc->fet_modeinfo == FET_100_HDX){
			FET_SET_MII(sc, PHY_BMCR, PHY_BMCR_SPEEDSEL); /* Speed 100 */
			printf("LineSpeed=100Mb,");
			sc->arpcom.ac_if.if_baudrate=100000000;
		}
		else{
			FET_CLR_MII(sc, PHY_BMCR, PHY_BMCR_SPEEDSEL); /* Speed 10 */
			printf("LineSpeed=10Mb,");
		}
		if (sc->fet_modeinfo == FET_100_FDX || sc->fet_modeinfo == FET_10_FDX){
			FET_SET_MII(sc, PHY_BMCR, PHY_BMCR_DUPLEX);  /* write mii Fullduplex mode */
			FET_SETBIT16(sc, FET_COMMAND, FET_CMD_FDX);                /* Set Chip Fullduplex mode */
                        sc->fet_full_duplex = 1;
			printf("Fullduplex\n");
		}
		else {
			FET_CLR_MII(sc, PHY_BMCR, PHY_BMCR_DUPLEX); /* write miiHalfduplex mode */
			FET_CLRBIT16(sc, FET_COMMAND, FET_CMD_FDX);               /* Set Chip Fullduplex mode */
                        sc->fet_full_duplex = 0;
			printf("Halfduplex\n");
		}
	}	/* end if (AssignFlag) */
	else {
		/* restart MII auto-negotiation*/
		FET_SET_MII(sc, PHY_BMCR, PHY_BMCR_AUTONEGRSTR);
		printf("Analyzing Media type of fet%d, this will take several seconds........",unit);
		DELAY(1000000);
		for (idx=0;idx<20;idx++)
		{
			DELAY(100000);
			if (fet_read_mii(sc, PHY_BMSR) & PHY_BMSR_AUTONEGCOMP)
				break;
		}
		printf("OK\n");

		/* query MII to know LineSpeed,duplex mode */
		LineSpeed = CSR_READ_1(sc, FET_MIISR) & FET_MIISR_SPEED;
		printf("fet%d: ",unit);

		if(LineSpeed==1)
			printf("Linespeed=10Mb,");
		else
		{
			printf("Linespeed=100Mb,");
			sc->arpcom.ac_if.if_baudrate=100000000;
		}
		if(sc->fet_chip_revid == REV_ID_VT3071_B){
			if (LineSpeed == 1)
				FET_CLRBIT(sc, FET_BCR0, FET_BCR0_LED100M);
			else
				FET_SETBIT(sc, FET_BCR0, FET_BCR0_LED100M);
			}

		FDXFlag = fet_query_auto(sc);
		if (FDXFlag == 1) {
		 	printf("Fullduplex\n");
		 	FET_SETBIT16(sc, FET_COMMAND, FET_CMD_FDX);       /* Set Chip Fullduplex mode */
                        sc->fet_full_duplex = 1;
		}
		else {
		  	printf("Halfduplex\n");
		  	FET_CLRBIT16(sc, FET_COMMAND, FET_CMD_FDX);      /* Set Chip Halfduplex mode */
                        sc->fet_full_duplex = 0;
		  	}
	  }
   
	/* special treatment on LED for various phys */
	if (sc->fet_chip_revid < REV_ID_VT3071_A)
		/* for NS or Mison phys, turn on bit 1 in register 17h, no effect to DAVICOM phy */
		FET_SET_MII(sc,0x17,0x0002);

	else if (sc->fet_chip_revid < REV_ID_VT3065_A)
			 /* for ESI phys, turn on bit 7 in register 17h */
			 FET_SET_MII(sc,0x17,0x0080);

	 /* turn on MII link change */
	MIICRbak = CSR_READ_1(sc, FET_MIICR);
	FET_CLRBIT(sc, FET_MIICR, FET_MIICR_MAUTO);
	DELAY(1000);
	CSR_WRITE_1(sc, FET_MIIADDR, FET_MIIADDR_MSRCEN | FET_MIIADDR_MAD0);
	DELAY(1000);
		  
	CSR_WRITE_1(sc, FET_MIICR, MIICRbak | FET_MIICR_MAUTO);
	
	/*
	 * Do ifmedia setup.
	 */
	ifmedia_init(&sc->ifmedia, 0, fet_ifmedia_upd, fet_ifmedia_sts);

	media = fet_media_standard;
	nmedia = sizeof(fet_media_standard) / sizeof(fet_media_standard[0]);
	defmedia = sc->ifmedia.ifm_media = (IFM_ETHER|IFM_AUTO);
	
	for (idx = 0; idx < nmedia; idx++) {
		 ifmedia_add(&sc->ifmedia, media[idx], 0, NULL);
	}

	fet_init(sc);	
	fet_stop(sc);

	ifmedia_set(&sc->ifmedia, defmedia);
	
	/*
	 * Call MI attach routines.
	 */
	if_attach(ifp);
	ether_ifattach(ifp);

#if NBPFILTER > 0
	bpfattach(ifp, DLT_EN10MB, sizeof(struct ether_header));
#endif

	at_shutdown(fet_shutdown, sc, SHUTDOWN_POST_SYNC);

fail:
	splx(s);
	return;
}
#endif
/*
 * Initialize the transmit descriptors.
 */
static int fet_list_tx_init(sc)
	struct fet_softc		*sc;
{
	struct fet_chain_data	*cd;
	struct fet_list_data	*ld;
	int			i;

	cd = &sc->fet_cdata;
	ld = sc->fet_ldata;
	for (i = 0; i < FET_TX_LIST_CNT; i++) {
		cd->fet_tx_chain[i].fet_ptr = &ld->fet_tx_list[i];
		if (i == (FET_TX_LIST_CNT - 1))
			cd->fet_tx_chain[i].fet_nextdesc = 
				&cd->fet_tx_chain[0];
		else
			cd->fet_tx_chain[i].fet_nextdesc =
				&cd->fet_tx_chain[i + 1];
	}

	cd->fet_tx_free = &cd->fet_tx_chain[0];
	cd->fet_tx_tail = cd->fet_tx_head = NULL;
	/* initial the number of free tx descriptorx */
        cd->fet_free_tx_count=FET_TX_LIST_CNT;
	return(0);
}


/*
 * Initialize the RX descriptors and allocate mbufs for them. Note that
 * we arrange the descriptors in a closed ring, so that the last descriptor
 * points back to the first.
 */
static int fet_list_rx_init(sc)
	struct fet_softc		*sc;
{
	struct fet_chain_data	*cd;
	struct fet_list_data	*ld;
	int			i;

	cd = &sc->fet_cdata;
	ld = sc->fet_ldata;

	for (i = 0; i < FET_RX_LIST_CNT; i++) {
		cd->fet_rx_chain[i].fet_ptr =
			(struct fet_desc *)&ld->fet_rx_list[i];
#if __FreeBSD_version >= 400000
		if (fet_newbuf(sc, &cd->fet_rx_chain[i], NULL) == ENOBUFS)
#elif __FreeBSD_version < 400000
		if (fet_newbuf(sc, &cd->fet_rx_chain[i]) == ENOBUFS)
#endif
			return(ENOBUFS);
		if (i == (FET_RX_LIST_CNT - 1)) {
			cd->fet_rx_chain[i].fet_nextdesc =
					&cd->fet_rx_chain[0];
			ld->fet_rx_list[i].fet_next =
					vtophys(&ld->fet_rx_list[0]);
		} else {
			cd->fet_rx_chain[i].fet_nextdesc =
					&cd->fet_rx_chain[i + 1];
			ld->fet_rx_list[i].fet_next =
					vtophys(&ld->fet_rx_list[i + 1]);
		}
	}

	cd->fet_rx_head = &cd->fet_rx_chain[0];

	return(0);
}

/*
 * Initialize an RX descriptor and attach an MBUF cluster. 
 */
#if __FreeBSD_version >= 400000
static int fet_newbuf(sc, c, m)
	struct fet_softc	*sc;
	struct fet_chain_onefrag	*c;
	struct mbuf		*m;
{
	struct mbuf		*m_new = NULL;

	if (m == NULL) {
		MGETHDR(m_new, M_DONTWAIT, MT_DATA);
		if (m_new == NULL) {
			printf("fet%d: no memory for rx list "
			    "-- packet dropped!\n", sc->fet_unit);
			return(ENOBUFS);
		}

		MCLGET(m_new, M_DONTWAIT);
		if (!(m_new->m_flags & M_EXT)) {
			printf("fet%d: no memory for rx list "
			    "-- packet dropped!\n", sc->fet_unit);
			m_freem(m_new);
			return(ENOBUFS);
		}
		m_new->m_len = m_new->m_pkthdr.len = MCLBYTES;
	} else {
		m_new = m;
		m_new->m_len = m_new->m_pkthdr.len = MCLBYTES;
		m_new->m_data = m_new->m_ext.ext_buf;
	}

	m_adj(m_new, sizeof(u_int64_t));

	c->fet_mbuf = m_new;
	c->fet_ptr->fet_status = FET_RXSTAT;
        /*chenyp */

	c->fet_ptr->fet_data = vtophys(mtod(m_new, caddr_t));
	c->fet_ptr->fet_ctl = FET_RXCTL | FET_RXLEN;

	return(0);
}

#elif __FreeBSD_version < 400000

static int fet_newbuf(sc, c)
	struct fet_softc		*sc;
	struct fet_chain_onefrag	*c;
{
	struct mbuf		*m_new = NULL;

	MGETHDR(m_new, M_DONTWAIT, MT_DATA);
	if (m_new == NULL) {
		printf("fet%d: no memory for rx list -- packet dropped!\n",
								sc->fet_unit);
		return(ENOBUFS);
	}

	MCLGET(m_new, M_DONTWAIT);
	if (!(m_new->m_flags & M_EXT)) {
		printf("fet%d: no memory for rx list -- packet dropped!\n",
								sc->fet_unit);
		m_freem(m_new);
		return(ENOBUFS);
	}

	c->fet_mbuf = m_new;
	c->fet_ptr->fet_status = FET_RXSTAT;
	c->fet_ptr->fet_data = vtophys(mtod(m_new, caddr_t));
	c->fet_ptr->fet_ctl = FET_RXCTL | FET_RXLEN;

	return(0);
}
#endif

/*
 * A frame has been uploaded: pass the resulting mbuf chain up to
 * the higher level protocols.
 */
static void fet_rxeof(sc)
	struct fet_softc		*sc;
{
    struct ether_header	*eh;
    struct mbuf		*m;
    struct ifnet		*ifp;
    struct fet_chain_onefrag	*cur_rx;
    int			total_len = 0;
    u_int32_t		rxstat;
    u_int32_t		wSAP=0, wLen=0, wActualLen=0;

    ifp = &sc->arpcom.ac_if;

	while(!((rxstat = sc->fet_cdata.fet_rx_head->fet_ptr->fet_status) &
							FET_RXSTAT_OWN)) {
#if __FreeBSD_version >= 400000
		struct mbuf		*m0 = NULL;
#endif										
		cur_rx = sc->fet_cdata.fet_rx_head;
		sc->fet_cdata.fet_rx_head = cur_rx->fet_nextdesc;

		m = cur_rx->fet_mbuf;
			
		total_len = FET_RXBYTES(cur_rx->fet_ptr->fet_status);

                /*
		 * If an error occurs, update stats, clear the
		 * status word and leave the mbuf cluster in place:
		 * it should simply get re-used next time this descriptor
	 	 * comes up in the ring.
		 */
		if (rxstat & FET_RXSTAT_RXERR) {
			ifp->if_ierrors++;
			printf("fet%d: rx error: ", sc->fet_unit);
			switch(rxstat & 0x000000FF) {
			case FET_RXSTAT_CRCERR:
				printf("crc error\n");
				break;
			case FET_RXSTAT_FRAMEALIGNERR:
				printf("frame alignment error\n");
				break;
			case FET_RXSTAT_FIFOOFLOW:
				printf("FIFO overflow\n");
				break;
			case FET_RXSTAT_GIANT:
				printf("received giant packet\n");
				break;
			case FET_RXSTAT_RUNT:
				printf("received runt packet\n");
				break;
			case FET_RXSTAT_BUSERR:
				printf("system bus error\n");
				break;
			case FET_RXSTAT_BUFFERR:
				printf("rx buffer error\n");
				break;
			default:
				printf("unknown rx error\n");
				break;
			}

#if __FreeBSD_version >= 400000
			fet_newbuf(sc, cur_rx, m);
#elif __FreeBSD_version < 400000			
			cur_rx->fet_ptr->fet_status = FET_RXSTAT;
			cur_rx->fet_ptr->fet_ctl = FET_RXCTL|FET_RXLEN;
#endif
			continue;
		}

                /* For conforming IEEE 802.3 spec
                 * If the incoming packet is IEE 802.3 frmae/IEEE 802.3 SNAP frame, get
                 * RX_Length in RDES0 from the incoming packet, subtract Ethernet header
                 * length and CRC length from it. Then, compare the result with L/T field
                 * of the packet. If they're not equal, descard this packet.
                 */

                wLen = (((unsigned char)(*(mtod(cur_rx->fet_mbuf, caddr_t) + 12)) )<< 8) + (unsigned char)(*(mtod(cur_rx->fet_mbuf, caddr_t) + 13)) ;
                if (wLen >= 46 && wLen <= 1500) {   /* IEEE 802.3/IEEE 802.3 SNAP frame */
                    wSAP = (((unsigned char)(*(mtod(cur_rx->fet_mbuf, caddr_t) + 14))) << 8) + (unsigned char)(*(mtod(cur_rx->fet_mbuf, caddr_t) + 15));
                    if (wSAP != 0xFFFF) {            /* exclude Novell's Ethernet 802.3 frame */
                        wActualLen = total_len - U_HEADER_LEN - U_CRC_LEN;         /* real packet length */
                        if (wLen != wActualLen ) {    /* if not equal, drop this frame */
#if __FreeBSD_version >= 400000
                            fet_newbuf(sc, cur_rx, m);
#elif __FreeBSD_version < 400000			
                            cur_rx->fet_ptr->fet_status = FET_RXSTAT;
                            cur_rx->fet_ptr->fet_ctl = FET_RXCTL|FET_RXLEN;
#endif
                            continue;
                        }
                    }
                }

		/* substract 4 bytes CRC */		
		total_len -= ETHER_CRC_LEN;

#if __FreeBSD_version >= 400000
		m0 = m_devget(mtod(m, char *) - ETHER_ALIGN,
		    total_len + ETHER_ALIGN, 0, ifp, NULL);
		fet_newbuf(sc, cur_rx, m);
		if (m0 == NULL) {
			ifp->if_ierrors++;
			continue;
		}
		m_adj(m0, ETHER_ALIGN);
		m = m0;

		ifp->if_ipackets++;
		eh = mtod(m, struct ether_header *);

#if __FreeBSD_version >=400000 && __FreeBSD_version < 410000
		/*
		 * Handle BPF listeners. Let the BPF user see the packet, but
		 * don't pass it up to the ether_input() layer unless it's
		 * a broadcast packet, multicast packet, matches our ethernet
		 * address or the interface is in promiscuous mode.
		 */
		if (ifp->if_bpf) {
			bpf_mtap(ifp, m);
			if (ifp->if_flags & IFF_PROMISC &&
				(bcmp(eh->ether_dhost, sc->arpcom.ac_enaddr,
						ETHER_ADDR_LEN) &&
					(eh->ether_dhost[0] & 1) == 0)) {
				m_freem(m);
				continue;
			}
		}

#ifdef BRIDGE
		if (do_bridge) {
			struct ifnet		*bdg_ifp;
			bdg_ifp = bridge_in(m);
			if (bdg_ifp != BDG_LOCAL && bdg_ifp != BDG_DROP)
				bdg_forward(&m, bdg_ifp);
			if (((bdg_ifp != BDG_LOCAL) && (bdg_ifp != BDG_BCAST) &&
			    (bdg_ifp != BDG_MCAST)) || bdg_ifp == BDG_DROP) {
				m_freem(m);
				continue;
			}
		}
#endif /* BRIDGE */
#endif

#elif __FreeBSD_version < 400000
		/*
		 * Try to conjure up a new mbuf cluster. If that
		 * fails, it means we have an out of memory condition and
		 * should leave the buffer in place and continue. This will
		 * result in a lost packet, but there's little else we
		 * can do in this situation.
		 */
		 
		if (fet_newbuf(sc, cur_rx) == ENOBUFS) {
			ifp->if_ierrors++;
			cur_rx->fet_ptr->fet_status = FET_RXSTAT;
			cur_rx->fet_ptr->fet_ctl = FET_RXCTL|FET_RXLEN;
			continue;
		}
                
		ifp->if_ipackets++;
		eh = mtod(m, struct ether_header *);
		m->m_pkthdr.rcvif = ifp;
		m->m_pkthdr.len = m->m_len = total_len;
#if NBPFILTER > 0
		/*
		 * Handle BPF listeners. Let the BPF user see the packet, but
		 * don't pass it up to the ether_input() layer unless it's
		 * a broadcast packet, multicast packet, matches our ethernet
		 * address or the interface is in promiscuous mode.
		 */
		if (ifp->if_bpf) {
			bpf_mtap(ifp, m);
			if (ifp->if_flags & IFF_PROMISC &&
				(bcmp(eh->ether_dhost, sc->arpcom.ac_enaddr,
						ETHER_ADDR_LEN) &&
					(eh->ether_dhost[0] & 1) == 0)) {
				m_freem(m);
				continue;
			}
		}
#endif
#endif
		/* Remove header from mbuf and pass it on. */
		m_adj(m, sizeof(struct ether_header));
		ether_input(ifp, eh, m);
	}

	return;
}

void fet_rxeoc(sc)
	struct fet_softc		*sc;
{
	fet_rxeof(sc);
	
	if (!fet_safe_rx_off(sc))
		printf("fet%d: RX shutdown error! \n", sc->fet_unit);
		
	CSR_WRITE_4(sc, FET_RXADDR, vtophys(sc->fet_cdata.fet_rx_head->fet_ptr));
	FET_SETBIT16(sc, FET_COMMAND, FET_CMD_RX_ON);	
	return;
}

/*
 * A frame was downloaded to the chip. It's safe for us to clean up
 * the list buffers.
 */

static void fet_txeof(sc)
	struct fet_softc		*sc;
{
	struct fet_chain		*cur_tx;
	struct ifnet		*ifp;
	register struct mbuf	*n;
	
	ifp = &sc->arpcom.ac_if;

	/* Clear the timeout timer. */
	ifp->if_timer = 0;

	/* Sanity check. */
	if (sc->fet_cdata.fet_tx_head == NULL)
		return;
	/*
	 * Go through our tx list and free mbufs for those
	 * frames that have been transmitted.
	 */
	while(sc->fet_cdata.fet_tx_head->fet_mbuf != NULL) {
		u_int32_t		txstat;
		cur_tx = sc->fet_cdata.fet_tx_head;
		if (cur_tx->fet_ptr->fet_ctl & FET_TXCTL_FIRSTFRAG) {
			txstat = cur_tx->fet_ptr->fet_status;
			if (txstat &FET_TXSTAT_ABT|| txstat &FET_TXSTAT_UDF)
			{
				while(CSR_READ_2(sc,FET_COMMAND) & FET_CMD_TX_ON)
					;
				FET_TXOWN(cur_tx) = FET_TXSTAT_OWN;
				CSR_WRITE_4(sc, FET_TXADDR, vtophys(cur_tx->fet_ptr));
					break;
			}

			if (txstat & FET_TXSTAT_OWN)
				break;

			if (txstat & FET_TXSTAT_ERRSUM) {
				ifp->if_oerrors++;
				if (txstat & FET_TXSTAT_DEFER)
					ifp->if_collisions++;
				if (txstat & FET_TXSTAT_LATECOLL)
					ifp->if_collisions++;
				}

			ifp->if_collisions +=(txstat & FET_TXSTAT_COLLCNT);

			ifp->if_opackets++;
		}
		if (cur_tx->fet_mbuf != NULL) {
			/*
			 * we do need to check non-first mbuf, since some of existing
			 * code does not call M_PREPEND properly.
			 * (example: call to bpf_mtap from drivers)
			 */
#if __FreeBSD_version >= 410000
			if ((cur_tx->fet_mbuf->m_flags & M_PKTHDR) != 0 && cur_tx->fet_mbuf->m_pkthdr.aux) {
				m_freem(cur_tx->fet_mbuf->m_pkthdr.aux);
				cur_tx->fet_mbuf->m_pkthdr.aux = NULL;
			}
#endif
			MFREE(cur_tx->fet_mbuf, n);
			cur_tx->fet_mbuf = NULL;
		}
		sc->fet_cdata.fet_free_tx_count++;

		if (sc->fet_cdata.fet_tx_head == sc->fet_cdata.fet_tx_tail) {
			sc->fet_cdata.fet_tx_head = NULL;
			sc->fet_cdata.fet_tx_tail = NULL;
			break;
		}
		sc->fet_cdata.fet_tx_head = cur_tx->fet_nextdesc;
	}
	return;
}

/*
 * TX 'end of channel' interrupt handler.
 */
static void fet_txeoc(sc)
	struct fet_softc		*sc;
{
	struct ifnet		*ifp;

	ifp = &sc->arpcom.ac_if;
	ifp->if_timer = 0;
	if (sc->fet_cdata.fet_tx_head == NULL) {
		ifp->if_flags &= ~IFF_OACTIVE;
		sc->fet_cdata.fet_tx_tail = NULL;
	}
	return;
}
/*
#if __FreeBSD_version >= 400000
static void fet_tick(xsc)
	void			*xsc;
{
	struct fet_softc	*sc;
	struct mii_data		*mii;
	int			s;

	s = splimp();

	sc = xsc;
	mii = device_get_softc(sc->fet_miibus);
	mii_tick(mii);

	sc->fet_stat_ch = timeout(fet_tick, sc, hz);

	splx(s);

	return;
}
#endif
*/
static void fet_intr(arg)
	void			*arg;
{
	struct fet_softc	*sc;
	struct ifnet		*ifp;
	u_int16_t		status;

	sc = arg;
	ifp = &sc->arpcom.ac_if;

	/* Supress unwanted interrupts. */
	if (!(ifp->if_flags & IFF_UP)) {
		fet_stop(sc);
		return;
	}

	/* Disable interrupts. */
	CSR_WRITE_2(sc, FET_IMR, 0x0000);

	for (;;) {

		status = CSR_READ_2(sc, FET_ISR);
		if (status)
			CSR_WRITE_2(sc, FET_ISR, status);

		if ((status & IMRShadow) == 0)
			break;

		if (status & FET_ISR_PRX)
			fet_rxeof(sc);

		if ((status & FET_ISR_RXE) || (status & FET_ISR_RU) ||
		    (status & FET_ISR_RU) || (status & FET_ISR_OVFI) ||
		    (status & FET_ISR_PKTRACE)) {
			fet_rxeof(sc);
			fet_rxeoc(sc);
		}

		if ((status & FET_ISR_BE )|| (status & FET_ISR_TU)) {
			fet_reset(sc);
			fet_init(sc);
			break;
		}
		
		if ( (status & FET_ISR_PTX)||(status & FET_ISR_TXE)||(status & FET_ISR_TX_ABTI)||(status & FET_ISR_UDFI)) {
			fet_txeof(sc);
			if ((status & FET_ISR_UDFI)||(status & FET_ISR_TX_ABTI)){ 
				ifp->if_oerrors++;
				if (sc->fet_cdata.fet_tx_head != NULL) {
					FET_SETBIT16(sc, FET_COMMAND, FET_CMD_TX_ON);
					FET_SETBIT16(sc, FET_COMMAND, FET_CMD_TDMD1);
				}
			}
			else
				fet_txeoc(sc);
		}

		if (status & FET_ISR_SRCI)
			fet_link_change(sc);
	}

	/* Re-enable interrupts. */
	CSR_WRITE_2(sc, FET_IMR, IMRShadow);

	if (ifp->if_snd.ifq_head != NULL) {
		fet_start(ifp);
	}

	return;

}

/*
 * Encapsulate an mbuf chain in a descriptor by coupling the mbuf data
 * pointers to the fragment pointers.
 */
static int fet_encap(sc, c, m_head)
	struct fet_softc		*sc;
	struct fet_chain		*c;
	struct mbuf		*m_head;
{
	int			frag = 0;
	struct fet_desc		*f = NULL;
	int			total_len;
	struct mbuf		*m;

	m = m_head;
	total_len = 0;

	/*
	 * NIC wants packet buffers to be longword
	 * aligned, but very often our mbufs aren't. Rather than
	 * waste time trying to decide when to copy and when not
	 * to copy, just do it all the time.
	 */
	if (m != NULL) {
		struct mbuf		*m_new = NULL;

		MGETHDR(m_new, M_DONTWAIT, MT_DATA);
		if (m_new == NULL) {
			printf("fet%d: no memory for tx list", sc->fet_unit);
			return(1);
		}
		if (m_head->m_pkthdr.len > MHLEN) {
			MCLGET(m_new, M_DONTWAIT);
			if (!(m_new->m_flags & M_EXT)) {
				m_freem(m_new);
				printf("fet%d: no memory for tx list",
						sc->fet_unit);
				return(1);
			}
		}
		m_copydata(m_head, 0, m_head->m_pkthdr.len,	
					mtod(m_new, caddr_t));
		m_new->m_pkthdr.len = m_new->m_len = m_head->m_pkthdr.len;
		m_freem(m_head);
		m_head = m_new;
		/*
		 * The Rhine chip doesn't auto-pad, so we have to make
		 * sure to pad short frames out to the minimum frame length
		 * ourselves.
		 */
		if (m_head->m_len < FET_MIN_FRAMELEN) {
			m_new->m_pkthdr.len += FET_MIN_FRAMELEN - m_new->m_len;
			m_new->m_len = m_new->m_pkthdr.len;
		}
		f = c->fet_ptr;
		f->fet_data = vtophys(mtod(m_new, caddr_t));
		f->fet_ctl = total_len = m_new->m_len;
		f->fet_ctl |= FET_TXCTL_TLINK|FET_TXCTL_FIRSTFRAG;
		f->fet_status = 0;
		frag = 1;
	}

	c->fet_mbuf = m_head;
	c->fet_ptr->fet_ctl |= FET_TXCTL_LASTFRAG|FET_TXCTL_FINT;
	c->fet_ptr->fet_next = vtophys(c->fet_nextdesc->fet_ptr);

	return(0);
}

/*
 * Main transmit routine. To avoid having to do mbuf copies, we put pointers
 * to the mbuf data regions directly in the transmit lists. We also save a
 * copy of the pointers since the transmit list fragment pointers are
 * physical addresses.
 */

static void fet_start(ifp)
	struct ifnet		*ifp;
{
	struct fet_softc	*sc;
	struct mbuf		*m_head = NULL, *m = NULL;
	struct fet_chain	*cur_tx = NULL, *start_tx, *first_tx = NULL;
	int 			frag_count, frag_need=0;

	sc = ifp->if_softc;
	
/*
	if (sc->fet_autoneg) {
		sc->fet_tx_pend = 1;
		return;
	}
*/
	/*
	 * check the transmit process is not proceeding now.
	 */
	if (ifp->if_flags & IFF_OACTIVE)
		return;
	/*
	 * Check for an available queue slot. If there are none,
	 * punt.
	 */
	if (sc->fet_cdata.fet_tx_free->fet_mbuf != NULL) {
		ifp->if_flags |= IFF_OACTIVE;
		return;
	}
	start_tx = sc->fet_cdata.fet_tx_free;
	while(sc->fet_cdata.fet_tx_free->fet_mbuf == NULL) {
		if (sc->fet_chip_revid >= REV_ID_VT3065_A) {
        	    struct mbuf *m_tmp;

            	    for(m_tmp=ifp->if_snd.ifq_head, frag_need=0; m_tmp!=NULL; m_tmp=m_tmp->m_next)
              	        frag_need++;
                    if( (sc->fet_cdata.fet_free_tx_count) < frag_need)
                    {
	       	             ifp->if_flags |= IFF_OACTIVE;
	                     break;
                    }
        	}

		IF_DEQUEUE(&ifp->if_snd, m_head);
		if (m_head == NULL)
			break;
		
		if (sc->fet_chip_revid < REV_ID_VT3065_A) {

			/* Pick a descriptor off the free list. */
			cur_tx = sc->fet_cdata.fet_tx_free;
			sc->fet_cdata.fet_tx_free = cur_tx->fet_nextdesc;
			/* Pack the data into the descriptor. */
			fet_encap(sc, cur_tx, m_head);

			if (cur_tx != start_tx)
				FET_TXOWN(cur_tx) = FET_TXSTAT_OWN;

#if __FreeBSD_version >= 400000 || (__FreeBSD_version < 400000 && NBPFILTER > 0)
			/*
		 	 * If there's a BPF listener, bounce a copy of this frame
		 	 * to him.
		 	 */
			if (ifp->if_bpf)		
				bpf_mtap(ifp, cur_tx->fet_mbuf);
#endif
			FET_TXOWN(cur_tx) = FET_TXSTAT_OWN;
			sc->fet_cdata.fet_free_tx_count--;
		}
		else {

			frag_count = 1;			 
			for (m = m_head; m != NULL; m = m->m_next) {
				/* Pick a descriptor off the free list. */
				cur_tx = sc->fet_cdata.fet_tx_free;
        
				sc->fet_cdata.fet_tx_free = cur_tx->fet_nextdesc;
				sc->fet_cdata.fet_free_tx_count--;
				if (frag_count == 1)
					first_tx = cur_tx;
			
				cur_tx->fet_ptr->fet_data = vtophys(mtod(m, caddr_t));

				/* pad the short frame to minimum frame length */
				/* ps: Pad those packets which only occupy 1 fragment */
				if (m->m_len < FET_MIN_FRAMELEN && frag_count == 1 && m->m_next == NULL) {
					cur_tx->fet_ptr->fet_ctl = FET_MIN_FRAMELEN;
					m->m_pkthdr.len += FET_MIN_FRAMELEN - m->m_len;
					m->m_len = m->m_pkthdr.len;
					}
				else
					cur_tx->fet_ptr->fet_ctl = m->m_len;
				
				if (frag_count == 1)
					cur_tx->fet_ptr->fet_ctl |= FET_TXCTL_FINT|FET_TXCTL_TLINK|		\
													FET_TXCTL_FIRSTFRAG;
				else
					cur_tx->fet_ptr->fet_ctl |= FET_TXCTL_FINT|FET_TXCTL_TLINK;

				cur_tx->fet_ptr->fet_status = 0;
				cur_tx->fet_mbuf = m;			
				cur_tx->fet_ptr->fet_next = vtophys(cur_tx->fet_nextdesc->fet_ptr);
				
				frag_count++;				

#if __FreeBSD_version >= 400000 || (__FreeBSD_version < 400000 && NBPFILTER > 0)
				/*
		 		* If there's a BPF listener, bounce a copy of this frame
		 		* to him.
		 		*/
				if (ifp->if_bpf)		
					bpf_mtap(ifp, cur_tx->fet_mbuf);
#endif
			} /* end for */
			cur_tx->fet_ptr->fet_ctl |=	FET_TXCTL_LASTFRAG;
			FET_TXOWN(first_tx) = FET_TXSTAT_OWN;
		} /* end else */
		FET_SETBIT16(sc, FET_COMMAND, FET_CMD_TDMD1);
	}
	/*
	 * If there are no frames queued, bail.
	 */
	if (cur_tx == NULL)
		return;
	sc->fet_cdata.fet_tx_tail = cur_tx;

	if (sc->fet_cdata.fet_tx_head == NULL)
		sc->fet_cdata.fet_tx_head = start_tx;

	/*
	 * Set a timeout in case the chip goes out to lunch.
	 */
	ifp->if_timer = 5;
	return;
}

/* Set flow control capability accroding to ANAR and ANLPAR register in MII */
/* The half duplex flow control capability is turn off now, because it's not in the spec.*/
/* Follow the table 28B-3 in the IEEE Standard 802.3, 2000 Edition to set */
/* full duplex flow control capability*/
static void flow_control_ability(sc)
	struct fet_softc	*sc;
{
    unsigned int PHYANAR_temp, PHYANLPAR_temp, MIISR_temp, FlowCR1_temp, Micr0_temp;

    if (sc->fet_chip_revid >= 0x40 && sc->fet_chip_revid < 0x80) {
       /* Read the old value of FlowCR1 register */
        Micr0_temp = CSR_READ_1(sc, FET_Micr0);

        /*check whether NIC is operated in full duplex mode */
        /* in full duplex mode*/
        if (sc->fet_full_duplex == 1) {
            /* read PAUSE and ASM_DIR in PHYANAR and MIISR register*/
            PHYANAR_temp = (fet_read_mii(sc, PHY_ANAR) & 0x0C00) >> 10;
            PHYANLPAR_temp = (fet_read_mii(sc, PHY_LPAR) & 0x0C00) >> 10;

            /* Local: ASM_DIR=1, PAUSE=0   Remote: ASM_DIR=1, PAUSE=1*/
            if ( (PHYANAR_temp & 0x02) && (!(PHYANAR_temp & 0x01)) && (PHYANLPAR_temp & 0x02) && (PHYANLPAR_temp & 0x01)) {
                /* Disable PAUSE receive */
                Micr0_temp = Micr0_temp & 0xF7;
            }
            /* Local: ASM_DIR=Don't care, PAUSE=1   Remote: ASM_DIR=Don't care, PAUSE=1*/
            else if (PHYANAR_temp & 0x01 && PHYANLPAR_temp & 0x01) {
                /* Enable PAUSE receive */
                Micr0_temp = Micr0_temp | 0x08;
            }
            /* Local: ASM_DIR=1, PAUSE=1   Remote: ASM_DIR=1, PAUSE=0*/
            else if ( (PHYANAR_temp & 0x02) && (PHYANAR_temp & 0x01) && (PHYANLPAR_temp & 0x02) && (!(PHYANLPAR_temp & 0x01))) {
                /* Enable PAUSE receive */
                Micr0_temp = Micr0_temp | 0x08;
            }
            /* Other conditions*/
            else {
                /* Disable PAUSE receive */
                Micr0_temp = Micr0_temp & 0xF7;
            }
        }
        /* in half duplex mode*/
        else {
            /* Disable PAUSE receive */
            Micr0_temp = Micr0_temp & 0xF7;        
        }
        /* Disable half duplex flow control */
        Micr0_temp = Micr0_temp & 0xFB;

        /* Disable full duplex PAUSE transmit */
        Micr0_temp = Micr0_temp & 0xEF;

        /* Set the Micr0 register*/
        CSR_WRITE_1(sc, FET_Micr0, Micr0_temp );
    }
    else if (sc->fet_chip_revid >= 0x80) {
        /* Read the old value of FlowCR1 register */
        FlowCR1_temp = CSR_READ_1(sc, FET_FlowCR1);


        /*check whether NIC is operated in full duplex mode */
        /* in full duplex mode*/
        if (sc->fet_full_duplex == 1) {
            /* read PAUSE and ASM_DIR in PHYANAR and MIISR register*/
            PHYANAR_temp = (fet_read_mii(sc, PHY_ANAR) & 0x0C00) >> 10;
            MIISR_temp = (CSR_READ_1(sc, FET_MIISR) & 0x60) >> 5;

            /* Local: ASM_DIR=1, PAUSE=0   Remote: ASM_DIR=1, PAUSE=1*/
            if ( (PHYANAR_temp & 0x02) && (!(PHYANAR_temp & 0x01)) && (MIISR_temp & 0x02) && (MIISR_temp & 0x01)) {
                /* Enable PAUSE transmit */
                FlowCR1_temp = FlowCR1_temp | 0x04;
                /* Disable PAUSE receive */
                FlowCR1_temp = FlowCR1_temp & 0xFD;
            }
            /* Local: ASM_DIR=Don't care, PAUSE=1   Remote: ASM_DIR=Don't care, PAUSE=1*/
            else if (PHYANAR_temp & 0x01 && MIISR_temp & 0x01) {
                /* Enable PAUSE transmit */
                FlowCR1_temp = FlowCR1_temp | 0x04;
                /* Enable PAUSE receive */
                FlowCR1_temp = FlowCR1_temp | 0x02;
            }
            /* Local: ASM_DIR=1, PAUSE=1   Remote: ASM_DIR=1, PAUSE=0*/
            else if ( (PHYANAR_temp & 0x02) && (PHYANAR_temp & 0x01) && (MIISR_temp & 0x02) && (!(MIISR_temp & 0x01))) {
                /* Disable PAUSE transmit */
                FlowCR1_temp = FlowCR1_temp & 0xFB;
                /* Enable PAUSE receive */
                FlowCR1_temp = FlowCR1_temp | 0x02;
            }
            /* Other conditions*/
            else {
                /* Disable PAUSE transmit */
                FlowCR1_temp = FlowCR1_temp & 0xFB;
                /* Disable PAUSE receive */
                FlowCR1_temp = FlowCR1_temp & 0xFD;
            }
        }
        /* in half duplex mode*/
        else {
            /* Disable PAUSE transmit */
            FlowCR1_temp = FlowCR1_temp & 0xFB;
            /* Disable PAUSE receive */
            FlowCR1_temp = FlowCR1_temp & 0xFD;        
        }
        /* Disable half duplex flow control */
        FlowCR1_temp = FlowCR1_temp & 0xFE;

        /* Set the FlowCR1 register*/
        CSR_WRITE_1(sc, FET_FlowCR1, FlowCR1_temp);
    }
}
static void fet_init(xsc)
	void			*xsc;
{
	struct fet_softc		*sc = xsc;
	struct ifnet		*ifp = &sc->arpcom.ac_if;	
#if __FreeBSD_version >= 400000
	struct mii_data		*mii;
#endif
	int			s;
	u_int16_t	FDXFlag;
#ifdef DEBUG
	u_int8_t	aa;
	char		c[5];
#endif

#if __FreeBSD_version < 400000
    if (sc->fet_autoneg)
        return;
#endif
      s = splimp();
#if __FreeBSD_version >= 400000
	mii = device_get_softc(sc->fet_miibus);

#endif

    /*
     * Cancel pending I/O and free all RX/TX buffers.
     */
    fet_stop(sc);

	    
    fet_reset(sc);

	/* set the MIIAD again because software reset will reset the value */
     FET_SETBIT(sc, FET_MIIADDR, FET_MIIADDR_MSRCEN | FET_MIIADDR_MAD0);

	/* set TCR RCR threshold */
	/* set DMAL to 110 , and not to change REQOPT set in attach fucntion */
	FET_SETBIT(sc, FET_BCR0, FET_BCR0_DMAL2 | FET_BCR0_DMAL1);	
	CSR_WRITE_1(sc, FET_BCR1, 0x00);    

	/* Set RX threshhold */
	FET_CLRBIT(sc, FET_RXCFG, FET_RXCFG_RX_THRESH);
	FET_SETBIT(sc, FET_RXCFG, FET_RXTHRESH_1024BYTES);
	/* Set TX threshhold */
	FET_CLRBIT(sc, FET_TXCFG, FET_TXCFG_TX_THRESH);
	FET_SETBIT(sc, FET_TXCFG, FET_TXTHRESH_512BYTES);

	/* Turn on bit3 (OFSET) in TCR during MAC initialization */
        FET_SETBIT(sc, FET_TXCFG, FET_TXCFG_OFSET);

	/* QPacket setting */
	if (sc->fet_chip_revid < REV_ID_VT3065_A)
            /* disable queue packet */
            FET_SETBIT(sc, FET_CFGB, FET_CFGB_QPKTDIS);
	else {		
            /* enable queue packet */
            FET_CLRBIT(sc, FET_CFGB, FET_CFGB_QPKTDIS);
        }

	/* set backoff algorithm ,disable the right-most 4-bit off CFGD[0] during initialization */
        FET_CLRBIT(sc, FET_CFGD,(FET_CFGD_CAP|FET_CFGD_CRADOM|FET_CFGD_MBA|FET_CFGD_BAKOPT));

	/* Init circular RX list. */
	if (fet_list_rx_init(sc) == ENOBUFS) {
		printf("fet%d: initialization failed: no "
			"memory for rx buffers\n", sc->fet_unit);
		fet_stop(sc);
		(void)splx(s);
		return;
	}

	/*
	 * Init tx descriptors.
	 */
	fet_list_tx_init(sc);

	/* If we want promiscuous mode, set the allframes bit. */
	if (ifp->if_flags & IFF_PROMISC)
		FET_SETBIT(sc, FET_RXCFG, FET_RXCFG_RX_PROMISC);
	else
		FET_CLRBIT(sc, FET_RXCFG, FET_RXCFG_RX_PROMISC);

	/* Set capture broadcast bit to capture broadcast frames. */
	if (ifp->if_flags & IFF_BROADCAST)
		FET_SETBIT(sc, FET_RXCFG, FET_RXCFG_RX_BROAD);
	else
		FET_CLRBIT(sc, FET_RXCFG, FET_RXCFG_RX_BROAD);


	/*
	 * Program the multicast filter, if necessary.
	 */
	fet_setmulti(sc);

	/*
	 * Load the address of the RX list.
	 */
	CSR_WRITE_4(sc, FET_RXADDR, vtophys(sc->fet_cdata.fet_rx_head->fet_ptr));		
	CSR_WRITE_4(sc, FET_TXADDR, vtophys(&sc->fet_ldata->fet_tx_list[0]));

	
    /* query fullduplex */
    FDXFlag = fet_query_auto(sc);

	if (FDXFlag == 1) {	 	
		FET_SETBIT16(sc, FET_COMMAND, FET_CMD_FDX);       /* Set Chip Fullduplex mode */		
                sc->fet_full_duplex = 1;
        }
	else {
		FET_CLRBIT16(sc, FET_COMMAND, FET_CMD_FDX);      /* Set Chip Halfduplex mode */		
                sc->fet_full_duplex = 0;
	}
        if (sc->fet_chip_revid >= 0x40)
            flow_control_ability (sc);

#ifdef DEBUG
	aa = CSR_READ_1(sc, FET_MIICR);
	printf("MIICR: %x \n",aa);
	aa = CSR_READ_1(sc, FET_MIIADDR);
	printf("MIIADDR: %x \n",aa);
	db_readline(c,4);
#endif

	/* Enable receiver and transmitter. */
	CSR_WRITE_1(sc, FET_COMMAND, FET_CMD_START|
				    FET_CMD_TX_ON|FET_CMD_RX_ON);
        FET_SETBIT16(sc, FET_COMMAND, FET_CMD_TX_NOPOLL);

         
	/*
	 * Enable interrupts.
	 */	
	CSR_WRITE_2(sc, FET_IMR, IMRShadow);

#if __FreeBSD_version >= 400000
	mii_mediachg(mii);
#endif
	ifp->if_flags |= IFF_RUNNING;
	ifp->if_flags &= ~IFF_OACTIVE;


	(void)splx(s);
/*
#if __FreeBSD_version >= 400000	
	sc->fet_stat_ch = timeout(fet_tick, sc, hz);
#endif
*/

	return;
}

/*
 * Set media options.
 */
static int fet_ifmedia_upd(ifp)
	struct ifnet		*ifp;
{
	struct fet_softc		*sc;
	struct ifmedia		*ifm = NULL;
        struct mii_data         *mii = NULL;

	sc = ifp->if_softc;
#if __FreeBSD_version >= 400000
        mii = device_get_softc(sc->fet_miibus);
        ifm = &mii->mii_media;
        if (ifp->if_flags & IFF_UP)
		fet_init(sc);
#elif __FreeBSD_version < 400000
        ifm = &sc->ifmedia;
	if (IFM_TYPE(ifm->ifm_media) != IFM_ETHER)
		return(EINVAL);
#endif
	if (IFM_TYPE(ifm->ifm_media) != IFM_ETHER)
		return (EINVAL);
/*
        printf("chenyp:fet_ifmedia_upd:ifm_media=%x\n",ifm->ifm_media);
        switch(IFM_SUBTYPE(ifm->ifm_media)) {
            case IFM_AUTO :
                
                break;
            case IFM_100_TX :
            case IFM_100_FX :
                if ((ifm->ifm_media & IFM_GMASK) == IFM_FDX)
                break;
            case IFM_10_FL :
            case IFM_10_T :
                break;
        }
*/
	/*
	if (IFM_SUBTYPE(ifm->ifm_media) == IFM_AUTO)
		fet_autoneg_mii(sc, FET_FLAG_SCHEDDELAY, 1);
	else
		fet_setmode_mii(sc, ifm->ifm_media);
	*/

	return(0);
}

/*
 * Report current media status.
 */
#if __FreeBSD_version >= 400000
static void fet_ifmedia_sts(ifp, ifmr)
	struct ifnet		*ifp;
	struct ifmediareq	*ifmr;
{
	struct fet_softc		*sc;
	struct mii_data		*mii;

	sc = ifp->if_softc;
	mii = device_get_softc(sc->fet_miibus);
	mii_pollstat(mii);
	ifmr->ifm_active = mii->mii_media_active;
	ifmr->ifm_status = mii->mii_media_status;

	return;
}
#elif __FreeBSD_version < 400000
static void fet_ifmedia_sts(ifp, ifmr)
	struct ifnet		*ifp;
	struct ifmediareq	*ifmr;
{
	struct fet_softc		*sc;
	u_int8_t	SRbak;
    	u_int16_t	FDXFlag;
	/*
	u_int16_t		advert = 0, ability = 0;
	*/

	sc = ifp->if_softc;

	ifmr->ifm_active = IFM_ETHER;

	/* query fullduplex */
	FDXFlag = fet_query_auto(sc);

	/* query speed */
	SRbak = CSR_READ_1(sc, FET_MIISR);

	if(FDXFlag)
	   ifmr->ifm_active |= (SRbak & FET_MIISR_SPEED)? (IFM_10_T & IFM_FDX): (IFM_100_TX & IFM_FDX);
	else
	   ifmr->ifm_active |= (SRbak & FET_MIISR_SPEED)? (IFM_10_T & IFM_HDX): (IFM_100_TX & IFM_HDX);

	return;
}
#endif

static int fet_ioctl(ifp, command, data)
	struct ifnet		*ifp;
	u_long			command;
	caddr_t			data;
{
	struct fet_softc		*sc = ifp->if_softc;
	struct ifreq		*ifr = (struct ifreq *) data;

#if __FreeBSD_version >= 400000
	struct mii_data		*mii;
#endif
	int			s, error = 0;

	s = splimp();
	switch(command) {
	case SIOCSIFADDR:
	case SIOCGIFADDR:
	case SIOCSIFMTU:
		error = ether_ioctl(ifp, command, data);
		break;
	case SIOCSIFFLAGS:
		if (ifp->if_flags & IFF_UP) {
			fet_init(sc);
		} else {
			if (ifp->if_flags & IFF_RUNNING)
				fet_stop(sc);
		}
		error = 0;
		break;
	case SIOCADDMULTI:
	case SIOCDELMULTI:
		fet_setmulti(sc);
		error = 0;
		break;
	case SIOCGIFMEDIA:
	case SIOCSIFMEDIA:
#if __FreeBSD_version >= 400000
		mii = device_get_softc(sc->fet_miibus);
		error = ifmedia_ioctl(ifp, ifr, &mii->mii_media, command);

#elif __FreeBSD_version < 400000
		error = ifmedia_ioctl(ifp, ifr, &sc->ifmedia, command);
#endif
		break;
	default:

		error = EINVAL;
		break;
	}

	(void)splx(s);
	return(error);
}

static void fet_watchdog(ifp)
	struct ifnet		*ifp;
{
	struct fet_softc		*sc;

	sc = ifp->if_softc;
	
	/*
	if (sc->fet_autoneg) {
		fet_autoneg_mii(sc, FET_FLAG_DELAYTIMEO, 1);
		return;
	}
	*/

	ifp->if_oerrors++;
	printf("fet%d: watchdog timeout\n", sc->fet_unit);

	/*
	if (!(fet_phy_readreg(sc, PHY_BMSR) & PHY_BMSR_LINKSTAT))
		printf("fet%d: no carrier - transceiver cable problem?\n",
								sc->fet_unit);
	*/

	fet_stop(sc);
	fet_reset(sc);
	fet_init(sc);

	if (ifp->if_snd.ifq_head != NULL)
		fet_start(ifp);

	return;
}

/*
 * Stop the adapter and free any mbufs allocated to the
 * RX and TX lists.
 */
static void fet_stop(sc)
	struct fet_softc	*sc;
{
	register int		i;
	struct ifnet		*ifp;

	ifp = &sc->arpcom.ac_if;
	ifp->if_timer = 0;
/*
#if __FreeBSD_version >= 400000
	untimeout(fet_tick, sc, sc->fet_stat_ch);
#endif
*/		
	FET_CLRBIT16(sc, FET_COMMAND, FET_CMD_TX_ON);

	if (!fet_safe_rx_off(sc))
		printf("fet%d: RX shutdown error! \n", sc->fet_unit);

	FET_SETBIT16(sc, FET_COMMAND, FET_CMD_STOP);

	CSR_WRITE_2(sc, FET_IMR, 0x0000);
	CSR_WRITE_4(sc, FET_TXADDR, 0x00000000);
	CSR_WRITE_4(sc, FET_RXADDR, 0x00000000);

	/*
	 * Free data in the RX lists.
	 */
	for (i = 0; i < FET_RX_LIST_CNT; i++) {
		if (sc->fet_cdata.fet_rx_chain[i].fet_mbuf != NULL) {
			m_freem(sc->fet_cdata.fet_rx_chain[i].fet_mbuf);
			sc->fet_cdata.fet_rx_chain[i].fet_mbuf = NULL;
		}
	}
	bzero((char *)&sc->fet_ldata->fet_rx_list,
		sizeof(sc->fet_ldata->fet_rx_list));

	/*
	 * Free the TX list buffers.
	 */
	for (i = 0; i < FET_TX_LIST_CNT; i++) {
		if (sc->fet_cdata.fet_tx_chain[i].fet_mbuf != NULL) {
			m_freem(sc->fet_cdata.fet_tx_chain[i].fet_mbuf);
			sc->fet_cdata.fet_tx_chain[i].fet_mbuf = NULL;
		}
	}

	bzero((char *)&sc->fet_ldata->fet_tx_list,
		sizeof(sc->fet_ldata->fet_tx_list));

	ifp->if_flags &= ~(IFF_RUNNING | IFF_OACTIVE);

	return;
}

/*
 * Stop all chip I/O so that the kernel's probe routines don't
 * get confused by errant DMAs when rebooting.
 */
#if __FreeBSD_version >= 400000
static void fet_shutdown(dev)
	device_t		dev;
{
	struct fet_softc	*sc;
	
	sc = device_get_softc(dev);
	fet_stop(sc);
	return;
}
 
#elif __FreeBSD_version < 400000
static void fet_shutdown(howto, arg)
	int			howto;
	void			*arg;
{
	struct fet_softc		*sc = (struct fet_softc *)arg;

	fet_stop(sc);

	return;
}

static struct pci_device fet_device = {
	"fet",
	fet_probe,
	fet_attach,
	&fet_count,
	NULL
};
DATA_SET(pcidevice_set, fet_device);
#endif
