/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  NetCentric Computing with Object Rexx                                   */
/*  Programming Example                                                     */
/*                                                                          */
/*    IBM Corporation 1998                                                  */
/*                                                                          */
/*  Agency.cmd  -  An Advanced TCP/IP Socket Server with sessions           */
/*                                                                          */
/*    ---> This server requires the asynchronous client client.cmd!         */
/*                                                                          */
/*    Parameters:                                                           */
/*      Port:   server port number (optional)                               */
/*                                                                          */
/*  Documentation: see agency.doc in directory docs                         */
/*                                                                          */
/*--------------------------------------------------------------------------*/

Parse Arg Port 
if Port = '' then Port = 1925          /* Default server port is 1925       */
aServer = .AgencyServer~new(Port)      /* create a new agency server        */ 
aServer~startAccepting                 /* method inherited from "tcpServer" */    
aServer~terminate                      /* terminate all client sessions     */    
aServer~shutdown                       /* shutdown the server               */                    

/****************************************************************************/

::REQUIRES "servers.frm"               /* Load the servers framework        */   
::REQUIRES "security.frm"              /* Load the security framework       */   

/*--------------------------------------------------------------------------*/
/* AgencyServer Class definition                                            */
/*--------------------------------------------------------------------------*/
::CLASS AgencyServer SUBCLASS tcpServer PUBLIC

/*--------------------------------------------------------------------------*/
::METHOD init
  expose Capabilities 
  use arg Port

  self~init:super(Port)                /* Run the superclass init           */
  self~Sessions= .bag~new              /* Create Sessions with no sessions  */                                          
                                       /* setup the authorization resource  */
  capabilities = .directory~new        /* -- this is a set of samples --    */
  capabilities["emaseSnepO"]= 0        /* maximum capabilities              */
  capabilities["none"]= .noway~new     /* minimum capabilities              */
  capabilities["tidua"]= .dumper~new(.stream~new("audit.log"))
  capabilities["ecalper"]= .replacer~new

/*--------------------------------------------------------------------------*/
::METHOD NewClient UNGUARDED           /* Over-write superclass method      */    
  expose Capabilities  
  use arg cSocket, Prefix              /* parm: client socket and prefix    */
                                       /* Create a session at server        */ 
  session = .Session~new(self, cSocket, Prefix, Capabilities)
  session~logon                        /* and start it                      */

  /* Note: NewClient must return NOTHING! It would block accepting new      */
  /*       clients otherwise!                                               */

/*--------------------------------------------------------------------------*/
::METHOD Sessions ATTRIBUTE            /* get collection of sessions        */

/*--------------------------------------------------------------------------*/
::METHOD terminate                     /* Terminate the whole server        */
         
  thisHost = .tcpHost~new
  message = "Server" thisHost~name "is shutting down now!"
  do session over self~Sessions        /* Logoff all sessions on server     */      
    session~logoff(message)
  end


/*--------------------------------------------------------------------------*/
/* Session Class definition                                                 */
/*--------------------------------------------------------------------------*/
::CLASS Session

/*--------------------------------------------------------------------------*/
::METHOD init
  expose server cSocket Platform NL 
  use arg server, cSocket, prefix, capabilities  

  self~name = prefix~word(1)           /* Get the client name from prefix   */
  platform = prefix~word(2)            /* Get the platform id from prefix   */
  password = prefix~word(3)            /* Get security enabler password     */
                                       /* Lookup the security manager       */ 
  self~SecurityManager= capabilities[password]
  if .nil = self~SecurityManager then  /* wrong pw -> minimum capabilities  */ 
    self~SecurityManager= capabilities['none']

  if platform = 'LINUX' | platform = 'AIX' then
    NL = '0a'x                         /* 'new line' character for UNIX     */   
  else
    NL = '0d0a'x                       /* 'new line' characters for nonUNIX */   
  server~Sessions~put(self)            /* register itself at the server     */
                                       
/*--------------------------------------------------------------------------*/
::METHOD name ATTRIBUTE

/*--------------------------------------------------------------------------*/
::METHOD SecurityManager ATTRIBUTE 

/*--------------------------------------------------------------------------*/
::METHOD logon UNGUARDED
  expose server cSocket Platform NL 
  reply                                /* return control to caller          */
                                       /* create an agency                  */
  agency = .Agency~new(cSocket, Platform, self~name, NL, self~SecurityManager)

  do forever                           /* Session handling loop             */ 
    CommandLines = cSocket~ReceiveData /* Receive command line from client  */
    if \cSocket~stillOpen then return  /* if problems, end client session   */ 

    do while CommandLines~length > 0 
      Parse Var CommandLines CommandLine '00'x CommandLines
                                       /* parse into command and options    */
      Parse Var CommandLine Command Option
      Option = option~strip            /* strip off leading/trailing blanks */

      if Option = '' then              /* if no option specified ...        */  
        optstring = '<none>'           /* define substitute                 */
      else
        optstring = Option             /* or use real option otherwise      */

      say "  *" Command "with option:" optstring "(client:" self~name")"
      cSocket~SendData("---Begin_of_transmission---"NL)
                                       /* Create message for the processor  */
      Message = .message~new(agency, Command, 'I', Option)
      Message~send                     /* now send it to him                */
                                       /* process result accordingly        */
      if Message~result = "#>>End_of_session<<#" then do
        self~logoff                    /* Logoff this session               */
        server~Sessions~remove(self)   /* deregister itself from the server */
        return
      end
                                       /* Initiate server shutdown          */ 
      if Message~result = "#>>Shutdown<<#" then do
                                       /* determine no of sessions running  */
        say "    Shutting down" server~Sessions~makearray~last "sessions!" 
        server~close                   /* close server request queue        */  
        return                         /* leave the session loop            */
      end

      if Message~result = "#>>Sessions<<#" then do 
        sessions = server~Sessions~makearray
        cSocket~SendData("There are" Sessions~last "client sessions:"NL) 
        do session over sessions       /* list all sessions on server       */      
          if 0 = session~SecurityManager then
            cSocket~SendData("*" session~name "with superuser capabilities"NL) 
          else                            
            cSocket~SendData("*" session~name "with security manager" session~SecurityManager || NL) 
        end
      end

      else if Message~result \= '' then/* Send the result to the client     */
        cSocket~SendData(Message~result || NL)
                                       /* Send end of answer back to client */
      cSocket~SendData("---End_of_transmission---"NL)
    end
  end

/*--------------------------------------------------------------------------*/
::METHOD logoff UNGUARDED 
  expose server cSocket NL 
                     
  if arg() > 0 then                    /* if there is a  message, ...       */          
    csocket~SendData(arg(1) || NL)     /* notify the client accordingly     */
                                       /* inform client that session ended  */
  csocket~SendData("#>>End_of_session<<#")
                                       /* Remove client from server         */  
  server~removeClient(cSocket, self~name)


/*--------------------------------------------------------------------------*/
/* Agency Class definition                                                  */
/*--------------------------------------------------------------------------*/
::CLASS Agency

/*--------------------------------------------------------------------------*/
::METHOD init                          /* Agency initialisation             */
  expose router cSocket Platform Name NL SecurityManager
  use arg cSocket, Platform, Name, NL, SecurityManager
                                       /* provide a router                  */
  router = .router~new(cSocket, NL)
  
/*--------------------------------------------------------------------------*/
::METHOD show UNGUARDED                /* Method handling show request      */  
  expose cSocket NL                    
  use arg options

  file = options~word(1)
  if file = "" then                    /* send message to client            */
    cSocket~SendData("You have to specify a filename with SHOW"NL) 
  else do
    stream = .stream~new(file)         /* create a stream object for file   */
    if stream~query("EXISTS") = "" then/* does file exist?                  */ 
      cSocket~SendData("The file '"file"' does not exist"NL) 
    else do
      cSocket~SendData("#>>BeginTxtMode<<#") 
      do line over stream              /* Walk through the stream by line   */
        cSocket~SendData(line || NL)   /* and send each line to the client  */
      end 
      cSocket~SendData("#>>EndTxtMode<<#") 
    end 
    stream~close 
  end 
  return ""                            /* the session can continue          */ 

/*--------------------------------------------------------------------------*/
::METHOD Hello                         /* Method handling hello command     */ 
  expose cSocket Name NL                     
                                       /* Do whatever is necessary for this */
  thisHost = .tcpHost~new
  if thisHost~name = cSocket~Hostname then 
    cSocket~SendData("Hi" Name"! It's me," thisHost~name || NL)
  else 
    cSocket~SendData("Hello" Name"! My name is" thisHost~name || NL)
  return ""                            /* the session can continue          */ 

/*--------------------------------------------------------------------------*/
::METHOD Quit                          /* Method handling the quit command  */
  expose router 

  router~terminate                     /* terminate the router services     */ 
  return "#>>End_of_session<<#"        /* tell the session to end           */ 

/*--------------------------------------------------------------------------*/
::METHOD Sessions                      /* Method handling sessions command  */  
  expose cSocket NL SecurityManager                   

  if SecurityManager = 0 then          /* If superuser capabilities         */
    return "#>>Sessions<<#"            /* tell the session to shutdown      */

  cSocket~SendData("You are not authorized to query sessions on server" .tcpHost~new~name"!"NL)
  return ""                            /* the session can continue          */ 

/*--------------------------------------------------------------------------*/
::METHOD Shutdown                      /* Method handling shutdown command  */  
  expose cSocket NL SecurityManager                   

  if SecurityManager = 0 then          /* If superuser capabilities         */
    return "#>>Shutdown<<#"            /* tell the session to shutdown      */

  cSocket~SendData("You are not authorized to shut down server" .tcpHost~new~name"!"NL)
  return ""                            /* the session can continue          */ 

/*--------------------------------------------------------------------------*/
::METHOD ?                             /* Method handling query             */
  expose cSocket NL

  cSocket~SendData("Valid commands:" cmdlist() || NL ) 
  return ""                            /* the session can continue          */ 

/*--------------------------------------------------------------------------*/
::METHOD ??                            /* Method handling query agents      */
  expose cSocket NL
  use arg search

  cSocket~SendData("Available agents:" agentlist(search) || NL ) 
  return ""                            /* the session can continue          */ 

/*--------------------------------------------------------------------------*/
::METHOD Unknown UNGUARDED             /* Method handling local agents      */
  expose router cSocket NL SecurityManager              
  use arg Command, Parms
                                       /* Were there any socket problems?   */ 
  if Command = -1 & command~length > 0 then do  
      Say "Lost client on socket" cSocket "; session closed"
      return "#>>End_of_session<<#"    /* tell the session to end           */ 
  end
  else do                              /* Handle local agents               */
                                       /* translate to lower case for UNIX  */
    command = command~translate(xrange('a', 'z'), xrange('A', 'Z'))

    if command~lastpos('.') = 0 then   /* was a file extension specified?   */
      agentfile = command'.orx'        /* no, try it with .orx extension    */  
    else 
      agentfile = command              /* use the original filename         */  

    fullname = .stream~new(agentfile)~query('EXISTS')
    if fullname = '' then do           /* Unknown command handling          */
      if arg() = 1 then                /* Immediate Unknown                 */
        Command = "UNKNOWN"
      else                             /* Unsup'd command routed to unknown */
        Say "    --> Processing UNKNOWN command for" Command
  
      cSocket~SendData("Invalid command:" Command || NL) 
      cSocket~SendData("Valid commands:" cmdlist() || NL ) 
      cSocket~SendData("Available agents:" agentlist() || NL ) 
      return ""                        /* the session can continue          */ 
    end  

    if SecurityManager \= 0 then       /* provide the security enablement   */
      say '    agent:' agentfile 'with security manager:' SecurityManager                                                   
    else 
      say '    agent:' agentfile 'with superuser capabilities' 
                                                                                
    router~route                       /* make router current destination   */ 

    signal on syntax                   /* to intercept any problems         */  
                                       /* create a local agent              */ 
    agent = .residentAgent~new(agentfile, SecurityManager)     
                                       /* process agent with opt parameters */
    agent~dispatch(Parms[1], cSocket, NL)
  end 
  router~transmit                      /* transmit to client                */
  return ""                            /* the session can continue          */ 

syntax:                                
  except = condition('o')              /* report the problem                */ 
  say 'Error in agent "'except~program'" line' except~position':'  
  do traceline over except~traceback 
    say traceline
  end 
  say sigl~right(6) '*-*' 'SOURCELINE'(sigl)
  say 'Error' except~rc except~errortext 
  say 'Error' except~code except~message

  router~transmit                      /* transmit to client                */
  return ""                            /* the session can continue          */ 
  
/*--------------------------------------------------------------------------*/
::METHOD sAgent UNGUARDED              /* Method handling sendagent command */
  expose router cSocket NL SecurityManager              
  use arg option

  if option = '' then do               /* No option, no agent!              */
    cSocket~SendData("An agent filename has to be specified" || NL)
    return ""                          /* the session can continue          */ 
  end

                                       /* request the agent code            */
  cSocket~SendData('#>>Agent file='option~word(1)'<<#')
  agentstring = cSocket~receiveFile()  /* receive the agent code and        */
  if agentstring = '' then             /* any valid agent received?         */
    return ""                          /* the session can continue          */ 
             
  if SecurityManager \= 0 then
    say '    transient agent with security manager:' SecurityManager
  else 
    say '    transient agent with superuser capabilities' 

  router~route                         /* make router current destination   */ 

  signal on syntax                     /* to intercept any problems         */  
                                       /* Create an agent from the method   */
  agent = .transientAgent~new(agentstring, SecurityManager)     
                                       /* process agent with opt parameters */
  agent~dispatch(option~subword(2), cSocket, NL)

  router~transmit                      /* transmit to client                */
  return ""                            /* the session can continue          */ 

syntax:                                
  except = condition('o')              /* report the problem                */ 
  say 'Error in agent "' || except~program || '" line' except~position':'  
  do traceline over except~traceback 
    say traceline
  end 
  say sigl~right(6) '*-*' 'SOURCELINE'(sigl)
  say 'Error' except~rc except~errortext 
  say 'Error' except~code except~message

  router~transmit                      /* transmit to client                */
  return ""                            /* the session can continue          */ 

/*--------------------------------------------------------------------------*/
::METHOD '='                           /* hide inherited methods            */ 
  forward message(unknown) array('=', '')
::METHOD '=='
  forward message(unknown) array('==', '')
::METHOD '\=' 
  forward message(unknown) array('\=', '')
::METHOD '\=='
  forward message(unknown) array('\==', '')
::METHOD '<>'
  forward message(unknown) array('<>', '')
::METHOD '><'
  forward message(unknown) array('><', '')
::METHOD copy
  forward message(unknown) array('copy', '')
::METHOD class
  forward message(unknown) array('class', '') 
::METHOD setmethod
  forward message(unknown) array('setmethod', '') 
::METHOD unsetmethod
  forward message(unknown) array('unsetmethod', '') 
::METHOD defaultname
  forward message(unknown) array('defaultname', '') 
::METHOD hasmethod
  forward message(unknown) array('hasmethod', '')   
::METHOD objectname
  forward message(unknown) array('objectname', '')  
::METHOD 'objectname='
  forward message(unknown) array('objectname=', '') 
::METHOD request
  forward message(unknown) array('request', '')     
::METHOD run
  forward message(unknown) array('run', '')       
::METHOD start
  forward message(unknown) array('start', '')     
::METHOD startat
  forward message(unknown) array('startat', '')   
::METHOD string
  forward message(unknown) array('string', '')


/*--------------------------------------------------------------------------*/
/* Router Class definition                                                  */
/*--------------------------------------------------------------------------*/
::CLASS Router

/*--------------------------------------------------------------------------*/
::METHOD init                          /* Router initialisation             */
  expose rstream cSocket NL
  use arg cSocket, NL
                                       /* create a stream for routing       */ 
  rstream = .stream~new(sysTempFileName('temp????'))  
                                       /* create a local monitor            */
  monitor = .monitor~new(.stream~new('STDOUT'))
  .local~setentry('output', monitor)   /* make it local for .output         */
  monitor = .monitor~new(.stream~new('STDERR'))
  .local~setentry('error', monitor)    /* make it local for .error          */


/*--------------------------------------------------------------------------*/
::METHOD route                         /* route to rstream destination      */
  expose rstream cSocket Platform Name NL SecurityManager

  .output~destination(rstream)         /* make this the current destination */ 
  .error~destination(rstream)          /* ... also for standard error       */ 
  rstream~open('WRITE REPLACE')        /* Open the stream file              */ 

/*--------------------------------------------------------------------------*/
::METHOD transmit                      /* Router transmission services      */
  expose rstream cSocket Platform Name NL SecurityManager

  .output~destination                  /* reset destination to STDOUT       */
  .error~destination                   /* reset destination to STDERR       */

  do line over rstream                 /* read from rstream until it's empty*/
    select                             /* if line contains control...       */ 
                                       /* GET control                       */              
      when line~abbrev("#>>Get ") then do
        parse var line '#>>Get file=' fname option'<<#' 
        rc = cSocket~SendData(line)    /* send the Get control              */
        if rc > 0 then do
          control = cSocket~ReceiveData/* receive the Acknowledge control   */
          if control = "#>>Acknowledge<<#" then   
                                       /* send file content to client       */ 
            rc = cSocket~SendFile(fname)
          else if control \= "#>>Error<<#" then   
                                       /* transmission problem              */ 
            say 'Received wrong control' control';',
                'Expected #>>Acknowledge<<# or #>>Error<<#'
        end 
      end                              
                                       /* PUT control                       */              
      when line~abbrev("#>>Put ") then do
        parse var line '#>>Put file=' fname mode '<<#' 
        rc = cSocket~SendData(line)    /* send Put control to client        */
        if rc > 0 then do              /* receive file content from client  */ 
          rc = cSocket~ReceiveFile(fname)
          if rc > 0 then               /* if ok, display success message    */
            message = 'File' fname 'successfully received'           
          else 
            message = 'Problems receiving file' fname 
          rc = cSocket~SendData(message || NL)
        end          
      end                              
                                       /* BEEP control                      */              
      when line~abbrev("#>>Beep ") then   
        rc = cSocket~SendData(line)
                                       /* PLAY control                      */              
      when line~abbrev("#>>Play ") then do
        parse var line '#>>Play file=' fname mode '<<#' 
                                       /* does file exist on server?        */
        fullname = .stream~new(fname)~query('EXISTS')
        if fullname = '' then               
          rc = cSocket~SendData("File '"fname"' does not exist"NL) 
        else do                        /* ok, transfer the file now         */
          rc = cSocket~SendData(line)  /* send the Play control             */
          if rc > 0 then do
            rc = cSocket~ReceiveData   /* receive the acknowledge           */
            if rc = "#>>Acknowledge<<#" then   
                                       /* send file content to client       */
              rc = cSocket~SendFile(fname)
          end
        end 
      end                              

      otherwise                        /* send line to client               */
        rc = cSocket~SendData(line || NL)
    end  
    if rc <= 0 then do
      say 'Router: transmission problem' 
      leave
    end 
  end 
  rstream~close                        /* close the TEMP file               */

/*--------------------------------------------------------------------------*/
::METHOD terminate                     /* Router termination services       */
  expose rstream                                         

  rc = sysFileDelete(rstream~string)   /* delete the router stream file     */ 

/*--------------------------------------------------------------------------*/
::ROUTINE cmdlist                      /* Generic fct providing cmdList     */
                                       
  hiddenCmds = .set~of('UNKNOWN')      /* define set of hidden commands     */ 
  hiddenCmdsSupplier = .object~methods
  do while hiddenCmdsSupplier~available
    hiddenCmds~put(hiddenCmdsSupplier~index)
    hiddenCmdsSupplier~next
  end 
  cmds = ''                            /* start with empty command list     */  
  commandSupplier = .Agency~methods    /* get instance methods of class     */ 
  do while commandSupplier~available   /* process all supplier entries      */
    cmd = commandSupplier~index        /* get command from the supplier     */
    if \hiddenCmds~hasindex(cmd) then  /* if a hidden command, skip         */
      cmds = cmds"," cmd               /* include this command in the list  */
    commandSupplier~next               /* move to the next supplier entry   */ 
  end
  
  return cmds~substr(3)                /* remove leading comma and return   */

/*--------------------------------------------------------------------------*/
::ROUTINE agentlist                    /* Generic fct providing agentlist   */

  if arg(1) = '' then 
    search = '*.orx'
  else 
    search = arg(1)'.orx'
  
  rc = sysfiletree(search, output.)    /* retrieve all .orx files           */ 
  if rc > 0 then return '<System ERROR: out of memory>'
   
  if output.0 = 0 then
    agents = '<none>'
  else do
    agents = ''
    do i = 1 to output.0               /* extract filename of each entry    */ 
      agent = filespec('name', output.i)
      pos = agent~lastpos('.')
      if pos > 0 then
        agent = agent~delstr(pos)
      agents = agents"," agent         /* include this command in the list  */
    end
    agents = agents~substr(3)
  end
  return agents                        /* return the list of local agents   */


/*--------------------------------------------------------------------------*/
/* Agent Class definitions                                                  */
/*--------------------------------------------------------------------------*/
::CLASS transientAgent 

::METHOD init                          /* Agent initialisation              */
  use arg agentstring, SecurityMgr                    

  agentcode = makearray(agentstring)   /* convert stream to array of lines  */ 

  signal on syntax          
                                       /* Create method object from array   */  
  agentmethod = .method~new('tempAgent', agentcode) 
  if SecurityMgr \= 0 then             /* provide the security enablement   */
    agentmethod~setSecurityManager(SecurityMgr)
                                       /* method available with 'dispatch'  */
  self~setmethod('DISPATCH', agentmethod)  
  return

syntax:
  raise propagate  
  /* Note: Make sure the security manager is an instance of a valid         */
  /*       security class; the system will block otherwise!                 */

/*--------------------------------------------------------------------------*/
::CLASS residentAgent 

::METHOD init                          /* Agent initialisation              */
  use arg fileName, SecurityMgr                    

  signal on syntax          
                                       /* Create method object from array   */  
  agentmethod = .method~newFile(fileName)
  agentmethod~setprotected             /* make it a protected method        */   
  if SecurityMgr \= 0 then             /* provide the security enablement   */
    agentmethod~setSecurityManager(SecurityMgr)
                                       /* method available with 'dispatch'  */
  self~setmethod('DISPATCH', agentmethod)  
  return

syntax:
  raise propagate  
  /* Note: Make sure the security manager is an instance of a valid         */
  /*       security class; the system will block otherwise!                 */

/*--------------------------------------------------------------------------*/
::ROUTINE makearray                    /* convert stream to array of lines  */  
  use arg Mstring                      /* multi-string delimited by NL      */
                    
  array = .array~new                   /* created an empty array            */
  n = 1                                
  do i = 1 while mstring~length > n    /* process all lines                 */
    pos = mstring~pos('0a'x, n)        /* determine position of the next NL */ 
    array[i] = mstring~substr(n, pos-n)~translate('', '0d'x)
    n = pos + 1                        /* update start position             */ 
  end 
  return array                         /* return the array of strings       */
