/*
*********************************************************************

SAMPLE NAME:  teamcenv.c

USAGE:        teamcenv                   -  This help message
              teamcenv ENVFILE           -  Print contents of environment file
              teamcenv ENVFILE PARAMETER -  Print value of specified parameter

COMPILATION:  cc -o viewExit viewExit.c

ENVIRONMENT VARIABLES:
                       none

DESCRIPTION:  This is the source code for the teamcenv tool.  It reads the
              contents of an environment file for parameters and their
              values.

*********************************************************************
*             IBM TeamConnection for OS/2
*                     Version 2 Release 0
*
*                       5622-717
*    (C) Copyright, IBM Corp., 1996. All Rights Reserved.
*             Licensed Materials - Property of IBM
*
*         US Government Users Restricted Rights
*     - Use, duplication or disclosure restricted by
*         GSA ADP Schedule Contract with IBM Corp.
*
*              IBM is a registered trademark of
*        International Business Machines Corporation
*********************************************************************
*
*           NOTICE TO USERS OF THE SOURCE CODE EXAMPLES
*
* INTERNATIONAL BUSINESS MACHINES CORPORATION PROVIDES THE SOURCE CODE
* EXAMPLES, BOTH INDIVIDUALLY AND AS ONE OR MORE GROUPS, "AS IS" WITHOUT
* WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT
* LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
* PARTICULAR PURPOSE.  THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE
* OF THE SOURCE CODE EXAMPLES, BOTH INDIVIDUALLY AND AS ONE OR MORE GROUPS,
* IS WITH YOU.  SHOULD ANY PART OF THE SOURCE CODE EXAMPLES PROVE
* DEFECTIVE, YOU (AND NOT IBM OR AN AUTHORIZED DEALER) ASSUME THE ENTIRE
* COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
*
*********************************************************************
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>

/* This is based on a limit used for actions in TeamC */
#define maxParmName 40

/* Function Declarations */
void envHelp(void);
int envGetFromEnvFile(FILE *envFile, char *parameterName);
FILE *envInitReadEnvFile(char *envFileName);

/*-----------------------------------------------------------------------------\
|  envHelp:                                                                    |
|  - Explain usage of teamcenv sample program                                  |
|  - Write output to standard error                                            |
\-----------------------------------------------------------------------------*/
void envHelp(void)
{
   fprintf(stderr,"teamcenv usage:\n");
   fprintf(stderr,"\tteamcenv                   -  This help message\n");
   fprintf(stderr,"\tteamcenv ENVFILE           -  Print contents of environment file\n");
   fprintf(stderr,"\tteamcenv ENVFILE PARAMETER -  Print value of specified parameter.\n");
   return;
}


/*-----------------------------------------------------------------------------\
|  envInitReadEnvFile:                                                         |
|  - Open temporary binary file for reading binary data                        |
|  - Return File Handle                                                        |
|  - temporary file automatically deleted when daemons killed                  |
|  * Need to pass file name to user through exit parameter                     |
\-----------------------------------------------------------------------------*/
FILE *envInitReadEnvFile(char *envFileName)
{
   FILE *envFile;

   /* Open temporary file */
   envFile = fopen(envFileName, "rb");
   if (envFile == NULL)
   {
       fprintf(stderr,"teamcenv: Error, could not open file \"%s\"\n",
          envFileName);
       return (FILE *)0;
    }

    return (envFile);
}


/*-----------------------------------------------------------------------------\
|  envGetFromEnvFile:                                                          |
|  - Write an entry to environment file                                        |
|  - Write binary:                                                             |
|    size of parameter, parameter string, size of value, value string          |
\-----------------------------------------------------------------------------*/
int envGetFromEnvFile(FILE *envFile, char *inName)
{
   int nNameLength;
   int nValueLength;
   char parameterName[maxParmName+1]; /* allow for maximum in TeamC (15 + NULL) */
   char parameterValue[16001]; /* allow for max in TeamC 16000 for remarks + NULL */

   /* Either dump all attributes or search for parameter identified by */
   /* inName */
   if (*inName == '\0')
   {
      printf("\
Parameter                                Value\n\
=======================================  ======================================\n");
      fread(&nNameLength, sizeof(int), 1, envFile);
      fread(parameterName, sizeof(char), nNameLength, envFile);
      *(parameterName+nNameLength)='\0';
      fread(&nValueLength, sizeof(int), 1, envFile);
      fread(parameterValue, sizeof(char), nValueLength, envFile);
      *(parameterValue+nValueLength)='\0';

      while (!feof(envFile))
      {
         strncat(parameterName, "                                       ",
            (maxParmName - strlen(parameterName)));
         *(parameterName+maxParmName) = '\0';
         printf("%s %s\n", parameterName, parameterValue);
         fread(&nNameLength, sizeof(int), 1, envFile);
         *(parameterName+nNameLength)='\0';
         fread(parameterName, sizeof(char), nNameLength, envFile);
         fread(&nValueLength, sizeof(int), 1, envFile);
         fread(parameterValue, sizeof(char), nValueLength, envFile);
         *(parameterValue+nValueLength)='\0';
      }
   }
   else /* Searching for one entry */
   {
      fread(&nNameLength, sizeof(int), 1, envFile);
      fread(parameterName, sizeof(char), nNameLength, envFile);
      *(parameterName+nNameLength)='\0';
      fread(&nValueLength, sizeof(int), 1, envFile);
      fread(parameterValue, sizeof(char), nValueLength, envFile);
      *(parameterValue+nValueLength)='\0';

      while ((strcmp(inName, parameterName) != 0) && (!feof(envFile)))
      {
         fread(&nNameLength, sizeof(int), 1, envFile);
         fread(parameterName, sizeof(char), nNameLength, envFile);
         *(parameterName+nNameLength)='\0';
         fread(&nValueLength, sizeof(int), 1, envFile);
         fread(parameterValue, sizeof(char), nValueLength, envFile);
         *(parameterValue+nValueLength)='\0';
      }
      if (!feof(envFile))
      {
         /* Return the value of the parameter */
         printf("%s\n", parameterValue);
      }
      else
      {
         fprintf(stderr, "teamcenv: Error, parameter \"%s\" not found\n", inName);
         return 1;
      }
   }

   return 0;
}


/*-----------------------------------------------------------------------------\
|  main:                                                                       |
|  - Verify number of arguments                                                |
|  - Call routines to open environement file and get parameter values          |
\-----------------------------------------------------------------------------*/
int main(int argc, char *argv[])
{
   int rc;
   FILE *envFile;
   struct stat statBuff;

   /* If no parameters, print help text */
   if (argc == 1 || argc > 3)
   {
      envHelp();
      exit (1);
   }
   else /* First parameter must be temporary file */
   {
      /* Check for empty file */

      stat(argv[1], &statBuff);
      if (statBuff.st_size == 0)
      {

         fprintf(stderr, "teamcenv: Info, file \"%s\" contains no values\n", argv[1]);
         exit (1);
      }

      /* Open file */
      envFile = envInitReadEnvFile(argv[1]);
      if (envFile == (FILE *)0)
      {
         exit (1);
      }
   }


   /* Get parameter or parameters */
   if (argc == 3)
   {
      /* Return value for parameter in argument 2 */
      rc = envGetFromEnvFile(envFile, argv[2]);
   }
   else
   {
      /* Return all parameters and their values */
      rc = envGetFromEnvFile(envFile, "");
   }

   fclose(envFile);
   return (rc);
}

