/********************************************************/
/*       TEAMC Windows Server Universal Testcase        */
/********************************************************/
/* This Rexx program executes the major TEAMC functions.*/
/* It is broken up into 6 sections.                     */
/* 1. TEAMC base functions:                             */
/* 2. TEAMC Build Tool: A new integrated build tool     */
/* 3. TEAMC Versioning: new versioning commands         */
/* 4. TEAMC Packaging Tool: A new integrated packaging  */
/*    and distribution tool.                            */
/* 5. TEAMC Build Tool: More build actions              */
/* 6. TEAMC reports:                                    */
/********************************************************/
/* Last update: 01/06/95 tso  release: 9412             */
/********************************************************/
/* Last update: 04/29/97 ncc  release: v204             */
/********************************************************/
/* Last update: 09/17/98      release: v310             */
/********************************************************/
parse arg param            /*param is the build directory or -nobuild*/
param = strip(param)

call initialize

/********************************************************/
/*Do build with the directory the user gave             */
/********************************************************/
if do_the_build then do
  if dir_was_given then
    'start teamcbld -p pool1 -e win -s -v -c ' buildDir
  else
    'start teamcbld -p pool1 -e win -s -v'

  call Univ_base        /* do all TEAMC base function */
  call Univ_build       /* do some builds             */
  call Univ_versioning  /* do Versioning stuff        */
  call Univ_package     /* call gatherer via build    */
  call Univ_more_build  /* do a build                 */
  call univ_report      /* do all the report commands */
end


/********************************************************/
/*User said don't do the build                          */
/********************************************************/
else do
  call Univ_base        /* do all TEAMC base function */
  call Univ_versioning  /* do Versioning stuff        */
  call univ_report      /* do all the report commands */
end

exit




/*************************************************/
/*              Univ_base                        */
/*************************************************/
/* Run base TEAMC commands                       */
/*************************************************/
Univ_base:

/* Core environment setup for test cases */
call runCmd "teamc User -create -login mickey -name ""Mickey Mouse"" -area DisneyLand +super -address mickey@" || hostname "-verbose "
call runCmd "teamc User -create -login dudley +super -address dudley@" || hostname "-verbose "
call runCmd "teamc User -create -login sparkle +super -address sparkle@"|| hostname "-verbose "
call runCmd "teamc User -create -login rocky   +super -address rocky@"  || hostname "-verbose "
call runCmd "teamc User -create -login sherman +super -address sherman@"|| hostname "-verbose "
call runCmd "teamc User -create -login atestr  +super -address atestr@" || hostname "-verbose "
call runCmd "teamc User -create -login minnie  +super -address minnie@" || hostname "-verbose "

call runCmd "teamc Host -create mickey@" || hostname           "-login mickey  -verbose "
call runCmd "teamc Host -create " superuser || "@" || hostname "-login mickey  -verbose "
call runCmd "teamc Host -create atestr@" || hostname           "-login mickey  -verbose "
call runCmd "teamc Host -create minnie@" || hostname           "-login minnie  -verbose "
call runCmd "teamc Host -create" superuser || "@" || hostname  "-login minnie  -verbose "
call runCmd "teamc Host -create dudley@" || hostname           "-login dudley  -verbose "
call runCmd "teamc Host -create" superuser || "@" || hostname  "-login dudley  -verbose "
call runCmd "teamc Host -create rocky@" || hostname            "-login rocky   -verbose  "
call runCmd "teamc Host -create" superuser || "@" || hostname  "-login rocky   -verbose  "
call runCmd "teamc Host -create atestr@" || hostname           "-login rocky   -verbose  "
call runCmd "teamc Host -create sherman@" || hostname          "-login sherman -verbose  "
call runCmd "teamc Host -create" superuser || "@" || hostname  "-login sherman -verbose  "
call runCmd "teamc Host -create atestr@" || hostname           "-login sherman -verbose  "
call runCmd "teamc Host -create sparkle@" || hostname          "-login sparkle -verbose  "
call runCmd "teamc Host -create" superuser || "@" || hostname  "-login sparkle -verbose  "
call runCmd "teamc Host -create atestr@" || hostname           "-login atestr  -verbose  "
call runCmd "teamc Host -create" superuser || "@" || hostname  "-login atestr  -verbose  "

call runCmd "teamc Component -create comp1    -process default -parent root  -owner mickey -desc ""Top component"" -verbose "
call runCmd "teamc Component -create subcomp1 -process default -parent comp1 -verbose "
call runCmd "teamc Component -create comp2    -process default -parent root  -owner dudley -verbose "

call runCmd "teamc Release -create MOUSTRAP -process track_full -environment mtvers1 -tester sparkle -component comp1 -owner minnie -approver minnie -verbose "
call runCmd "teamc Release -create JEEZWHIZ -process prototype  -environment mtvers1 -tester sparkle -component comp2 -owner rocky -approver rocky -verbose "
call runCmd "teamc Release -create NOTRACK  -process no_track -component comp1 -owner mickey -desc ""use for workareas"" -verbose "

call runCmd "teamc Environment -create mtvers1 -tester sparkle -release NOTRACK -verbose "

/* use these 5 lines if authorit.ld has not been changed from the original English */
/*  also  remove +super from all but Mickey in the first 7 lines of univ_base */
/* call runCmd "teamc Access -create -component comp1 -login minnie  -authority projectlead   -verbose " */
/* call runCmd "teamc Access -create -component comp1 -login sparkle -authority componentlead -verbose " */
/* call runCmd "teamc Access -create -component comp2 -login dudley  -authority componentlead -verbose " */
/* call runCmd "teamc Access -create -component comp2 -login rocky   -authority developer     -verbose " */
/* call runCmd "teamc Access -create -component comp2 -login mickey  -authority general       -verbose " */

call runCmd "teamc Notify -create -login mickey minnie -interest builder -component comp1 -verbose"

call runCmd "teamc Approver -create -release MOUSTRAP -login mickey -verbose "
call runCmd "teamc Approver -create -release JEEZWHIZ -login dudley mickey -verbose  "

call runCmd "teamc Defect -open -become atestr -prefix d -severity 3 -phaseFound development -symptom incorrect_i/o -component subcomp1 -remarks ""This is a defect in subcomp1"" -verbose  "
call runCmd "teamc Defect -accept 1 -answer program_defect -verbose  "
call runCmd "teamc Defect -assign 1 -owner minnie -verbose  "

call runCmd "teamc Defect -open -become rocky -prefix d -severity 3 -phaseFound development -symptom incorrect_i/o -component comp2 -remarks ""This is a defect in comp2"" -verbose  "

call runCmd "teamc Feature -open -prefix f -remarks ""Feature change for comp2"" -component comp2 -verbose  "
call runCmd "teamc Feature -design 3 -remarks ""This should be the feature for the change"" -verbose  "

call runCmd "teamc Workarea -create -defect 1 -release MOUSTRAP -verbose  "

call runCmd "teamc Approval -accept -workarea 1 -release MOUSTRAP -approver mickey -verbose  "
call runCmd "teamc Approval -accept -workarea 1 -release MOUSTRAP -approver minnie -verbose  "

call runCmd "teamc Fix -create -workarea 1 -component subcomp1 -release MOUSTRAP -verbose "

call runCmd "teamc Part -create afile.c -workarea 1 -component subcomp1 -release MOUSTRAP -verbose "

/* refresh the track from the release */
call runCmd "teamc Workarea -freeze  1 -release MOUSTRAP -verbose"
call runCmd "teamc Workarea -refresh 1 -release MOUSTRAP -verbose"

call runCmd "teamc Fix -complete -workarea 1 -component subcomp1 -release MOUSTRAP  -verbose "

call runCmd "teamc driver -create adriver -release MOUSTRAP -become minnie -type production -verb  "
call runCmd "teamc driverMember -create -driver adriver -workarea 1 -release MOUSTRAP -verbose  "

/* create some workareas to be used later */
call runCmd "teamc Workarea -create -name Dumbo         -release NOTRACK -verbose "
call runCmd "teamc Workarea -create -name SpaceMountain -release NOTRACK -verbose "


return


/*************************************************/
/*              Univ_Build                       */
/*************************************************/
/* Add some builders, parsers, and files         */
/* Note: The build of hello.exe is now actually  */
/*  kicked off by the gather build in the        */
/*  UNIV_PACKAGE routine below.                  */
/*-----------------------------------------------*/
/*  The initial build structure looks like this  */
/*  after adding the files to TEAMC.             */
/*                                               */
/*  hello.exe  (Clink)                           */
/*     |                                         */
/*     |_ hello.obj  (Ccompile)                  */
/*            |                                  */
/*            |_ hello.c  (Cparser)              */
/*                 |                             */
/*                 |_  hello.h (Cparser)         */
/*                                               */
/*  hello2.h                                     */
/*                                               */
/*-----------------------------------------------*/
/*  After the file -build the structure should   */
/*  look like this.                              */
/*                                               */
/*  hello.exe  (Clink)                           */
/*     |                                         */
/*     |_ hello.obj  (Ccompile)                  */
/*            |                                  */
/*            |_ hello.c  (Cparser)              */
/*                 |                             */
/*                 |_  hello.h (Cparser)         */
/*                 |     |                       */
/*                 |     |_ hello2.h             */
/*                 |                             */
/*                 |_  stdio.h                   */
/*                                               */
/*-----------------------------------------------*/
/* What build did:                               */
/*  1. Initially check the timestamps of each    */
/*     file to see if it should be parsed or     */
/*     built.  Since this is an initial build,   */
/*     all files were 'out-of-date'.             */
/*  2. Parse the hello.c file with the Cparser   */
/*     to determine other dependencies that      */
/*     should be included in the build. Build    */
/*     found stdio.h in hello.c and a            */
/*     placeholder file for stdio.h and a        */
/*     relationship to it.  hello.h was also     */
/*     found but it is already in the library.   */
/*  3. Parse hello.h.  Builds finds a dependency */
/*     for hello2.h and adds a relationship.     */
/*     Since hello2.h is in TEAMC already, it    */
/*     will be materialized for build.           */
/*  4. Parse the hello2.c file with the Cparser  */
/*     to determine other dependencies that      */
/*     should be included in the build. Build    */
/*     found hello2.h but it is already in the   */
/*     library. Only a dependency relationship   */
/*     is added between hello2.c and hello2.h    */
/*  5. Builder Ccompile is run for hello.obj.    */
/*     Inputs (hello.c) and dependents (hello.c, */
/*     hello2.h) are extracted from TEAMC to the */
/*     directory where the build server was      */
/*     started and the compile is done.          */
/*     If the compile is successful, then the    */
/*     output (hello.obj) is stored back into    */
/*     TEAMC.                                    */
/*  6. hello2.obj is compiled similarly to       */
/*     hello.obj in step 5.                      */
/*  7. Builder CLink is run for hello.exe.       */
/*     Inputs (hello.obj, hello2.obj) are        */
/*     extracted fromTEAMC and the link is done. */
/*     If the link is successful, then the       */
/*     output (hello.exe) is stored back into    */
/*     TEAMC.                                    */
/*************************************************/
Univ_build:

/* add a builder that executes an icc compile only */
call runCmd 'teamc builder -create Ccompile -release NOTRACK -script compC.exe -binary -from fhbwcomp.exe -condition "==" -value 0 -environment win -timeout 10 -verbose '
call runCmd 'teamc builder -extract Ccompile -release NOTRACK -to extract.fil -verbose '
call runCmd "dir extract.fil"   /* call it again and log it */
call runCmd 'teamc builder -create Clink    -release NOTRACK -script linkC.exe -binary -from fhbwlink.exe -condition "<=" -value 4 -environment win -timeout 10 -verbose'

/* add a C parser */
call runCmd 'teamc parser  -create Cparser  -release NOTRACK -command fhbwpars.exe -verbose'

call runCmd "teamc part -create hello.exe  -binary -empty -component comp1 -release NOTRACK -workarea Dumbo -builder Clink -remarks ""For testing build""  -verb "
call runCmd "teamc part -create hello.obj  -binary -empty -component comp1 -release NOTRACK -workarea Dumbo -parent hello.exe -input -builder Ccompile -verb  "
call runCmd "teamc part -create hello.c    -text   -component comp1 -release NOTRACK -workarea Dumbo -parent hello.obj -input -parser Cparser -verb "
call runCmd "teamc part -create hello2.obj -binary -empty -component comp1 -release NOTRACK -workarea Dumbo -parent hello.exe -input -builder Ccompile -verb  "
call runCmd "teamc part -create hello2.c   -text   -component comp1 -release NOTRACK -workarea Dumbo -parent hello2.obj -input -parser Cparser -verb "
call runCmd "teamc part -create hello.h    -text   -component comp1 -release NOTRACK -workarea Dumbo -parent hello.c   -dependent -parser Cparser -verb"
call runCmd "teamc part -create hello2.h   -text   -component comp1 -release NOTRACK -workarea Dumbo -verb "

/* add the files with builders and parsers associated with them */

/* hello.obj and hello2.obj will also be built   */
call runCmd 'teamc part -build  hello.exe -release NOTRACK -workarea  Dumbo  -verbose -pool pool1 '

/* view translator messages for the build of gathtgt */
call runCmd "teamc part -viewmsg hello.obj   -release NOTRACK -work Dumbo -verbose"
call runCmd "teamc part -viewmsg hello2.obj  -release NOTRACK -work Dumbo -verbose"
call runCmd "teamc part -viewmsg hello.exe   -release NOTRACK -work Dumbo -verbose"

/* extract hello.exe from the library and execute it */
call runCmd "teamc part -extract hello.exe  -release NOTRACK -work Dumbo -verbose"
call runCmd "hello"   /* call it again and log it */

return
/*************************************************/
/*              Univ_Versioning                  */
/*************************************************/
/* Do some miscellaneous versioning              */
/*-----------------------------------------------*/
Univ_Versioning:

/* freeze, add a file to, and commit workarea Dumbo */
call runCmd "teamc Workarea -freeze Dumbo -release NOTRACK -verbose"
call runCmd "teamc Part -create bfile.c -component comp2 -release NOTRACK -workarea Dumbo -verbose "

call runCmd "teamc Workarea -integrate Dumbo -release NOTRACK -verbose"

/* refresh to see changes introduced by previous commit */
call runCmd "teamc Workarea -refresh SpaceMountain -release NOTRACK -verbose"

call runCmd "teamc part -view bfile.c -release NOTRACK -workarea Dumbo -verbose"

/* do some versioning freezes and refreshes on track and driver workareas */
call runCmd "teamc driver -freeze  adriver    -release MOUSTRAP -verbose"
call runCmd "teamc driver -refresh adriver    -release MOUSTRAP -verbose"

return

/*************************************************/
/*              Univ_package                     */
/*************************************************/
/* Gather the files from the hello application   */
/* generated above. The source files are gathered*/
/* to \octo1\SRC directory, the hello.obj and    */
/* hello.exe to  \octo1\BIN. Due to the -t flag, */
/* the target directory is first deleted if it   */
/* exists, and then re-created, ensuring that    */
/* only the target directory results.            */
/* The -d flag allows duplicate files.           */
/*************************************************/
/*  The initial build structure for packaging    */
/*  looks like this                              */
/*                                               */
/*  gathtgt    (gatherer)                        */
/*     |                                         */
/*     |_ univpack.pkf                           */
/*     |                                         */
/*     |_ hello.exe                              */
/*     |                                         */
/*     |_ hello.obj                              */
/*     |                                         */
/*     |_ hello2.obj                             */
/*     |                                         */
/*     |_ hello.c                                */
/*     |                                         */
/*     |_ hello.h                                */
/*     |                                         */
/*     |_ hello2.h                               */
/*                                               */
/*************************************************/
Univ_package:

/* define gather builder */
call runCmd 'teamc builder -create gatherer -release NOTRACK -script  gather.exe -binary  -from gather.exe  -parameters " -d -t" -condition "==" -value 0 -environment win -timeout 10 -verbose'

/* define output of gather build process */
call runCmd 'teamc part -create gathtgt.out -none -component comp1 -release  NOTRACK -workarea  SpaceMountain -builder gatherer -verbose'

/* adding files to gathtgt: first, add the package file */
call runCmd 'teamc part -create univpack.pkf -text -component comp1 -release NOTRACK -workarea  SpaceMountain   -verbose -parent gathtgt.out -input'

/* adding other files to gathtgt */
call runCmd 'teamc part -connect hello.exe -release  NOTRACK -workarea  SpaceMountain -verbose -parent gathtgt.out -input'
call runCmd 'teamc part -connect hello.obj -release  NOTRACK -workarea  SpaceMountain -verbose -parent gathtgt.out -input'
call runCmd 'teamc part -connect hello2.obj -release  NOTRACK -workarea  SpaceMountain -verbose -parent gathtgt.out -input'
call runCmd 'teamc part -connect hello.c   -release  NOTRACK -workarea  SpaceMountain -verbose -parent gathtgt.out -input'
call runCmd 'teamc part -connect hello.h   -release  NOTRACK -workarea  SpaceMountain -verbose -parent gathtgt.out -input'
call runCmd 'teamc part -connect hello2.h  -release  NOTRACK -workarea  SpaceMountain -verbose -parent gathtgt.out -input'

/* let's modify hello.h, this should cause hello.obj to be marked out-of-date */
call runCmd 'teamc part -checkout hello.h   -release  NOTRACK -workarea  SpaceMountain -verbose'
call runCmd 'teamc part -checkin  hello.h   -release  NOTRACK -workarea  SpaceMountain -verbose'

/* call the gather build function */
/* hello.exe, hello.obj will also be built because hello.h was modified  */
/* but hello2.obj will not be built, it has not changed.                 */
call runCmd 'teamc part -build  gathtgt.out -release NOTRACK -workarea  SpaceMountain  -verbose -pool pool1 '

/* view translator messages for the build of gathtgt */
call runCmd "teamc part -viewmsg hello.obj   -release NOTRACK -work SpaceMountain -verbose"
call runCmd "teamc part -viewmsg hello.exe   -release NOTRACK -work SpaceMountain -verbose"
call runCmd "teamc part -viewmsg gathtgt.out -release NOTRACK -work SpaceMountain -verbose"

/* run the hello.exe that was processed by gather */
/* Note: this command only works if client and buildserver are on the same machine */
call runCmd "\octo1\bin\hello"   /* call it again and log it */

/* let's modify hello.h, then the workarea -commit needs the -force option   */
/* since hello.exe and hello.obj were not built successful before the commit. */
call runCmd 'teamc part -checkout hello.h   -release  NOTRACK -workarea  SpaceMountain -verbose'
call runCmd 'teamc part -checkin  hello.h   -release  NOTRACK -workarea  SpaceMountain -verbose'

/* Commit the current workarea */
call runCmd "teamc Workarea -integrate SpaceMountain -release NOTRACK -force -verbose"

return

/*************************************************/
/*              Univ_more_builld                 */
/*************************************************/
/* More build actions                            */
/*************************************************/
Univ_more_build:

/* create a new workarea in release NOTRACK */
call runCmd 'teamc workarea -create -name HauntedHouse -release NOTRACK -verbose'

/* report on the files in the new workarea  */
call runCmd "teamc Report -view PartView -workarea HauntedHouse -release NOTRACK -verbose "

/* hello.exe, hello.obj were updated in workarea SpaceMountain but were  */
/* not rebuilt.  They are still out_of_date, so lets build them in the   */
/* new workarea.                                                         */
call runCmd 'teamc part -build  hello.exe -release NOTRACK -workarea  HauntedHouse -verbose -pool pool1 '

/* view translator messages for the build of hello  */
call runCmd "teamc part -viewmsg hello.obj   -release NOTRACK -work HauntedHouse -verbose"
call runCmd "teamc part -viewmsg hello.exe   -release NOTRACK -work HauntedHouse -verbose"

/* do some other miscellaneous build commands */
call runCmd 'teamc part -touch  hello.exe -release NOTRACK -workarea  HauntedHouse -verbose '
call runCmd 'teamc part -build  hello.exe -report -release NOTRACK -workarea  HauntedHouse -verbose -pool pool1 '
call runCmd 'teamc part -childInfoView  hello.exe -release NOTRACK -workarea  HauntedHouse -verbose '

/*Try to Commit the current workarea */
call runCmd "teamc Workarea -integrate HauntedHouse -release NOTRACK -force -verbose"

return


/*************************************************/
/*              Univ_report                      */
/*************************************************/
/* Run all the reports for all views that are    */
/*  available                                    */
/*************************************************/
univ_report:

call runCmd "teamc Report -view accessdownview  -raw -verbose "
call runCmd "teamc Report -view accessview      -raw -verbose "
call runCmd "teamc Report -view approvalview    -raw -verbose "
call runCmd "teamc Report -view approverview    -raw -verbose "
call runCmd "teamc Report -view authority       -raw -verbose "
call runCmd "teamc Report -view bchangeview     -raw -verbose "
call runCmd "teamc Report -view cfgcomproc      -raw -verbose "
call runCmd "teamc Report -view cfgrelproc      -raw -verbose "
call runCmd "teamc Report -view changeView      -raw -verbose "
call runCmd "teamc Report -view compview        -raw -verbose "
call runCmd "teamc Report -view config          -raw -verbose "
call runCmd "teamc Report -view defectview      -raw -verbose "
call runCmd "teamc Report -view defectdownview  -raw -verbose "
call runCmd "teamc Report -view envview         -raw -verbose "
call runCmd "teamc Report -view featureview     -raw -verbose "
call runCmd "teamc Report -view featuredownview -raw -verbose "
call runCmd "teamc Report -view partfullview    -raw -verbose "
call runCmd "teamc Report -view partsoutview    -raw -verbose "
call runCmd "teamc Report -view fixview         -raw -verbose "
call runCmd "teamc Report -view hostview        -raw -verbose "
call runCmd "teamc Report -view interest        -raw -verbose "
call runCmd "teamc Report -view driverview       -raw -verbose "
call runCmd "teamc Report -view drivermemberview -raw -verbose   "
call runCmd "teamc Report -view noteview        -raw -verbose "

call runCmd "teamc Report -view notifyview      -raw -verbose "
call runCmd "teamc Report -view notifydownview  -raw -verbose "
call runCmd "teamc Report -view NotifyUpView    -raw -verbose "

call runCmd "teamc Report -view releaseview     -raw -verbose "
call runCmd "teamc Report -view sizeview        -raw -verbose "
call runCmd "teamc Report -view testview        -raw -verbose "
call runCmd "teamc Report -view users           -raw -verbose "
call runCmd "teamc Report -view verifyview      -raw -verbose "

call runCmd "teamc Report -testClient"
call runCmd "teamc Report -testServer"

call runCmd "teamc Report -view BuilderView     -raw -verbose "
call runCmd "teamc Report -view PartView        -raw -workarea Dumbo -release NOTRACK -verbose "
call runCmd "teamc Report -view PartView        -raw -workarea NULL -release NOTRACK -verbose "
call runCmd "teamc Report -view bPartView       -raw -workarea Dumbo -release NOTRACK -where ""baseName = 'hello.exe'"" -verbose "
call runCmd "teamc Report -view bPartView       -raw -workarea NULL -release NOTRACK -where ""baseName = 'gathtgt.out'"" -verbose "
call runCmd "teamc Report -view CollisionView   -raw -verbose "
call runCmd "teamc Report -view ParserView      -raw -verbose "
call runCmd "teamc Report -view versionView     -raw -verbose "
call runCmd "teamc Report -view WorkareaView    -raw -verbose "

call reset_Notify    /* delete notification for FVT regression */

return


/*************************************************/
/* Setup up the test environment                 */
/*************************************************/
initialize:

do_the_build  = 1  /*assume we want to do the build*/
dir_was_given = 1  /*and the user provided a temporary build directory name*/
buildDir = ""

if param = '?' | param = '-help' then
  call help                             /* this exits */
else
  if param = '-nobuild' then
    do_the_build   = 0                    /*Don't do the build*/
  else
    if param = "" then
      dir_was_given  = 0                    /*no user-specified build directory*/
    else
      buildDir = param                      /* directory to use for build is passed */

env = 'Environment'
rexx = 'rexx'

/* setup the log */
logfile = 'univwin.log'
'copy univwin.log univwin.lo2 1>NUL '
'del univwin.log 1>NUL '

/********************************/
/* check that variables are set */
/********************************/
superuser = value('TC_BECOME',,env)
'HOSTNAME > hostname'
hostname = linein('hostname')
call lineout 'hostname'
'del hostname'

if superuser = '' then do
  BEEP(450,250)
  say ''
  say 'TC_BECOME variable is not set.'
  say ' please set it to a TEAMC userid that is a superuser'
  say ' and then rerun UNIVWIN'
  exit 8
end
if hostname = '' then do
  BEEP(450,250)
  say ''
  say 'HOSTNAME variable is not set.'
  say ' please set it to your TCP/IP hostname'
  say ' and then rerun UNIVWIN'
  exit 8
end

/**************************/
/* setup the files used   */
/**************************/

/* delete the files first */
'attrib hello*.* -r 1>NUL '
'attrib *file*.* -r 1>NUL '
'attrib univpack.pkf -r >NUL '
'del afile.c   1>NUL '
'del bfile.c   1>NUL '
'del hello.c   1>NUL '
'del hello.obj 1>NUL '
'del hello.exe 1>NUL '
'del hello.h   1>NUL '
'del hello2.*  1>NUL '
'del hello.$*  1>NUL '
'del univpack.pkf >NUL '
'del \octo1\bin\*.* /q >NUL '
'rd \octo1\bin >NUL '

/* create the files used in Univ_base */
call LINEOUT afile.c, "Date is "date('w')" "date('n')
call LINEOUT afile.c, "Time is "time('c')
call LINEOUT afile.c

call LINEOUT bfile.c, "Date is "date('w')" "date('n')
call LINEOUT bfile.c, "Time is "time('c')
call LINEOUT bfile.c

/* create the files that are used in Univ_build  */
call LINEOUT hello.c, "/*   HELLO.C -- Hello, world */"
call LINEOUT hello.c, " "
call LINEOUT hello.c, "#include <stdio.h>"
call LINEOUT hello.c, '#include "hello.h" '
call LINEOUT hello.c, " "
call LINEOUT hello.c, "int main()"
call LINEOUT hello.c, "{"
call LINEOUT hello.c, "      printf(""Hello! world\n"");"
call LINEOUT hello.c, "      printf(""I was built by the Team Connection\n"");"
call LINEOUT hello.c, "      printf(""integrated Build Tool!!!\n"");"
call LINEOUT hello.c, "      return 0;"
call LINEOUT hello.c, "}"
call LINEOUT hello.c

call LINEOUT hello.h, "/* HELLO.H */"
call LINEOUT hello.h, '#include "hello2.h"'
call LINEOUT hello.h

call LINEOUT hello2.h, "/* HELLO2.H */"
call LINEOUT hello2.h

/* create the hello2.c file */
call LINEOUT hello2.c, '/* hello2 */'
call LINEOUT hello2.c, '#include "hello2.h"'
call LINEOUT hello2.c, 'hello2()'
call LINEOUT hello2.c, '{'
call LINEOUT hello2.c, '  return 0;'
call LINEOUT hello2.c, '}'
call LINEOUT hello2.c


/* create the gather package file used in Univ_package */
call LINEOUT univpack.pkf, "#|Date is "date('w')" "date('n')"|#"
call LINEOUT univpack.pkf, "(DATA (PACKAGEFORMAT GATHER )  "
call LINEOUT univpack.pkf, "                               "
call LINEOUT univpack.pkf, "(TARGETROOT \octo1 )           "
call LINEOUT univpack.pkf, "                               "
call LINEOUT univpack.pkf, "(RULE                          "
call LINEOUT univpack.pkf, "  (SOURCE                      "
call LINEOUT univpack.pkf, "    *.exe                      "
call LINEOUT univpack.pkf, "    *.obj                      "
call LINEOUT univpack.pkf, "    )                          "
call LINEOUT univpack.pkf, "  (TARGET BIN    )             "
call LINEOUT univpack.pkf, ")                              "
call LINEOUT univpack.pkf, "                               "
call LINEOUT univpack.pkf, "(RULE                          "
call LINEOUT univpack.pkf, "  (SOURCE                      "
call LINEOUT univpack.pkf, "    *.C                        "
call LINEOUT univpack.pkf, "    *.H                        "
call LINEOUT univpack.pkf, "    )                          "
call LINEOUT univpack.pkf, "  (TARGET SRC    )             "
call LINEOUT univpack.pkf, ")                              "
call LINEOUT univpack.pkf, ")                              "
call LINEOUT univpack.pkf


return


/********************************************************/
/*              Reset_Notify                            */
/********************************************************/
/*  Deletes the only Notification registered in TEAMC.   */
/*  Must do this to make sure that old regression       */
/*  testcases that use this setup will compare correctly*/
/********************************************************/
reset_Notify:

call runCmd "teamc Notify -delete -login mickey minnie -interest builder -component comp1 -verbose"

return

/********************************************************/
/*                      runCmd                          */
/********************************************************/
/*  Executes a command and logs the results to a file   */
/*  'univwin.log'                                       */
/*                                                      */
/*  It will also beep if there is an unexpected result. */
/********************************************************/
runCmd:

parse arg  cmd

call lineout logfile, ' '
call lineout logfile

/* log and run the command */
"echo === " cmd
"echo === " cmd ">>" logfile
cmd ">buf.out"

saverc = RC

/* check if result is expected */
if saverc \= 0 then do
  BEEP(440,200)

  /* type results to the screen */
  "echo --- rc=" saverc
  "type  buf.out"
end  /* then do */

/* type results to the logfile */
"echo --- rc=" saverc ">>" logfile
"type buf.out >>" logfile

call lineout logfile, " "
call lineout logfile

return

/********************************************************/
/*                 commentCmd                           */
/********************************************************/
/*  Places a comment in the log                         */
/*  'univwin.log'                                       */
/*                                                      */
/*                                                      */
/********************************************************/
commentCmd:

parse arg  Comment

call lineout logfile, ' '
call lineout logfile

/* log and run the command */
"echo *** " Comment  "***"
"echo *** " Comment  "***" ">>" logfile

call lineout logfile, " "
call lineout logfile

return



help:

say' '
say'     *-----------------------------------------*'
say'     * TEAMC WINDOWS Server Universal Testcase *'
say'     *-----------------------------------------*'
say' '
say 'This Rexx program executes the major TEAMC functions.'
say 'It will run commands for the following areas'
say''
say '1. TEAMC base functions: All the major functions'
say '2. TEAMC Build Tool: A new integrated build tool'
say '3. TEAMC Versioning: new versioning commands'
say '4. TEAMC Packaging Tool: A new integrated packaging'
say '   and distribution tool. '
say '5. TEAMC Build Tool: more build actions'
say '6. TEAMC reports '
say''
say' SYNTAX:  UNIVWIN  <buildDir || -nobuild || ? || -help>'
say''
say'   ? or -help means display this screen'
say'   -nobuild means no build is desired'
say'   buildDir should be the directory to use for the build.'
say''
say' Results of the program are written to a file UNIVWIN.LOG'

exit 0
