package com.ibm.vap.Persistence;


/**
 * PersistentHomeCollection
 *
 * Licensed Material - Property of IBM 
 * IBM(R) VisualAge(TM) for Java(TM) Version 2.0 
 * (C) Copyright IBM Corp. 1997, 1998 - All Rights Reserved. 
 * US Government Users Restricted Rights - Use, duplication or disclosure 
 * restricted by GSA ADP Schedule Contract with IBM Corp. 
 *
 */
import javax.ejb.*;
import java.util.Vector;
import java.util.Enumeration;
import com.ibm.vap.Transactions.*;
import com.ibm.vap.Isolation.*;
import java.rmi.RemoteException;
import com.ibm.vap.common.VapReadFailureException;
import com.ibm.vap.common.VapObjectLockedException;
import com.ibm.vap.common.VapRemoteObjectLockedException;
import com.ibm.vap.common.VapTransactionRequiredException;
import com.ibm.vap.common.ClassShortName;
import com.ibm.vap.common.VapMethodInvocation;
import java.lang.reflect.Method;
import java.lang.reflect.InvocationTargetException;
   
public abstract class PersistentHomeCollection extends TransientHomeCollection implements VapEJBHome
{
	protected DOCache cache;
	protected ServiceObject so = null;
	protected DataStore dataStore = null;
	private static java.util.ResourceBundle resobjectExtender = java.util.ResourceBundle.getBundle("objectExtender");  //$NON-NLS-1$
/**
 * Answers a collection containing all the instances within this home.
 * Answers an empty Vector if nothing found.
 */
public Vector allInstances() throws RemoteException, VapReadFailureException
{
	return Transaction.getCurrent().getIsolationPolicy().allInstancesFrom(this);
}
/**
 * Answers a collection containing all the instances within this home.
 * Answers an empty Vector if nothing found. The objects are locked in the database.
 */
public Vector allInstancesWithLock() throws RemoteException, VapReadFailureException
{
	return this.getIsolationImplementor().allInstancesWithLockFrom(this);
}
/**
 * API:
 * Create an instance of the class which lives in this home.
 */
public void cacheDataObject(DataObject cacheEntry) {
	getCache().putDO(cacheEntry.getKey(), cacheEntry);
	cacheEntry.setHome(this);
}
/**
 * Create a new instance of the class which lives in this home.
 * Specialized to invoke createPersistent(), to get a persistent instance.
 */
public VapEJBObject create()
{
	return this.createPersistent();
}
/**
 * Create an instance with the passed key.
 */
public VapEJBObject createFromKey(Key aKey)
{
	return this.createPersistentFromKey(aKey);
}
/* 
 * Answer the data object cache
 */
public DOCache createNewCache() 
{
	DOCache cache ;

	cache = new DOCache();
	cache.setHome(this);	
	return cache;
} 
/**
* Create a persistent instance of the class which lives in this home.
 */
public VapEJBObject createPersistent()
{
	return this.createFrom(null,null,Version.createNewPersistentState());
}
/**
* Create a persistent instance of the class which lives in this home.
 */
public VapEJBObject createPersistentFromKey(Key aKey)
{
	return this.createFrom(aKey,null,Version.createNewPersistentState());
}
/**
 * Reload a version for an ejb object from a do
 */
public void createReloaded(VapEJBObject anEjbObject,DataObject aDataObject)
{
	Transaction currentTransaction;
	Version newVersion;
	VersionState state;

	currentTransaction = Transaction.getCurrent();
	state = Version.createOldPersistentState();
	newVersion = this.createVersionFrom(anEjbObject,null,aDataObject,state,currentTransaction);
	try {currentTransaction.getView().addVersion(newVersion);} catch (RemoteException e) {};
	newVersion.initializeLinks();
}
/**
 * Create a new instance for a retrieved object
 */
public VapEJBObject createRetrieved(DataObject aDataObject)
{
	return aDataObject.getHome().createFrom(null,aDataObject,Version.createOldPersistentState());
}
/**
 * Use this method to call a custom query named aQueryName for this PersistentHomeCollection.
 * <aQueryName> will be the method name on the query pool class.
 */
public Vector customQuery(String aQueryName) throws RemoteException,VapReadFailureException
{
	return this.customQuery(aQueryName, null);
}
/**
 * Use this method to call a custom query named aQueryName for this PersistentHomeCollection.
 * The args parameter will be an array containing the arguments in the same order as they are
 * passed to the query named aQueryName on the service object.
 * <aQueryName> will be the method name on the query pool class.
 */
public Vector customQuery(String aQueryName, Object[] args) throws RemoteException,VapReadFailureException
{
	return Transaction.getCurrent().getIsolationPolicy().customQueryFrom(this, aQueryName, args);
}
/**
 * This method is called automatically if pessimistic locking is enabled for this class.
 * Do not use this method directly, unless you always want this query to lock.
 * The objects are locked in the database.
 * See customQuery.
 */
public Vector customQueryWithLock(String aQueryName, Object[] args) throws RemoteException, VapReadFailureException
{
	return this.getIsolationImplementor().customQueryWithLockFrom(this, aQueryName, args);
}
/* 
 * Discard all caches.
 */
public void discardCache() 
{
	Relationship relationship;
	Enumeration relationships;

	if (cache != null)
		cache.discardCache();
		
	relationships = currentRelationships().elements();
	while (relationships.hasMoreElements())
	{
		relationship = (Relationship)relationships.nextElement();
		relationship.discardCache();
	}
}      
/**
 * Insert the passed version
 */
public ServiceResult executeInsert(Version aVersion, Session aSession)
{
	ServiceObject serviceObject;
	ServiceResult serviceResult;

	serviceObject = this.getServiceObject();
	synchronized(so)
	{
		serviceObject.setSession(aSession);
		serviceResult = serviceObject.insert(aVersion.getBean());
	}	

	return serviceResult;	
}
/**
 * Lock the passed version
 */
public ServiceResult executeLock(Version aVersion, Session aSession)
{
	ServiceObject serviceObject;
	ServiceResult serviceResult = null;
	DataObject dataObject;
	VapEJBObject ejbObject;
	Key key;

	if ((serviceObject = this.getServiceObject()) != null)
		synchronized(so)
		{
			key = aVersion.getVersionIdentifier().getKey();
			serviceObject.setSession(aSession);
			serviceResult = serviceObject.findByKeyWithLock(key);
			if (!serviceResult.isError())
				if ((dataObject = (DataObject)serviceResult.result()) != null)
				{
					ejbObject = aVersion.getEjbObject();
					ejbObject.getBom().setCachedTransaction(aVersion.getTransaction());
					ejbObject.getBom().setCachedVersion(aVersion);
					this.refresh(ejbObject,dataObject);
				}
		}	

	return serviceResult;	
}
/**
 * Refresh the passed version
 */
public ServiceResult executeRefresh(Version aVersion, Session aSession)
{
	ServiceObject serviceObject;
	ServiceResult serviceResult = null;
	DataObject dataObject;
	VapEJBObject ejbObject;
	Key key;

	if ((serviceObject = this.getServiceObject()) != null)
		synchronized(so)
		{
			key = aVersion.getVersionIdentifier().getKey();
			serviceObject.setSession(aSession);
			serviceResult = serviceObject.findByKey(key);
			if (!serviceResult.isError())
				if ((dataObject = (DataObject)serviceResult.result()) != null)
				{
					ejbObject = aVersion.getEjbObject();
					ejbObject.getBom().setCachedTransaction(aVersion.getTransaction());
					ejbObject.getBom().setCachedVersion(aVersion);
					this.refresh(ejbObject,dataObject);
				}
		}	

	return serviceResult;	
}
/**
 * Insert the passed version
 */
public ServiceResult executeRemove(Version aVersion, Session aSession)
{
	ServiceObject serviceObject;
	ServiceResult serviceResult;

	serviceObject = this.getServiceObject();
	synchronized(so)
	{
		serviceObject.setSession(aSession);
		serviceResult = serviceObject.delete(aVersion.getBean());
	}	

	return serviceResult;	
}
/**
 * Update the passed version
 */
public ServiceResult executeUpdate(Version aVersion, Session aSession)
{
	ServiceObject serviceObject;
	ServiceResult serviceResult;

	try {aVersion.getBean().ejbStore();} catch(RemoteException e) {};
	serviceObject = this.getServiceObject();
	synchronized(so)
	{
		serviceObject.setSession(aSession);
		serviceResult = serviceObject.update(aVersion.getBean());
	}	

	return serviceResult;	
}
/**
 * Lookup an instance of the class which lives in this home with the passed Key.
 * Search current view, if not found search cache,
 * if not found invoke service and create retrieved BO.
 * 
 */
public EJBObject findByPrimaryKey(Key aKey) throws RemoteException, VapReadFailureException
{
	return Transaction.getCurrent().getIsolationPolicy().findByPrimaryKey(aKey,this);
}
/**
 * Lookup an instance of the class. Lock the instance in the database.
 * For existing objects, lookupOidFromView: -> versionForRead: path will lock the object.
 * We can ignore the cache, because potentilly locked (root) objects have already been
 * removed from there
 */
public EJBObject findByPrimaryKeyWithLock(Key aKey) throws RemoteException, VapReadFailureException
{
	return this.getIsolationImplementor().findByPrimaryKeyWithLock(aKey,this);
}
/* 
 * Return our Data Object cache
 */
public DOCache getCache() 
{
	if (cache == null)
		cache = this.createNewCache();
		
	return cache;
}
/**
 * Answer the datastore
 */
public DataStore getDataStore() 
{
	return dataStore;
}
/**
 * Answer the isolation implementor
 * Override to get ask our service object if we have one.
 */
public ObjectIsolationImplementor getIsolationImplementor()
{
	ServiceObject serviceObject;

	if (isolationImplementor == null)
		if ((serviceObject = this.serviceObjectInstance()) != null)
			isolationImplementor = serviceObject.getIsolationImplementor();
		else
			isolationImplementor = this.createDefaultIsolationImplementor();

	return isolationImplementor;
}
/* 
 * Return a new instance of our Service Object
 */
 	public ServiceObject getServiceObject() 
 	{
	 	if (serviceObjectInstance() == null)
		{
			if ( dataStore == null ) 
			{
				String errorMessage = resobjectExtender.getString("VapNoActiveDataStore");
				// Find the position of the %1 and substitute it for the class name
				int positionOfArgument = errorMessage.indexOf("%1") ;
				errorMessage = 
					errorMessage.substring( 0 ,positionOfArgument - 1 ) +
					" " +
					getClass().getName() +
					" " +
					errorMessage.substring( positionOfArgument + 2 , errorMessage.length());
				throw new RuntimeException(errorMessage);	
			}
		return null;
		}
	 	else
	 		return (ServiceObject) serviceObjectInstance().clone();
 	}                       
/* 
 * Return the current session
 */
public Session getSession() 
{
	try
	{
 		return Transaction.getCurrent().getSessionOn(this.getDataStore());
 	}
 	catch (java.rmi.RemoteException e)
 	{
 		return null;
 	}		
}   
/**
 * Lookup an instance of the class from the DO cache
 */
public DataObject lookupOidFromCache(OID anOid)
{
	return this.searchCacheForKey(anOid.getKey());
}
/**
 * Lookup an instance of the class which lives in this home with the passed Key.
 * 
 * Override to add search of DO cache.
 */
protected VapEJBObject lookupOidFromViewWithNoLock(OID anOid)
{
	VapEJBObject ejbObject = null;
	Version version=null;

	try {
		version = Transaction.getCurrentView().primGetVersionForRead(anOid);}
	catch (RemoteException e) {}

	if (version != null)
	{
		ejbObject = version.getEjbObject();
		ejbObject.getBom().setCachedTransaction(version.getTransaction());
		ejbObject.getBom().setCachedVersion(version);
	}

	return ejbObject;
}

/**
 * Answer all the instances owned by this Home
 * Returns an empty Vector if nothing found, or null on an error
 */
public Vector primAllInstances() throws RemoteException,VapReadFailureException
{
	ServiceObject serviceObject;
	ServiceResult serviceResult;
	VapEJBObject ejbObject;
	DataObject dataObject;
	OID oid;
	Vector ejbObjects = null;
	Enumeration results;

	serviceObject = this.getServiceObject();
	serviceObject.setSession(this.getSession());
	serviceResult = serviceObject.allInstances();

	if (serviceResult.isError())
		throw new VapReadFailureException(serviceResult.errors());
	else
	{
		ejbObjects = new Vector();
		results = serviceObject.oids().elements();
		while (results.hasMoreElements())
		{
			oid = (OID)results.nextElement();
			dataObject = this.lookupOidFromCache(oid);
			if ((ejbObject = this.lookupOidFromView(oid)) != null)
				dataObject.getHome().refresh(ejbObject,dataObject);
			else
				ejbObject = this.createRetrieved(dataObject);
		
			ejbObjects.addElement(ejbObject);
		}
	}	

	return ejbObjects;
}
/**
 * Answer all the instances owned by this Home
 * Returns an empty Vector if nothing found, or null on an error
 */
public Vector primAllInstancesWithLock() throws RemoteException,VapReadFailureException,VapRemoteObjectLockedException
{
	ServiceObject serviceObject;
	ServiceResult serviceResult;
	VapEJBObject ejbObject;
	DataObject dataObject;
	OID oid;
	Vector ejbObjects = null;
	Enumeration results;

	serviceObject = this.getServiceObject();
	serviceObject.setSession(this.getSession());
	serviceResult = serviceObject.allInstancesWithLock();

	if (serviceResult.isError())
		if (serviceResult.isLockError())
			throw new VapRemoteObjectLockedException(serviceResult.errors());
		else
			throw new VapReadFailureException(serviceResult.errors());
	else
	{
		ejbObjects = new Vector();
		results = serviceObject.oids().elements();
		while (results.hasMoreElements())
		{
			oid = (OID)results.nextElement();
			dataObject = this.lookupOidFromCache(oid);
			if ((ejbObject = this.lookupOidFromViewWithNoLock(oid)) != null)
				dataObject.getHome().refresh(ejbObject,dataObject);
			else
				ejbObject = this.createRetrieved(dataObject);
			
			ejbObjects.addElement(ejbObject);
		}
	}	

	return ejbObjects;
}
/**
 * This method is used by the custom quering framework to execute a custom
 * query named aQueryName with arguments args.  You should not be using this method
 * directly.  See customQuery.
 */
public Vector primCustomQuery(String aQueryName, Object[] args) throws RemoteException,VapReadFailureException
{
	ServiceObject serviceObject;
	ServiceResult serviceResult;
	VapEJBObject ejbObject;
	DataObject dataObject;
	Enumeration results;
	OID oid;
	Vector ejbObjects = null;

	serviceObject = this.getServiceObject();
	serviceObject.setSession(this.getSession());
	serviceResult = serviceObject.customQuery(aQueryName, args);
	
	if (serviceResult.isError())
		throw new VapReadFailureException(serviceResult.errors());
	else
	{
		ejbObjects = new Vector();
		results = serviceObject.oids().elements();
		while (results.hasMoreElements())
		{
			oid = (OID)results.nextElement();
			dataObject = this.lookupOidFromCache(oid);
			if ((ejbObject = this.lookupOidFromView(oid)) != null)
				dataObject.getHome().refresh(ejbObject,dataObject);
			else
				ejbObject = this.createRetrieved(dataObject);
		
			ejbObjects.addElement(ejbObject);
		}
	}	

	return ejbObjects;
}
/**
 * This method is used by the custom quering framework to execute a custom
 * query named aQueryName with arguments args.  You should not be using this method
 * directly.  See customQuery.
 */
public Vector primCustomQueryWithLock(String aQueryName, Object[] args) throws RemoteException,VapReadFailureException, VapRemoteObjectLockedException
{
	ServiceObject serviceObject;
	ServiceResult serviceResult;
	VapEJBObject ejbObject;
	DataObject dataObject;
	Enumeration results;
	OID oid;
	Vector ejbObjects = null;

	serviceObject = this.getServiceObject();
	serviceObject.setSession(this.getSession());
	serviceResult = serviceObject.customQueryWithLock(aQueryName, args);
	
	if (serviceResult.isError())
		if (serviceResult.isLockError())
			throw new VapRemoteObjectLockedException(serviceResult.errors());
		else
			throw new VapReadFailureException(serviceResult.errors());
	else
	{
		ejbObjects = new Vector();
		results = serviceObject.oids().elements();
		while (results.hasMoreElements())
		{
			oid = (OID)results.nextElement();
			dataObject = this.lookupOidFromCache(oid);
			if ((ejbObject = this.lookupOidFromView(oid)) != null)
				dataObject.getHome().refresh(ejbObject,dataObject);
			else
				ejbObject = this.createRetrieved(dataObject);
		
			ejbObjects.addElement(ejbObject);
		}
	}	

	return ejbObjects;
}
/**
 * Lookup an instance of the class which lives in this home with the passed Key.
 * Search current view, if not found search cache,
 * if not found invoke service and create retrieved BO.
 * 
 */
public VapEJBObject primFindByPrimaryKey(Key aKey) throws RemoteException,VapReadFailureException
{
	VapEJBObject ejbObject = null;
	DataObject dataObject;
	OID oid;
	ServiceObject serviceObject;
	ServiceResult serviceResult;

	oid = new OID(this,aKey);

	if ((ejbObject = this.lookupOidFromView(oid)) == null)
		if ((dataObject = this.lookupOidFromCache(oid)) != null)
			ejbObject = this.createRetrieved(dataObject);
		else if ((serviceObject = this.getServiceObject()) != null)
		{
			serviceObject.setSession(this.getSession());
			serviceResult = serviceObject.findByKey(aKey);
			if (serviceResult.isError())
				throw new VapReadFailureException(serviceResult.errors());
			else
				if ((dataObject = (DataObject)serviceResult.result()) != null)
					ejbObject = this.createRetrieved(dataObject);
				else
					throw new VapReadFailureException();
		}

	return ejbObject;
}
/***
 * Lookup an instance of the class which lives in this home with the passed Key.
 * Search current view, if not found search cache,
 * if not found invoke service and create retrieved BO.
 **/
public VapEJBObject primFindByPrimaryKeyWithLock(Key aKey) throws RemoteException,VapReadFailureException
{
	VapEJBObject ejbObject = null;
	DataObject dataObject;
	OID oid;
	ServiceObject serviceObject;
	ServiceResult serviceResult;

	oid = new OID(this,aKey);

	if ((ejbObject = this.lookupOidFromView(oid)) == null)
		if ((serviceObject = this.getServiceObject()) != null)
		{
			serviceObject.setSession(this.getSession());
			serviceResult = serviceObject.findByKeyWithLock(aKey);
			if (serviceResult.isError())
				if (serviceResult.isLockError())
					throw new VapRemoteObjectLockedException(serviceResult.errors());
				else
					throw new VapReadFailureException(serviceResult.errors());
			else if ((dataObject = (DataObject)serviceResult.result()) != null)
				ejbObject = this.createRetrieved(dataObject);
		}

	return ejbObject;
}
/**
 * 
 */
public Vector primQueryNoArgs(String aQueryName) throws RemoteException,VapReadFailureException
{
	ServiceObject serviceObject;
	ServiceResult serviceResult;
	VapEJBObject ejbObject;
	DataObject dataObject;
	OID oid;
	Vector ejbObjects = null;
	Enumeration results;
	Method aMethod;
	Class aClass;

	serviceObject = this.getServiceObject();
	serviceObject.setSession(this.getSession());
	aClass = serviceObject.getClass();
	try {
		aMethod = aClass.getMethod(aQueryName, null);
		serviceResult = (ServiceResult) aMethod.invoke(serviceObject, null);
	} 
	catch (NoSuchMethodException e) {throw new RuntimeException(resobjectExtender.getString("ch_noMethod") + aClass);} //$NON-NLS-1$
	catch (InvocationTargetException e) {throw new RuntimeException(resobjectExtender.getString("ch_invocation") + aClass);}		 //$NON-NLS-1$
	catch (IllegalAccessException e3) {throw new RuntimeException(resobjectExtender.getString("ch_illegalAcess") + aClass);} //$NON-NLS-1$

	if (serviceResult.isError())
		throw new VapReadFailureException(serviceResult.errors());
	else
	{
		ejbObjects = new Vector();
		results = serviceObject.oids().elements();
		while (results.hasMoreElements())
		{
			oid = (OID)results.nextElement();
			dataObject = this.lookupOidFromCache(oid);
			if ((ejbObject = this.lookupOidFromView(oid)) != null)
				dataObject.getHome().refresh(ejbObject,dataObject);
			else
				ejbObject = this.createRetrieved(dataObject);
		
			ejbObjects.addElement(ejbObject);
		}
	}	

	return ejbObjects;
}
/**
 * Refresh an object from a do
 */
public void refresh(VapEJBObject anEjbObject,DataObject aDataObject)
{
	Transaction currentTransaction;
	Version oldVersion=null, newVersion;

	currentTransaction = Transaction.getCurrent();
	try {
		oldVersion = anEjbObject.getBom().getVersionForRead();}
	catch (RemoteException e) {}

	newVersion = this.createVersionFrom(anEjbObject,null,aDataObject,Version.createOldPersistentState(),currentTransaction);
	newVersion.reconcileLinksWith(oldVersion);
	oldVersion.setBean(newVersion.getBean());
	oldVersion.getState().setInternallySynchronized(false);
	oldVersion.incrementModificationLevel();
	oldVersion.getState().setExternallySynchronized(true);
	anEjbObject.getBom().setCachedVersion(oldVersion);
}
/**
 * Reload a version for an ejb object
 */
public void reload(VapEJBObject anEjbObject) throws RemoteException,VapReadFailureException
{
	DataObject dataObject;
	OID oid;
	ServiceObject serviceObject;
	ServiceResult serviceResult;

	if (this.getDataStore() == null)
		throw new VapReadFailureException();

	if ((oid = anEjbObject.getBom().primGetObjectIdentifier()) == null)
		throw new VapReadFailureException();

	if ((dataObject = this.lookupOidFromCache(oid)) != null)
		this.createReloaded(anEjbObject,dataObject);
	else
		if ((serviceObject = this.getServiceObject()) != null)
			synchronized (serviceObject)
			{
				serviceObject.setSession(this.getSession());
				serviceResult = serviceObject.findByKey(oid.getKey());
				if (serviceResult.isError())
					throw new VapReadFailureException(serviceResult.errors());
				else
					if ((dataObject = (DataObject)serviceResult.result()) != null)
						this.createReloaded(anEjbObject,dataObject);
			}
}
/**
 * Release any resources we may be holding
 */
public void reset()
{
	super.reset();
	if (cache != null)
		cache.removeAll();
	cache = null;
	dataStore = null;
	so = null;
}
/**
 * Search for a cached DO with the specified Key.
 * If found, return the cached data object
 */
public DataObject searchCacheForKey(Key aKey)
{
	DataObject dataObject = null;

	if ((dataObject = this.getCache().getDO(aKey)) == null)
		dataObject = this.searchSubCachesFor(aKey);

	return dataObject;
}
/**
 * Search our children for a cached DO with the specified Key
 */
public DataObject searchSubCachesFor(Key aKey)
{
	return null;
}
/* 
 * Return a new instance of our Service Object
 */
 	public ServiceObject serviceObject() 
 	{
 		return (ServiceObject) getServiceObject().clone();
 	}   
/* 
 * Return the original, pristine instance of our Service Object.
 */
 	public ServiceObject serviceObjectInstance() 
 	{
 		return so;
 	}    
/* 
 * Set our Service Object
 */
 	public void serviceObjectInstance(ServiceObject services) 
 	{
 		so = services;
 		so.setHome(this);
 	}  
/***
 * Set our Data Store
 **/
public void setDataStore(DataStore aDataStore) 
{
	dataStore = aDataStore;
} 
/* 
 * Answer a nice short name.
 */
 	public String toString() 
 	{
 	 	return "a " + ClassShortName.name(this);
 	} 
}