/****************************************************************************
*  OS2TURBO preloader prototype by Sam Emrick, emrick@us.ibm.com 11/01/2001 *

This program implements a module preloader for OS2WEB Mozilla web browser.
This program is an entirely original work by the author.

The program operates by way of DosLoadModule of a pre-defined list of modules.
In addition to DosLoadModule, the program then queries the ordinal entry points
of the modules and then accesses the address of each entry point in order to
cause the memory page of that entry point to become paged-in and resident in
memory. In this way the program preloads much of the modules used in OS2WEB
execution. After preloading the modules, then the program suspends execution
by infinite wait on a named semaphore. In this manner, the OS2TURBO holds the
OS2WEB modules in memory before and after the execution of OS2WEB.

The UNLOAD option is provided to allow a second invocation of this program to
communicate with, and signal termination of, the first instance of the program.
This allows DETACH OS2TURBO -LOAD to be used to load, then as required,
OS2TURBO -UNLOAD will cause the detached instance of OS2TURBO to unload and
exit. Both -UNLOAD and -LOAD may be specified, to allow a concurrent unload
and reload of modules. This might be used to, for example, reload OS2WEB
modules from a different path.

Notes: As currently implemented, OS2TURBO loads modules from LIBPATH. This
may not be the module location loading as used by OS2WEB. Need to check this.

The list of module names to be loaded is 'hardcoded' in the file os2turbo.h
and specified to the program by means of #include <os2turbo.h>. This list of
files need to be reviewed. Perhaps a less hardcoded list should be implemented.
****************************************************************************/

#define INCL_DOS
#include <os2.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <sys\timeb.h>
#define ERREXIT(msg) {printf(msg);return -1;}

int main(int argc, char *argv[]) {

/* The list of modules to be loaded */
#define nummods 48
 char *mods[nummods] = {
#include "os2turbo.h"
 };

/* The list of supported command line parameters */
 char *parms[6] = { "-DEBUG", "-LOAD", "-UNLOAD", "-HELP", "-?", "-PATH" };
/* Flags to indicate options */
 int debug=0; int doload=0; int dounload=0; int dohelp=0; int dopath=0;

/* Vars for EventSem usage */
 char *semname = "\\SEM32\\OS2TURBO\\OS2TURBO";
 unsigned long mysem = 0;
/* Vars for DosLoadModule usage */
 unsigned long handle=0;
 unsigned long rc=0;
 char *basepath = "";
 char *filepath = "";
/* Vars for DosQueryProcAddr usage */
 PFN modaddr;
/* Vars for DosQueryMem usage */
 unsigned long memsize=0;
 unsigned long memend=0;
 unsigned long memflags=0;
 unsigned char buffer[512];
 volatile unsigned char cpybuf;
 unsigned int bas1, bas2, bas3, bas4, bas5;
/* Help file vars */
 char *myhelp = "os2turbo.hlp";
 char *pchar;
 FILE *hfile;
/* Loop control vars */
 unsigned int i=0;
 unsigned int j=0;
 unsigned int base=0;
 unsigned char* baseptr=0;

/* Parse command line args, set flags */
 if (argc==1) dohelp=1;
 for (i=1; i < argc; i++) {
    argv[i] = strupr(argv[i]);
    if (0==strcmp(argv[i],parms[0])) debug=1;
    else if (0==strcmp(argv[i],parms[1])) doload=1;
    else if (0==strcmp(argv[i],parms[2])) dounload=1;
    else if (0==strcmp(argv[i],parms[3])) dohelp=1;
    else if (0==strcmp(argv[i],parms[4])) dohelp=1;
    else if (0==strcmp(argv[i],parms[5])) dopath=i;
    } // end for

/* If requested, process -HELP or -? */
 if (dohelp) {
    /* print from help file if available */
    if (NULL != (hfile=fopen(myhelp,"r"))) {
       while (fread(buffer, sizeof(char), 511, hfile) >0) {
          printf(buffer);
          strnset(buffer,0,512); /* clear buf, easier way? */
          } // end while
       } // end if
    /* If no help file, print this message instead */
    else {
       printf("OS2TURBO help file not found, but... \n"\
              "parms are: \n"                          \
              "  -HELP    to get this textual help. \n"\
              "  -LOAD    to sysload OS2WEB modules.\n"\
              "  -UNLOAD  to un-load OS2WEB modules.\n"\
              "  -PATH x  to specify fully qualified\n"\
              "            path to OS2WEB.EXE file. \n"\
              "  -DEBUG   to turn on debug messages.\n"\
              "Any combination of parms are allowed.\n");
       } // end else
    } // end if dohelp

/* If requested, process -UNLOAD */
 if (dounload) {
   /* If already loaded, sem create will fail */
   if (0 == DosCreateEventSem(semname, &mysem, 1, 0))
      ERREXIT("OS2TURBO Not loaded, so cannot unload.\n");

   /* Open the sem and post, other instance is waiting for it */
   if (0 != DosOpenEventSem(semname, &mysem))
      ERREXIT("OS2TURBO UNLOAD DosOpenEventSem error.\n");

   if (0 != DosPostEventSem(mysem))
      ERREXIT("OS2TURBO UNLOAD DosPostEventSem error.\n");

   /* Close sem (because we might want to reload) */
   if (0 != DosCloseEventSem(mysem))
      ERREXIT("OS2TURBO UNLOAD DosCloseEventSem error.\n");

   /* Give other instance time to execute, exit, and destroy the sem */
   DosSleep(100L);
   } // end if dounload

/* If requested, process -PATH */
 if (dopath) {
  basepath = argv[dopath+1];
  i=strlen(basepath);
  if (basepath[i-4]=='.' && basepath[i-3]=='E' && basepath[i-2]=='X' && basepath[i-1]=='E') {
    /* path parm ends with .exe or .EXE so strip back to previous backslash */
     pchar = strrchr(basepath, '\\');
     pchar++;
     *pchar = '\0';
     }
   else if (basepath[i-1]!='\\') {
    /* path parm ends with neither .EXE nor backslash so add a backslash */
     basepath = strcat(basepath,"\\");
     }
  } // end if dopath

/* If requested, process -LOAD */
 if (doload) {
   DosSetMaxFH(120L); /* insure sufficient file handles */
   /* create the shared interprocess sem used to signal quit */
   if (0 != DosCreateEventSem(semname, &mysem, 1, 0))
      ERREXIT("OS2TURBO Cannot load, already loaded.\n");

/* Set up extended libpath, form is "path;path\COMPONENTS;.\;" */
   filepath = strcpy(buffer,basepath);
   filepath = strcat(filepath,";");
   filepath = strcat(filepath,basepath);
   filepath = strcat(filepath,"COMPONENTS;.\\;\0");
   rc = DosSetExtLIBPATH(filepath,1);
   if (debug) printf("DosSetExtLIBPATH %s rc=%u \n", filepath,rc);

   /* loop through list loading named modules */
   for (i=0; i<nummods; i++) {
      bas1=0; bas2=0; bas3=0; bas4=0; bas5=0;
      buffer[0]='\0';
      filepath = strcpy(buffer,basepath);
      filepath = strcat(filepath,mods[i]);
      filepath = strcat(filepath,".DLL\0");

      rc = DosLoadModule(buffer,sizeof(buffer),filepath,&handle);
        if (debug) printf("DosLoadModule[%u] %s rc= %u\n",i,filepath,rc);
      if (rc!=0) {
        buffer[0]='\0'; /* clear buf, easier way? */
        filepath = strcpy(buffer,basepath);
        filepath = strcat(filepath,"COMPONENTS\\");
        filepath = strcat(filepath,mods[i]);
        filepath = strcat(filepath,".DLL\0");
        rc = DosLoadModule(buffer,sizeof(buffer),filepath,&handle);
        if (debug) printf("DosLoadModule[%u] %s rc= %u\n",i,filepath,rc);
        }
      if (rc==0) {
        /* Loop through module ordinal entry points */
        for (j=1; (0==DosQueryProcAddr(handle,j,0,&modaddr) && j<512); j++) {
//        if (debug) printf("DosQueryProcAddr %u %x\n",j, modaddr);

          /* calc 64K aligned addr previous to entry point */
          base=(( (unsigned long)modaddr) & 0xFFFF0000);
          /* filter previously loaded base addresses */
          if (base!=bas1 && base!=bas2 && base!=bas3 && base!=bas4 && base!=bas5) {

            /* get size and flags for this memory area */
            memsize=0x0fffffff;
            DosQueryMem((void*)base,&memsize,&memflags);
//          if (debug) printf("DosQueryMem %u %u %u\n",base, memsize, memflags);

            /* if not first page of object, back off addr and retry */
            while (memflags < 0x10000) {
              base=base-65536;
              memsize=0x0fffffff;
              DosQueryMem((void*)base,&memsize,&memflags);
              } // end while

            /* again filter previously loaded base addresses */
            /* also filter any readwrite pages (usually not shared) */
//          if (memflags&0x02 && base!=bas1 && base!=bas2 && base!=bas3 && base!=bas4 && base!=bas5) {
            if (base!=bas1 && base!=bas2 && base!=bas3 && base!=bas4 && base!=bas5) {
            /* not previously loaded, so cache the base address (first 5) */
              if (bas1==0) bas1=base;
              else if (bas2==0) bas2=base;
              else if (bas3==0) bas3=base;
              else if (bas4==0) bas4=base;
              else if (bas5==0) bas5=base;

              /* finally, now loop through object pages, force page-in */
              memend=base+memsize;
              while(base<memend) {
//              if (debug) printf("end= %X base= %X \n",memend, base);
                baseptr=(unsigned char*)base;
                cpybuf=*baseptr;
                base+=4096;
                }
              if (debug) {
                char tbuf[9]; struct timeb tb; ftime(&tb);
                printf("%s.%03d file[%u] %s hnd %u ord %d addr %X bytes %X flags %X\n",\
                   _strtime(tbuf),tb.millitm,i,filepath,handle,j,base,memsize,memflags);
                } // end if debug
              } // end if base (inner)
            } // end if base (outer)
          } // end for j (ordinals)
        } // end if rc==0
      else if (debug) printf("ERROR=%u Loading module[%d] %s\n",rc,i,mods[i]);
      } // end for i

   /* Go into infinite wait for the post signaling us to quit */
   if (debug) printf("\nDone Loading, going into wait.\n");
   if (0 != DosWaitEventSem(mysem, (unsigned long) -1L))
      ERREXIT("OS2TURBO LOAD DosWaitEventSem error.\n");

   /* Another instance of us must have posted it. Close the sem */
   if (0 != DosCloseEventSem(mysem))
      ERREXIT("OS2TURBO LOAD DosCloseEventSem error.\n");
   } // end if doload

/* When the program exits, any loaded modules will unload automaticly */
 return 0;
}
