/*
 * gdict Dictionary for UNIX
 * Copyright (C) 2003 Alexander Zapjagaev <frustum@frustum.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "mainwindow.h"

static void file_dictionary(mainwindow_t *mainwin);
static void file_quit(mainwindow_t *mainwin);
static void help_about(mainwindow_t *mainwin);

/* 
 */
static GtkItemFactoryEntry menubar_entries[] = {
	{ N_("/File"), NULL, NULL, 0, "<Branch>" },
	{ N_("/File/Dictionaries"), "<ctrl>D", file_dictionary, 0, NULL },
	{ N_("/File/---"), NULL, NULL, 0, "<Separator>" },
	{ N_("/File/Quit"), "<ctrl>Q", file_quit, 0, NULL },
	{ N_("/Help"), NULL, NULL, 0, "<LastBranch>" },
	{ N_("/Help/About"), NULL, help_about, 0, NULL }
};

/*   /file/dictionary
 */
static void file_dictionary(mainwindow_t *mainwin) {
	dict_window_create(mainwin);
}

/*   /file/quit
 */
static void file_quit(mainwindow_t *mainwin) {
	gtk_widget_destroy(mainwin->window);
}

/*   /help/about
 */
static void help_about(mainwindow_t *mainwin) {
	about_create();
}

/*  
 */
static void window_destroy(GtkWidget *widget,mainwindow_t *mainwin) {
	gtk_main_quit();
}

/*  
 */
static void focus_in_event(GtkWidget *widget,GdkEventFocus *event,mainwindow_t *mainwin) {
	if(widget == mainwin->clist) mainwin->focus = FOCUS_CLIST;
	else mainwin->focus = FOCUS_ENTRY;
}

/*    clist
 */
static void clist_select_row(GtkWidget *widget,int row,int column,
	GdkEventButton *event,mainwindow_t *mainwin) {
	char *text,*text_utf8,*trans;
	/*         */
	if(mainwin->focus != FOCUS_CLIST) return;
	gtk_clist_get_text(GTK_CLIST(widget),row,0,&text_utf8);
	gtk_entry_set_text(GTK_ENTRY(mainwin->entry),text_utf8);
	text = g_convert(text_utf8,-1,LOCALE,"UTF-8",NULL,NULL,NULL);
	trans = dict_word_find(mainwin->root,text,&mainwin->index);
	if(event && event->type == GDK_2BUTTON_PRESS) {
		if(trans) trans_window_create(text,trans);
	} else {
		if(trans) trans_window_change(text,trans);
	}
	g_free(text);
}

/*     clist
 */
static int clist_press_event(GtkWidget *widget,GdkEventKey *event,mainwindow_t *mainwin) {
	if(event->keyval == GDK_Return) {
		char *text_utf8,*text;
		gtk_clist_get_text(GTK_CLIST(mainwin->clist),mainwin->index,0,&text_utf8);
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget),"key_press_event");
		text = g_convert(text_utf8,-1,LOCALE,"UTF-8",NULL,NULL,NULL);
		char *trans = dict_word_find(mainwin->root,text,&mainwin->index);
		if(trans) trans_window_create(text,trans);
		g_free(text);
		return 1;
	}
	if(event->keyval == GDK_Escape) {
		trans_window_destroy();
		return 1;
	}
	return 0;
}

/*  entry
 */
static void entry_changed(GtkWidget *widget,mainwindow_t *mainwin) {
	char *text,*text_utf8,*trans;
	/*         */
	if(mainwin->focus != FOCUS_ENTRY) return;
	text_utf8 = (char*)gtk_entry_get_text(GTK_ENTRY(widget));
	text = g_convert(text_utf8,-1,LOCALE,"UTF-8",NULL,NULL,NULL);
	trans = dict_word_find(mainwin->root,text,&mainwin->index);
	if(mainwin->index != -1) {
		GtkAdjustment *adj;
		float value;
		adj = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(mainwin->scrolled_window));
		value = (adj->upper - adj->lower) * mainwin->index / mainwin->num_words - adj->page_size / 2.0;
		if(value < adj->lower) value = adj->lower;
		if(value > adj->upper - adj->page_size) value = adj->upper - adj->page_size;
		if(fabs(adj->value - value) > adj->page_size / 2.0) gtk_adjustment_set_value(adj,value);
		gtk_clist_select_row(GTK_CLIST(mainwin->clist),mainwin->index,0);
	} else {
		char buf[1024];
		strcpy(buf,text);
		if(strlen(buf) >= 1) {
			buf[strlen(buf) - 1] = '\0';
			text_utf8 = g_convert(buf,-1,"UTF-8",LOCALE,NULL,NULL,NULL);
			gtk_entry_set_text(GTK_ENTRY(widget),text_utf8);
			g_free(text_utf8);
		}
	}
	if(trans) trans_window_change(text,trans);
	g_free(text);
}

/*   
 *  return, up, down
 */
static int entry_key_press_event(GtkWidget *widget,GdkEventKey *event,mainwindow_t *mainwin) {
	char *text;
	if(event->keyval == GDK_Return) {
		char *text_utf8;
		gtk_clist_get_text(GTK_CLIST(mainwin->clist),mainwin->index,0,&text_utf8);
		gtk_entry_set_text(GTK_ENTRY(mainwin->entry),text_utf8);
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget),"key_press_event");
		text = g_convert(text_utf8,-1,LOCALE,"UTF-8",NULL,NULL,NULL);
		char *trans = dict_word_find(mainwin->root,text,&mainwin->index);
		if(trans) trans_window_create(text,trans);
		g_free(text);
		return 1;
	}
	if(event->keyval == GDK_Up || event->keyval == GDK_Page_Up) {
		if(event->keyval == GDK_Up) mainwin->index--;
		else mainwin->index -= 5;
		if(mainwin->index < 0) mainwin->index = 0;
		gtk_clist_get_text(GTK_CLIST(mainwin->clist),mainwin->index,0,&text);
		gtk_entry_set_text(GTK_ENTRY(mainwin->entry),text);
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget),"key_press_event");
		return 1;
	}
	if(event->keyval == GDK_Down || event->keyval == GDK_Page_Down) {
		if(event->keyval == GDK_Down) mainwin->index++;
		else mainwin->index += 5;
		if(mainwin->index > mainwin->num_words - 1) mainwin->index = mainwin->num_words - 1;
		gtk_clist_get_text(GTK_CLIST(mainwin->clist),mainwin->index,0,&text);
		gtk_entry_set_text(GTK_ENTRY(mainwin->entry),text);
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget),"key_press_event");
		return 1;
	}
	if(event->keyval == GDK_Escape) {
		trans_window_destroy();
		return 1;
	}
	return 0;
}

/*   
 */
mainwindow_t *main_window_create(void) {
	mainwindow_t *mainwin;
	GtkWidget *window;
	GtkWidget *vbox;
	GtkWidget *menubar;
	GtkWidget *scrolled_win;
	GtkWidget *clist;
	GtkWidget *entry;
	
	mainwin = calloc(1,sizeof(mainwindow_t));
	
	/* window */
	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(window),PACKAGE"-"VERSION);
	gtk_signal_connect(GTK_OBJECT(window),"destroy",
		GTK_SIGNAL_FUNC(window_destroy),mainwin);
	
	/* vbox */
	vbox = gtk_vbox_new(FALSE,0);
	gtk_container_add(GTK_CONTAINER(window),vbox);
	
	/* menubar */
	menubar = menubar_create(window,menubar_entries,
		sizeof(menubar_entries) / sizeof(menubar_entries[0]),"<main>",mainwin);
	gtk_box_pack_start(GTK_BOX(vbox),menubar,FALSE,TRUE,0);
	
	/* scrolled_win */
	scrolled_win = gtk_scrolled_window_new(NULL,NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_win),
		GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
	gtk_widget_set_usize(scrolled_win,200,220);
	gtk_box_pack_start(GTK_BOX(vbox),scrolled_win,TRUE,TRUE,0);

	/* clist */
	clist = gtk_clist_new(1);
	gtk_clist_set_shadow_type(GTK_CLIST(clist),GTK_SHADOW_OUT);
	gtk_clist_set_selection_mode(GTK_CLIST(clist),GTK_SELECTION_BROWSE);
	gtk_clist_set_column_width(GTK_CLIST(clist),0,150);
	gtk_container_add(GTK_CONTAINER(scrolled_win),clist);
	gtk_signal_connect(GTK_OBJECT(clist),"focus_in_event",
		GTK_SIGNAL_FUNC(focus_in_event),mainwin);
	gtk_signal_connect(GTK_OBJECT(clist),"select_row",
		GTK_SIGNAL_FUNC(clist_select_row),mainwin);
	gtk_signal_connect(GTK_OBJECT(clist),"key_press_event",
		GTK_SIGNAL_FUNC(clist_press_event),mainwin);
	
	/* entry */
	entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(vbox),entry,FALSE,TRUE,0);
	gtk_signal_connect(GTK_OBJECT(entry),"focus_in_event",
		GTK_SIGNAL_FUNC(focus_in_event),mainwin);
	gtk_signal_connect(GTK_OBJECT(entry),"changed",
		GTK_SIGNAL_FUNC(entry_changed),mainwin);
	gtk_signal_connect(GTK_OBJECT(entry),"key_press_event",
		GTK_SIGNAL_FUNC(entry_key_press_event),mainwin);
	
	gtk_widget_grab_focus(entry);

	gtk_widget_show_all(window);
	
	mainwin->window = window;
	mainwin->scrolled_window = scrolled_win;
	mainwin->clist = clist;
	mainwin->entry = entry;
	
	return mainwin;
}

/*  
 */
void main_window_dict_load(mainwindow_t *mainwin,char *name) {
	if(mainwin->root) {
		if(dict_add(mainwin->root,name) == -1) return;
	} else {
		mainwin->root = dict_load(name);
		if(!mainwin->root) return;
	}
	mainwin->num_dicts++;
	mainwin->dicts = realloc(mainwin->dicts,sizeof(char*) * mainwin->num_dicts);
	mainwin->dicts[mainwin->num_dicts - 1] = strdup(name);
}

/*  
 */
void main_window_dict_free(mainwindow_t *mainwin) {
	int i;
	if(!mainwin->dicts) return;
	for(i = 0; i < mainwin->num_dicts; i++) free(mainwin->dicts[i]);
	free(mainwin->dicts);
	mainwin->dicts = NULL;
	mainwin->num_dicts = 0;
	dict_free(mainwin->root);
	mainwin->root = NULL;
	mainwin->num_words = 0;
	mainwin->index = -1;
}

/*    clist
 */
static void main_window_clist_load_preorder(dict_t *n,GtkWidget *clist) {
	int i;
	if(n->word) {
		char *utf8 = g_convert(n->word,-1,"UTF-8",LOCALE,NULL,NULL,NULL);
		char **data = &utf8;
		gtk_clist_append(GTK_CLIST(clist),data);
		g_free(utf8);
	}
	for(i = 0; i < n->num_childs; i++) {
		main_window_clist_load_preorder(n->childs[i],clist);
	}
}

/*    clist
 */
void main_window_clist_load(mainwindow_t *mainwin) {
	if(!mainwin->root) return;
	mainwin->num_words = dict_index(mainwin->root);
	main_window_clist_load_preorder(mainwin->root,mainwin->clist);
}

/*  clist
 */
void main_window_clist_free(mainwindow_t *mainwin) {
	gtk_clist_clear(GTK_CLIST(mainwin->clist));
}
