/******************************************************************************
//               INTEL CORPORATION PROPRIETARY INFORMATION
//  This software is supplied under the terms of a license agreement or
//  nondisclosure agreement with Intel Corporation and may not be copied
//  or disclosed except in accordance with the terms of that agreement.
//        Copyright (c) 2000 Intel Corporation. All Rights Reserved.
//
//  VSS:
//		$Workfile: Filtering.c $
//      $Revision: 1 $
//      $Date: 03-09-10 13:20 $
//      $Archive: /ipp_se/primitive/ipp4/ipp40/beta/bulverde/install/arm_linux/sample/Filtering/Filtering.c $
//
//  Description:
//      IPP filtering example
//
******************************************************************************/

/************
Header files
*************/

/* Linux audio output device interface */
#include	<fcntl.h>					
#include	<unistd.h>
#include	<sys/ioctl.h>
#include	<linux/soundcard.h>
#include	<sched.h>

/* Console, file, and string I/O */
#include	<stdio.h>
#include	<string.h>

/* Standard IPP definitions */
#include	"ippdefs.h"
#include	"ippSP.h"

/* Interface to IPP-based examples */
#include	"filter.h"

/* Local constants */
#define		FS				11025		/* Sample rate */
#define		N				2048		/* Buffer size */

/* Initial conditions */
#define		INIT_SIGMA		1000		/* Startup standard deviation for noise generator */

/* User interface constants */
#define		UI_STACK_SIZE	2500		/* User interface stack size, in 32-bit words */					
#define		MSGBUF_LEN      1000		/* User interface keyboard buffer size, in bytes */
#define		PBUF_LEN		100			/* User interface parsing buffer */

/*************
 Prototypes 
**************/
void			DisplayConsoleStartupBanner		(void);
void			ProcessCommandLineArguments		(int, char **, FILE **);
void			RenderSound						(int *, Ipp16s *, int);
int				OpenAudioDevice					(int *, int, int);
void			CloseOutputFile					(FILE *);
void			CloseAudioDevice				(int);
void			WriteOutputFile					(FILE *, Ipp16s *, int);
int				UserInterface					(void);

/*************
 Global data 
**************/
int g_Quit=0;					/* Quit flag, set asynchronously by the user interface */

/*************************************
 Console-mode application entry point 
**************************************/
/******************************************************************************************
// Name:			 main
//
// Description:		 Signal generation sample code
//
// Input Arguments:  argc				- Standard C argument count
//					 argv				- Standard C argument list
//
*******************************************************************************************/
int main(int argc, char **argv)
{
	FILE *fpo;					/* Output file */							
	Ipp16s sig[N];				/* Signal buffer */
	Ipp16s noise[N];			/* Noise buffer */
	int AudioDevice;			/* Pointer to the (Linux) audio output device */
	int StackUI[UI_STACK_SIZE];	/* User interface stack (child thread) */
	int snr;					/* Signal-to-noise ratio, in dB */

	/****************************************************
	Startup banner, parse command line, open I/O devices  
	*****************************************************/
	DisplayConsoleStartupBanner();	
	ProcessCommandLineArguments(argc,argv,&fpo);

	/* Spawn user interface thread */
	__clone(UserInterface,&(StackUI[UI_STACK_SIZE]),CLONE_VM|CLONE_FS);

	/* Open audio device if no output file was specified */
	if (fpo==NULL)
		OpenAudioDevice(&AudioDevice,2,FS);

	/* Establish initial conditions */
	/* Tone on, noise off */
	SetNoiseSigma(INIT_SIGMA);
	SetToneFrequency(DEFAULT_TONE_FRQ,FS);
	SetToneAmplitude(DEFAULT_TONE_AMP);
	SelectOutputMode(PASS_THRU);
	SelectFilterType(FIR);

	/* Generate signal */
	while(!g_Quit)
	{
		GenerateSignal(sig,noise,N);
		/* Copy to audio device or output file */
		if (fpo==NULL)
			RenderSound(&AudioDevice,sig,N);
		else
			WriteOutputFile(fpo,sig,N);

		/* Display SNR */
		GetSNR(&snr);
		printf("SNR = %d dB\r",snr);
	}

	/* Close I/O device */	
	if (fpo!=NULL)
		CloseOutputFile(fpo);
	else
		CloseAudioDevice(AudioDevice);

	/* Shtdown message */
	printf("Done.\n\n");

} /* End of main() */

/********************************
 Console management and file I/O 
*********************************/

/*********************************************************************************************
// Name:			 DisplayConsoleStartupBanner
//
// Description:		 Display startup message on the text console
//
// Input Arguments:  None.
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
void DisplayConsoleStartupBanner()
{
	int i;
	for(i=0;i<50;i++) printf("\n");
	printf("\n\nIntel(R) Integrated Performance Primitives (IPP)\nfor Intel(R) XScale(TM) Microarchitecture.\n\n");
	printf("Filtering Example, Version 1.00\n\n");
	printf("Copyright (C) 2000-01, Intel Corporation.  All Rights Reserved.\n\n");
}

/*********************************************************************************************
// Name:			 ProcessCommandLineArguments
//
// Description:		 Extracts run-time parameters from the command line
//					 Opens I/O files
//
// Input Arguments:  argc				- Standard C argument count
//					 argv				- Standard C argument list
//					 fpo				- Pointer to the output file stream
//
// Output Arguments: None				
//
// Returns:			 None
**********************************************************************************************/
void ProcessCommandLineArguments(int argc, char **argv, FILE **fpo)
{
	/* Usage */
	if (((argc>1)&&(strcmp(argv[1],"--help")==0))||(argc>2))
	{
		printf("Usage: Filtering <output file> [--help], \n\nwhere\n\n");
		printf("<output file> is the desired binary pcm output file\n\n");
		printf("No output file forces use of the audio device for output.\n\n");
		printf("--help displays usage\n\n");
		printf("\n");
		exit(0);
	}

	/* Get desired output device 
	   No argument = use audio device
	   Argument present = output file name
	 */  
    /* Open output file */
	if (argc>=2)
	{
		if ((*fpo=fopen(argv[1],"wb"))==NULL)
		{
			printf("Unable to open %s\n",argv[1]);
			exit(1);
		}
		else
			printf("Output file:\t%s\n",argv[1]);	
	}
	else
	{
		*fpo=NULL;
		printf("Output:\tAudio device\n");
	}
}

/*********************************************************************************************
// Name:			 CloseOutputFile
//
// Description:		 Close PCM binary output file
//
// Input Arguments:  fpo				- Pointer to the output file stream
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
void CloseOutputFile(FILE *fpo)
{
	if (fpo!=NULL) fclose(fpo);
}

/*****************************
 Linux Audio Device Interface
******************************/

/***************************************************************************************
// Name:			 RenderSound
//
// Description:		 Write one granule of pcm audio samples to the audio output device. 
//
// Input Arguments:  fp					- Pointer to the audio output device. 
//					 pcm				- Pointer to a buffer of interleaved pcm audio.
//					 channels			- Number of channels contained in the buffer.
//					 len				- PCM buffer length, in samples (not bytes)
//
// Returns:			 None.
//
// Notes:			 Stereophonic data is interleaved as follows:  R0L0R1L1 ... R575L575; 
//					 For channels=1, right channel data is replicated on to the left.
****************************************************************************************/
void RenderSound(int *fp, Ipp16s *pcm, int len)
{
	static Ipp16s obuf[4*2*N];
	static Ipp16s rsin[4*N],rsout[4*N];
	int i,j;

	/* Filter parameters */
	int RS_TAPS=38;
	static Ipp16s DelayLineRS[80];
	static int DelayLineIndexRS;
	static int init=0;
	Ipp16s rstaps[38]=
	{  -150, -215, -258, -192, 15, 328, 641, 802, 680, 234, -428, -1051, -1306, -908, 260,
		2063, 4124, 5925, 6974, 6974, 5925, 4124, 2063, 260, -908, -1306, -1051, -428, 234,
		680, 802, 641, 328, 15, -192, -258, -215, -150
	};

	if (!init)
	{
		init=1;
		ippsZero_16s(DelayLineRS,80);
		DelayLineIndexRS=0;
	}

	/* Upsample from 11.025 to 44.1 kHz */
	for(i=j=0;i<4*len;i+=4,j++)
		rsin[i]=pcm[j];

	/* Interpolate */
	ippsFIR_Direct_16s(rsin,rsout,4*len,rstaps,RS_TAPS,DelayLineRS,&DelayLineIndexRS);		

	/* Replicate signal on both left and right channels */
	for(i=j=0;i<8*len;i+=2,j++)
		obuf[i]=obuf[i+1]=rsout[j];

	/* Pass audio output buffer to the audio output device */
	write(*fp,obuf,sizeof(Ipp16s)*len*8);
}

/***************************************************************************************
// Name:			 OpenAudioDevice
//
// Description:		 Initialize Linux audio output device. 
//
// Input Arguments:  dev				- Pointer to the audio output device. 
//					 channels			- Number of audio channels.
//
// Returns:			 Status				 0	 = success, device ready
//										-100 = unable to open device
//										-101 = unable to select 16-bit stereo PCM
//										-102 = unable to select 44100 Hz sample rate
//										-103 = unable to select specified # of channels 
//
****************************************************************************************/
int OpenAudioDevice(int *dev, int channels, int SampleRate)
{
	long param;
	int status=0;

	/* Open audio device (UDA1341 on the Linux SA-1110 Microprocessor Evaluation Module) */
	if ((*dev=open("/dev/dsp",O_RDWR))==-1) status=-100;

	/* Request 16-bit, little-endian, PCM data format */
	param = AFMT_S16_LE;
	if ((ioctl(*dev,SNDCTL_DSP_SETFMT,&param))!=0) { close (*dev); status=-101; }
  
	/* Request sample rate */
	param=SampleRate;
	if ((ioctl (*dev,SNDCTL_DSP_SPEED,&param))!=0) { close (*dev); status=-102; }

	/* Request number of channels */
	param = channels;
	if ((ioctl(*dev,SNDCTL_DSP_CHANNELS,&param))!=0) { close (*dev); status=-103; }

	/* Trap unsuccessful attempts to open audio device */
	if (status<0) { fprintf(stderr,"\nError opening audio device.\n"); fflush(stderr); exit(1); }
	else return 0;
}

/*********************************************************************************************
// Name:			 CloseAudioDevice
//
// Description:		 Close audio output device
//
// Input Arguments:  AudioDevice		- Pointer to the audio output device
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
void CloseAudioDevice(int AudioDevice)
{
	close(AudioDevice);
}

/*********************************************************************************************
// Name:			 WriteOutputFile
//
// Description:		 Write PCM buffer contents to a binary file
//
// Input Arguments:  fpo				- Pointer to the output file stream
//					 pcm				- PCM output buffer pointer
//					 N					- Number of samples to write
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
void WriteOutputFile(FILE *fpo, Ipp16s *pcm, int len)
{
	if (fpo!=NULL) 
		fwrite(pcm,sizeof(Ipp16s),len,fpo);
}

/*********************************************************************************************
// Name:			 User interface thread
//
// Description:		 Keyboard character-based user interface
//					 Thread terminates with the parent
//
// Returns:			 None				
**********************************************************************************************/
int UserInterface()
{
	char msg[MSGBUF_LEN], tstr[PBUF_LEN];	/* Keyboard message and parsing buffer */
	int i,freq;								/* Buffer indices and control params */
	Ipp16s sigma,a;
	int NoiseEnable=0,ToneEnable=0;
	int OutputSelect=PASS_THRU;
	int FilterSelection=FIR;
	while(1)
	{
		/* Get user input from the keyboard */
		gets(msg);

		/* Parse and act on user input */
		switch(msg[0])
		{
			/* a = set tone amplitude */
			case 'a':
			case 'A':
				strncpy(tstr,&(msg[1]),5);
				a=atoi(tstr);
				SetToneAmplitude(a);
				printf("Tone amplitude (Q15): %d\n",a);
				break;
			/* f = set tone frequency */
			case 'f':
			case 'F':
				strncpy(tstr,&(msg[1]),5);
				freq=atoi(tstr);
				SetToneFrequency(freq,FS);
				printf("Tone frequency: %d Hz\n",freq);
				break;
			/* h = help */
			case 'h':
			case 'H':
				printf("\n\nCommands:\n");
				printf("a<Q15 amplitude> - set tone amplitude\n");
				printf("f<frequency, Hz> - set tone frequency\n");
				printf("h                - display command list\n");
				printf("i                - toggle FIR/IIR filter select\n");
				printf("o                - toggle pass_thru/LPF/HPF output select\n");
				printf("v<sigma>         - set noise standard deviation\n");
				printf("q                - QUIT\n");
				printf("\n\n");
				break;
			/* i = toggle FIR/IIR filter selection */
			case 'i':
			case 'I':
				FilterSelection=(~FilterSelection)&0x1;
				SelectFilterType(FilterSelection);
				switch(FilterSelection)
				{
					case FIR:
						printf("FIR filter selected\n");
						break;
					case IIR:
						printf("IIR filter selected\n");
						break;
				}
				break;
			/* o = toggle PASS_THRU/LPF/HPF output select */
			case 'o':
			case 'O':
				OutputSelect+=1;
				if (OutputSelect>HPF)
					OutputSelect=0;
				SelectOutputMode(OutputSelect);	
				switch(OutputSelect)
				{
					case PASS_THRU:
						printf("Unfiltered signal + noise output selected\n");
						break;
					case LPF:
						printf("LPF output selected\n");
						break;
					case HPF:
						printf("HPF output selected\n");
						break;
				}
				break;
			/* q = Quit */
			case 'q':
			case 'Q':
				g_Quit=1;
				break;
			/* v<sigma> = adjust Gaussian noise stddev to the value sigma */
			case 'v':
			case 'V':
				strncpy(tstr,&(msg[1]),5);
				sigma=atoi(tstr);
				SetNoiseSigma(sigma);
				printf("Gaussian sigma: %d\n",sigma);
				break;
			default:
				break;
		}

		/* Clear keyboard message buffer */
		for(i=0;i<MSGBUF_LEN;i++)
			msg[i]='\0';
		/* Clear parse buffer */
		for(i=0;i<PBUF_LEN;i++)
			tstr[i]='\0';
	}
}
