/*****************************************************************

Copyright (c) 1996-2000 the kicker authors. See file AUTHORS.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <qcursor.h>
#include <qlayout.h>
#include <qpainter.h>
#include <qstyle.h>
#include <qpixmapcache.h>
#include <qtooltip.h>

#include <kpushbutton.h>
#include <kapplication.h>
#include <kdebug.h>
#include <kglobal.h>
#include <klocale.h>

#include "container_applet.h"
#include "kickerSettings.h"

#include "applethandle.h"

static const char* const up_xpm[]={
"5 5 2 1",
"# c black",
". c None",
".....",
"..#..",
".###.",
"#####",
"....."};

static const char* const down_xpm[]={
"5 5 2 1",
"# c black",
". c None",
".....",
"#####",
".###.",
"..#..",
"....."};

static const char* const left_xpm[]={
"5 5 2 1",
"# c black",
". c None",
"...#.",
"..##.",
".###.",
"..##.",
"...#."};

static const char* const right_xpm[]={
"5 5 2 1",
"# c black",
". c None",
".#...",
".##..",
".###.",
".##..",
".#..."};

AppletHandle::AppletHandle(AppletContainer* parent, const char* name)
    : QWidget(parent, name),
      _parent(parent),
      _fadeout_handle(false),
      _drawIt(false),
      m_popupDirection(KPanelApplet::Up)
{
   setBackgroundOrigin( AncestorOrigin );
   setMinimumSize(widthForHeight(0), heightForWidth(0));
   _layout = new QBoxLayout( this, QBoxLayout::BottomToTop, 0, 0 );

   _drag = new AppletHandleDrag( this );
   _drag->installEventFilter(this);
   _layout->addWidget( _drag );

   _parent->installEventFilter( this );

   _menuButton = new AppletHandleButton( this );
   _menuButton->setPixmap( xpmPixmap( up_xpm, "up" ));
   _menuButton->installEventFilter(this);
   _layout->addWidget( _menuButton );

   connect(_menuButton, SIGNAL(pressed()), this, SLOT(menuButtonPressed()));

   QToolTip::add(this, i18n("%1 applet handle").arg(parent->info().name()));
   QToolTip::add(_menuButton, i18n("%1 menu").arg(parent->info().name()));
   resetLayout();
}

int AppletHandle::heightForWidth( int /* w */ ) const
{
    int size = style().pixelMetric(QStyle::PM_DockWindowHandleExtent, this);

    if (KickerSettings::transparent())
    {
        size += 2;
    }

    return size;
}

int AppletHandle::widthForHeight( int /* h */ ) const
{
    int size = style().pixelMetric(QStyle::PM_DockWindowHandleExtent, this);

    if (KickerSettings::transparent())
    {
        size += 2;
    }

    return size;
}

void AppletHandle::setPopupDirection(KPanelApplet::Direction d)
{
    if (d == m_popupDirection)
    {
        return;
    }

    m_popupDirection = d;

    // Set a minimum size so that fade out handles does not cause
    // applets to move around.
    setMinimumSize(_menuButton->minimumSizeHint());

    switch (m_popupDirection)
    {
        case KPanelApplet::Up:
            _layout->setDirection(QBoxLayout::BottomToTop);
            _menuButton->setPixmap(xpmPixmap( up_xpm, "up" ));
            break;
        case KPanelApplet::Down:
            _layout->setDirection(QBoxLayout::TopToBottom);
            _menuButton->setPixmap(xpmPixmap( down_xpm, "down"));
            break;
        case KPanelApplet::Left:
            _layout->setDirection(QBoxLayout::RightToLeft);
            _menuButton->setPixmap(xpmPixmap( left_xpm, "left"));
            break;
        case KPanelApplet::Right:
            _layout->setDirection(QBoxLayout::LeftToRight);
            _menuButton->setPixmap(xpmPixmap( right_xpm, "right"));
            break;
    }
    _layout->activate();
}

void AppletHandle::resetLayout()
{
  if (_fadeout_handle && !_drawIt) {
    _drag->hide();
    _menuButton->hide();
  } else {
    _drag->show();
    _menuButton->show();
  }
}

void AppletHandle::setFadeOutHandle(bool v)
{
  _fadeout_handle = v;
  resetLayout();
}

bool AppletHandle::eventFilter (QObject *o, QEvent *e)
{
    if (o == parent())
    {
        switch (e->type())
        {
            case QEvent::Enter:
                _drawIt = true;
                resetLayout();
                break;

            case QEvent::Leave:
            {
                QWidget* w = dynamic_cast<QWidget*>(o);

                bool nowDrawIt = false;
                if (w)
                {
                    // a hack for applets that have out-of-process
                    // elements (e.g the systray) so that the handle
                    // doesn't flicker when moving over those elements
                    if (w->rect().contains(w->mapFromGlobal(QCursor::pos())))
                    {
                        nowDrawIt = true;
                    }
                }

                if (nowDrawIt != _drawIt)
                {
                    _drawIt = nowDrawIt;
                    resetLayout();
                }
                break;
            }

            default:
                break;
        }

        return QWidget::eventFilter( o, e );
    }

    switch (e->type())
    {
        case QEvent::MouseButtonPress:
        {
            QMouseEvent* ev = (QMouseEvent*) e;
            if (ev->button() == RightButton)
            {
                if (_menuButton->isDown())
                {
                    toggleMenuButtonOff();
                }
                else
                {
                    _menuButton->setDown(true);
                    menuButtonPressed();
                }
                return true;
            }
            else if (o == _drag)
            {
                emit moveApplet( _parent->mapFromGlobal( ev->globalPos() ) );
                return true;
            }
        }
        break;

        default:
            return QWidget::eventFilter(o, e);    // standard event processing
            break;
    }

    return false;
}

void AppletHandle::menuButtonPressed()
{
    if (!_menuButton->isOn())
    {
        _menuButton->setOn(true);
        emit showAppletMenu();
    }
}

bool AppletHandle::onMenuButton(const QPoint& point) const
{
    return (childAt(mapFromGlobal(point)) == _menuButton);
}

void AppletHandle::toggleMenuButtonOff()
{
    _menuButton->setOn(false);
    _menuButton->setDown(false);
}

QPixmap AppletHandle::xpmPixmap( const char* const xpm[], const char* _key )
{
   QString key = QString( "$kde_kicker_applethandle_" ) + _key;
   if( QPixmap* pm = QPixmapCache::find( key ))
       return *pm;
   QPixmap pm( const_cast< const char** >( xpm ));
   QPixmapCache::insert( key, pm );
   return pm;
}

AppletHandleDrag::AppletHandleDrag( AppletHandle* parent, const char* name)
    : QWidget( parent, name )
    , _parent( parent )
{
   setBackgroundOrigin( AncestorOrigin );
}

QSize AppletHandleDrag::minimumSizeHint() const
{
    int wh = style().pixelMetric(QStyle::PM_DockWindowHandleExtent, this);

    if (KickerSettings::transparent())
    {
        wh += 2;
    }

    if (orientation() == Horizontal)
    {
        return QSize(wh, 0);
    }

    return QSize(0, wh);
}

QSizePolicy AppletHandleDrag::sizePolicy() const
{
    if (orientation() == Horizontal)
    {
        return QSizePolicy( QSizePolicy::Fixed, QSizePolicy::Preferred );
    }

    return QSizePolicy( QSizePolicy::Preferred, QSizePolicy::Fixed );
}

void AppletHandleDrag::paintEvent(QPaintEvent *)
{
    QPainter p(this);

    QStyle::SFlags flags = QStyle::Style_Default;
    flags |= QStyle::Style_Enabled;
    if (orientation() == Horizontal)
    {
        flags |= QStyle::Style_Horizontal;
    }

    QRect r = rect();

    if (!KickerSettings::transparent())
    {
        if (orientation() == Horizontal)
        {
            if (kapp->reverseLayout())
            {
                // paint it on the right-hand side
                style().drawPrimitive(QStyle::PE_PanelDockWindow,
                                      &p, QRect(width() - 2, 0, 2, height()),
                                      colorGroup(), QStyle::Style_Horizontal);
                r.rRight() -= 2;
            }
            else
            {
                // paint it on the left-hand side
                style().drawPrimitive(QStyle::PE_PanelDockWindow,
                                      &p, QRect(0, 0, 2, height()),
                                      colorGroup(), QStyle::Style_Horizontal);
                r.rLeft() += 2;
            }
        }
        else
        {
            //vertical, paint it on top
            style().drawPrimitive(QStyle::PE_PanelDockWindow,
                                  &p, QRect(0, 0, width(), 2),
                                  colorGroup(), QStyle::Style_Horizontal);
            r.rTop() += 2;
        }
    }

    style().drawPrimitive(QStyle::PE_DockWindowHandle, &p, r,
                          colorGroup(), flags);
}

AppletHandleButton::AppletHandleButton( AppletHandle *parent, const char * name )
  : QPushButton( parent, name )
  , _parent( parent )
{
    setBackgroundOrigin( AncestorOrigin );
    moveMouse = false;
    setFocusPolicy( NoFocus );
    setToggleButton( true );
    setToggleType(QButton::Toggle);
}

AppletHandleButton::~AppletHandleButton()
{
}

QSize AppletHandleButton::minimumSizeHint() const
{
    int height = style().pixelMetric(QStyle::PM_DockWindowHandleExtent, this);
    int width = height;

    if (KickerSettings::transparent())
    {
        width += 2;
    }

    if (orientation() == Horizontal)
    {
        return QSize(width, height);
    }

    return QSize(height, width);
}

QSizePolicy AppletHandleButton::sizePolicy() const
{
    return QSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
}

void AppletHandleButton::drawButton( QPainter* p )
{
    QPixmap* bgPixmap = colorGroup().brush(QColorGroup::Background).pixmap();
    if (bgPixmap)
    {
        QPoint origin = backgroundOffset();
        p->drawTiledPixmap(0, 0, width(), height(), *bgPixmap, origin.x(), origin.y());
    }
    else
    {
        QBrush brush = colorGroup().brush(QColorGroup::Background);
        p->fillRect(rect(), brush);
    }

    if (!KickerSettings::transparent())
    {
        if (orientation() == Horizontal)
        {
            if (kapp->reverseLayout())
            {
                // paint it on the right-hand side
                style().drawPrimitive(QStyle::PE_PanelDockWindow,
                                      p, QRect(width() - 2, 0, 2, height()),
                                      colorGroup(), QStyle::Style_Horizontal);
            }
            else
            {
                style().drawPrimitive(QStyle::PE_PanelDockWindow,
                                      p, QRect(0, 0, 2, height()),
                                      colorGroup(), QStyle::Style_Horizontal);
            }
        }
        else
        {
            //vertical, paint it on top
            style().drawPrimitive(QStyle::PE_PanelDockWindow,
                                  p, QRect(0, 0, width(), 2),
                                  colorGroup(), QStyle::Style_Horizontal);
        }
    }

    int w = width();
    int h = height();
    if (orientation() == Horizontal)
    {
        if (!kapp->reverseLayout())
        {
            p->translate(2, 0);
        }
        w -= 2;
    }
    else
    {
        p->translate(0, 2);
        h -= 2;
    }

    p->drawPixmap((w - pixmap()->width()) / 2,
                  (h - pixmap()->height()) / 2,
                  *pixmap());

    --w;
    --h;

    if (moveMouse && !isDown())
    {
        p->setPen(white);
        p->moveTo(0, h);
        p->lineTo(0, 0);
        p->lineTo(w, 0);

        p->setPen(colorGroup().dark());
        p->lineTo(w, h);
        p->lineTo(0, h);
    }

    if (isOn() || isDown())
    {
        p->setPen(colorGroup().dark());
        p->moveTo(0, h);
        p->lineTo(0, 0);
        p->lineTo(w, 0);

        p->setPen(white);
        p->lineTo(w, h);
        p->lineTo(0, h);
    }
}

void AppletHandleButton::enterEvent( QEvent * )
{
  moveMouse = true;
  repaint();
}

void AppletHandleButton::leaveEvent( QEvent * )
{
  moveMouse = false;
  repaint();
}

#include "applethandle.moc"
