/* vi: set sw=4 ts=4 ai: */

// #define MODULE

#define READ_AFTER_WRITE

/**********************************************************************
*  linux/drivers/misc/eeprom-lh79x.c
*
*  Provide Microchip 93LC46B 64 x 16 EEPROM access for LH7x EVB boards
*
*  Copyright (C) 2002  Lineo, Inc.
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License (GPL) version 2
*  as published by the Free Software Foundation.
*
*  References:
*     SHARP_EVB_DISPLAY_BOARD_REV2.pdf
*     93LC46.pdf (Microchip 1K Microwire(R) EEPROM chip spec)
*
**********************************************************************/

#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/proc_fs.h>
#include <linux/smp_lock.h>
#include <linux/miscdevice.h>
#include <linux/poll.h>
#include <linux/delay.h>

#undef DEBUG
#undef VERBOSE
#define DRVNAME "eeprom-lh79x"
#include <linux/verbosedebug.h>

#include <linux/version.h>
#ifdef MODULE
char kernel_version[] = UTS_RELEASE;
#endif /* MODULE */

#include <asm/arch/hardware.h>

#include "ssp.h"

#define SIXmsJIFFIES	(((HZ*6)/1000)+2)

/**********************************************************************
* Define EEPROM Control macros for "Microchip 93LC46B Microwire Serial EEPROM"
**********************************************************************/

/* Erase one 16 bit word at addr */
#define EEPROM_ERASE(addr)	(0x01C0 | (addr & 0x3F))
/* Erase entire eeprom */
#define EEPROM_ERAL()		(0x0120)
/* Erase/Write disable  */
#define EEPROM_EWDS()		(0x0100)
/* Erase/Write enable */
#define EEPROM_EWEN()		(0x0130)
/* Read one 16 bit word at addr */
#define EEPROM_READ(addr)	(0x0180 | (addr & 0x3F))
/* (Erase and) Write one 16 bit word at addr */
#define EEPROM_WRITE(addr)	(0x0140 | (addr & 0x3F))
/* Write one 16 bit value throught entire eeprom */
#define EEPROM_WRAL()		(0x0110)

/**********************************************************************
* Define our eeprom context structure
**********************************************************************/

#define EEPROM_SIZE_16BIT	64
#define EEPROM_SIZE_8BIT	128

typedef struct eepromContext_t eepromContext_t;
struct eepromContext_t {
	union {
		uint16_t w[EEPROM_SIZE_16BIT];	/* Actual device size */
		u_char   c[EEPROM_SIZE_8BIT];
	} cache;
	union {
		uint16_t w[EEPROM_SIZE_16BIT];	/* Actual device size */
		u_char   c[EEPROM_SIZE_8BIT];
	} state;
	wait_queue_head_t read_and_write_wait;

	void          *sspContext;
	void          (*write) (void *sspContext, unsigned int data);
	unsigned int  (*read) (void *sspContext);
	int           (*lock)(void *sspContext, int device);
	int           (*unlock)(void *sspContext, int device);
	void          (*ssp_chipselect_automatic)(void);
	void          (*ssp_chipselect_manual)(void);
	void          (*ssp_chipselect_enable)(void);
	void          (*ssp_chipselect_disable)(void);
	void          (*ssp_flush_tx_fifo)(void *sspContext);
	void          (*ssp_flush_rx_fifo)(void *sspContext);
	void          (*ssp_busy_wait)(void);
};
static eepromContext_t eepromContext_l;

#define CACHE_STATE_VALID_8			0x01
#define CACHE_STATE_MODIFIED_8		0x02

#define CACHE_STATE_VALID_16		0x0101
#define CACHE_STATE_MODIFIED_16		0x0202

/**********************************************************************
* Function: eeprom_lh79x_erase_write_enable
* Function: eeprom_lh79x_erase_write_disable
**********************************************************************/
static void
eeprom_lh79x_erase_write_enable(eepromContext_t *eepromContext)
{
	void *sspContext = eepromContext->sspContext;

	/* Lock the SSP before accessing it */
	eepromContext->lock(sspContext, SSP_DEV_EEPROM);

	eepromContext->ssp_busy_wait(); /* Wait for the SSP to not be busy */
	eepromContext->ssp_flush_rx_fifo(sspContext);

	/* EEPROM Writes must be done using manual control of the ChipSelect */
	eepromContext->ssp_chipselect_manual();
	eepromContext->ssp_chipselect_enable();

	eepromContext->write(sspContext, EEPROM_EWEN());  /* Enable Erase/Write */

	eepromContext->ssp_busy_wait(); /* Wait for the SSP to not be busy */
	eepromContext->ssp_flush_rx_fifo(sspContext);

	/* Reset back to automatic control of the EEPROM ChipSelect */
	eepromContext->ssp_chipselect_disable();
	eepromContext->ssp_chipselect_automatic();

	/* Unlock the SSP after it has been locked */
	eepromContext->unlock(sspContext, SSP_DEV_EEPROM);

	return;
}

static void
eeprom_lh79x_erase_write_disable(eepromContext_t *eepromContext)
{
	void *sspContext = eepromContext->sspContext;

	/* Lock the SSP before accessing it */
	eepromContext->lock(sspContext, SSP_DEV_EEPROM);

	eepromContext->ssp_busy_wait(); /* Wait for the SSP to not be busy */
	eepromContext->ssp_flush_rx_fifo(sspContext);

	/* EEPROM Writes must be done using manual control of the ChipSelect */
	eepromContext->ssp_chipselect_manual();
	eepromContext->ssp_chipselect_enable();

	eepromContext->write(sspContext, EEPROM_EWDS());  /* Disable Erase/Write */

	eepromContext->ssp_busy_wait(); /* Wait for the SSP to not be busy */
	eepromContext->ssp_flush_rx_fifo(sspContext);

	/* Reset back to automatic control of the EEPROM ChipSelect */
	eepromContext->ssp_chipselect_disable();
	eepromContext->ssp_chipselect_automatic();

	/* Unlock the SSP after it has been locked */
	eepromContext->unlock(sspContext, SSP_DEV_EEPROM);

	return;
}

/**********************************************************************
* Function: eeprom_lh79x_read_device_word
**********************************************************************/
static void
eeprom_lh79x_read_device_word(eepromContext_t *eepromContext, int offset_w)
{
	void *sspContext = eepromContext->sspContext;
	uint16_t word = 0;

	/* Lock the SSP before accessing it */
	eepromContext->lock(sspContext, SSP_DEV_EEPROM);

	eepromContext->ssp_busy_wait(); /* Wait for the SSP to not be busy */
	eepromContext->ssp_flush_rx_fifo(sspContext);

	/* EEPROM Reads must be done using manual control of the ChipSelect */
	eepromContext->ssp_chipselect_manual();
	eepromContext->ssp_chipselect_enable();

	/* Read eeprom into cache */
	/* Note: We shift to take care of the "dummy 0" the eeprom sends */
	eepromContext->write(sspContext, (EEPROM_READ(offset_w))<<1);
	/* Following is a Dummy/Invalid command to allow the eeprom to be read */
	eepromContext->write(sspContext, 0);
	eepromContext->ssp_busy_wait(); /* Wait for the SSP to not be busy */
	word = eepromContext->read(sspContext);		/* Dummy Word */
	word = eepromContext->read(sspContext);		/* Real Word */

	/* Reset back to automatic control of the EEPROM ChipSelect */
	eepromContext->ssp_chipselect_disable();
	eepromContext->ssp_chipselect_automatic();

	/* Unlock the SSP after it has been locked */
	eepromContext->unlock(sspContext, SSP_DEV_EEPROM);

	/* Modify the state of the cache data */
	eepromContext->cache.w[offset_w] = word;
	eepromContext->state.w[offset_w] |= CACHE_STATE_VALID_16;

	schedule();		/* Give the rest of the system a chance to work */

	return;
}

/**********************************************************************
* Function: eeprom_lh79x_write_device_word
**********************************************************************/
static void
eeprom_lh79x_write_device_word(eepromContext_t *eepromContext, int offset_w)
{
	void *sspContext = eepromContext->sspContext;
	uint16_t word;
	long timeoutJiffies;

	/* Get the modified cache data to write to the eeprom */
	word = eepromContext->cache.w[offset_w];

	/* Lock the SSP before accessing it */
	eepromContext->lock(sspContext, SSP_DEV_EEPROM);

	eepromContext->ssp_busy_wait(); /* Wait for the SSP to not be busy */
	eepromContext->ssp_flush_rx_fifo(sspContext);

	/* EEPROM Writes must be done using manual control of the ChipSelect */
	eepromContext->ssp_chipselect_manual();
	eepromContext->ssp_chipselect_enable();

	/* Write modified cache data to the eeprom */
	eepromContext->write(sspContext, EEPROM_WRITE(offset_w));
	eepromContext->write(sspContext, word);

	/* Reset back to automatic control of the EEPROM ChipSelect */
	eepromContext->ssp_chipselect_disable();
	eepromContext->ssp_chipselect_automatic();

	eepromContext->ssp_busy_wait(); /* Wait for the SSP to not be busy */
	eepromContext->ssp_flush_rx_fifo(sspContext);

	/* Unlock the SSP after it has been locked */
	eepromContext->unlock(sspContext, SSP_DEV_EEPROM);

	/* Modify the state of the cache data */
	eepromContext->state.w[offset_w]  &= ~CACHE_STATE_MODIFIED_16;

#ifdef READ_AFTER_WRITE
	/* Force the modified cache data to be reread from the eeprom */
	eepromContext->state.w[offset_w]  &= ~CACHE_STATE_VALID_16;
#endif /* READ_AFTER_WRITE */

	/* Cause ~ 6ms delay (actually does 10 on the Sharp LH79520) */
	timeoutJiffies = SIXmsJIFFIES;
	while (timeoutJiffies > 0) {
		__set_current_state(TASK_UNINTERRUPTIBLE);
		timeoutJiffies = schedule_timeout(timeoutJiffies);
	}
	// __set_current_state(TASK_RUNNING);

	schedule();		/* Give the rest of the system a chance to work */

	return;
}

/**********************************************************************
* Function: eeprom_lh79x_read_device
**********************************************************************/
static void
eeprom_lh79x_read_device(eepromContext_t *eepromContext)
{
	uint16_t *scwp;
	int offset_w;

	scwp = eepromContext->state.w;
	for (offset_w = 0; offset_w < EEPROM_SIZE_16BIT; offset_w++) {
		if ((*scwp & CACHE_STATE_VALID_16) != CACHE_STATE_VALID_16) {
			eeprom_lh79x_read_device_word(eepromContext, offset_w);
		}
		scwp++;
	}

	return;
}

/**********************************************************************
* Function: eeprom_lh79x_write_device
**********************************************************************/
static void
eeprom_lh79x_write_device(eepromContext_t *eepromContext)
{
	uint16_t *scwp;
	int offset_w;
	int timeoutJiffies;

	/* Enable erase/write of the eeprom */
	eeprom_lh79x_erase_write_enable(eepromContext);

	/* Just to get the timeouts in a desirable sequence */
	/* and so we don't get a "partial" timeout we do the following... */
	/* Cause ~ 6ms delay (actually does 10 on the Sharp LH79520) */
	timeoutJiffies = SIXmsJIFFIES;
	while (timeoutJiffies > 0) {
		__set_current_state(TASK_UNINTERRUPTIBLE);
		timeoutJiffies = schedule_timeout(timeoutJiffies);
	}
	// __set_current_state(TASK_RUNNING);

	scwp = eepromContext->state.w;
	for (offset_w = 0; offset_w < EEPROM_SIZE_16BIT; offset_w++) {
		if (*scwp & CACHE_STATE_MODIFIED_16) {
			eeprom_lh79x_write_device_word(eepromContext, offset_w);
		}
		scwp++;
	}

	/* Disable erase/write of the eeprom */
	eeprom_lh79x_erase_write_disable(eepromContext);

#ifdef READ_AFTER_WRITE
	/*
	* NOW ... Update the eeprom cache.
	* ( Read the actual contents of the eeprom that were
	* modified instead of relying on what we wrote. )
	*/
	eeprom_lh79x_read_device(eepromContext);
#endif /* READ_AFTER_WRITE */

	return;
}

/**********************************************************************
* *****************************************************
* *** User space "file operation" driver interfaces ***
* *****************************************************
* Function: lh79x_eeprom_llseek
* Function: lh79x_eeprom_read
* Function: lh79x_eeprom_write
* Function: lh79x_eeprom_poll				( NOT USED --- YET )
* Function: lh79x_eeprom_ioctl				( NOT USED --- YET )
* Function: lh79x_eeprom_open
* Function: lh79x_eeprom_flush				( NOT USED --- YET )
* Function: lh79x_eeprom_release			( NOT USED --- YET )
* Function: lh79x_eeprom_fsync				( NOT USED --- YET )
* Function: lh79x_eeprom_fasync				( NOT USED --- YET )
* Function: lh79x_eeprom_lock				( NOT USED --- YET )
**********************************************************************/

static loff_t
lh79x_eeprom_llseek(struct file *filp, loff_t offset, int origin)
{
	// eepromContext_t *eepromContext = filp->private_data;
	loff_t new_offset = -EINVAL;

	switch (origin) {
		case 0:		/* SEEK_SET == 0, Offset from the start */
			new_offset = offset;
			break;
		case 1:		/* SEEK_CUR == 1, Offset from the current position */
			new_offset = filp->f_pos + offset;
			break;
		case 2:		/* SEEK_END == 2, Offset from the end */
			new_offset = EEPROM_SIZE_8BIT - offset;
			break;
	}
	if ((new_offset < 0) || (new_offset > EEPROM_SIZE_8BIT)) {
		new_offset = -EINVAL;
	} else {
		filp->f_pos = new_offset;
	}

	return(new_offset);
}

static ssize_t
lh79x_eeprom_read(struct file *filp, char *buffer,
		size_t count, loff_t *offsetp)
{
	eepromContext_t *eepromContext = filp->private_data;
	int sts = 0;
	int err;

	vdprintk("ENTER: lh79x_eeprom_read(%d:%d)\n", *offsetp, count);
	/* Ensure we still have data to read (relative to the offset we are at) */
	if (*offsetp < EEPROM_SIZE_8BIT) {
		/* Adjust the size to be read if necessary */
		if ((*offsetp + count) > EEPROM_SIZE_8BIT) {
			count = EEPROM_SIZE_8BIT - *offsetp;
		}
		/* Ensure the eeprom cache is valid */
		eeprom_lh79x_read_device(eepromContext);
		/* Return the contents of the eeprom cache */
		err = copy_to_user(buffer, &eepromContext->cache.c[*offsetp], count);
		if ( ! err) {
			*offsetp += count;
			sts = count;
		} else {
			sts = -EFAULT;
		}
	}
	vdprintk("LEAVE: lh79x_eeprom_read(%d)\n", sts);

	return(sts);
}

static ssize_t
lh79x_eeprom_write(struct file *filp, const char *buffer,
		size_t count, loff_t *offsetp)
{
	eepromContext_t *eepromContext = filp->private_data;
	u_char newcache_c[EEPROM_SIZE_8BIT];
	int sts = 0;
	int err;

	vdprintk("ENTER: lh79x_eeprom_write(%d:%d)\n", *offsetp, count);
	/* Ensure we still have room to write (relative to the offset we are at) */
	if (*offsetp < EEPROM_SIZE_8BIT) {
		/* Adjust the size to be written if necessary */
		if ((*offsetp + count) > EEPROM_SIZE_8BIT) {
			count = EEPROM_SIZE_8BIT - *offsetp;
		}
		/* Ensure the eeprom cache is valid to start with */
		eeprom_lh79x_read_device(eepromContext);
		/* Get the new contents of the eeprom cache */
		err = copy_from_user(&newcache_c[*offsetp], buffer, count);
		if ( ! err) {
			u_char *ccp, *nccp, *sccp;
			int i;
			/*
			* Transfer the new cache contents into the cache
			* marking what has changed.
			*/
			ccp = &eepromContext->cache.c[*offsetp];
			nccp = &newcache_c[*offsetp];
			sccp = &eepromContext->state.c[*offsetp];
			for (i = 0; i < count; i++) {
				if (*ccp != *nccp) {
					*ccp = *nccp;
					*sccp |= CACHE_STATE_MODIFIED_8;
				}
				ccp++;
				nccp++;
				sccp++;
			}
			/* Write the modified cache into the eeprom */
			eeprom_lh79x_write_device(eepromContext);
			*offsetp += count;
			sts = count;
		} else {
			sts = -EFAULT;
		}
	}
	vdprintk("LEAVE: lh79x_eeprom_write(%d)\n", sts);

	return(sts);
}

#if (0) /* NOT USED --- YET */
static unsigned int
lh79x_eeprom_poll(struct file *filp, struct poll_table_struct *wait)
{
	eepromContext_t *eepromContext = filp->private_data;
	int sts = 0;

	return(sts);
}
#endif /* NOT USED --- YET */

#if (0) /* NOT USED --- YET */
static int
lh79x_eeprom_ioctl(struct inode *, struct file *filp, unsigned int cmd, unsigned long arg)
{
	eepromContext_t *eepromContext = filp->private_data;
	int sts = 0;

	return(sts);
}
#endif /* NOT USED --- YET */

static int
lh79x_eeprom_open(struct inode *inode, struct file *filp)
{
	eepromContext_t *eepromContext = &eepromContext_l;
	int sts = 0;

	filp->private_data = eepromContext;

	return(sts);
}

#if (0) /* NOT USED --- YET */
static int
lh79x_eeprom_flush(struct file *filp)
{
	eepromContext_t *eepromContext = filp->private_data;
	int sts = 0;

	return(sts);
}
#endif /* NOT USED --- YET */

#if (0) /* NOT USED --- YET */
static int
lh79x_eeprom_release(struct inode *inode, struct file *filp)
{
	eepromContext_t *eepromContext = filp->private_data;
	int sts = 0;

	return(sts);
}
#endif /* NOT USED --- YET */

#if (0) /* NOT USED --- YET */
static int
lh79x_eeprom_fsync(struct file *filp, struct dentry *dentry, int datasync)
{
	eepromContext_t *eepromContext = filp->private_data;
	int sts = 0;

	return(sts);
}
#endif /* NOT USED --- YET */

#if (0) /* NOT USED --- YET */
static int
lh79x_eeprom_fasync(int fd, struct file *filp, int on)
{
	eepromContext_t *eepromContext = filp->private_data;
	int sts = 0;

	return(sts);
}
#endif /* NOT USED --- YET */

#if (0) /* NOT USED --- YET */
static int
lh79x_eeprom_lock(struct file *filp, int XXX, struct file_lock *file_lock)
{
	eepromContext_t *eepromContext = filp->private_data;
	int sts = 0;

	return(sts);
}
#endif /* NOT USED --- YET */

/**********************************************************************
* Define (fill in) the user space file operations for this driver
* and initialize the eeprom driver as a "miscdevice":
*       Character device
*       Major(10) --- Non-serial mice, misc features
*       Minor(22) --- /dev/eeprom		( Microchip 93LC46B 64 x 16 EEPROM )
**********************************************************************/
static struct file_operations lh79x_eeprom_fops = {
	owner:      		THIS_MODULE,
	llseek:				lh79x_eeprom_llseek,
	read:				lh79x_eeprom_read,
	write:				lh79x_eeprom_write,
//	poll:				lh79x_eeprom_poll,
//	ioctl:				lh79x_eeprom_ioctl,
	open:				lh79x_eeprom_open,
//	flush:				lh79x_eeprom_flush,
//	release:			lh79x_eeprom_release,
//	fsync:				lh79x_eeprom_fsync,
//	fasync:				lh79x_eeprom_fasync,
//	lock:				lh79x_eeprom_lock,
};

static struct miscdevice lh79x_eeprom_dev = {
minor: 22,
name: "eeprom",
fops: &lh79x_eeprom_fops,
};

/**********************************************************************
* Function: lh79x_eeprom_make_ssp_association
*
* Purpose:
*	Make the association between the eeprom driver and the ssp driver
**********************************************************************/
static int lh79x_eeprom_make_ssp_association(eepromContext_t *eepromContext)
{
	int sts = 0;
	void *vp;

/* NOTE: -EOPNOTSUPP == Operation not supported on transport endpoint */
#define ASSOCIATION_ERROR   -EOPNOTSUPP

	dprintk("ENTER: lh79x_eeprom_make_ssp_association()\n");

	vp = ssp_request_pointer(SSP_DEV_EEPROM, "sspContext");
	if ( ! vp ) 
		sts = ASSOCIATION_ERROR;
	eepromContext->sspContext = vp;

	vp = ssp_request_pointer(SSP_DEV_EEPROM, "write");
	if ( ! vp ) 
		sts = ASSOCIATION_ERROR;
	eepromContext->write = vp;

	vp = ssp_request_pointer(SSP_DEV_EEPROM, "read");
	if ( ! vp ) 
		sts = ASSOCIATION_ERROR;
	eepromContext->read = vp;

	vp = ssp_request_pointer(SSP_DEV_EEPROM, "lock");
	if ( ! vp ) 
		sts = ASSOCIATION_ERROR;
	eepromContext->lock = vp;

	vp = ssp_request_pointer(SSP_DEV_EEPROM, "unlock");
	if ( ! vp ) 
		sts = ASSOCIATION_ERROR;
	eepromContext->unlock = vp;

	vp = ssp_request_pointer(SSP_DEV_EEPROM, "chipselect_enable");
	if ( ! vp ) 
		sts = ASSOCIATION_ERROR;
	eepromContext->ssp_chipselect_enable = vp;

	vp = ssp_request_pointer(SSP_DEV_EEPROM, "chipselect_disable");
	if ( ! vp ) 
		sts = ASSOCIATION_ERROR;
	eepromContext->ssp_chipselect_disable = vp;

	vp = ssp_request_pointer(SSP_DEV_EEPROM, "chipselect_manual");
	if ( ! vp ) 
		sts = ASSOCIATION_ERROR;
	eepromContext->ssp_chipselect_manual = vp;

	vp = ssp_request_pointer(SSP_DEV_EEPROM, "chipselect_automatic");
	if ( ! vp ) 
		sts = ASSOCIATION_ERROR;
	eepromContext->ssp_chipselect_automatic = vp;

	vp = ssp_request_pointer(SSP_DEV_EEPROM, "flush_tx_fifo");
	if ( ! vp ) 
		sts = ASSOCIATION_ERROR;
	eepromContext->ssp_flush_tx_fifo = vp;

	vp = ssp_request_pointer(SSP_DEV_EEPROM, "flush_rx_fifo");
	if ( ! vp ) 
		sts = ASSOCIATION_ERROR;
	eepromContext->ssp_flush_rx_fifo = vp;

	vp = ssp_request_pointer(SSP_DEV_EEPROM, "ssp_busy_wait");
	if ( ! vp ) 
		sts = ASSOCIATION_ERROR;
	eepromContext->ssp_busy_wait = vp;

	dprintk("LEAVE: lh79x_eeprom_make_ssp_association(%d)\n", sts);

	return(sts);
}

/**********************************************************************
* Function: lh79x_eeprom_init
*
* Purpose:
*	Register & Initialize the module
**********************************************************************/
static int lh79x_eeprom_init(void)
{
	eepromContext_t *eepromContext = &eepromContext_l;
	int sts = 0;
	dprintk("ENTER: lh79x_eeprom_init()\n");
	init_waitqueue_head(&eepromContext->read_and_write_wait);
	/* Retrieve the service information from the SSP driver */
	sts = lh79x_eeprom_make_ssp_association(eepromContext);
	if (sts == 0) {
		/* Ensure the eeprom cache is valid */
		eeprom_lh79x_read_device(eepromContext);
		sts = misc_register(&lh79x_eeprom_dev);
	}
	dprintk("LEAVE: lh79x_eeprom_init(%d)\n", sts);
	return(sts);
}

/**********************************************************************
* Function: lh79x_eeprom_exit
*
* Purpose:
*	Un-Register & Cleanup the module
**********************************************************************/
static void lh79x_eeprom_exit(void)
{
	dprintk("ENTER: lh79x_eeprom_exit()\n");
	misc_deregister(&lh79x_eeprom_dev);
	dprintk("LEAVE: lh79x_eeprom_exit()\n");
	return;
}

module_init(lh79x_eeprom_init);
module_exit(lh79x_eeprom_exit);

MODULE_AUTHOR("Jim Gleason / Lineo, Inc.");
MODULE_DESCRIPTION("Microchip 93LC46B 64 x 16 EEPROM access for LH7x EVB");
MODULE_LICENSE("Copyright (c) 2002 Lineo, Inc.");

