/* vi: set sw=4 ts=4 ai: */

//#define MODULE

/**********************************************************************
*  linux/drivers/misc/lh7x-7seg.c
*
*  Provide ADS_784x 7-Segment access for LH7x EVB boards
*
*  Copyright (C) 2002  Lineo, Inc.
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License (GPL) version 2
*  as published by the Free Software Foundation.
*
**********************************************************************/

/**********************************************************************
* To light up the 7-segment display, write a 16-bit value to
*
*	cpld->seven_seg.
*
* The high-order byte is the most significant 7-segment digit,
* and the low-order byte is the lsb.
*
* NOTE: The 7-segment display bars are bit-mapped.
* NOTE: The 7-segment display bars are ACTIVE LOW.
*
*     _   ==   a
*    | |  ==  f b
*     -   ==   g
*    | |  ==  e c
*     -.  ==   d dot
*
*    a    0x01
*    b    0x02
*    c    0x04
*    d    0x08
*    e    0x10
*    f    0x20
*    g    0x40
*    dot  0x80		(also known as dp)
*
* The data to write looks like this:
*
*	static u_char lednum[] =
*	{ 0xC0, 0xF9, 0xA4, 0xB0, 0x99, 0x92, 0x82, 0xF8,  // 0-7
*	  0x80, 0x98, 0x88, 0x83, 0xC6, 0xA1, 0x86, 0x8E,  // 8-F
*	  0xBF, // hyphen
*	  0xFF, // (blank)
*	};
*
* SO - to make "7F" show up, do this:
*
*	cpld->seven_seg = 0xf88e;
*
* NOTE: When read, the 7-segment display does not return valid data.
*
**********************************************************************/

#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/proc_fs.h>
#include <linux/smp_lock.h>
#include <linux/miscdevice.h>
#include <linux/poll.h>

#undef DEBUG
#undef VERBOSE
#define DRVNAME "lh7x-7seg"
#include <linux/verbosedebug.h>

#include <linux/version.h>
#ifdef MODULE
char kernel_version[] = UTS_RELEASE;
#endif /* MODULE */

#include <asm/arch/hardware.h>
#include <asm/arch/cpld.h>
#include <asm/arch/lh7x-7seg.h>

static cpldRegs_t *cpld = (cpldRegs_t *)CPLD_BASE;

/**********************************************************************
* Define our Seven Segment context structure
**********************************************************************/
typedef struct sevenSegmentContext_t sevenSegmentContext_t;
struct sevenSegmentContext_t {
	struct fasync_struct *fasync;
	wait_queue_head_t read_and_write_wait;

	int inEscape;		/* 1 == We are in an escape sequence, 0 == NOT */
	int accessMode;		/* See ACCESSMODE_xxx definitions below */
	int rawData;		/* Raw == 1, Cooked == 0 */
	uint16_t			currentRawVal;
};
static sevenSegmentContext_t sevenSegmentContext_l;

#define ESCAPE		27

#define ACCESSMODE_SHIFT			0
#define ACCESSMODE_LSB			1
#define ACCESSMODE_MSB			2
#define ACCESSMODE_DAFAULT		ACCESSMODE_SHIFT

/**********************************************************************
* Define our Seven Segment Data
**********************************************************************/

typedef struct sevenSegmentData_t sevenSegmentData_t;
struct sevenSegmentData_t {
	int val;
	u_char raw_val;
};

#define SSD_BLANK	((u_char)~(0x00))

static sevenSegmentData_t sevenSegmentData[] = {
	{'0',  (u_char)~(SSD_A | SSD_B | SSD_C | SSD_D | SSD_E | SSD_F) },
	{'1',  (u_char)~(SSD_B | SSD_C) },
	{'2',  (u_char)~(SSD_A | SSD_B | SSD_G | SSD_E | SSD_D) },
	{'3',  (u_char)~(SSD_A | SSD_B | SSD_G | SSD_C | SSD_D) },
	{'4',  (u_char)~(SSD_F | SSD_G | SSD_B | SSD_C) },
	{'5',  (u_char)~(SSD_A | SSD_F | SSD_G | SSD_C | SSD_D) },
	{'6',  (u_char)~(SSD_A | SSD_F | SSD_E | SSD_D | SSD_C | SSD_G) },
	{'7',  (u_char)~(SSD_A | SSD_B | SSD_C) },
	{'8',  (u_char)~(SSD_A | SSD_B | SSD_C | SSD_D | SSD_E | SSD_F | SSD_G) },
	{'9',  (u_char)~(SSD_G | SSD_F | SSD_A | SSD_B | SSD_C) },
	{'A',  (u_char)~(SSD_E | SSD_F | SSD_A | SSD_B | SSD_C | SSD_G) },
	{'a',  (u_char)~(SSD_E | SSD_F | SSD_A | SSD_B | SSD_C | SSD_G) },
	{'B',  (u_char)~(SSD_F | SSD_E | SSD_D | SSD_C | SSD_G) },
	{'b',  (u_char)~(SSD_F | SSD_E | SSD_D | SSD_C | SSD_G) },
	{'C',  (u_char)~(SSD_A | SSD_F | SSD_E | SSD_D) },
	{'c',  (u_char)~(SSD_A | SSD_F | SSD_E | SSD_D) },
	{'D',  (u_char)~(SSD_B | SSD_C | SSD_D | SSD_E | SSD_G) },
	{'d',  (u_char)~(SSD_B | SSD_C | SSD_D | SSD_E | SSD_G) },
	{'E',  (u_char)~(SSD_A | SSD_F | SSD_G | SSD_E | SSD_D) },
	{'e',  (u_char)~(SSD_A | SSD_F | SSD_G | SSD_E | SSD_D) },
	{'F',  (u_char)~(SSD_A | SSD_F | SSD_G | SSD_E) },
	{'f',  (u_char)~(SSD_A | SSD_F | SSD_G | SSD_E) },
	{'H',  (u_char)~(SSD_F | SSD_E | SSD_G | SSD_B | SSD_C) },
	{'h',  (u_char)~(SSD_F | SSD_E | SSD_G | SSD_B | SSD_C) },
	{'I',  (u_char)~(SSD_F | SSD_E) },
	{'i',  (u_char)~(SSD_F | SSD_E) },
	{'Y',  (u_char)~(SSD_F | SSD_G | SSD_B | SSD_C | SSD_D) },
	{'y',  (u_char)~(SSD_F | SSD_G | SSD_B | SSD_C | SSD_D) },
	{'-',  (u_char)~(SSD_G) },
	{'_',  (u_char)~(SSD_D) },
	{'.',  (u_char)~(SSD_DOT) },
	{' ',  SSD_BLANK },
	{0x00, SSD_BLANK },
	{  -1, (u_char)~(0x00) }		/* End Of Data --- Must Be Last */
};

/**********************************************************************
* Function: val_to_raw_val
**********************************************************************/
static u_char val_to_raw_val(u_char val)
{
	sevenSegmentData_t *data;
	u_char raw_val = 0xFF;	/* Assume a blank if not found */
	for (data = sevenSegmentData; data->val != -1; data++) {
		if (val == data->val) {
			raw_val = data->raw_val;
			break;
		}
	}
	return(raw_val);
}

/**********************************************************************
* Function: raw_val_to_val
**********************************************************************/
static u_char raw_val_to_val(u_char raw_val)
{
	sevenSegmentData_t *data;
	u_char val = ' ';	/* Assume a blank if not found */
	for (data = sevenSegmentData; data->val != -1; data++) {
		if (raw_val == data->raw_val) {
			val = data->val;
			break;
		}
	}
	return(val);
}

/**********************************************************************
* Function: lh79x_7seg_read_raw_display
* Function: lh79x_7seg_read_raw_display_lsb
* Function: lh79x_7seg_read_raw_display_msb
**********************************************************************/
uint16_t lh79x_7seg_read_raw_display(void)
{
	sevenSegmentContext_t *sevenSegmentContext = &sevenSegmentContext_l;
	uint16_t raw_val;

	/*
	* NOTE: The device does not read so we have to remember...
	*/
	raw_val = sevenSegmentContext->currentRawVal;
	vdprintk("lh79x_7seg_read_raw_display(0x%04X)\n", raw_val);

	return(raw_val);
}

u_char lh79x_7seg_read_raw_display_lsb(void)
{
	uint16_t raw_val;
	u_char raw_lsb;

	raw_val = lh79x_7seg_read_raw_display();
	raw_lsb = (u_char)(raw_val & 0xFF);

	return(raw_lsb);
}

u_char lh79x_7seg_read_raw_display_msb(void)
{
	uint16_t raw_val;
	u_char raw_msb;

	raw_val = lh79x_7seg_read_raw_display();
	raw_msb = (u_char)((raw_val >> 8) & 0xFF);

	return(raw_msb);
}

/**********************************************************************
* Function: lh79x_7seg_read_display
* Function: lh79x_7seg_read_display_lsb
* Function: lh79x_7seg_read_display_msb
**********************************************************************/
uint16_t lh79x_7seg_read_display(void)
{
	uint16_t raw_val, val;
	u_char raw_lsb, lsb;
	u_char raw_msb, msb;

	raw_val = lh79x_7seg_read_raw_display();
	raw_lsb = (u_char)( raw_val       & 0xFF);
	raw_msb = (u_char)((raw_val >> 8) & 0xFF);
	lsb = raw_val_to_val(raw_lsb);
	msb = raw_val_to_val(raw_msb);
	val = (uint16_t)((msb << 8) | lsb);

	return(val);
}

u_char lh79x_7seg_read_display_lsb(void)
{
	u_char raw_lsb, lsb;

	raw_lsb = lh79x_7seg_read_raw_display_lsb();
	lsb = raw_val_to_val(raw_lsb);

	return(lsb);
}

u_char lh79x_7seg_read_display_msb(void)
{
	u_char raw_msb, msb;

	raw_msb = lh79x_7seg_read_raw_display_msb();
	msb = raw_val_to_val(raw_msb);

	return(msb);
}

/**********************************************************************
* Function: lh79x_7seg_write_raw_display
* Function: lh79x_7seg_write_raw_display_lsb
* Function: lh79x_7seg_write_raw_display_msb
**********************************************************************/
void lh79x_7seg_write_raw_display(uint16_t raw_val)
{
	sevenSegmentContext_t *sevenSegmentContext = &sevenSegmentContext_l;

	vdprintk("lh79x_7seg_write_raw_display(0x%04X)\n", raw_val);
	/*
	* NOTE: The device does not read so we have to remember...
	*/
	sevenSegmentContext->currentRawVal = raw_val;
	cpld->seven_seg = raw_val;

	return;
}

void lh79x_7seg_write_raw_display_lsb(u_char raw_lsb)
{
	uint16_t raw_val;

	raw_val = lh79x_7seg_read_raw_display();
	raw_val &= 0xFF00;
	raw_val &= ((uint16_t)raw_lsb) & 0x00FF;
	lh79x_7seg_write_raw_display(raw_val);

	return;
}

void lh79x_7seg_write_raw_display_msb(u_char raw_msb)
{
	uint16_t raw_val;

	raw_val = lh79x_7seg_read_raw_display();
	raw_val &= 0x00FF;
	raw_val &= (((uint16_t)raw_msb) << 8) & 0xFF00;
	lh79x_7seg_write_raw_display(raw_val);

	return;
}

/**********************************************************************
* Function: lh79x_7seg_write_display
* Function: lh79x_7seg_write_display_lsb
* Function: lh79x_7seg_write_display_msb
* Function: lh79x_7seg_write_display_str
**********************************************************************/
void lh79x_7seg_write_display(uint16_t val)
{
	u_char raw_lsb, lsb;
	u_char raw_msb, msb;
	uint16_t raw_val;

	lsb = (u_char)( val       & 0xFF);
	msb = (u_char)((val >> 8) & 0xFF);
	raw_lsb = val_to_raw_val(lsb);
	raw_msb = val_to_raw_val(msb);
	raw_val = (uint16_t)((raw_msb << 8) | raw_lsb);
	lh79x_7seg_write_raw_display(raw_val);

	return;
}

void lh79x_7seg_write_display_lsb(u_char lsb)
{
	u_char raw_lsb;

	raw_lsb = val_to_raw_val(lsb);
	lh79x_7seg_write_raw_display_lsb(raw_lsb);

	return;
}

void lh79x_7seg_write_display_msb(u_char msb)
{
	u_char raw_msb;

	raw_msb = val_to_raw_val(msb);
	lh79x_7seg_write_raw_display_msb(raw_msb);

	return;
}

void lh79x_7seg_write_display_str(u_char *str)
{
	uint16_t val;
	uint16_t c16;
	u_char c;

	if (str) {
		/* This is basically a read, shift, write lsb loop */
		for ( ; *str; str++) {
			c = *str;
			if (c == '\n') continue;
			if (c == '\r') continue;
			val = lh79x_7seg_read_display();
			val <<= 8;
			val &= 0xFF00;
			c16 = c & 0x00FF;
			val |= c16;
			lh79x_7seg_write_display(val);
		}
	}
	return;
}

/**********************************************************************
* Function: sevenSegment_read
* Function: sevenSegment_poll
* Function: sevenSegment_open
* Function: sevenSegment_fasync
* Function: sevenSegment_release
* ************************************
* *** User space driver interfaces ***
* ************************************
**********************************************************************/

/*
* NOTE: The read algorithm is wide open for interpretation.
*       I am not sure what the behaviour ought to be here.
*/
static ssize_t sevenSegment_read(struct file *filp, char *buffer,
		size_t _count, loff_t *ppos)
{
	sevenSegmentContext_t *sevenSegmentContext = filp->private_data;
	DECLARE_WAITQUEUE(wait, current);
	ssize_t sizeRead = 0;
	char *ptr = buffer;
	int err = 0;
	int count = (int)_count;
	uint16_t c16;
	u_char c;
	int dataSize;
	int dataByteCount;

	switch (sevenSegmentContext->accessMode) {
		case ACCESSMODE_LSB:
		case ACCESSMODE_MSB:
			dataSize = 8;
			break;
		case ACCESSMODE_SHIFT:
		default:
			dataSize = 16;
			break;
	}
	dataByteCount = dataSize / 8;

	add_wait_queue(&sevenSegmentContext->read_and_write_wait, &wait);
	while (count >= dataByteCount) {
		err = -ERESTARTSYS;
		if (signal_pending(current))
			break;
		/* NOTE: We always have data */
		switch (sevenSegmentContext->accessMode) {
			case ACCESSMODE_LSB:
				c = lh79x_7seg_read_display_lsb();
				break;
			case ACCESSMODE_MSB:
				c = lh79x_7seg_read_display_msb();
				break;
			case ACCESSMODE_SHIFT:
			default:
				c16 = lh79x_7seg_read_display();
				/* Flip the bytes so they get returned in the correct order */
				c = (u_char)(c16 >> 8);
				c16 = ((c16 << 8) & 0xFF00) | ((uint16_t)c);
				break;
		}
		if (dataSize == 8) {
			err = copy_to_user(ptr, &c, sizeof(c));
			if (err)
				break;
			ptr += sizeof(c);
			count -= sizeof(c);	
		} else /* (dataSize == 16) */ {
			err = copy_to_user(ptr, &c16, sizeof(c16));
			if (err)
				break;
			ptr += sizeof(c16);
			count -= sizeof(c16);	
		}
	}
	remove_wait_queue(&sevenSegmentContext->read_and_write_wait, &wait);
	sizeRead = (ptr == buffer ? err : ptr - buffer);
	return(sizeRead);
}

static ssize_t sevenSegment_write(struct file *filp, const char *buffer,
		size_t _count, loff_t *ppos)
{
	sevenSegmentContext_t *sevenSegmentContext = filp->private_data;
	ssize_t sizeWritten = 0;
	const char *ptr = buffer;
	int count;
	u_char c;
	uint16_t c16;
	uint16_t val;
	int err = 0;

	for (count = (int)_count; count > 0; count--) {
		/* NOTE: We always have room for the data */
		get_user(c, ptr++);
		/* Ignore new_line or carriage_return characters */
		if (c == '\n') continue;
		if (c == '\r') continue;
		vdprintk("JMG: (e:%d, r:%d, a:%d) Attempting to write (%c) == (0x%02X)\n",
				sevenSegmentContext->inEscape,
				sevenSegmentContext->rawData,
				sevenSegmentContext->accessMode,
				c, c);
		if (sevenSegmentContext->inEscape) {
			sevenSegmentContext->inEscape = 0;
			if ( c != ESCAPE ) {
				switch (c) {
					case 'r':	/* Raw Data */
					case 'R':	/* Raw Data */
						sevenSegmentContext->rawData = 1;
						break;
					case 'c':	/* Coooked Data */
					case 'C':	/* Coooked Data */
						sevenSegmentContext->rawData = 0;
						break;
					case 'l':	/* LSB */
					case 'L':	/* LSB */
						sevenSegmentContext->accessMode = ACCESSMODE_LSB;
						break;
					case 'm':	/* MSB */
					case 'M':	/* MSB */
						sevenSegmentContext->accessMode = ACCESSMODE_MSB;
						break;
					case 's':	/* Shift */
					case 'S':	/* Shift */
					case 'n':	/* Normal */
					case 'N':	/* Normal */
					default:
						sevenSegmentContext->accessMode = ACCESSMODE_SHIFT;
				}
				continue;
			}
		} else if ( c == ESCAPE ) {
			sevenSegmentContext->inEscape = 1;
			continue;
		}
		if (sevenSegmentContext->rawData) {
			val = lh79x_7seg_read_raw_display();
		} else {
			val = lh79x_7seg_read_display();
		}
		switch (sevenSegmentContext->accessMode) {
			case ACCESSMODE_LSB:
				val &= 0xFF00;
				c16 = c & 0x00FF;
				val |= c16;
				break;
			case ACCESSMODE_MSB:
				val &= 0x00FF;
				c16 = ((uint16_t)c << 8) & 0xFF00;
				val |= c16;
				break;
			case ACCESSMODE_SHIFT:
			default:
				val <<= 8;
				val &= 0xFF00;
				c16 = c & 0x00FF;
				val |= c16;
				break;
		}
		vdprintk("JMG: Writing (0x%04X)\n", val);
		if (sevenSegmentContext->rawData) {
			lh79x_7seg_write_raw_display(val);
		} else {
			lh79x_7seg_write_display(val);
		}
	}
	filp->f_dentry->d_inode->i_mtime = CURRENT_TIME;
	sizeWritten = (ptr == buffer ? err : ptr - buffer);
	return(sizeWritten);
}

static unsigned int sevenSegment_poll(struct file *filp, poll_table *wait)
{
	sevenSegmentContext_t *sevenSegmentContext = filp->private_data;
	/* We ALWAYS have data waiting ;) */
	int sts = POLLIN | POLLRDNORM;
	poll_wait(filp, &sevenSegmentContext->read_and_write_wait, wait);
	return(sts);
}

static int sevenSegment_open(struct inode *inode, struct file *filp)
{
	sevenSegmentContext_t *sevenSegmentContext = &sevenSegmentContext_l;
	int sts = 0;
	filp->private_data = sevenSegmentContext;
	return(sts);
}

static int sevenSegment_fasync(int fd, struct file *filp, int on)
{
	sevenSegmentContext_t *sevenSegmentContext = filp->private_data;
	int sts;
	sts = fasync_helper(fd, filp, on, &sevenSegmentContext->fasync);
	return(sts);
}

static int sevenSegment_release(struct inode *inode, struct file *filp)
{
	lock_kernel();
	sevenSegment_fasync(-1, filp, 0);
	unlock_kernel();
	return(0);
}

/**********************************************************************
* Define (fill in) the user space file operations for this driver
* and initialize the Seven Segment driver as a "miscdevice":
*       Character device
*       Major(10) --- Non-serial mice, misc features
*       Minor(21) --- /dev/7seg		(7-segment display)
**********************************************************************/
static struct file_operations sevenSegment_fops = {
owner:      THIS_MODULE,
read:       sevenSegment_read,
write:      sevenSegment_write,
poll:       sevenSegment_poll,
open:       sevenSegment_open,
fasync:     sevenSegment_fasync,
release:    sevenSegment_release,
};

static struct miscdevice sevenSegment_dev = {
minor: 21,
name: "7seg",
fops: &sevenSegment_fops,
};

/**********************************************************************
* Function: lh79x_7seg_init
*
* Purpose:
*	Register & Initialize the module
**********************************************************************/
static int lh79x_7seg_init(void)
{
	sevenSegmentContext_t *sevenSegmentContext = &sevenSegmentContext_l;
	int sts = 0;
	dprintk("ENTER: lh79x_7seg_init()\n");
	init_waitqueue_head(&sevenSegmentContext->read_and_write_wait);
	sts = misc_register(&sevenSegment_dev);
	lh79x_7seg_write_display_str((u_char *)"HI");
	dprintk("LEAVE: lh79x_7seg_init(%d)\n", sts);
	return(sts);
}

/**********************************************************************
* Function: lh79x_7seg_exit
*
* Purpose:
*	Un-Register & Cleanup the module
**********************************************************************/
static void lh79x_7seg_exit(void)
{
	dprintk("ENTER: lh79x_7seg_exit()\n");
	misc_deregister(&sevenSegment_dev);
	lh79x_7seg_write_display_str((u_char *)"BY");
	dprintk("LEAVE: lh79x_7seg_exit()\n");
	return;
}

module_init(lh79x_7seg_init);
module_exit(lh79x_7seg_exit);

MODULE_AUTHOR("Jim Gleason / Lineo, Inc.");
MODULE_DESCRIPTION("Seven Segment Display Driver for Sharp LH7x EVB");
MODULE_LICENSE("Copyright (c) 2002 Lineo, Inc.");

