/*
 * av500_hd.c - hard disk access
 *
 * author: Niklas Schroeter
 *         <niki@nikishome.de>
 * date:   11.06.2003
 *
 */

#include <linux/config.h>
#include <linux/delay.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/pci.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/blk.h>

#include <asm/io.h>
#include <asm/arch/dma.h>
#include <asm/arch/ck.h>
#include <asm/arch/gpio.h>
#include <asm/irq.h>

/* #undef the register definitions */
#undef HD_ALTSTATUS
#undef HD_DATA
#undef HD_ERROR
#undef HD_NSECTOR
#undef HD_SECTOR
#undef HD_HCYL
#undef HD_LCYL
#undef HD_DEV_HEAD
#undef HD_STATUS
#undef HD_RESET

#include "av500_hd.h"

/* write register */
#define HD_COMMAND 	HD_STATUS
#define HD_FEATURES 	HD_ERROR
#define HD_CONTROL	HD_ALTSTATUS

#define USE_LBA_MODE
#define USE_DMA_MODE

/* COMMANDS */
#define HD_IDENTIFY_DEVICE	0xec

#define HD_READ_DMA             0xc8
#define HD_WRITE_DMA            0xca
#define HD_READ_SECTORS		0x20
#define HD_WRITE_SECTORS	0x30
#define HD_READ_MULTIPLE        0xc4
#define HD_WRITE_MULTIPLE       0xc5
#define HD_SET_MULTIPLE_MODE    0xc6
#define HD_STANDBY_IMMEDIATE	0xe0
#define HD_IDLE_IMMEDIATE       0xe1
#define HD_SLEEP                0xe6
#define HD_SET_FEATURES         0xef
#define HD_FEATURE_KEEPSETTINGS 0x66
#define HD_FEATURE_ENABLE_AAM	0x42
#define HD_FEATURE_ENABLE_APM	0x05
#define HD_FEATURE_DISABLE_APM	0x85
#define HD_DEVICE_RESET		0x08

#define HD_STAT_ERR(x) 		(x & 0x01)
#define HD_STAT_DRQ(x) 		(x & 0x08)
#define HD_STAT_DRDY(x)		(x & 0x40)
#define HD_STAT_BSY(x)		(x & 0x80)

#ifdef USE_LBA_MODE
#define HD_LBA		0x40
#else
#define HD_LBA		0x00
#endif

//#define DEBUG

#ifdef DEBUG
#define DBG(f, a...)	printk(f, ## a)
#else
#define DBG(f, a...)	do { } while (0)
#endif

/* DMA data structures */
#define AV500_IDE_DMA   eExt0
static struct av500_trans_data {
	unsigned short		hd_status;
	unsigned int		transfer_count;
	unsigned long		part_offset;
	struct tq_struct	dpc_task;
	struct request*		kernel_req;
	struct timer_list	cmd_timeout;
} av500_hd_trans_data;

/* for IDENTIFY DEVICE */
static struct hd_driveid driveid;

static const int use_lba = 1;

static int av500_hd_cmd_read;
static int av500_hd_cmd_write;

int av500_hd_multcount;

extern void av500_report_hdpower(int);

static void av500_hd_interrupt(int irq, void* data, struct pt_regs *regs);
static int av500_hd_start_pio_write(struct av500_trans_data* trans_data);
static int av500_hd_start_pio_read(struct av500_trans_data* trans_data);
extern void av500_hd_reset(void);

static inline void av500_hd_delay(void)
{
	if (likely(in_interrupt())) {
		udelay(100);
		return;
	}
	set_current_state(TASK_UNINTERRUPTIBLE);
	schedule_timeout(0);
}

static inline int av500_hd_isBusy(void)
{
	unsigned char tmp = HD_IN_BYTE(HD_ALTSTATUS);
	return HD_STAT_BSY(tmp);
}

static inline int av500_hd_isError(void)
{
	unsigned char tmp = HD_IN_BYTE(HD_ALTSTATUS);
	return HD_STAT_ERR(tmp);
}

static inline int av500_hd_dataReady(void)
{
	unsigned char tmp = HD_IN_BYTE(HD_ALTSTATUS);
	return HD_STAT_DRQ(tmp);
}

static int waitifbusy(void)
{
    int i;

    for (i=0;i<50000;++i)
    {
        /* wait for BSY=0 */
        if (av500_hd_isBusy()) {
                av500_hd_delay();
                continue;
        }
        if (av500_hd_isError())
                return 0;
        return 1;
    }
    printk(KERN_ERR "av500_hd.c: timeout waiting for BSY=0\n");
    return 0;
}

static inline void av500_dma_ack(int ack)
{
        if (ack)
                outw_p(0, DMA_ACK_1);
        else
                outw_p(0, DMA_ACK_0);
}

static inline void av500_hdclock_high(void)
{
    unsigned int tmp;

    /* set FCLKDIV=0, RDWST=8(10), WRWST=3, WELEN=7 */
    tmp  = inl(BUS_CONFIG_REG) & ~(3UL<<0|15UL<<4|15UL<<8|15UL<<12);
    tmp |= (8UL<<4 | 3UL<<8 | 7UL<<12);
    outl(tmp, BUS_CONFIG_REG);
}

static inline void av500_hdclock_low(void)
{
    unsigned int tmp;

    /* set FCLKDIV=1, RDWST=5(7), WRWST=3, WELEN=6 */
    tmp  = inl(BUS_CONFIG_REG) & ~(3UL<<0|15UL<<4|15UL<<8|15UL<<12);
    tmp |= (1UL<<0 | 5UL<<4 | 3UL<<8 | 6UL<<12);
    outl(tmp, BUS_CONFIG_REG);
}

static inline void enable_hdirq(void)
{
        /* read HD_STATUS first to acknowledge any pending interrupts */
        av500_hd_trans_data.hd_status = HD_IN_BYTE(HD_STATUS);
	HD_OUT_BYTE(0x00, HD_CONTROL);
        enable_irq(HDD_IRQ);
}

static inline void disable_hdirq(void)
{
	/* disable interrupts both on host and device */
	HD_OUT_BYTE(0x02, HD_CONTROL);
        disable_irq(HDD_IRQ);
}

#define WAITIFBUSY	waitifbusy()
#define WAITDEVICE	while(!av500_hd_devReady())
#define WAITIFNODATA	while(!av500_hd_dataReady())

#ifdef DEBUG
void av500_hd_memdump(unsigned char *buf, unsigned int count)
{
    int i, j, s;
    unsigned char c;
    for (s=0; s<count; ++s)
    {
	for (j=0; j<32; ++j)
	{
	    for (i=0; i<16; ++i)
	    {
		printk("%02x ", buf[s*512+j*16+i]);
	    }

	    printk("  ");
	    for (i=0; i<16; ++i)
	    {
		c = buf[s*512+j*16+i];

		if (c>=0x20 && c<=0x7e)
	    	    printk("%c", buf[s*512+j*16+i]);
		else
		    printk(".");
	    }

	    printk("\n");
	}
    }
}

#endif

static inline void av500_start_cmd_timeout(struct av500_trans_data *data)
{
	data->cmd_timeout.expires = jiffies + 5*HZ;
	data->cmd_timeout.data    = (unsigned long)data;
	add_timer(&data->cmd_timeout);
}

static inline void av500_stop_cmd_timeout(struct av500_trans_data *data)
{
        del_timer_sync(&data->cmd_timeout);
}

static inline void rollback_req(struct request* req)
{
	struct buffer_head* bh = req->bh;
	
	req->sector = req->hard_sector;
	req->nr_sectors = req->hard_nr_sectors;
	req->current_nr_sectors = bh->b_size >> 9;
	req->buffer = bh->b_data;
}

static void av500_hd_cmd_timeout(unsigned long data)
{
	struct av500_trans_data *trans_data = (struct av500_trans_data*)data;
	struct request* req = trans_data->kernel_req;
	unsigned char drive_stat;
	
	drive_stat = HD_IN_BYTE(HD_ALTSTATUS);
	printk(KERN_ERR "hd_cmd_timeout cmd:%s, sect:%li, count:%li\n", req->cmd == READ ? "READ":"WRITE", req->sector, req->nr_sectors);
	printk(KERN_ERR "HD_STATUS = %02x\n", drive_stat);
	printk(KERN_ERR "transfer_count: %i\n", trans_data->transfer_count);

	if (!HD_STAT_ERR(drive_stat)) {
		/* no error status */
		if (!HD_STAT_BSY(drive_stat)) {
			/* drive not busy, maybe the command was completed? */
			if (req->cmd == WRITE) {
				if (!HD_STAT_DRQ(drive_stat)) {
					/* drive wants no data, command must have been completed already */
					printk(KERN_DEBUG "command complete, missed interrupt?\n");
					
					/* play safe and reset the drive before issuing the next command */
					av500_hd_hw_reset();
					av500_hd_set_multcount(av500_hd_multcount);
					av500_hd_set_features();
					
					av500_hd_interrupt(HDD_IRQ, (void*)data, 0);
				} else {
					/* 
					 * The drive expects us to send more data, error in bus communication
					 * or missed interrupt in a multi-write
					 */
					printk(KERN_DEBUG "drive waiting for data, missed write?\n");
					printk(KERN_DEBUG "retrying access\n");
					
					disable_hdirq();
					av500_hd_hw_reset();
					av500_hd_set_multcount(av500_hd_multcount);
					av500_hd_set_features();
					
					rollback_req(req);
					av500_hd_start_pio_write(trans_data);
				}
			} else
			if (req->cmd == READ) {
				if (HD_STAT_DRQ(drive_stat)) {
					/* not busy and data ready - command must have been completed */
					printk(KERN_DEBUG "command complete, missed interrupt?\n");
					av500_hd_interrupt(HDD_IRQ, (void*)data, 0);
				}
			}
		} else {
			/* drive is still busy after 5 seconds, something went wrong */
			printk(KERN_DEBUG "command not completed after 5 seconds, retrying\n");
			
			disable_hdirq();
			
			av500_hd_hw_reset();
			av500_hd_set_multcount(av500_hd_multcount);
			av500_hd_set_features();
			
			if (req->cmd == READ)
				av500_hd_start_pio_read(trans_data);
			else
			if (req->cmd == WRITE) {
				rollback_req(req);
				av500_hd_start_pio_write(trans_data);
			}
		}
	} else {
		printk(KERN_ERR "command failed, retrying\n");
		
		disable_hdirq();
		
		av500_hd_hw_reset();
		av500_hd_set_multcount(av500_hd_multcount);
		av500_hd_set_features();

		if (req->cmd == READ)
			av500_hd_start_pio_read(trans_data);
		else
		if (req->cmd == WRITE) {
			rollback_req(req);
			av500_hd_start_pio_write(trans_data);
		}
	}
}

static inline int av500_hd_command(int cmd)
{
    HD_OUT_BYTE(cmd, HD_COMMAND);
    
    /* this should be around 400ns */
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();
    nop();

    if (av500_hd_isError())
    {
	DBG("hard disk error after command!\n");
	return -1;
    }

    return 0;
}


static int av500_hd_start_pio_read(struct av500_trans_data* trans_data)
{
	struct request *req       = trans_data->kernel_req;
	unsigned int sector_count = req->nr_sectors;
	unsigned long start       = req->sector + trans_data->part_offset;
	
	int err = -EIO;

        DBG(KERN_DEBUG "hd_start_pio_read: start:%lu count:%u\n", start, sector_count);

        if (!WAITIFBUSY)
                goto out;

        HD_OUT_BYTE(  start       , HD_SECTOR);
        HD_OUT_BYTE(( start >>  8), HD_LCYL);
        HD_OUT_BYTE(( start >> 16), HD_HCYL);
        HD_OUT_BYTE(( start >> 24)| HD_LBA, HD_DEV_HEAD);
        HD_OUT_BYTE(sector_count  , HD_NSECTOR);

        enable_hdirq();

        if (av500_hd_command(av500_hd_cmd_read) < 0) {
                DBG("pio_read_sectors: command failed!\n");
                disable_hdirq();
                goto out;
        }

        if (sector_count > av500_hd_multcount)
                sector_count = av500_hd_multcount;

        trans_data->transfer_count = sector_count;
	av500_start_cmd_timeout(trans_data);
	
        err = -EINPROGRESS;
out:
        return err;
}


static int av500_hd_read_sector(unsigned short* buf, unsigned long sector_count)
{
        int i;

	av500_hdclock_high();
        for (i=0; i < 256*sector_count;) {
                buf[i++] = HD_IN_WORD(HD_DATA);
                /* play safe, read cycles are critical
                 * at 150MHz, 2 nops are around 27ns */
                nop();
                nop();
	}
	av500_hdclock_low();
	
        return 0;
}


/*
 * av500_hd_read_sectors()
 * 
 * IN:
 *	I/O request from block device layer to be serviced
 *
 */
int av500_hd_read_sectors(struct request* req, unsigned long part_offset)
{
        DBG(KERN_DEBUG "pio_read_sectors: start:%lu count:%lu\n", req->sector, req->nr_sectors);

	av500_hd_trans_data.kernel_req  = req;
	av500_hd_trans_data.part_offset = part_offset;
	
        return av500_hd_start_pio_read(&av500_hd_trans_data);
}

/*
 * This is the WRITE path
 */
static int av500_hd_write_sector(const unsigned short* buf, unsigned long sector_count)
{
        int i;

	av500_hdclock_high();
        for (i=0; i < 256*sector_count;) {
                HD_OUT_WORD(buf[i++], HD_DATA);
                nop();
                nop();
        }
	av500_hdclock_low();
	
        return 0;
}

static int av500_hd_start_pio_write(struct av500_trans_data* trans_data)
{
	struct request *req         = trans_data->kernel_req;
	unsigned int sector_count   = req->nr_sectors;
	unsigned long start         = req->sector + trans_data->part_offset;
	int timeout;

        int err = -EIO;

        DBG(KERN_DEBUG "+++ hd_start_pio_write: start:%lu count:%u\n", start, sector_count);

        if (!WAITIFBUSY)
                goto out;

        HD_OUT_BYTE(  start       , HD_SECTOR);
        HD_OUT_BYTE(( start >>  8), HD_LCYL);
        HD_OUT_BYTE(( start >> 16), HD_HCYL);
        HD_OUT_BYTE(( start >> 24)| HD_LBA, HD_DEV_HEAD);
        HD_OUT_BYTE(sector_count, HD_NSECTOR);

        enable_hdirq();

        if (av500_hd_command(av500_hd_cmd_write) < 0) {
                DBG("pio_write_sectors: command failed!\n");
                disable_hdirq();
                goto out;
        }
	
	timeout = 100;
	while ( !HD_STAT_DRQ(HD_IN_BYTE(HD_ALTSTATUS)) && (timeout-- > 0) )
		av500_hd_delay();

	if (timeout == 0) {
		disable_hdirq();
		goto out;
	}
	
	av500_start_cmd_timeout(trans_data);
	
	err = -EINPROGRESS;	
	queue_task( &(trans_data->dpc_task), &tq_immediate );
	mark_bh(IMMEDIATE_BH);

out:
        return err;
}


int av500_hd_write_sectors(struct request *req, unsigned long part_offset)
{
        DBG(KERN_DEBUG "+++ pio_write_sectors: start:%lu count:%lu\n", req->sector, req->nr_sectors);

	av500_hd_trans_data.kernel_req  = req;
	av500_hd_trans_data.part_offset = part_offset;

        return av500_hd_start_pio_write(&av500_hd_trans_data);
}


static int av500_hd_complete_pio(struct av500_trans_data* trans_data)
{
	struct request *req          = trans_data->kernel_req;
	unsigned long transfer_count = trans_data->transfer_count;
	int complete                 = 0;

	DBG(KERN_DEBUG "hd_complete_pio(%s): transfer_count:%lu\n", req->cmd == READ ? "READ" : "WRITE", transfer_count);

	if (req->cmd == READ) {
		while (transfer_count > 0) {
			/* see how much data we have ready to transfer */			
			int sectors = req->current_nr_sectors;
			
			if (complete)
				printk(KERN_DEBUG "hd_complete_pio: request complete but still sectors in transfer?\n");

			/* not a problem. the current buffer size might well be bigger than our multi-count */
			if (sectors > transfer_count)
				sectors = transfer_count;

			/* there's no way this can fail */
			av500_hd_read_sector((unsigned short*)req->buffer, sectors);

			transfer_count -= sectors;
			req->current_nr_sectors -= sectors;
			req->nr_sectors -= sectors;
			req->buffer += sectors << 9;
			req->sector += sectors;

			if (req->current_nr_sectors == 0)
				complete = !end_that_request_first(req, 1, "av500_ide");

			/* if there are buffers left we'll repeat until transfer_count == 0 */
		}

		trans_data->transfer_count = 0;
		return complete;
	}

	if (req->cmd == WRITE) {
		if (!transfer_count)
			return 1;
			
		while (transfer_count > 0) {
			int sectors = req->current_nr_sectors;
			
			/* not a problem. the current buffer size might well be bigger than our multi-count */
			if (sectors > transfer_count)
				sectors = transfer_count;

			/* there's no way this can fail */
			av500_hd_write_sector((unsigned short*)req->buffer, sectors);

			transfer_count -= sectors;
			req->current_nr_sectors -= sectors;
			req->nr_sectors -= sectors;
			req->buffer += sectors << 9;
			req->sector += sectors;

			if (req->current_nr_sectors == 0)
				end_that_request_first(req, 1, "av500_ide");
		}
		trans_data->transfer_count = 0;
		return 0;
	}

	/* if this is ever reached, it's a BUG */
	return -1;
}

static void av500_hd_pio_callback(void* data)
{
	static atomic_t enter_count = ATOMIC_INIT(0);
	struct av500_trans_data* trans_data = (struct av500_trans_data*)data;
	struct request *req = trans_data->kernel_req;
	int complete;

	/*
	 * We have a race condition here for WRITE requests:
	 * as soon as the data transfer to the drive is completed, this function
	 * might become re-entered from the ISR. We protect against this by
	 * maintaining an atomic enter counter.
	 *
	 * Another solution might be to call this function exclusively from IMMEDIATE_BH,
	 * that way the kernel should serialize the function entry (but it's slower!).
	 */
	atomic_inc(&enter_count);
	if (atomic_read(&enter_count) > 1) {
		DBG(KERN_DEBUG "av500_hd_pio_callback(%s), already here %i times\n", req->cmd == READ ? "READ" : "WRITE", atomic_read(&enter_count));
		return;
	}

	DBG(KERN_DEBUG "av500_hd_pio_callback(%s)\n", req->cmd == READ ? "READ" : "WRITE");
	
	do {
		unsigned char drive_stat = HD_IN_BYTE(HD_ALTSTATUS);

		if (req->cmd == WRITE) {
			if (!HD_STAT_DRQ(drive_stat) && req->nr_sectors > 0) {
				printk(KERN_ERR "av500_hd_pio_callback: ERROR - DATA and !DRQ?\n");
				/* MUST decrease enter_count on function exit! */
				atomic_dec(&enter_count);
				return;
			}

			trans_data->transfer_count = req->nr_sectors;
			if (trans_data->transfer_count > av500_hd_multcount)
				trans_data->transfer_count = av500_hd_multcount;
		}
		
		/* complete the request, check if it ended with a drive error */
		if (!HD_STAT_ERR(drive_stat))
			complete = av500_hd_complete_pio(trans_data);
		else
			complete = !end_that_request_first(req, 0, "av500_ide");
	
		DBG(KERN_DEBUG "av500_hd_pio_callback: complete:%i\n", complete);
		
		if ( complete == 0 ) {
			if (req->cmd == READ) {
				trans_data->transfer_count = req->nr_sectors;
				if (trans_data->transfer_count > av500_hd_multcount)
					trans_data->transfer_count = av500_hd_multcount;
			}
		} else
		if ( complete == 1 ) {
			/* request completely serviced, signal to upper layers */
			av500_stop_cmd_timeout(trans_data);
			disable_hdirq();
			av500_ide_next_request(req);
		}
	} while (!atomic_dec_and_test(&enter_count));
}

/* interrupt service for HDD_IRQ */
static void av500_hd_interrupt(int irq, void* data, struct pt_regs *regs)
{
	static int write_intr = 0;
        struct av500_trans_data* trans_data = (struct av500_trans_data*)data;
        struct request* req                 = trans_data->kernel_req;

        /* read status and acknowledge the interrupt */
        trans_data->hd_status = HD_IN_BYTE(HD_STATUS);

	DBG(KERN_DEBUG "HDD_IRQ, HD_STATUS=0x%02x\n", trans_data->hd_status);
	
        if (HD_STAT_BSY(trans_data->hd_status)) {
        	/* Interrupt from device but BSY=1. Spurious? Ignore. */
        	printk(KERN_DEBUG "av500_hd.c: spurious interrupt, HD_STATUS=0x%02x\n", trans_data->hd_status);
        	return;
        }

#if 0
	if (req->cmd == WRITE) {
		write_intr++;
		if (write_intr == 1000) {
			printk(KERN_DEBUG "av500_hd.c: FORCE WRITE TIMEOUT! status=0x%02x\n", trans_data->hd_status);
			write_intr = 0;
			return;
		}
	}
#endif

	av500_hd_pio_callback(data);
}

void av500_hd_set_features(void)
{
        if (!WAITIFBUSY)
                return;

        HD_OUT_BYTE(HD_FEATURE_KEEPSETTINGS, HD_FEATURES);
        if (av500_hd_command(HD_SET_FEATURES) < 0)
                printk(KERN_INFO "failed setting feature KEEPSETTINGS\n");
		
	HD_OUT_BYTE(HD_FEATURE_ENABLE_AAM, HD_FEATURES);
	HD_OUT_BYTE(128, HD_NSECTOR);
	if (av500_hd_command(HD_SET_FEATURES) < 0)
		printk(KERN_INFO "failed setting feature Acoustic Management\n");
	
	HD_OUT_BYTE(HD_FEATURE_ENABLE_APM, HD_FEATURES);
	HD_OUT_BYTE(128, HD_NSECTOR);
	if (av500_hd_command(HD_SET_FEATURES) < 0)
		printk(KERN_INFO "failed setting feature APM\n");
}


void av500_hd_set_multcount(int multcount)
{
        av500_hd_multcount = 1;
        av500_hd_cmd_read  = HD_READ_SECTORS;
        av500_hd_cmd_write = HD_WRITE_SECTORS;

	if (!WAITIFBUSY) {
		printk(KERN_INFO "failed setting multcount\n");
                return;
	}

        HD_OUT_BYTE(multcount & 0xff, HD_NSECTOR);

        if (av500_hd_command(HD_SET_MULTIPLE_MODE) < 0) {
                printk(KERN_INFO "failed setting multcount\n");
                return;
        }

        av500_hd_cmd_read  = HD_READ_MULTIPLE;
        av500_hd_cmd_write = HD_WRITE_MULTIPLE;
        av500_hd_multcount = multcount;
}

void av500_hd_hw_reset(void)
{
        HD_OUT_BYTE(0, HD_RESET_0);
        av500_hd_delay();
        HD_OUT_BYTE(0, HD_RESET_1);
        av500_hd_delay();
        WAITIFBUSY;
        if (!av500_hd_devReady())
        	printk(KERN_ERR "av500_hd.c: no DRDY after device reset!");
}

void av500_hd_reset(void)
{
        HD_OUT_BYTE(0x06, HD_CONTROL); /* reset & irq off */
        av500_hd_delay();
        HD_OUT_BYTE(0x02, HD_CONTROL); /* irq off, deassert reset */
        av500_hd_delay();
        WAITIFBUSY;
        if (!av500_hd_devReady())
        	printk(KERN_ERR "av500_hd.c: no DRDY after device reset!");
}

int av500_hd_idle(void)
{
        if (!WAITIFBUSY)
                return -1;

        return av500_hd_command(HD_IDLE_IMMEDIATE);
}

int av500_hd_standby(void)
{
	volatile int status;
	if (!WAITIFBUSY)
		return -1;
	
	if (av500_hd_command(HD_STANDBY_IMMEDIATE) < 0)
		return -1;
		
	if (!WAITIFBUSY)
		return -1;

	if (av500_hd_isError())
		return -1;
		
	status = HD_IN_BYTE(HD_STATUS);
	return 0;
}

int av500_hd_sleep(void)
{
	volatile int status;
	
        if (!WAITIFBUSY)
                return -1;

        if (av500_hd_command(HD_SLEEP) < 0)
                return -1;

        udelay(5);

        if (!WAITIFBUSY)
                return -1;

	if (av500_hd_isError())
		return -1;
		
        status = HD_IN_BYTE(HD_STATUS);
	return 0;
}

int av500_hd_identify(unsigned long long *max_sectors)
{
    int i;
    unsigned long long lba_sectors;

    if (!WAITIFBUSY)
        return -EBUSY;

    if (av500_hd_command(HD_IDENTIFY_DEVICE) < 0)
        return -EIO;
        
    if (!WAITIFBUSY)
    	return -EIO;

    if (av500_hd_read_sector((unsigned short*)&driveid, 1) < 0)
	return -EIO;

    /* display name */
    printk(KERN_DEBUG "drive model: ");
    for(i=0; i<40; i+=2)
	printk("%c%c", driveid.model[i+1], driveid.model[i]);

    printk("\n" KERN_DEBUG "firmware: ");
    for(i=0; i<8; i+=2)
	printk("%c%c", driveid.fw_rev[i+1],driveid.fw_rev[i]);

    /* check LBA capacity */
    lba_sectors = driveid.lba_capacity;
    if (driveid.lba_capacity_2 != 0)
        lba_sectors = driveid.lba_capacity_2;

    printk(" LBA: %llu\n", lba_sectors);

    *max_sectors = lba_sectors;

    if (driveid.max_multsect != 0) {
        printk(KERN_DEBUG "drive supports multcount %i\n", driveid.max_multsect);
        av500_hd_set_multcount(driveid.max_multsect);
    }
    printk(KERN_DEBUG "recommended DMA cycle time %i(ns)\n", driveid.eide_dma_time);
    printk(KERN_DEBUG "minimum PIO cycle time %i(ns)\n", driveid.eide_pio);

    av500_hd_set_features();

    return 0;
}

/* lowlevel driver initialization */
int av500_hd_init(void)
{
        av500_dma_ack(0);
	
	hd_led_force_off();

        if (request_irq(HDD_IRQ, av500_hd_interrupt, SA_SHIRQ, "av500_ide", (void*)&av500_hd_trans_data) != 0) {
                printk(KERN_ERR "av500_ide: request irq failed\n");
                return -1;
        }

        disable_hdirq();
	
	av500_hd_trans_data.dpc_task.routine = av500_hd_pio_callback;
	av500_hd_trans_data.dpc_task.data    = &av500_hd_trans_data;
	
	init_timer(&av500_hd_trans_data.cmd_timeout);
	av500_hd_trans_data.cmd_timeout.function = av500_hd_cmd_timeout;
	
        return 0;
}

void av500_hd_release(void)
{
        free_irq(HDD_IRQ, (void*)&av500_hd_trans_data);
}
