/*
 * av500_hd.c - hard disk access
 *
 * author: Niklas Schroeter
 *         <niki@nikishome.de>
 * date:   11.06.2003
 *
 */

#include <linux/config.h>
#include <linux/delay.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/hdreg.h>
#include <linux/pci.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/blk.h>

#include <asm/io.h>
#include <asm/arch/dma.h>
#include <asm/arch/ck.h>
#include <asm/arch/gpio.h>
#include <asm/irq.h>

/* #undef the register definitions */
#undef HD_ALTSTATUS
#undef HD_DATA
#undef HD_ERROR
#undef HD_NSECTOR
#undef HD_SECTOR
#undef HD_HCYL
#undef HD_LCYL
#undef HD_DEV_HEAD
#undef HD_STATUS
#undef HD_RESET

#include "av500_hd.h"

/* write register */
#define HD_COMMAND 	HD_STATUS
#define HD_FEATURES 	HD_ERROR
#define HD_CONTROL	HD_ALTSTATUS

#define USE_LBA_MODE
#define USE_DMA_MODE

/* COMMANDS */
#define HD_IDENTIFY_DEVICE	0xec

#define HD_READ_DMA             0xc8
#define HD_WRITE_DMA            0xca
#define HD_READ_SECTORS		0x20
#define HD_WRITE_SECTORS	0x30
#define HD_READ_MULTIPLE        0xc4
#define HD_WRITE_MULTIPLE       0xc5
#define HD_SET_MULTIPLE_MODE    0xc6
#define HD_STANDBY_IMMEDIATE	0xe0
#define HD_IDLE_IMMEDIATE       0xe1
#define HD_SLEEP                0xe6
#define HD_SET_FEATURES         0xef
#define HD_FEATURE_KEEPSETTINGS 0x66
#define HD_FEATURE_ENABLE_AAM	0x42
#define HD_FEATURE_ENABLE_APM	0x05
#define HD_FEATURE_DISABLE_APM	0x85

#ifdef USE_LBA_MODE
#define HD_LBA		0x40
#else
#define HD_LBA		0x00
#endif

//#define DEBUG

#ifdef DEBUG
#define DBG(f, a...)	printk(f, ## a)
#else
#define DBG(f, a...)	do { } while (0)
#endif

/* DMA data structures */
#define AV500_IDE_DMA   eExt0
static struct av500_trans_data {
        unsigned short  hd_status;
        unsigned int    transfer_count;
	unsigned long   part_offset;
	struct tq_struct dpc_task;
	struct request* request;
} av500_hd_trans_data;

/* for IDENTIFY DEVICE */
static struct hd_driveid driveid;

static const int use_lba = 1;

static int av500_hd_cmd_read;
static int av500_hd_cmd_write;
int av500_hd_multcount;

#define HD_OUT_BYTE(v,p) outw_p( (v) & 0xff,p)
#define HD_OUT_WORD(v,p) outw_p(v,p)
#define HD_IN_BYTE(p) (inw_p(p) & 0xff)
#define HD_IN_WORD(p) inw_p(p)

extern void av500_report_hdpower(int);

static inline void av500_hd_delay(void)
{
        if (likely(in_interrupt())) {
                udelay(100);
                return;
        }

        set_current_state(TASK_INTERRUPTIBLE);
        schedule_timeout(0);
}

static inline int av500_hd_devReady(void)
{
    return ((HD_IN_BYTE(HD_ALTSTATUS) & 0x40) != 0);
}

static inline int av500_hd_isBusy(void)
{
    unsigned char tmp;

    tmp = HD_IN_BYTE(HD_ALTSTATUS);

    return ((tmp & 0x80) != 0);
}

static inline int av500_hd_isError(void)
{
    return (HD_IN_BYTE(HD_ALTSTATUS) & 0x01);
}

static inline int av500_hd_dataReady(void)
{
    unsigned char tmp = HD_IN_BYTE(HD_ALTSTATUS);
    return (tmp & 0x08);
}

static int waitifbusy(void)
{
    int i;

    for (i=0;i<50000;++i)
    {
        /* wait for BSY=0 */
        if (av500_hd_isBusy()) {
                av500_hd_delay();
                continue;
        }
        if (av500_hd_isError())
                return 0;
        return 1;
    }
    DBG("timeout waiting for BSY=0\n");
    return 0;
}

static inline void av500_dma_ack(int ack)
{
        if (ack)
                outw_p(0, DMA_ACK_1);
        else
                outw_p(0, DMA_ACK_0);
}

static inline void av500_hdclock_high(void)
{
    unsigned int tmp;

    /* set FCLKDIV=0, RDWST=8(10), WRWST=3, WELEN=7 */
    tmp  = inl(BUS_CONFIG_REG) & ~(3UL<<0|15UL<<4|15UL<<8|15UL<<12);
    tmp |= (8UL<<4 | 3UL<<8 | 7UL<<12);
    outl(tmp, BUS_CONFIG_REG);
}

static inline void av500_hdclock_low(void)
{
    unsigned int tmp;

    /* set FCLKDIV=1, RDWST=5(7), WRWST=3, WELEN=6 */
    tmp  = inl(BUS_CONFIG_REG) & ~(3UL<<0|15UL<<4|15UL<<8|15UL<<12);
    tmp |= (1UL<<0 | 5UL<<4 | 3UL<<8 | 6UL<<12);
    outl(tmp, BUS_CONFIG_REG);
}

static inline void enable_hdirq(void)
{
        /* read HD_STATUS first to acknowledge any pending interrupts */
        av500_hd_trans_data.hd_status = HD_IN_BYTE(HD_STATUS);
	HD_OUT_BYTE(0x00, HD_CONTROL);
        enable_irq(HD_IRQ);
}

static inline void disable_hdirq(void)
{
	/* disable interrupts both on host and device */
	HD_OUT_BYTE(0x02, HD_CONTROL);
        disable_irq(HD_IRQ);
}

#define WAITIFBUSY	waitifbusy()
#define WAITDEVICE	while(!av500_hd_devReady())
#define WAITIFNODATA	while(!av500_hd_dataReady())

#ifdef DEBUG
void av500_hd_memdump(unsigned char *buf, unsigned int count)
{
    int i, j, s;
    unsigned char c;
    for (s=0; s<count; ++s)
    {
	for (j=0; j<32; ++j)
	{
	    for (i=0; i<16; ++i)
	    {
		printk("%02x ", buf[s*512+j*16+i]);
	    }

	    printk("  ");
	    for (i=0; i<16; ++i)
	    {
		c = buf[s*512+j*16+i];

		if (c>=0x20 && c<=0x7e)
	    	    printk("%c", buf[s*512+j*16+i]);
		else
		    printk(".");
	    }

	    printk("\n");
	}
    }
}

#endif

static inline int av500_hd_command(int cmd)
{
    HD_OUT_BYTE(cmd, HD_COMMAND);

    if (av500_hd_isError())
    {
	DBG("hard disk error after command!\n");
	return -1;
    }

    return 0;
}


static int av500_hd_start_pio_read(struct av500_trans_data* trans_data)
{
	struct request *req       = trans_data->request;
	unsigned int sector_count = req->nr_sectors;
	unsigned long start       = req->sector + trans_data->part_offset;
	
	int err = -EIO;

        DBG(KERN_DEBUG "hd_start_pio_read: start:%lu count:%u\n", start, sector_count);

        if (sector_count > av500_hd_multcount)
                sector_count = av500_hd_multcount;

        if (!WAITIFBUSY)
                goto out;

        HD_OUT_BYTE(  start       , HD_SECTOR);
        HD_OUT_BYTE(( start >>  8), HD_LCYL);
        HD_OUT_BYTE(( start >> 16), HD_HCYL);
        HD_OUT_BYTE(( start >> 24)| HD_LBA, HD_DEV_HEAD);
        HD_OUT_BYTE(sector_count  , HD_NSECTOR);

        enable_hdirq();

        if (av500_hd_command(av500_hd_cmd_read) < 0) {
                DBG("pio_read_sectors: command failed!\n");
                disable_hdirq();
                goto out;
        }

        trans_data->transfer_count = sector_count;

        err = -EINPROGRESS;
out:
        return err;
}


static int av500_hd_read_sector(unsigned short* buf, unsigned long sector_count)
{
        int i;

	DBG(KERN_DEBUG "hd_read_sector(%p, %lu)\n", buf, sector_count);
	
	if (!WAITIFBUSY) {
		DBG("hd_read_sector: timeout!\n");
                return -1;
	}

        if (!av500_hd_dataReady()) {
                DBG("hd_read_sector: no data available!\n");
                return -1;
        }

	av500_hdclock_high();
        for (i=0; i < 256*sector_count;) {
                buf[i++] = HD_IN_WORD(HD_DATA);
	}
	av500_hdclock_low();
	
        return 0;
}


/*
 * av500_hd_read_sectors()
 * 
 * IN:
 *	I/O request from block device layer to be serviced
 *
 */
int av500_hd_read_sectors(struct request* req, unsigned long part_offset)
{
        DBG(KERN_DEBUG "pio_read_sectors: start:%lu count:%lu\n", req->sector, req->nr_sectors);

	av500_hd_trans_data.request     = req;
	av500_hd_trans_data.part_offset = part_offset;
	
        return av500_hd_start_pio_read(&av500_hd_trans_data);
}

/*
 * This is the WRITE path
 */
static int av500_hd_write_sector(const unsigned short* buf, unsigned long sector_count)
{
        int i;

        if (!WAITIFBUSY)
                return -1;

        if (!av500_hd_dataReady()) {
                DBG("av500_hd_write_sector: hd not ready!\n");
                return -1;
        }

	av500_hdclock_high();
        for (i=0; i < 256*sector_count;) {
                HD_OUT_WORD(buf[i++], HD_DATA);
        }
	av500_hdclock_low();
	
        return 0;
}

static int av500_hd_start_pio_write(struct av500_trans_data* trans_data)
{
	struct request *req         = trans_data->request;
	unsigned int transfer_count = req->nr_sectors;
	unsigned long start         = req->sector + trans_data->part_offset;
        int i;

        int err = -EIO;

        DBG(KERN_DEBUG "hd_start_pio_write: start:%lu count:%u\n", start, transfer_count);

        if (transfer_count > av500_hd_multcount)
                transfer_count = av500_hd_multcount;

        if (!WAITIFBUSY)
                goto out;

        HD_OUT_BYTE(  start       , HD_SECTOR);
        HD_OUT_BYTE(( start >>  8), HD_LCYL);
        HD_OUT_BYTE(( start >> 16), HD_HCYL);
        HD_OUT_BYTE(( start >> 24)| HD_LBA, HD_DEV_HEAD);
        HD_OUT_BYTE(transfer_count, HD_NSECTOR);

        enable_hdirq();

        if (av500_hd_command(av500_hd_cmd_write) < 0) {
                DBG("pio_write_sectors: command failed!\n");
                disable_hdirq();
                goto out;
        }

	trans_data->transfer_count = transfer_count;
	
	queue_task( &(trans_data->dpc_task), &tq_immediate );
	mark_bh(IMMEDIATE_BH);

	err = -EINPROGRESS;
out:
        return err;
}


int av500_hd_write_sectors(struct request *req, unsigned long part_offset)
{
        DBG(KERN_DEBUG "pio_write_sectors: start:%lu count:%lu\n", req->sector, req->nr_sectors);

	av500_hd_trans_data.request     = req;
	av500_hd_trans_data.part_offset = part_offset;

        return av500_hd_start_pio_write(&av500_hd_trans_data);
}


static int av500_hd_complete_pio(struct av500_trans_data* trans_data)
{
	struct request *req          = trans_data->request;
	unsigned long transfer_count = trans_data->transfer_count;
	int complete                 = 0;

	DBG(KERN_DEBUG "hd_complete_pio: transfer_count:%lu\n", transfer_count);

	if (req->cmd == READ) {
		while (transfer_count > 0) {
			/* see how much data we have ready to transfer */			
			int sectors = req->current_nr_sectors;
			
			if (complete)
				printk(KERN_DEBUG "hd_complete_pio: request complete but still sectors in transfer?\n");

			if (sectors > transfer_count) {
				printk(KERN_DEBUG "hd_complete_pio: %i sectors left for buffer, but only %u left to read!\n",
				       sectors, transfer_count);
				sectors = transfer_count;
			}

			if (av500_hd_read_sector((unsigned short*)req->buffer, sectors) < 0) {
				end_that_request_first(req, 0, "av500_ide");
				return -1;
			}

			transfer_count -= sectors;
			req->current_nr_sectors -= sectors;
			req->buffer += sectors << 9;

			if (req->current_nr_sectors == 0) {
				complete = !end_that_request_first(req, 1, "av500_ide");
			}

			/* if there are buffers left we'll repeat until transfer_count == 0 */
		}

		trans_data->transfer_count = transfer_count;

		return complete;
	}

	if (req->cmd == WRITE) {

		/* if we enter here with transfer_count == 0, we've been called from HD IRQ */
		if (transfer_count == 0)
			return 1; /* WRITE command completed */
		
		while (transfer_count > 0) {
			int sectors = req->current_nr_sectors;
			if (sectors > transfer_count) {
				printk(KERN_DEBUG "hd_start_pio_write: %i sectors left in buffers, but only %u can be transferred!\n",
				       sectors, transfer_count);
				sectors = transfer_count;
			}

			if (av500_hd_write_sector((unsigned short*)req->buffer, sectors) < 0) {
				end_that_request_first(req, 0, "av500_ide");
				return -1;
			}

			transfer_count -= sectors;
			req->current_nr_sectors -= sectors;
			req->buffer += sectors << 9;

			if (req->current_nr_sectors == 0) {
				end_that_request_first(req, 1, "av500_ide");
			}
		}

		trans_data->transfer_count = transfer_count;
		
		return -1;
	}

	/* if this is ever reached, it's a BUG */
	return -1;
}

static void av500_hd_pio_callback(void* data)
{
	struct av500_trans_data* trans_data = (struct av500_trans_data*)data;
	struct request *req = trans_data->request;
	int complete;
	
	/* complete the request and start a new one */
	complete = av500_hd_complete_pio(trans_data);
	if ( complete == 0 ) {
		disable_hdirq();
		if (req->nr_sectors > 0) {
			if (req->cmd == READ)
				av500_hd_start_pio_read(trans_data);
			else
			if (req->cmd == WRITE)
				av500_hd_start_pio_write(trans_data);
			return;
		}
	} else
	if ( complete == 1 ) {
		/* request completely serviced, signal to upper layers */
		disable_hdirq();
		av500_ide_next_request(req);
	}
}

/* interrupt service for HD_IRQ */
static void av500_hd_interrupt(int irq, void* data, struct pt_regs *regs)
{
        struct av500_trans_data* trans_data = (struct av500_trans_data*)data;

        /* read status and acknowledge the interrupt */
        trans_data->hd_status = HD_IN_BYTE(HD_STATUS);

        if (trans_data->transfer_count == 0) {
		if (trans_data->request->cmd == READ) {
			/* spurious interrupt */
			printk(KERN_DEBUG "av500_hd.c: spurious interrupt, HD_STATUS=0x%x02\n", trans_data->hd_status);
		} else {
			DBG(KERN_DEBUG "hd_interrupt: write complete, HD_STATUS=0x%x02\n", trans_data->hd_status);
			av500_hd_pio_callback(data);
		}
		return;
        }

	queue_task( &(trans_data->dpc_task), &tq_immediate );
	mark_bh(IMMEDIATE_BH);
}

void av500_hd_set_features(void)
{
        if (!WAITIFBUSY)
                return;

        HD_OUT_BYTE(HD_FEATURE_KEEPSETTINGS, HD_FEATURES);
        if (av500_hd_command(HD_SET_FEATURES) < 0)
                printk(KERN_INFO "failed setting feature KEEPSETTINGS\n");
		
	HD_OUT_BYTE(HD_FEATURE_ENABLE_AAM, HD_FEATURES);
	HD_OUT_BYTE(128, HD_NSECTOR);
	if (av500_hd_command(HD_SET_FEATURES) < 0)
		printk(KERN_INFO "failed setting feature Acoustic Management\n");
	
	HD_OUT_BYTE(HD_FEATURE_ENABLE_APM, HD_FEATURES);
	HD_OUT_BYTE(128, HD_NSECTOR);
	if (av500_hd_command(HD_SET_FEATURES) < 0)
		printk(KERN_INFO "failed setting feature APM\n");
}


void av500_hd_set_multcount(int multcount)
{
        av500_hd_multcount = 1;
        av500_hd_cmd_read  = HD_READ_SECTORS;
        av500_hd_cmd_write = HD_WRITE_SECTORS;

	if (!WAITIFBUSY) {
		printk(KERN_INFO "failed setting multcount\n");
                return;
	}

        HD_OUT_BYTE(multcount & 0xff, HD_NSECTOR);

        if (av500_hd_command(HD_SET_MULTIPLE_MODE) < 0) {
                printk(KERN_INFO "failed setting multcount\n");
                return;
        }

        av500_hd_cmd_read  = HD_READ_MULTIPLE;
        av500_hd_cmd_write = HD_WRITE_MULTIPLE;
        av500_hd_multcount = multcount;
}

#if defined(CONFIG_ARCHOS_AV500) && defined(CONFIG_OMAP_INNOVATOR)
void av500_hd_hw_reset(void)
{
        HD_OUT_BYTE(0, HD_RESET);
        av500_hd_delay();
        HD_OUT_BYTE(1, HD_RESET);
        av500_hd_delay();
        WAITIFBUSY;
        WAITDEVICE;
}
#elif defined(CONFIG_ARCHOS_AV500)
void av500_hd_hw_reset(void)
{
        HD_OUT_BYTE(0, HD_RESET_0);
        av500_hd_delay();
        HD_OUT_BYTE(0, HD_RESET_1);
        av500_hd_delay();
        WAITIFBUSY;
        WAITDEVICE;
}
#endif

void av500_hd_reset(void)
{
        HD_OUT_BYTE(0x06, HD_CONTROL); /* reset & irq off */
        av500_hd_delay();
        HD_OUT_BYTE(0x00, HD_CONTROL); /* irq on */
        av500_hd_delay();
        WAITIFBUSY;
        WAITDEVICE;
}

int av500_hd_idle(void)
{
        if (!WAITIFBUSY)
                return -1;

        return av500_hd_command(HD_IDLE_IMMEDIATE);
}

int av500_hd_standby(void)
{
	if (!WAITIFBUSY)
		return -1;
	
	return av500_hd_command(HD_STANDBY_IMMEDIATE);
}

int av500_hd_sleep(void)
{
	volatile int status;
	
        if (!WAITIFBUSY)
                return -1;

        if (av500_hd_command(HD_SLEEP) < 0)
                return -1;

        if (!WAITIFBUSY)
                return -1;

        status = HD_IN_BYTE(HD_STATUS);
        //printk(KERN_DEBUG "av500_hd_sleep() status:%02x\n", status);
	
	return 0;
}

void av500_hd_power(int status)
{
	if (status) {
		outw(0, HD_RESET_1);
		outw(0, HD_SWITCH_1);
		omap_gpio_dir(9, GPIO_DIR_OUT);
		omap_gpio_clr(9);
		av500_report_hdpower(1);
	} else {
		omap_gpio_set(9);
		omap_gpio_dir(9, GPIO_DIR_OUT);
		outw(0, HD_SWITCH_0);
		outw(0, HD_RESET_0);
		av500_report_hdpower(0);
	}
}

void av500_usb2_power(int status)
{
        if (status) {
                outw(0, USB_SWITCH_1);
		omap_gpio_set(15);
        } else {
                outw(0, USB_SWITCH_0);
		omap_gpio_clr(15);
	}
}

int av500_hd_identify(unsigned long long *max_sectors)
{
    int i;
    unsigned long long lba_sectors;

    if (!WAITIFBUSY)
        return -EBUSY;

    if (av500_hd_command(HD_IDENTIFY_DEVICE) < 0)
        return -EIO;

    if (av500_hd_read_sector((unsigned short*)&driveid, 1) < 0)
	return -EIO;

    /* display name */
    printk(KERN_DEBUG "drive model: ");
    for(i=0; i<40; i+=2)
	printk("%c%c", driveid.model[i+1], driveid.model[i]);

    printk("\n" KERN_DEBUG "firmware: ");
    for(i=0; i<8; i+=2)
	printk("%c%c", driveid.fw_rev[i+1],driveid.fw_rev[i]);

    /* check LBA capacity */
    lba_sectors = driveid.lba_capacity;
    if (driveid.lba_capacity_2 != 0)
        lba_sectors = driveid.lba_capacity_2;

    printk(" LBA: %llu\n", lba_sectors);

    *max_sectors = lba_sectors;

    if (driveid.max_multsect != 0) {
        printk(KERN_DEBUG "drive supports multcount %i\n", driveid.max_multsect);
        av500_hd_set_multcount(driveid.max_multsect);
    }
    printk(KERN_DEBUG "recommended DMA cycle time %i(ns)\n", driveid.eide_dma_time);
    printk(KERN_DEBUG "minimum PIO cycle time %i(ns)\n", driveid.eide_pio);

    av500_hd_set_features();

    return 0;
}

/* lowlevel driver initialization */
int av500_hd_init(void)
{
        av500_dma_ack(0);
	
	hd_led_force_off();

        if (request_irq(HD_IRQ, av500_hd_interrupt, SA_SHIRQ, "av500_ide", (void*)&av500_hd_trans_data) != 0) {
                printk(KERN_ERR "av500_ide: request irq failed\n");
                return -1;
        }

        disable_hdirq();
	
	av500_hd_trans_data.dpc_task.routine = av500_hd_pio_callback;
	av500_hd_trans_data.dpc_task.data    = &av500_hd_trans_data;

        return 0;
}

void av500_hd_release(void)
{
        free_irq(HD_IRQ, (void*)&av500_hd_trans_data);
}
