/*
 * Copyright (c) 2004 Nokia. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the
 * distribution.
 *
 * Neither the name of Nokia nor the names of its contributors may be
 * used to endorse or promote products derived from this software
 * without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include <assert.h>
#include <string.h>
#include <glib.h>

#include "osb.h" // URLProtectionSpace
#include "uri.h"
#include "HttpSofia.h"
#include "GLibHelpers.h"

extern "C" {
  #include <wengine.h>
#include <http.h>
#include <http_tag.h>
#include <su_source.h>
#include <http_header.h>
#include <tport_tag.h>

}

#include "osb.h" // URLProtectionSpace

class SofiaRequest : public HttpRequest
{
    friend class SofiaFactoryS;
public:
    SofiaRequest(const gchar * aurl,
                 const gchar * cookies,
                 /* HACK -- remove me after sofia is fixed. */
                 const gchar * proxyURL,
                 /* END HACK */                 
                 Type atype, WESession* asession);
    ~SofiaRequest();

    void execute();
    void stop();

  
    void event();
    
    void setPostData(const gchar * contentType, GByteArray *);
    void authenticate(const gchar * user, const gchar * pass);
protected:
    void setHost(const gchar * host);
private:
    void doAuth();

    WERequest *wr;
    WESession *session;
    bool headersSent;
    
    GByteArray * postData;

    /* HACK -- remove me after sofia is fixed. */
    gchar * proxy;
    /* END HACK */

    /** HTTP Host header. */
    gchar * host;
};


extern "C" { 
static void we_event_cb (WERequestMagic *user_data, WERequest *wr);
}

SofiaFactoryS& SofiaFactory()
{
    static SofiaFactoryS sofia_instance;

    return sofia_instance;
}


SofiaFactoryS::SofiaFactoryS()
{
    if (we_init() == -1)                    /* Initialize Web Engine */ 
    {
	we_perror("we_init");
	g_warning("Couldn't initialize sofia web engine");
	return;
    }
    engine  = we_engine_create(TAG_END());
    /*
     * WETAG_CACHE_DIR(cache), 
     * WETAG_COOKIE_JAR(cookies),
     */
    
    root = su_root_source_create(NULL);

    gs = su_root_gsource(root);
    g_source_attach(gs, NULL);

    session = we_session_create(engine,
 				::we_event_cb,
 				WETAG_ROOT(root),
                                WETAG_MAX_REDIR(0),  
				WETAG_MFLAGS(MSG_FLG_EXTRACT_COPY),
                                /* FIXME: This is broken with the latest version of sofia.
                                   May cause MSG_FLAG_ERROR asserts in sofia. */
                                WETAG_STREAMING(0),
				/* unless accept is defined, domino servers give warning, which results
				 * in error flag in sofia code which results in assert in wengine */
				HTTPTAG_ACCEPT_STR("*/*"),
                                NTHTAG_ERROR_MSG(1), 					
                                TAG_END());

    
    if (!session) {
	g_warning("couldn't create sofia web engine session");
    }
}

void SofiaFactoryS::setProxy(const gchar * proto, const gchar * proxyURL)
{
    assignToString(&proxy, proxyURL);
   
    const url_string_t* weproxy = URL_STRING_MAKE(proxy);
    if (!proxy)
	weproxy = NULL;

    we_session_set_params(session, NTHTAG_PROXY(weproxy), TAG_END());
}

bool SofiaFactoryS::canProvide(const gchar * url) const
{
  return (url && (!strncmp(url, "http://", 7) || !strncmp(url, "https://", 8)));
}

SofiaFactoryS::~SofiaFactoryS()
{
    if (session) we_session_destroy(session);
    if (engine) we_engine_destroy(engine);    
    g_source_destroy(gs);
    su_root_destroy(root);
}

HttpRequest* SofiaFactoryS::createRequest(HttpRequestListener* listener,
                                          OSB::URLCredentialStorage* credentials,
                                          const gchar * url,
                                          const gchar * cookies,
                                          HttpRequest::Type type = HttpRequest::GET)
{
    SofiaRequest* req = new SofiaRequest(url, cookies,
                                         /* HACK -- remove me after sofia is fixed. */
                                         proxy,
                                         /* END HACK */                 
                                         type, session);
    req->setListener(listener);

    GURI * uri = gnet_uri_new(url);
    if (uri->port != 80) {
      gchar host[1024];
      g_snprintf(host, 1024, "%s:%d", uri->hostname, uri->port);
      
    } else {
      req->setHost(uri->hostname);
    }

    gnet_uri_delete(uri);
    
    OSB::URLProtectionSpace space(url, "",
				  OSB::URLProtectionSpace::Default, 
				  OSB::URLProtectionSpace::NoProxy);

    const OSB::URLCredential * cred = credentials?credentials->defaultCredential(space):0;
    
    if (cred) {
        req->authenticate(cred->user(), cred->password());
    }
    
    return req;
}

extern "C" { 
static void
we_event_cb (WERequestMagic *user_data, WERequest *wr)
{
    assert(user_data);
    assert(wr);
    SofiaRequest* req = static_cast<SofiaRequest*>(user_data);
    req->event();
}
}

SofiaRequest::SofiaRequest(const gchar * aurl,
                           const gchar * acookies, 
                           /* HACK -- remove me after sofia is fixed. */
                           const gchar * proxyURL,
                           /* END HACK */
			   Type atype, 
			   WESession* asession)
    :HttpRequest(aurl, acookies, atype)
    ,wr(0)
    ,session(asession)
    ,headersSent(false)
    ,postData(0)
    /* HACK -- remove me after sofia is fixed. */
    ,proxy(g_strdup(proxyURL))
    /* END HACK */                 
    ,host(0)
{
}

void SofiaRequest::setHost(const gchar * host)
{
    assignToString(&this->host, host);
}

void SofiaRequest::setPostData(const gchar * contentType, GByteArray * data) 
{
    // sofia wants to have post fields in extra headers.
  if (!strcmp(contentType, "application/x-www-form-urlencoded") ||
      !strncmp(contentType, "multipart/form-data", 19)) {
        postData = g_byte_array_sized_new(data->len);
        memcpy( postData->data, data->data, data->len );
        postData->len = data->len;
    } else { 
#if DEBUG
	g_printerr("Content type '%s' not supported", contentType);
#endif
	assert(0);
  }
}

SofiaRequest::~SofiaRequest()
{
    if (wr) {
	we_request_release(wr);
	wr = 0;
    }

    if (host) {
      g_free(host);
      host = 0;
    }

    if (postData) {     
        g_byte_array_free(postData, true);
        postData = 0;
    }

    session = 0;    
}

void SofiaRequest::execute()
{
    if (state() != STATE_CREATED) {
#if DEBUG
	g_printerr("execute failed, already running");
#endif
	error();   
	return;
    }
    
    /*  We might start getting events after the we_request -call */
    started();
    
    const char *cag = userAgent();
    const char *cref = referrer();
    const char *chost = host;

    /* HACK -- remove me after sofia is fixed.
       Currently proxy isn't inherited from session and
       therefore we need to duplicate this information for every
       request -- psalmi */
    url_string_t* weproxy = URL_STRING_MAKE(proxy);    
    
    /* END HACK */

    wr = we_request(session, 
		    this,  
		    (type() == POST)? "POST" : "GET",
		    (const url_string_t*) m_url, 
		    TPTAG_REUSE(0),
                    HTTPTAG_HOST_STR(chost),
                    /* HACK -- remove me after sofia is fixed. */
                    NTHTAG_PROXY(weproxy),
                    /* END HACK */
  		    HTTPTAG_USER_AGENT_STR(cag),
  		    HTTPTAG_REFERER_STR(cref),
		    HTTPTAG_HEADER_STR((const char *) (postData?postData->data:0L)),
                    // crashes.  -- psalmi
		    TAG_END());
    if (!wr) { 
#if DEBUG
	g_printerr("execute failed, couldn't create request");
#endif
	error();
	return;
    }
}

void SofiaRequest::stop()
{
    if (state() != STATE_RUNNING)
	return;

    finished();    
}

void SofiaRequest::doAuth()
{
    msg_auth_t *au;
    http_t *http = NULL;
    char const *phrase = NULL;
    int status;
    char const *scheme, *realm;

    status = we_request_status(wr, &phrase, &http);

    if ((au = http->http_www_authenticate))
	m_authIsProxy = false;
    else if ((au = http->http_proxy_authenticate))
	m_authIsProxy = true;
    else
	return;

    scheme = au->au_scheme;
    if (!scheme) {
	return;
    }

    realm = msg_params_find(au->au_params, "realm=");
    if (!realm) {
	return;
    }
    assignToString(&m_authRealm, realm);
    assignToString(&m_authScheme, scheme);

    listener->authenticate(this);
}

void SofiaRequest::authenticate(const gchar * user, const gchar * pass)
{
    int retval;

    retval = we_session_authenticate(session, 
				     WETAG_AUTH_SCHEME(m_authScheme),
				     WETAG_AUTH_REALM(m_authRealm),
				     WETAG_AUTH_USER(user),
				     WETAG_AUTH_PASSWORD(pass),
				     TAG_END());
}


static gchar * extractHeaderValue(const http_header_t *h) {
  char s[1024];
  msg_header_field_e(s, sizeof(s), (msg_header_t*)h, 0);
  s[sizeof(s) - 1] = '\0';

  return g_strdup(s);
}

void SofiaRequest::event()
{
    http_t *http = NULL;
    char const *phrase = NULL;
    int status;
    bool fin = false; 

    msg_payload_t *pl;
    assert(state() == STATE_RUNNING);

    status = we_request_status(wr, &phrase, &http);    

    if (status == we_status_resolving) {
	listener->resolving(this);
    } else if (status == we_status_authenticating) {
        doAuth();
    } else {
        if (status >= we_status_headers) {
            
            if (http && http->http_status && !headersSent) {
                if (http->http_location) {
                    gchar * location =
                      extractHeaderValue(reinterpret_cast<http_header_t*>(http->http_location));
                    HttpHeaderLocation hdr(location);
                    listener->header(this, &hdr);                   
                    m_url = location;
                    // maybe it would be better to derive reset and
                    // put this check there.  -- psalmi
                    if (wr) {
                        we_request_release(wr);
                        wr = 0;
                    }
                    reset();
                    execute();
                    return;
                }
                
                if (http->http_content_type) {
                    gchar * contentType =
                      extractHeaderValue(reinterpret_cast<http_header_t*>(http->http_content_type));
                    HttpHeaderContentType hdr(contentType);
                    listener->header(this, &hdr);
                    g_free(contentType);
                }
                
                listener->headersEnd(this, 200);
                headersSent = true;
            }
            
            if (status >= we_status_http) {
                fin = !we_request_is_streaming(wr);
                
                while ((pl = we_request_get_chunk(wr))) {
                    listener->data(this, pl->pl_data, pl->pl_len);
                    we_request_release_chunk(wr, pl);
                }

                if (fin) finished();
	    }
        }
    }
}
