/*
 * Copyright (c) 2004 Nokia. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the
 * distribution.
 *
 * Neither the name of Nokia nor the names of its contributors may be
 * used to endorse or promote products derived from this software
 * without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "ResourceLoadListener.h"

#include "GLibHelpers.h"

extern "C" {
	static void _str_free(gpointer data)
	{
		gchar* str = (gchar*) data;
		g_free(str);
	}
}

class ResourceLoadResponse : public WebCoreResponse
{
    gchar* _mimeType;
    GHashTable* _headers;
    int _statusCode;

public:
    ResourceLoadResponse()
		:_mimeType(0), _statusCode(-1)	{
		_headers = g_hash_table_new_full(g_str_hash, g_str_equal, _str_free, _str_free);
	}

    ~ResourceLoadResponse() {
		g_hash_table_destroy(_headers);
		if (_mimeType) g_free(_mimeType);
	}

    void setMimeType(const gchar* mime)	{
		assignToString(&_mimeType, mime);
	}

    void setStatusCode(int code) {
		_statusCode = code;
	}

    int statusCode() {
		return _statusCode;
	}

    const gchar* mimeType()	{
		return _mimeType;
	}

    GHashTable* allHeaderFields() {
		return _headers;
	}

    GTime expiresTime()	{
		return 0;
	}

    void addHeader(const HttpHeader *header) {
		assert(header);
		g_hash_table_insert(_headers, g_strdup(header->key()), g_strdup(header->value()));
    }
};


/**
 * Idea of ResourceLoader is to hold WebCoreResourceLoader until corresponding
 * handle is cancelled.
 * after that, WebCoreResourceLoader is released
 */
ResourceLoadListener::ResourceLoadListener(BridgeImpl* b, WebCoreResourceLoader* aloader)
    :m_bridge(b)
    ,m_loader(aloader)
    ,m_response(new ResourceLoadResponse)
    ,m_request(0)
    ,m_status(0)
    ,m_resp_sent(false)
    ,m_started(false)
    ,m_auth_attempted(false)
    ,m_cancelled(false)
{
    assert(m_bridge);
    assert(m_loader);
}

ResourceLoadListener::~ResourceLoadListener()
{
#ifdef DEBUG
	g_printerr("%s :%p\n",__PRETTY_FUNCTION__, this);
#endif
}

bool ResourceLoadListener::header(const HttpRequest* r, const HttpHeader *header)
{
    switch (header->type()) {
    case HttpHeader::ContentType:
    {
		const HttpHeaderContentType *ct
			= static_cast<const HttpHeaderContentType*>(header);
		m_response->setMimeType(ct->contentType());
		break;
    }
    case HttpHeader::ContentLength:
    {
		const HttpHeaderContentLength *cl
			= static_cast<const HttpHeaderContentLength*>(header);

		m_status.setSize(cl->contentLength());
		break;
    }
    case HttpHeader::Invalid:
    default:
    {
		break;
    }
    }

    m_response->addHeader(header);
    return true;
}

bool ResourceLoadListener::data(const HttpRequest*, const char* data, int len)
{
    assert(m_resp_sent);

    m_loader->addData(data, len);
    m_status.addReceived(len);
    m_bridge->emitResourceLoadStatus(&m_status);

    return true;
}

bool ResourceLoadListener::started(const HttpRequest*)
{
    assert(!started);
    m_started = true;
    m_bridge->emitResourceLoadStarted(&m_status);
    return true;
}

bool ResourceLoadListener::finished(const HttpRequest* r)
{
    assert(m_loader);

    if (m_response->statusCode() >= 400)
		error(r);
    else
	m_loader->finish();

    m_loader = 0;

    m_bridge->emitResourceLoadFinished(&m_status);
    if (m_bridge->numPendingOrLoadingRequests() == 0) {
		m_bridge->emitFrameLoadFinished(0);
    }

    return true;
}

bool ResourceLoadListener::error(const HttpRequest*)
{
    m_status.setError();
    m_loader->reportError();
	m_loader = 0;

    return true;
}

bool ResourceLoadListener::headersEnd(const HttpRequest * request, int astatus)
{
    assert(m_loader);
    assert(!m_resp_sent);

    m_response->setStatusCode(astatus);

    m_resp_sent = true;
    m_loader->receivedResponse(m_response);
    m_bridge->emitResourceLoadHeaders(&m_status);

    return false;
}


bool ResourceLoadListener::authenticate(HttpRequest* request)
{
    OSB::URLCredentialStorage* creds = m_bridge->credentials();
    OSB::URLProtectionSpace space(request->url(),
								  request->authRealm(),
								  OSB::URLProtectionSpace::Default,
								  OSB::URLProtectionSpace::NoProxy);

    const OSB::URLCredential *cred = creds->defaultCredential(space);
    if (m_auth_attempted) {
		// authentication failed -- clear auth cache for that space
		if (cred)
			creds->removeCredential(*cred, space);
		cred = 0;
    }

    m_auth_attempted = true;

    if (!cred) {
		gchar* user = 0;
		gchar* password = 0;
		bool ret = m_bridge->authenticate(request->authRealm(), &user, &password);
		if (ret && user && password) {
			// update credentials
			OSB::URLCredential newcred(user, password, OSB::URLCredential::ForSession);
			creds->setCredential(newcred, space);
			request->authenticate(newcred.user(), newcred.password());
		}
		if (user) g_free(user);
		if (password) g_free(password);
		return ret;
    }

    // try to apply authentication
    m_request->authenticate(cred->user(), cred->password());
    return true;
}


WebCoreResourceHandle* ResourceLoadListener::handle(HttpRequest* req)
{
    m_request = req;
    return this;
}

// from WebCoreResourceHandle
void ResourceLoadListener::cancel()
{
	m_loader->cancel();
	m_loader = 0;

    delete(m_request);
    m_request = 0;
}
