/****************************************************************************
**
** Copyright (C) 1992-2004 Trolltech AS. All rights reserved.
**
** This file is part of the core module of the Qt Toolkit.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
**   information about Qt Commercial License Agreements.
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#ifndef QPROCESS_H
#define QPROCESS_H

#include <qiodevice.h>
#include <qstringlist.h>

#if defined(Q_OS_WIN32)
#include <qt_windows.h>
typedef PROCESS_INFORMATION* Q_PID;
#else
typedef Q_LONG Q_PID;
#endif

class QProcessPrivate;

class Q_CORE_EXPORT QProcess : public QIODevice
{
    Q_OBJECT
public:
    enum ProcessError {
        FailedToStart, //### file not found
        Crashed,
        Timedout,
        ReadError,
        WriteError,
        UnknownError
    };
    enum ProcessState {
        NotRunning,
        Starting,
        Running,
        Finishing
    };
    enum ProcessChannel {
        StandardOutput,
        StandardError
    };

    QProcess(QObject *parent = 0);
    virtual ~QProcess();

    void start(const QString &program, const QStringList &arguments = QStringList());

    ProcessChannel inputChannel() const;
    void setInputChannel(ProcessChannel channel);

    QString workingDirectory() const;
    void setWorkingDirectory(const QString &dir);

    void setEnvironment(const QStringList &environment);
    QStringList environment() const;

    QProcess::ProcessError processError() const;
    QProcess::ProcessState processState() const;

    Q_PID pid() const;

    bool waitForStarted(int msecs = 30000);
    bool waitForReadyRead(int msecs = 30000);
    bool waitForBytesWritten(int msecs = 30000);
    bool waitForFinished(int msecs = 30000);

    void terminate();

    QByteArray readAllStandardOutput();
    QByteArray readAllStandardError();

    int exitCode() const;

    // QIODevice
    Q_LONGLONG bytesAvailable() const;
    bool isSequential() const;
    bool canReadLine() const;
    void close();
    bool flush();

signals:
    void started();
    void finishing();
    void finished(int exitCode);
    void error(QProcess::ProcessError error);
    void stateChanged(QProcess::ProcessState state);

    void readyReadStandardOutput();
    void readyReadStandardError();

protected:
    void setProcessState(QProcess::ProcessState state);

    // QIODevice
    Q_LONGLONG readData(char *data, Q_LONGLONG maxlen);
    Q_LONGLONG writeData(const char *data, Q_LONGLONG len);

private:
    Q_DECLARE_PRIVATE(QProcess)
    Q_DISABLE_COPY(QProcess)

    Q_PRIVATE_SLOT(d, void canReadStandardOutput())
    Q_PRIVATE_SLOT(d, void canReadStandardError())
    Q_PRIVATE_SLOT(d, void canWrite())
    Q_PRIVATE_SLOT(d, void startupNotification())
    Q_PRIVATE_SLOT(d, void processDied())
    Q_PRIVATE_SLOT(d, void notified())
    friend class QProcessManager;
};

#endif
