/*--------------------------------*-C-*---------------------------------*
 * File:	xdefaults.C
 *----------------------------------------------------------------------*
 *
 * All portions of code are copyright by their respective author/s.
 * Copyright (c) 1994      Robert Nation <nation@rocket.sanders.lockheed.com>
 *				- original version
 * Copyright (c) 1997,1998 mj olesen <olesen@me.queensu.ca>
 * Copyright (c) 2003-2006 Marc Lehmann <pcg@goof.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *----------------------------------------------------------------------*/

#include "../config.h"
#include "rxvt.h"
#include "version.h"

#ifdef KEYSYM_RESOURCE
# include "keyboard.h"
#endif

/* place holders used for parsing command-line options */
#define Optflag_Reverse              0x40000000UL
#define Optflag_Boolean              0x80000000UL
#define Optflag_mask                 0x3fffffffUL

/* monolithic option/resource structure: */
/*
 * `string' options MUST have a usage argument
 * `switch' and `boolean' options have no argument
 * if there's no desc (ription), it won't appear in rxvt_usage ()
 */

/* INFO () - descriptive information only */
#define INFO(opt, arg, desc)					\
    {0, -1, NULL, (opt), (arg), (desc)}

/* STRG () - command-line option, with/without resource */
#define STRG(rsp, kw, opt, arg, desc)				\
    {0, (rsp), (kw), (opt), (arg), (desc)}

/* RSTRG () - resource/long-option */
#define RSTRG(rsp, kw, arg)					\
    {0, (rsp), (kw), NULL, (arg), NULL}

/* BOOL () - regular boolean `-/+' flag */
#define BOOL(rsp, kw, opt, flag, desc)				\
    { (Optflag_Boolean | (flag)), (rsp), (kw), (opt), NULL, (desc)}

/* SWCH () - `-' flag */
#define SWCH(opt, flag, desc)					\
    { (flag), -1, NULL, (opt), NULL, (desc)}

/* convenient macros */
#define optList_strlen(i)						\
    (optList[i].flag ? 0 : (optList[i].arg ? strlen (optList[i].arg) : 1))
#define optList_isBool(i)						\
    (optList[i].flag & Optflag_Boolean)
#define optList_isReverse(i)						\
    (optList[i].flag & Optflag_Reverse)
#define optList_size							\
    (sizeof (optList) / sizeof (optList[0]))

static const struct
  {
    const unsigned long flag;	/* Option flag */
    const int       doff;	/* data offset */
    const char     *kw;		/* keyword */
    const char     *opt;	/* option */
    const char     *arg;	/* argument */
    const char     *desc;	/* description */
  }
optList[] = {
              STRG (Rs_display_name, NULL, "d", NULL, NULL),	/* short form */
              STRG (Rs_display_name, NULL, "display", "string", "X server to contact"),
              STRG (Rs_term_name, "termName", "tn", "string", "value of the TERM environment variable"),
              STRG (Rs_geometry, NULL, "g", NULL, NULL),	/* short form */
              STRG (Rs_geometry, "geometry", "geometry", "geometry", "size (in characters) and position"),
              SWCH ("C", Opt_console, "intercept console messages"),
              SWCH ("iconic", Opt_iconic, "start iconic"),
              SWCH ("ic", Opt_iconic, NULL),	/* short form */
              BOOL (Rs_reverseVideo, "reverseVideo", "rv", Opt_reverseVideo, "reverse video"),
              BOOL (Rs_loginShell, "loginShell", "ls", Opt_loginShell, "login shell"),
              BOOL (Rs_jumpScroll, "jumpScroll", "j", Opt_jumpScroll, "jump scrolling"),
              BOOL (Rs_pastableTabs, "pastableTabs", "ptab", Opt_pastableTabs, "tab characters are pastable"),
#if HAVE_SCROLLBARS
              RSTRG (Rs_scrollstyle, "scrollstyle", "mode"),
              BOOL (Rs_scrollBar, "scrollBar", "sb", Opt_scrollBar, "scrollbar"),
              BOOL (Rs_scrollBar_right, "scrollBar_right", "sr", Opt_scrollBar_right, "scrollbar right"),
              BOOL (Rs_scrollBar_floating, "scrollBar_floating", "st", Opt_scrollBar_floating, "scrollbar without a trough"),
              RSTRG (Rs_scrollBar_align, "scrollBar_align", "mode"),
              STRG (Rs_scrollBar_thickness, "thickness", "sbt", "number", "scrollbar thickness/width in pixels"),
#endif
              BOOL (Rs_scrollTtyOutput, "scrollTtyOutput", NULL, Opt_scrollTtyOutput, NULL),
              BOOL (Rs_scrollTtyOutput, NULL, "si", Optflag_Reverse | Opt_scrollTtyOutput, "scroll-on-tty-output inhibit"),
              BOOL (Rs_scrollTtyKeypress, "scrollTtyKeypress", "sk", Opt_scrollTtyKeypress, "scroll-on-keypress"),
              BOOL (Rs_scrollWithBuffer, "scrollWithBuffer", "sw", Opt_scrollWithBuffer, "scroll-with-buffer"),
#if TRANSPARENT
              BOOL (Rs_transparent, "inheritPixmap", "ip", Opt_transparent, "inherit parent pixmap"),
              SWCH ("tr", Opt_transparent, NULL),
# if TINTING
              STRG (Rs_color + Color_tint, "tintColor", "tint", "color", "tint color"),
# endif
#endif
#if OFF_FOCUS_FADING
              STRG (Rs_fade, "fading", "fade", "%", "fade colors x% percent when rxvt-unicode is losing focus"),
              STRG (Rs_color + Color_fade, "fadeColor", "fadecolor", "color", "target color for off-focus fading"),
#endif
#if TINTING
              STRG (Rs_shade, "shading", "sh", "%", "shade background by x% when tinting."),
#endif
              BOOL (Rs_utmpInhibit, "utmpInhibit", "ut", Opt_utmpInhibit, "utmp inhibit"),
#ifndef NO_BELL
              BOOL (Rs_visualBell, "visualBell", "vb", Opt_visualBell, "visual bell"),
# if ! defined(NO_MAPALERT) && defined(MAPALERT_OPTION)
              BOOL (Rs_mapAlert, "mapAlert", NULL, Opt_mapAlert, NULL),
# endif
#endif
#ifdef META8_OPTION
              BOOL (Rs_meta8, "meta8", NULL, Opt_meta8, NULL),
#endif
#ifdef MOUSE_WHEEL
              BOOL (Rs_mouseWheelScrollPage, "mouseWheelScrollPage", NULL, Opt_mouseWheelScrollPage, NULL),
#endif
#if ENABLE_FRILLS
              BOOL (Rs_tripleclickwords, "tripleclickwords", "tcw", Opt_tripleclickwords, "triple click word selection"),
              BOOL (Rs_insecure, "insecure", "insecure", Opt_insecure, "enable possibly insecure escape sequences"),
              BOOL (Rs_cursorUnderline, "cursorUnderline", "uc", Opt_cursorUnderline, "underline cursor"),
#endif
#if CURSOR_BLINK
              BOOL (Rs_cursorBlink, "cursorBlink", "bc", Opt_cursorBlink, "blinking cursor"),
#endif
#ifdef POINTER_BLANK
              BOOL (Rs_pointerBlank, "pointerBlank", "pb", Opt_pointerBlank, "switch off pointer after delay"),
#endif
              STRG (Rs_color + Color_bg, "background", "bg", "color", "background color"),
              STRG (Rs_color + Color_fg, "foreground", "fg", "color", "foreground color"),
              RSTRG (Rs_color + minCOLOR + 0, "color0", "color"),
              RSTRG (Rs_color + minCOLOR + 1, "color1", "color"),
              RSTRG (Rs_color + minCOLOR + 2, "color2", "color"),
              RSTRG (Rs_color + minCOLOR + 3, "color3", "color"),
              RSTRG (Rs_color + minCOLOR + 4, "color4", "color"),
              RSTRG (Rs_color + minCOLOR + 5, "color5", "color"),
              RSTRG (Rs_color + minCOLOR + 6, "color6", "color"),
              RSTRG (Rs_color + minCOLOR + 7, "color7", "color"),
              RSTRG (Rs_color + minBrightCOLOR + 0, "color8", "color"),
              RSTRG (Rs_color + minBrightCOLOR + 1, "color9", "color"),
              RSTRG (Rs_color + minBrightCOLOR + 2, "color10", "color"),
              RSTRG (Rs_color + minBrightCOLOR + 3, "color11", "color"),
              RSTRG (Rs_color + minBrightCOLOR + 4, "color12", "color"),
              RSTRG (Rs_color + minBrightCOLOR + 5, "color13", "color"),
              RSTRG (Rs_color + minBrightCOLOR + 6, "color14", "color"),
              RSTRG (Rs_color + minBrightCOLOR + 7, "color15", "color"),
#ifndef NO_BOLD_UNDERLINE_REVERSE
              RSTRG (Rs_color + Color_BD, "colorBD", "color"),
              RSTRG (Rs_color + Color_IT, "colorIT", "color"),
              RSTRG (Rs_color + Color_UL, "colorUL", "color"),
              RSTRG (Rs_color + Color_RV, "colorRV", "color"),
#endif				/* ! NO_BOLD_UNDERLINE_REVERSE */
#if ENABLE_FRILLS
              RSTRG (Rs_color + Color_underline, "underlineColor", "color"),
#endif
#ifdef KEEP_SCROLLCOLOR
              RSTRG (Rs_color + Color_scroll, "scrollColor", "color"),
              RSTRG (Rs_color + Color_trough, "troughColor", "color"),
#endif				/* KEEP_SCROLLCOLOR */
#ifdef OPTION_HC
              STRG (Rs_color + Color_HC, "highlightColor", "hc", "color", "highlight color"),
#endif
#ifndef NO_CURSORCOLOR
              STRG (Rs_color + Color_cursor, "cursorColor", "cr", "color", "cursor color"),
              /* command-line option = resource name */
              RSTRG (Rs_color + Color_cursor2, "cursorColor2", "color"),
#endif				/* NO_CURSORCOLOR */
              STRG (Rs_color + Color_pointer_fg, "pointerColor", "pr", "color", "pointer color"),
              STRG (Rs_color + Color_pointer_bg, "pointerColor2", "pr2", "color", "pointer bg color"),
              STRG (Rs_color + Color_border, "borderColor", "bd", "color", "border color"),
#ifdef XPM_BACKGROUND
              RSTRG (Rs_path, "path", "search path"),
              STRG (Rs_backgroundPixmap, "backgroundPixmap", "pixmap", "file[;geom]", "background pixmap"),
#endif
              /* fonts: command-line option = resource name */
              STRG (Rs_font, "font", "fn", "fontname", "normal text font"),
#if ENABLE_STYLES
              STRG (Rs_boldFont, "boldFont", "fb", "fontname", "bold font"),
              STRG (Rs_italicFont, "italicFont", "fi", "fontname", "italic font"),
              STRG (Rs_boldItalicFont, "boldItalicFont", "fbi", "fontname", "bold italic font"),
              BOOL (Rs_intensityStyles, "intensityStyles", "is", Opt_intensityStyles, "font styles imply intensity changes"),
#endif
#ifdef USE_XIM
              STRG (Rs_inputMethod, "inputMethod", "im", "name", "name of input method"),
              STRG (Rs_preeditType, "preeditType", "pt", "style", "input style: style = OverTheSpot|OffTheSpot|Root"),
              STRG (Rs_imLocale, "imLocale", "imlocale", "string", "locale to use for input method"),
              STRG (Rs_imFont, "imFont", "imfont", "fontname", "fontset for styles OverTheSpot and OffTheSpot"),
#endif				/* USE_XIM */
              STRG (Rs_name, NULL, "name", "string", "client instance, icon, and title strings"),
              STRG (Rs_title, "title", "title", "string", "title name for window"),
              STRG (Rs_title, NULL, "T", NULL, NULL),	/* short form */
              STRG (Rs_iconName, "iconName", "n", "string", "icon name for window"),
              STRG (Rs_saveLines, "saveLines", "sl", "number", "number of scrolled lines to save"),
#if ENABLE_XEMBED
              STRG (Rs_embed, NULL, "embed", "windowid", "window id to embed terminal in"),
#endif
#if XFT
              STRG (Rs_depth, "depth", "depth", "number", "depth of visual to request"),
#endif
#if ENABLE_FRILLS
              RSTRG (Rs_transient_for, "transient-for", "windowid"),
              BOOL (Rs_override_redirect, "override-redirect", "override-redirect", Opt_override_redirect, "set override-redirect on the terminal window"),
              STRG (Rs_pty_fd, NULL, "pty-fd", "fileno", "file descriptor of pty to use"),
              BOOL (Rs_hold, "hold", "hold", Opt_hold, "retain window after shell exit"),
              STRG (Rs_ext_bwidth, "externalBorder", "w", "number", "external border in pixels"),
              STRG (Rs_ext_bwidth, NULL, "bw", NULL, NULL),
              STRG (Rs_ext_bwidth, NULL, "borderwidth", NULL, NULL),
              STRG (Rs_int_bwidth, "internalBorder", "b", "number", "internal border in pixels"),
              BOOL (Rs_borderLess, "borderLess", "bl", Opt_borderLess, "borderless window"),
              BOOL (Rs_skipBuiltinGlyphs, "skipBuiltinGlyphs", "sbg", Opt_skipBuiltinGlyphs, "do not use internal glyphs"),
              STRG (Rs_lineSpace, "lineSpace", "lsp", "number", "number of extra pixels between rows"),
#endif
#ifdef POINTER_BLANK
              RSTRG (Rs_pointerBlankDelay, "pointerBlankDelay", "number"),
#endif
#ifndef NO_BACKSPACE_KEY
              RSTRG (Rs_backspace_key, "backspacekey", "string"),
#endif
#ifndef NO_DELETE_KEY
              RSTRG (Rs_delete_key, "deletekey", "string"),
#endif
              RSTRG (Rs_selectstyle, "selectstyle", "mode"),
#ifdef PRINTPIPE
              RSTRG (Rs_print_pipe, "print-pipe", "string"),
#endif
              STRG (Rs_modifier, "modifier", "mod", "modifier", "meta modifier = alt|meta|hyper|super|mod1|...|mod5"),
#ifdef CUTCHAR_RESOURCE
              RSTRG (Rs_cutchars, "cutchars", "string"),
#endif				/* CUTCHAR_RESOURCE */
              RSTRG (Rs_answerbackstring, "answerbackString", "string"),
#ifndef NO_SECONDARY_SCREEN
              BOOL (Rs_secondaryScreen, "secondaryScreen", "ssc", Opt_secondaryScreen, "enable secondary screen"),
              BOOL (Rs_secondaryScroll, "secondaryScroll", "ssr", Opt_secondaryScroll, "enable secondary screen scroll"),
#endif
#if ENABLE_PERL
              RSTRG (Rs_perl_lib, "perl-lib", "string"), //, "colon-separated directories with extension scripts"),TODO
              RSTRG (Rs_perl_eval, "perl-eval", "perl-eval"), // "string", "code to be evaluated after all extensions have been loaded"),TODO
              RSTRG (Rs_perl_ext_1, "perl-ext-common", "string"), //, "colon-separated list of perl extensions to enable"),TODO
              STRG (Rs_perl_ext_2, "perl-ext", "pe", "string", "colon-separated list of perl extensions to enable for this instance"),
#endif
#ifndef NO_RESOURCES
              INFO ("xrm", "string", "X resource"),
#endif
              INFO ("e", "command arg ...", "command to execute")
            };

#undef INFO
#undef STRG
#undef RSTRG
#undef SWCH
#undef BOOL

static const char releasestring[] = "rxvt-unicode (" RXVTNAME ") v" VERSION " - released: " DATE "\n";
static const char optionsstring[] = "options: "
#if ENABLE_PERL
                                    "perl,"
#endif
#if XFT
                                    "xft,"
#endif
#if ENABLE_STYLES
                                    "styles,"
#endif
#if ENABLE_COMBINING
                                    "combining,"
#endif
#if TEXT_BLINK
                                    "blink,"
#endif
#if ISO_14755
                                    "iso14755,"
#endif
#if UNICODE_3
                                    "unicode3,"
#endif
                                    "encodings=eu+vn"
#if ENCODING_JP
                                    "+jp"
#endif
#if ENCODING_JP_EXT
                                    "+jp-ext"
#endif
#if ENCODING_KR
                                    "+kr"
#endif
#if ENCODING_ZH
                                    "+zh"
#endif
#if ENCODING_ZH_EXT
                                    "+zh-ext"
#endif
                                    ","
#if OFF_FOCUS_FADING
                                    "fade,"
#endif
#if defined(XPM_BACKGROUND)
                                    "XPM,"
#endif
#if defined(TRANSPARENT)
                                    "transparent,"
#endif
#if TINTING
                                    "tint,"
#endif
#if defined(USE_XIM)
                                    "XIM,"
#endif
#if defined(NO_BACKSPACE_KEY)
                                    "no_backspace,"
#endif
#if defined(NO_DELETE_KEY)
                                    "no_delete,"
#endif
#if EIGHT_BIT_CONTROLS
                                    "8bitctrls,"
#endif
#if defined(ENABLE_FRILLS)
                                    "frills,"
#endif
#if defined(SELECTION_SCROLLING)
                                    "selectionscrolling,"
#endif
#if MOUSE_WHEEL
                                    "wheel,"
#endif
#if MOUSE_SLIP_WHEELING
                                    "slipwheel,"
#endif
#if defined(SMART_RESIZE)
                                    "smart-resize,"
#endif
#if defined(CURSOR_BLINK)
                                    "cursorBlink,"
#endif
#if defined(POINTER_BLANK)
                                    "pointerBlank,"
#endif
#if defined(NO_RESOURCES)
                                    "NoResources,"
#endif
                                    "scrollbars="
#if !defined(HAVE_SCROLLBARS)
                                    "NONE"
#else
# if defined(PLAIN_SCROLLBAR)
                                    "plain"
#  if defined(RXVT_SCROLLBAR) || defined(NEXT_SCROLLBAR) || defined(XTERM_SCROLLBAR)
                                    "+"
#  endif
# endif
# if defined(RXVT_SCROLLBAR)
                                    "rxvt"
#  if defined(NEXT_SCROLLBAR) || defined(XTERM_SCROLLBAR)
                                    "+"
#  endif
# endif
# if defined(NEXT_SCROLLBAR)
                                    "NeXT"
#  if defined(XTERM_SCROLLBAR)
                                    "+"
#  endif
# endif
# if defined(XTERM_SCROLLBAR)
                                    "xterm"
# endif
#endif
                                    "\nUsage: ";		/* Usage */

#define INDENT 18

const char rxvt_term::resval_undef [] = "<undef>";
const char rxvt_term::resval_on []    = "on";
const char rxvt_term::resval_off []   = "off";

/*{{{ usage: */
/*----------------------------------------------------------------------*/
static void
rxvt_usage (int type)
{
  unsigned int i, col;

  rxvt_log ("%s%s%s", releasestring, optionsstring, RESNAME);

  switch (type)
    {
      case 0:			/* brief listing */
        rxvt_log (" [-help] [--help]\n");

        for (col = 1, i = 0; i < optList_size; i++)
          if (optList[i].desc != NULL)
            {
              int len = 0;

              if (!optList_isBool (i))
                {
                  len = optList_strlen (i);
                  if (len > 0)
                    len++;	/* account for space */
                }
#ifdef DEBUG_STRICT
              assert (optList[i].opt != NULL);
#endif
              len += 4 + strlen (optList[i].opt) + (optList_isBool (i) ? 2: 0);
              col += len;
              if (col > 79)
                {	/* assume regular width */
                  rxvt_log ("\n");
                  col = 1 + len;
                }

              rxvt_log (" [-%s%s", (optList_isBool (i) ? "/+" : ""), optList[i].opt);
              if (optList_strlen (i))
                rxvt_log (" %s]", optList[i].arg);
              else
                rxvt_log ("]");
            }
        break;

      case 1:			/* full command-line listing */
        rxvt_log (" [options] [-e command args]\n\nwhere options include:\n");

        for (i = 0; i < optList_size; i++)
          if (optList[i].desc != NULL)
            {
#ifdef DEBUG_STRICT
              assert (optList[i].opt != NULL);
#endif
              rxvt_log ("  %s%s %-*s%s%s\n",
                         (optList_isBool (i) ? "-/+" : "-"), optList[i].opt,
                         (INDENT - strlen (optList[i].opt)
                          + (optList_isBool (i) ? 0 : 2)),
                         (optList[i].arg ? optList[i].arg : ""),
                         (optList_isBool (i) ? "turn on/off " : ""),
                         optList[i].desc);
            }
        rxvt_log ("\n  --help to list long-options");
        break;

      case 2:			/* full resource listing */
        rxvt_log (" [options] [-e command args]\n\n"
                   "where resources (long-options) include:\n");

        for (i = 0; i < optList_size; i++)
          if (optList[i].kw != NULL)
            rxvt_log ("  %s: %*s%s\n",
                    optList[i].kw,
                    (INDENT - strlen (optList[i].kw)), "", /* XXX */
                    (optList_isBool (i) ? "boolean" : optList[i].arg));
#ifdef KEYSYM_RESOURCE
        rxvt_log ("  " "keysym.sym" ": %*s%s\n",
                (INDENT - sizeof ("keysym.sym") + 1), "", /* XXX */
                "keysym");
#endif
        rxvt_log ("\n  -help to list options");
        break;
    }

  rxvt_log ("\n\n");
  rxvt_exit_failure ();
  /* NOTREACHED */
}

/*}}} */

/*{{{ get command-line options before getting resources */
void
rxvt_term::get_options (int argc, const char *const *argv)
{
  int i, bad_option = 0;

  for (i = 1; i < argc; i++)
    {
      unsigned int entry, longopt = 0;
      const char *flag, *opt;

      opt = argv[i];

      if (*opt == '-')
        {
          flag = resval_on;

          if (*++opt == '-')
            longopt = *opt++;	/* long option */
        }
      else if (*opt == '+')
        {
          flag = resval_off;

          if (*++opt == '+')
            longopt = *opt++;	/* long option */
        }
      else
        {
          bad_option = 1;
          rxvt_warn ("\"%s\": malformed option.\n", opt);
          continue;
        }

      if (!strcmp (opt, "help"))
        rxvt_usage (longopt ? 2 : 1);

      if (!strcmp (opt, "h"))
        rxvt_usage (0);

      /* feature: always try to match long-options */
      for (entry = 0; entry < optList_size; entry++)
        if ((optList[entry].kw && !strcmp (opt, optList[entry].kw))
            || (!longopt
                && optList[entry].opt && !strcmp (opt, optList[entry].opt)))
          break;

      if (entry < optList_size)
        {
          if (optList_isReverse (entry))
            flag = flag == resval_on ? resval_off : resval_on;

          if (optList_strlen (entry))
            {
              /*
               * special cases are handled in main.c:main () to allow
               * X resources to set these values before we settle for
               * default values
               */

              if (optList[entry].doff != -1)
                {
                  if (flag == resval_on && !argv [i+1])
                    rxvt_fatal ("option '%s' needs an argument, aborting.\n", argv [i]);

                  rs[optList[entry].doff] = flag == resval_on ? argv[++i] : resval_undef;
                }
            }
          else
            {		/* boolean value */
              set_option (optList[entry].flag & Optflag_mask, flag == resval_on);

              if (optList[entry].doff != -1)
                rs[optList[entry].doff] = flag;
            }
        }
      else
#ifdef KEYSYM_RESOURCE
        /* if (!strncmp (opt, "keysym.", sizeof ("keysym.") - 1)) */
        if (rxvt_Str_match (opt, "keysym."))
          {
            const char *str = argv[++i];

            if (str != NULL)
              parse_keysym (opt + sizeof ("keysym.") - 1, str);
          }
        else
#endif
          {
            bad_option = 1;
            rxvt_warn ("\"%s\": unknown or malformed option.\n", opt);
          }
    }

  if (bad_option)
    rxvt_usage (0);
}

/*}}} */

#ifndef NO_RESOURCES
/*----------------------------------------------------------------------*/

# ifdef KEYSYM_RESOURCE
/*
 * Define key from XrmEnumerateDatabase.
 *   quarks will be something like
 *      "rxvt" "keysym" "0xFF01"
 *   value will be a string
 */
/* ARGSUSED */
int
rxvt_define_key (XrmDatabase *database UNUSED,
                 XrmBindingList bindings UNUSED,
                 XrmQuarkList quarks,
                 XrmRepresentation *type UNUSED,
                 XrmValue *value,
                 XPointer closure UNUSED)
{
  int last;

  for (last = 0; quarks[last] != NULLQUARK; last++)	/* look for last quark in list */
    ;

  last--;
  GET_R->parse_keysym (XrmQuarkToString (quarks[last]), (char *)value->addr);//D//TODO
  return False;
}

/*
 * look for something like this (XK_Delete)
 * rxvt*keysym.0xFFFF: "\177"
 *
 * arg will be
 *      NULL for ~/.Xdefaults and
 *      non-NULL for command-line options (need to allocate)
 */
#define NEWARGLIM	500	/* `reasonable' size */

struct keysym_vocabulary_t
{
  const char    *name;
  unsigned short len;
  unsigned short value;
};
keysym_vocabulary_t keysym_vocabulary[] =
{
  { "ISOLevel3", 9, Level3Mask    },
  { "AppKeypad", 9, AppKeypadMask },
  { "Control",   7, ControlMask   },
  { "NumLock",   7, NumLockMask   },
  { "Shift",     5, ShiftMask     },
  { "Meta",      4, MetaMask      },
  { "Lock",      4, LockMask      },
  { "Mod1",      4, Mod1Mask      },
  { "Mod2",      4, Mod2Mask      },
  { "Mod3",      4, Mod3Mask      },
  { "Mod4",      4, Mod4Mask      },
  { "Mod5",      4, Mod5Mask      },
  { "I",         1, Level3Mask    },
  { "K",         1, AppKeypadMask },
  { "C",         1, ControlMask   },
  { "N",         1, NumLockMask   },
  { "S",         1, ShiftMask     },
  { "M",         1, MetaMask      },
  { "A",         1, MetaMask      },
  { "L",         1, LockMask      },
  { "1",         1, Mod1Mask      },
  { "2",         1, Mod2Mask      },
  { "3",         1, Mod3Mask      },
  { "4",         1, Mod4Mask      },
  { "5",         1, Mod5Mask      },
};

int
rxvt_term::parse_keysym (const char *str, const char *arg)
{
  int n, sym;
  unsigned int state = 0;
  const char *pmodend = NULL;
  char *newarg = NULL;
  char newargstr[NEWARGLIM];

  if (arg == NULL)
    {
      if ((n = rxvt_Str_match (str, "keysym.")) == 0)
        return 0;

      str += n;		/* skip `keysym.' */
      if (!(pmodend = strchr (str, ':')))
        return -1;
    }
  else
    pmodend = str + strlen(str);

  for (--pmodend; str < pmodend; --pmodend)
    if (*pmodend == '-')
      break;

  while (str < pmodend)
    {
      unsigned int i;

      for (i=0; i < sizeof (keysym_vocabulary) / sizeof (keysym_vocabulary_t); ++i)
        {
          if (strncmp (str, keysym_vocabulary [i].name, keysym_vocabulary [i].len) == 0)
            {
              state |= keysym_vocabulary[i].value;
              str += keysym_vocabulary[i].len;
              break;
            }
        }

      if (i >= sizeof (keysym_vocabulary) / sizeof (keysym_vocabulary_t))
        return -1;

      if (*str == '-')
        ++str;
    }

  /* some scanf () have trouble with a 0x prefix */
  if (str[0] == '0' && toupper (str[1]) == 'X')
    {
      str += 2;

      if (arg)
        {
          if (sscanf (str, (strchr (str, ':') ? "%x:" : "%x"), &sym) != 1)
            return -1;
        }
      else
        {
          if (sscanf (str, "%x:", &sym) != 1)
            return -1;

          /* cue to ':', it's there since sscanf () worked */
          strncpy (newargstr, strchr (str, ':') + 1, NEWARGLIM - 1);
          newargstr[NEWARGLIM - 1] = '\0';
          newarg = newargstr;
        }
    }
  else
    {
      /*
       * convert keysym name to keysym number
       */
      strncpy (newargstr, str, NEWARGLIM - 1);
      newargstr[NEWARGLIM - 1] = '\0';

      if (arg == NULL)
        {
          if ((newarg = strchr (newargstr, ':')) == NULL)
            return -1;

          *newarg++ = '\0';	/* terminate keysym name */
        }

      if ((sym = XStringToKeysym (newargstr)) == None)
        return -1;
    }

  if (newarg == NULL)
    {
      strncpy (newargstr, arg, NEWARGLIM - 1);
      newargstr[NEWARGLIM - 1] = '\0';
      newarg = newargstr;
    }

  rxvt_Str_trim (newarg);
  if (*newarg == '\0' || (n = rxvt_Str_escaped (newarg)) == 0)
    return -1;

  keyboard->register_user_translation (sym, state, newarg);
  return 1;
}

# endif				/* KEYSYM_RESOURCE */
#endif				/* NO_RESOURCES */

static char *
get_res (XrmDatabase database, const char *program, const char *option)
{
  char resource[512];
  char *type;
  XrmValue result;

  snprintf (resource, sizeof (resource), "%s.%s", program, option);
  XrmGetResource (database, resource, resource, &type, &result);

  return result.addr;
}

const char *
rxvt_term::x_resource (const char *name)
{
  XrmDatabase database = XrmGetDatabase (dpy);

  const char *p = get_res (database, rs[Rs_name], name);
  const char *p0 = get_res (database, "!INVALIDPROGRAMMENAMEDONTMATCH!", name);

  if (p == NULL || (p0 && strcmp (p, p0) == 0))
    {
      p = get_res (database, RESCLASS, name);
#ifdef RESFALLBACK
      if (p == NULL || (p0 && strcmp (p, p0) == 0))
        p = get_res (database, RESFALLBACK, name);
#endif
    }

  if (p == NULL && p0)
    p = p0;

  return p;
}

void
rxvt_term::extract_resources ()
{
#ifndef NO_RESOURCES
  /*
   * Query resources for options that affect us
   */
  for (int entry = 0; entry < optList_size; entry++)
    {
      int s;
      const char *kw = optList[entry].kw;

      if (kw == NULL || rs[optList[entry].doff] != NULL)
        continue; // previously set

      const char *p = x_resource (kw);

      if (p)
        {
          p = strdup (p);
          allocated.push_back ((void *)p);
          rs[optList[entry].doff] = p;

          if (optList_isBool (entry))
            {
              s = strcasecmp (p, "TRUE") == 0
                  || strcasecmp (p, "YES") == 0
                  || strcasecmp (p, "ON") == 0
                  || strcasecmp (p, "1") == 0;

              if (optList_isReverse (entry))
                s = !s;

              set_option (optList[entry].flag & Optflag_mask, s);
            }
        }
    }

  /*
   * [R5 or later]: enumerate the resource database
   */
#  ifdef KEYSYM_RESOURCE
  XrmDatabase database = XrmGetDatabase (dpy);
  XrmName name_prefix[3];
  XrmClass class_prefix[3];

  name_prefix[0] = XrmStringToName (rs[Rs_name]);
  name_prefix[1] = XrmStringToName ("keysym");
  name_prefix[2] = NULLQUARK;
  class_prefix[0] = XrmStringToName (RESCLASS);
  class_prefix[1] = XrmStringToName ("Keysym");
  class_prefix[2] = NULLQUARK;
  /* XXX: Need to check sizeof (rxvt_t) == sizeof (XPointer) */
  XrmEnumerateDatabase (database, name_prefix, class_prefix,
                        XrmEnumOneLevel, rxvt_define_key, NULL);
#   ifdef RESFALLBACK
  name_prefix[0] = XrmStringToName (RESFALLBACK);
  name_prefix[1] = XrmStringToName ("keysym");
  class_prefix[0] = XrmStringToName (RESFALLBACK);
  class_prefix[1] = XrmStringToName ("Keysym");
  /* XXX: Need to check sizeof (rxvt_t) == sizeof (XPointer) */
  XrmEnumerateDatabase (database, name_prefix, class_prefix,
                        XrmEnumOneLevel, rxvt_define_key, NULL);
#   endif
#  endif

#endif				/* NO_RESOURCES */
}

/*----------------------- end-of-file (C source) -----------------------*/
